"""
Toposens python bindings generator.
"""

import os
import subprocess

import cffi

import argparse

PATH = "."
OUTPUT_NAME = "toposens-sensor-library-python-bindings.so"

PYTHON_MODULE_NAME = "toposens_cffi"
INCLUDE_PATH = ["include/toposens/","../../core-libraries/toposens/"]
LIBRARY = "toposens-sensor-library"
LIBRARY_PATH = ["/usr/lib/"]

PYTHON_MODULE_NAME_INTERNAL = "toposens_cffi_internal"
INCLUDE_PATH_INTERNAL = ["include/toposens/","../../core-libraries/toposens/"]
LIBRARY_INTERNAL = "toposens-sensor-library"
LIBRARY_PATH_INTERNAL = ["/usr/lib/"]

# pycparser requires a preprocessed file as input. GCC is called here with the -E flag
# to create it. The -DTOPO_PYCPARSER flag is set to filter out includes in the source
# code that can't be parsed by pycparser.
def preprocess(source, header_path):
    """
    Function returning a preprocessed header file.
    """

    # Call gcc with our custom define, stop after preprocessing (-E)
    # and don't include linemarkers (-P)
    command = ["gcc","-DTOPO_PYCPARSER","-E","-P","-D__extension__="]

    # add all paths in INCLUDE_PATH to the gcc command line as header directories(-I flag)
    for header in header_path:
        command.append("-I" + header)

    # Tell gcc to parse stdin
    command.append("-")

    return subprocess.run(command, input=source, stdout=subprocess.PIPE, universal_newlines=True, check=True).stdout

def create_bindings():

    ffi = cffi.FFI()

    HEADER_FILE_NAME = os.path.join(PATH, "include/toposens/", "sensor_lib.h")
    with open(HEADER_FILE_NAME, encoding="UTF-8") as h_file:
        ffi.cdef(preprocess(h_file.read(), INCLUDE_PATH))

    print("Preprocess done")
    ffi.set_source(
        PYTHON_MODULE_NAME,
        """
        #include <stdbool.h>
        #include "sensor_lib.h"
        #include "custom_structs.h"
        #include "message_flags.h"
        #include "toposens/errors.h"
        """,
        include_dirs=INCLUDE_PATH,
        libraries=[LIBRARY],
        library_dirs=LIBRARY_PATH,
        extra_link_args=["-Wl,-rpath,."],
    )

    ffi.compile(target=OUTPUT_NAME)

def create_bindings_internal():

    ffi = cffi.FFI()

    HEADER_FILE_NAME = os.path.join(PATH, "include/toposens/", "sensor_lib.h")
    with open(HEADER_FILE_NAME, encoding="UTF-8") as h_file:
        ffi.cdef(preprocess(h_file.read()))

    # tasks.py
    ffi.set_source(
        PYTHON_MODULE_NAME_INTERNAL,
        """
        #include <stdbool.h>
        #include "sensor_lib.h"
        #include "sensor_lib_internal.h"
        #include "custom_structs.h"
        #include "message_flags.h"
        #include "toposens/errors.h"
        """,
        include_dirs=INCLUDE_PATH_INTERNAL,
        libraries=[LIBRARY_INTERNAL],
        library_dirs=LIBRARY_PATH_INTERNAL,
        extra_link_args=["-Wl,-rpath,."],
    )

    ffi.compile(target=OUTPUT_NAME)

if __name__ == '__main__':

    parser = argparse.ArgumentParser()
    parser.add_argument("-i", "--include_path", help="Header include path to use", action='append')
    parser.add_argument("-l", "--library_path", help="Library path to use",        action='append')
    parser.add_argument("-n", "--name",         help="Name of the output object",  action='store')
    parser.add_argument("-p", "--path",         help="Library path to use",        action='store')

    args = parser.parse_args()

    if args.include_path:
        INCLUDE_PATH = args.include_path

    if args.library_path:
        LIBRARY_PATH = args.library_path

    if args.path:
       PATH = args.path

    if args.name:
       OUTPUT_NAME = args.name

    INCLUDE_PATH = [os.path.join(PATH, p) for p in INCLUDE_PATH]

    create_bindings()
