/** **************************************************************************************
 * @file       uartinterface.c
 * @ingroup    Uart-Interface
 * @copyright  Copyright (c) Toposens GmbH 2021. All rights reserved.
 * @brief      This file contains the linux-uart interface functions
 ******************************************************************************************/

#define _GNU_SOURCE
/*---- "module header" ----------------------------------------------------------------*/
#include "toposens/linux/uartinterface.h"
/*---- <system includes> --------------------------------------------------------------*/
#include <fcntl.h>  //Used for UART
#include <pthread.h>
#include <semaphore.h>
#include <signal.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/poll.h>
#include <termios.h>  //Used for UART
#include <unistd.h>

/*---- "library includes" -------------------------------------------------------------*/
#include "toposens/uart_wrapper.h"
/*---- "project includes" -------------------------------------------------------------*/

/*---- local macros and constants -----------------------------------------------------*/
/** **************************************************************************************
  @brief      Defines how many seconds the ReadUart function will wait for an incoming can-frame
******************************************************************************************/
#define UART_TIMEOUT 1
/** **************************************************************************************
  @brief      Defines how many us the ReadUart function will wait for an incoming can-frame
******************************************************************************************/
#define UART_TIMEOUT_US       0
#define UART_MAX_PAYLOAD_SIZE 10
#define UART_LOOP_WAIT_TIME   500
/*---- local types --------------------------------------------------------------------*/
typedef struct UartInterface_t
{
    uint8_t InterfaceId_u8;
    char InterfaceName[DEVICE_NAME_LEN];
    uint32_t InterfaceBitrate_u32;
} UartInterface_t;
/*---- local functions prototypes -----------------------------------------------------*/

/** ******************************************************************************
  @ingroup      Uart-Interface
  @brief        Setup of uart-device with given paramters. Currently via system-call.
  @param[in]    *InterfaceName_cp interface-name that shall be configured
  @param[in]    InterfaceBitrate_u32 the interface should be working with
  @return       int 0 if setup was successful
                int 1 in case of an error
*******************************************************************************/
static int
SetupInterface(const char* InterfaceName_cp, uint32_t InterfaceBitrate_u32);

/** ******************************************************************************
  @ingroup      Uart-Interface
  @brief        Creates the Receiver Thread
  @param[in]    InterfaceId_u8
  @param[in]    InterfaceBitrate_u32
  @param[in]    *InterfaceName_cp
*******************************************************************************/
static void
StartReceiverThread(uint8_t InterfaceId_u8, uint32_t InterfaceBitrate_u32, const char* InterfaceName_cp);

/** ******************************************************************************
  @ingroup      Uart-Interface
  @brief        Reads the while-loop condition in a safe way
  @return       int 0 if receiver should stop
                int 1 if receiver should be running
*******************************************************************************/
int
ReceiverThreadShouldRunUART_b();

/** ******************************************************************************
  @ingroup      Uart-Interface
  @brief        Tries to read CAN frames as long as it is desired. Reacts to abort commands within
the set CAN timeout
  @param[in]    *arg
*******************************************************************************/
void*
Receiver(void* arg);

/** ******************************************************************************
  @ingroup      Uart-Interface
  @brief        Should be called periodically to fetch incoming messages from the socket
  @param[in]    *Msg_pu8
  @param[in]    InterfaceFS_i
  @return       can_frame Read-CAN-Frame. If no message was received after timeout, it will retrun
an empty frame.
*******************************************************************************/
static ssize_t
ReadMsg(uint8_t* Msg_pu8, int InterfaceFS_i);

/** ******************************************************************************
  @ingroup      Uart-Interface
  @param[in]    *Payload_pu8
  @param[in]    Length_u8
*******************************************************************************/
static void
PrintUARTPayload(uint8_t* Payload_pu8, uint8_t Length_u8);

/** ******************************************************************************
  @ingroup      Uart-Interface
  @brief        Only for debugging - print the content of an UART-Payload to stdout
  @param[in]    Payload_pu8 pointer to payload that should be printed
  @param[in]    Length_u8 length of payload
*******************************************************************************/
static void
PrintPayload(uint8_t* Payload_pu8, uint8_t Length_u8);
/*---- local inline functions ---------------------------------------------------------*/

/*---- local variables (static) -------------------------------------------------------*/
// NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
static pthread_t receiver_thread_id;
// NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
static bool ReceiverThreadShouldRun = false;
// NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
static sem_t thread_sem;
// NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
static pthread_mutex_t mutex_uart_receiver = PTHREAD_MUTEX_INITIALIZER;
// NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
static void (*ReadCallback)(uint8_t* UARTMsg_pu8, uint16_t DataSize_u16, uint8_t InterfaceId_u8) = NULL;
static int uart_fs;
static bool ConnectedToUART_b                    = false;
static int UART_fs[MAX_NUMBER_OF_SENSORS_ON_BUS] = {-1};

static int timeout_fd[2];
/*---- public variables ---------------------------------------------------------------*/

/*---- functions ----------------------------------------------------------------------*/

void
RegisterUARTReadCallback(void (*Callback)(uint8_t* UARTMsg_pu8, uint16_t UARTMsgSize_u16, uint8_t InterfaceId_u8))
{
    MPRINTF("RegisterReadCallback\n");
    ReadCallback = Callback;
}

static ssize_t
ReadMsg(uint8_t* Msg_pu8, int InterfaceFS_i)
{
    ssize_t Length = 0;

    struct pollfd pfd[2];

    pfd[0].fd     = InterfaceFS_i;
    pfd[0].events = POLLIN;

    pfd[1].fd     = timeout_fd[0];
    pfd[1].events = POLLIN;

    // Wait for an event on the interface or on the main thread to unblock us
    poll(pfd, 2, -1);

    if (pfd[1].revents & POLLIN)
    {
        return 0;
    }

    Length = read(InterfaceFS_i, Msg_pu8, UART_MAX_BUF_LEN);
    return Length;
}

int
WriteUARTPayload(uint8_t* Payload_pu8, uint8_t Length_u8, uint8_t InterfaceId_u8)
{
    PrintUARTPayload(Payload_pu8, Length_u8);
    int16_t BytesWritten_i16 = 0;
    BytesWritten_i16         = write(UART_fs[InterfaceId_u8], Payload_pu8, Length_u8);
    if (BytesWritten_i16 < 0)
    {
        perror("Write");
        return 1;
    }
    MPRINTF("Wrote %d Bytes\n", BytesWritten_i16);
    return 0;
}

int
SetupUARTPort(const char* InterfaceName_cp, uint32_t InterfaceBitrate_u32, uint8_t InterfaceId_u8)
{
    // We create a pipe to be used between the main thread and the receiving one
    // which blocks on a read. By making the read also block on the pipe (using
    // poll) we can unblock it when we want to shut it down.
    pipe2(timeout_fd, O_CLOEXEC);

    sem_init(&thread_sem, 0, 0);
    pthread_mutex_lock(&mutex_uart_receiver);
    ReceiverThreadShouldRun = true;
    pthread_mutex_unlock(&mutex_uart_receiver);
    StartReceiverThread(InterfaceId_u8, InterfaceBitrate_u32, InterfaceName_cp);
    sem_wait(&thread_sem);

    ConnectedToUART_b = true;
    return 0;
}

int
DeinitUARTPort()
{
    pthread_mutex_lock(&mutex_uart_receiver);
    ReceiverThreadShouldRun = false;
    pthread_mutex_unlock(&mutex_uart_receiver);

    // Wake up the blocked receiving thread
    write(timeout_fd[1], "a", 1);
    close(timeout_fd[1]);

    (void)pthread_join(receiver_thread_id, NULL);
    return 0;
}

static void
PrintPayload(uint8_t* Payload_pu8, uint8_t Length_u8)
{
    for (int i = 0; i < Length_u8; i++)
    {
        MPRINTF("%02X ", Payload_pu8[i]);
    }
    MPRINTF("\r\n");
}

static int
SetupInterface(const char* InterfaceName_cp, uint32_t InterfaceBitrate_u32)
{
    MPRINTF("SetupInterface\n");

    uart_fs = -1;
    uart_fs = open(InterfaceName_cp, O_RDWR | O_NOCTTY | O_CLOEXEC);

    if (uart_fs == -1)
    {
        MPRINTF("Error - Unable to open UART.  Ensure it is not in use by another application\n");
    }
    else
    {
        struct termios options;
        memset(&options, 0x00, sizeof(options));
        tcgetattr(uart_fs, &options);
        options.c_cflag = InterfaceBitrate_u32 | CS8 | CLOCAL | CREAD;  //<Set baud rate
        options.c_iflag = IGNPAR;
        options.c_oflag = 0;
        options.c_lflag = 0;
        tcflush(uart_fs, TCIFLUSH);
        tcsetattr(uart_fs, TCSANOW, &options);
    }
    return uart_fs;
}

static void
StartReceiverThread(uint8_t InterfaceId_u8, uint32_t InterfaceBitrate_u32, const char* InterfaceName_cp)
{
    MPRINTF("StartReceiverThread\n");
    UartInterface_t* NewInterface_pt = malloc(sizeof(*NewInterface_pt));
    memcpy(NewInterface_pt->InterfaceName, InterfaceName_cp, DEVICE_NAME_LEN);
    NewInterface_pt->InterfaceId_u8       = InterfaceId_u8;
    NewInterface_pt->InterfaceBitrate_u32 = InterfaceBitrate_u32;
    int err                               = 0;
    err                                   = pthread_create(&(receiver_thread_id), NULL, &Receiver, NewInterface_pt);
    if (err != 0)
    {
        MPRINTF("\ncan't create thread :[%s]", strerror(err));
    }
    else
    {
        MPRINTF("UART-Receiver Thread created successfully\n");
    }
}

int
ReceiverThreadShouldRunUART_b()
{
    int value = 0;

    pthread_mutex_lock(&mutex_uart_receiver);
    value = ReceiverThreadShouldRun;
    pthread_mutex_unlock(&mutex_uart_receiver);

    return value;
}

void*
Receiver(void* arg)
{
    UartInterface_t* NewInterface_pt = (UartInterface_t*)arg;
    UartInterface_t NewInterface;
    memcpy(&NewInterface, NewInterface_pt, sizeof(NewInterface));
    free(arg);
    uint32_t Bitrate_u32 = NewInterface.InterfaceBitrate_u32;
#ifndef NOREALPRINTF
    uint8_t InterfaceId_u8 = NewInterface.InterfaceId_u8;
#endif
    char* InterfaceName = NewInterface.InterfaceName;
    MPRINTF("--------------------\n");
    MPRINTF("Add New Interface: \n");
    MPRINTF("Bitrate: %d\n", Bitrate_u32);
#ifndef NOREALPRINTF
    MPRINTF("InterfaceId_u8: %d\n", InterfaceId_u8);
#endif
    MPRINTF("InterfaceName: %s\n", InterfaceName);
    int InterfaceFS_i = SetupInterface(InterfaceName, Bitrate_u32);
    MPRINTF("InterfaceFS_i: %d\n", InterfaceFS_i);
    MPRINTF("--------------------\n");
    UART_fs[NewInterface.InterfaceId_u8] = InterfaceFS_i;
    if (InterfaceFS_i > 0)
    {
        MPRINTF("Interface setup successful\n");
    }
    else
    {
        MPRINTF("Interface setup failed\n");
    }
    sem_post(&thread_sem);
    uint8_t Payload_pu8[UART_MAX_BUF_LEN] = {0};
    while (ReceiverThreadShouldRunUART_b())
    {
        ssize_t Size = ReadMsg(Payload_pu8, InterfaceFS_i);
        MPRINTF("Test Size: %zd\n", Size);
        if (Size > 0)
        {
            PrintUARTPayload(Payload_pu8, Size);
            if (ReadCallback != NULL)
            {
                if (Size < UINT8_MAX)
                {
                    uint8_t Size_u8 = (uint8_t)Size;
                    ReadCallback(Payload_pu8, Size_u8, NewInterface.InterfaceId_u8);
                }
            }
        }
        usleep(UART_LOOP_WAIT_TIME);
    }
    close(InterfaceFS_i);
    close(timeout_fd[0]);
    MPRINTF("End Receiver\n");
    return NULL;
}

static void
PrintUARTPayload(uint8_t* Payload_pu8, uint8_t Length_u8)
{
    MPRINTF("Payload: \n ");
    for (uint8_t Idx_u8 = 0; Idx_u8 < Length_u8; Idx_u8++)
    {
        MPRINTF("%02X ", Payload_pu8[Idx_u8]);
    }
    MPRINTF("\n");
}
