#!/bin/bash
# Generates API documentation via doxygen, replacing local files temporarily

BASEDIR=$(realpath $(dirname "${BASH_SOURCE[0]:-$0}")/..)  # Absolute path to package directory
DOXYFILE="$BASEDIR/Doxyfile"             # Temporarily modified with package version
DOC_README="$BASEDIR/doc/doxygen.md"     # README provided for doxygen
DOC_CSS="$BASEDIR/doc/html/doxygen.css"  # CSS file generated by doxygen

# Custom CSS added to generated docs
EXTRA_CSS=$(cat << EOF
pre { margin-top: 0; }
#projectbrief {
  position: absolute;
  top: 20px;
  right: 10px;
}
EOF
)

# Temporary renames, as {original path: renamed path}
declare -A RENAMES

cd "$BASEDIR"

# Rename existing Markdown/reStructuredText files to omit them from processing
shopt -s nullglob  # Remove missing globs from list
for f in *.md *.rst ; do
  echo "Renaming $f to $f.bak"
  mv "$f" "$f.bak"
  RENAMES[$f]="$f.bak"
done
shopt -u nullglob  # Restore default behaviour

echo "Providing ${DOC_README/${BASEDIR}\//} as README.md"
cp "$DOC_README" "$DOC_README.bak"
mv "$DOC_README" "$BASEDIR/README.md"
RENAMES["$DOC_README"]="$DOC_README.bak"

echo "Backing up ${DOXYFILE/${BASEDIR}\//} before adding version number"
cp "$DOXYFILE" "$DOXYFILE.bak"
VERSION=$(grep __version__ "$BASEDIR"/src/$(basename "$BASEDIR")/__init__.py | cut -d'=' -f2)
echo "" >> "$DOXYFILE"
echo "PROJECT_NUMBER = $VERSION" >> "$DOXYFILE"
RENAMES["$DOXYFILE"]="$DOXYFILE.bak"

echo
doxygen || true

echo
echo "Adding custom CSS"
if [[ -f "$DOC_CSS" ]] ; then echo "$EXTRA_CSS" >> "$DOC_CSS" ; else echo "No such file: $DOC_CSS" ; fi


for f in "${!RENAMES[@]}" ; do
  echo "Restoring ${f/${BASEDIR}\//}"
  mv "${RENAMES[$f]}" "$f"
done
