// -*- C++ -*-
/*!
 * @file CORBAManager.h
 * @brief CORBA manager for doil
 * @date $Date$
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2008
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id$
 *
 */
#ifndef DOIL_CORBA_CORBAMANAGER_H
#define DOIL_CORBA_CORBAMANAGER_H

#include <coil/Properties.h>
#include <coil/Mutex.h>
#include <coil/Guard.h>
#include <doil/IORB.h>
#include <doil/ServantFactory.h>
#include <doil/ProxyFactory.h>
#include <doil/corba/CORBA.h>
#include <doil/corba/CORBAServantBase.h>
#include <doil/corba/CORBAProxyBase.h>
#include <doil/ObjectManager.h>

namespace doil
{
namespace CORBA
{
  /*!
   * @if jp
   * @class CORBAManager 饹
   *
   * CORBA ORB  IORB ǥ󥿡ե Singleton 饹
   *
   * @else
   * @class CORBAManager class
   *
   * This class is a singleton class that interfaces IORB.
   *
   * @endif
   */
  class CORBAManager
    : public doil::IORB
  {
  public:
    //
    //for unit tests
    //
//    doil::LocalBase* toLocal(::CORBA::Object_ptr& obj);
    doil::LocalBase* toLocal(::CORBA::Object_ptr& obj)
    {
        return NULL;
    }
    /*!
     * @if jp
     * @brief ؿ
     * @else
     * @brief initializer
     * @endif
     */
    static CORBAManager* init(coil::Properties prop)
      throw();

    /*!
     * @if jp
     * @brief 󥹥󥹼ؿ
     * @else
     * @brief getting instance
     * @endif
     */
    static CORBAManager& instance()
      throw();

    //------------------------------------------------------------
    // IORB interfaces
    //------------------------------------------------------------
    /*!
     * @if jp
     * @brief ORB ̾
     *
     * ORB ֤̾
     * δؿɬ "corba" ֤
     *
     * @else
     * @brief Getting ORB's name
     *
     * This operation will return ORB's name.
     *
     * @endif
     */
    virtual const char* name()
      throw();

    virtual void run()
      throw();
    virtual void shutdown()
      throw();

    /*!
     * @if jp
     * @brief Servant  Factory Ͽ
     *
     * Servant  Factory Ͽ롣
     *
     * @return OK:             ｪλ
     *         ALREADY_EXISTS: Ϳ줿եȥϤǤϿѤ
     *         INVALID_ARGS:   ΰİʾ夬
     *         UNKNOWN:        ʥ顼
     *
     * @else
     * @brief Register servant's factory
     *
     * This operation will register servant's factory.
     *
     * @return OK:             Normal return
     *         ALREADY_EXISTS: Given factory already exists.
     *         INVALID_ARGS:   One or more given arguments are invalid.
     *         UNKNOWN:        Unknown error
     *
     * @endif
     */
    virtual ReturnCode_t
    registerFactory(const char* id,
                    doil::ServantNewFunc new_func,
                    doil::ServantDeleteFunc delete_func)
      throw();
#if 1
    //
    //for unit tests
    //
    virtual ReturnCode_t
    registerProxyFactory(const char* id,
                    doil::ProxyNewFunc new_func,
                    doil::ProxyDeleteFunc delete_func)
      throw();
#endif
    /*!
     * @if jp
     * @brief ֥Ȥactivate
     *
     * Ϳ줿֥Ȥ򥢥ƥֲ롣
     * Ϳ줿֥ȤIDȹפ륵ХȤΥեȥ꤬
     * ϿѤߤǤɬפ롣⤷֥ȤIDȹפեȥ
     * ¸ߤʤ硢INVALID_ARGS 顼֤롣
     *
     * @return OK:             ｪλ
     *         ALREADY_EXISTS: Ϳ줿֥ȤϤǤ¸ߤ
     *         INVALID_ARGS:   
     *         UNKNOWN:        ʥ顼
     *
     * @else
     * @brief Activate object
     *
     * This operation will activate given object
     * The servant that has same ID with the given object's ID should 
     * exist in this ORB servant's map. INVALID_ARGS error code will return
     * if the factory does not exist.
     *
     * @return OK:             Normal return
     *         ALREADY_EXISTS: Given object already exists.
     *         INVALID_ARGS:   The given arguments is invalid.
     *         UNKNOWN:        Unknown error
     *
     * @endif
     */
    virtual ReturnCode_t activateObject(doil::ImplBase* impl)
      throw();

    /*!
     * @if jp
     * @brief ֥Ȥactivate
     *
     * Ϳ줿֥ȤͿ줿ХȤȶ˥ƥֲ롣
     * ֥Ȥ򥢥ƥֲ륵ХȤ¸ߤɬפϤʤ
     * Ϳ줿ХȤIDͿ줿֥ȤID
     * פƤɬפ롣פʤ INVALID_ARGS 顼֤롣
     *
     * @return OK:             ｪλ
     *         ALREADY_EXISTS: Ϳ줿֥ȤϤǤ¸ߤ
     *         INVALID_ARGS:   
     *         UNKNOWN:        ʥ顼
     *
     * @else
     * @brief Activate object
     *
     * This operation will activate given object
     * The servant's factory that activate the given object does not need to
     * exists in this ORB manager. However, the given servant's ID and
     * the given object's ID have to be same. INVALID_ARGS error will be
     * returned if these IDs are not same. 
     *
     * @return OK:             Normal return
     *         ALREADY_EXISTS: Given object already exists.
     *         INVALID_ARGS:   The given arguments is invalid.
     *         UNKNOWN:        Unknown error
     *
     * @endif
     */
    virtual ReturnCode_t activateObject(doil::ImplBase* impl,
                                        doil::ServantBase* servant)
      throw();

    /*!
     * @if jp
     * @brief ֥Ȥdeactivate
     *
     * Ϳ줿֥Ȥ󥢥ƥֲ롣
     * ֥Ȥ֥ȥޥåפˤʤ NOT_FOUND 顼֤롣
     *
     * @return OK:             ｪλ
     *         NOT_FOUND:      Ϳ줿֥Ȥϥޥå׾¸ߤʤ
     *         INVALID_ARGS:   
     *         UNKNOWN:        ʥ顼
     *
     * @else
     * @brief Deactivate object
     *
     * This operation will deactivate the given object.
     * NOT_FOUND error will be returned if the given object does not exist
     * in the ative object map.
     *
     * @return OK:             Normal return
     *         NOT_FOUND:      Given object does not exist in the map.
     *         INVALID_ARGS:   The given arguments is invalid.
     *         UNKNOWN:        Unknown error
     *
     * @endif
     */
    virtual ReturnCode_t deactivateObject(doil::ImplBase* impl)
      throw();

    /*!
     * @if jp
     * @brief ֥Ȥdeactivate
     *
     * Ϳ줿̾Υ֥Ȥ󥢥ƥֲ롣֥̾
     * ֥ȥޥåפˤʤ NOT_FOUND 顼֤롣
     *
     * @return OK:             ｪλ
     *         NOT_FOUND:      Ϳ줿֥Ȥϥޥå׾¸ߤʤ
     *         INVALID_ARGS:   
     *         UNKNOWN:        ʥ顼
     *
     * @else
     * @brief Deactivate object
     *
     * This operation will deactivate the given name's object.
     * NOT_FOUND error will be returned if the given object does not exist
     * in the ative object map.
     *
     * @return OK:             Normal return
     *         NOT_FOUND:      Given object does not exist in the map.
     *         INVALID_ARGS:   The given arguments is invalid.
     *         UNKNOWN:        Unknown error
     *
     * @endif
     */
    virtual ReturnCode_t deactivateObject(const char* name)
      throw();

    /*!
     * @if jp
     * @brief Impl֥Ȥ̾Ǽ
     *
     * Ϳ줿̾Υ֥Ȥ롣
     * ֥Ȥޥåפˤʤ NOT_FOUND 顼֤롣
     *
     * @return OK:             ｪλ
     *         NOT_FOUND:      ֥Ȥϥޥå׾¸ߤʤ
     *         INVALID_ARGS:   
     *         UNKNOWN:        ʥ顼
     *
     * @else
     * @brief Getting object by name
     *
     * This operation will return a object by given name.  NOT_FOUND
     * error will be returned if the given name's object does not
     * exist in the ative object map.
     *
     * @return OK:             Normal return
     *         NOT_FOUND:      The object does not exist in the map.
     *         INVALID_ARGS:   The given arguments is invalid.
     *         UNKNOWN:        Unknown error
     *
     * @endif
     */
    virtual ImplBase* getImpl(const char* name)
      throw();

    /*!
     * @if jp
     * @brief Impl֥ȤServantǼ
     *
     * Ϳ줿֥Ȥб륵ХȤ롣
     * ֥Ȥޥåפˤʤ NOT_FOUND 顼֤롣
     *
     * @return OK:             ｪλ
     *         NOT_FOUND:      ֥Ȥϥޥå׾¸ߤʤ
     *         INVALID_ARGS:   
     *         UNKNOWN:        ʥ顼
     *
     * @else
     * @brief Getting impl object by servant
     *
     * This operation will return an impl object by given servant.
     * NOT_FOUND error will be returned if the given servant's impl object
     * does not exist in the ative object map.
     *
     * @return OK:             Normal return
     *         NOT_FOUND:      The object does not exist in the map.
     *         INVALID_ARGS:   The given arguments is invalid.
     *         UNKNOWN:        Unknown error
     *
     * @endif
     */
    virtual ImplBase* toImpl(doil::ServantBase* servant)
      throw();

    /*!
     * @if jp
     * @brief Servant֥Ȥ
     *
     * Ϳ줿̾Servant֥Ȥ롣
     * ֥Ȥޥåפˤʤ NOT_FOUND 顼֤롣
     *
     * @return OK:             ｪλ
     *         NOT_FOUND:      ֥Ȥϥޥå׾¸ߤʤ
     *         INVALID_ARGS:   
     *         UNKNOWN:        ʥ顼
     *
     * @else
     * @brief Getting servant object by name
     *
     * This operation will return an servant object by given name.
     * NOT_FOUND error will be returned if the given name's servant object
     * does not exist in the ative object map.
     *
     * @return OK:             Normal return
     *         NOT_FOUND:      The object does not exist in the map.
     *         INVALID_ARGS:   The given arguments is invalid.
     *         UNKNOWN:        Unknown error
     *
     * @endif
     */
    virtual ServantBase* getServant(const char* name)
      throw();

    /*!
     * @if jp
     * @brief Servant֥Ȥ
     *
     * Ϳ줿impl֥ȤServant֥Ȥ롣
     * ֥Ȥޥåפˤʤ NOT_FOUND 顼֤롣
     *
     * @return OK:             ｪλ
     *         NOT_FOUND:      ֥Ȥϥޥå׾¸ߤʤ
     *         INVALID_ARGS:   
     *         UNKNOWN:        ʥ顼
     *
     * @else
     * @brief Getting servant object by impl object
     *
     * This operation will return an servant object by given impl object.
     * NOT_FOUND error will be returned if the given name's servant object
     * does not exist in the ative object map.
     *
     * @return OK:             Normal return
     *         NOT_FOUND:      The object does not exist in the map.
     *         INVALID_ARGS:   The given arguments is invalid.
     *         UNKNOWN:        Unknown error
     *
     * @endif
     */
    virtual ServantBase* toServant(doil::ImplBase* lobj)
      throw();

    //------------------------------------------------------------
    // CORBAManager interfaces
    //------------------------------------------------------------
    /*!
     * @if jp
     * @brief ObjectϢդ줿ImplѴ
     *
     * @else
     * @brief Convert CORBA Object_ptr to Impl related to it.
     *
     * @endif
     */
    doil::ImplBase* toImpl(::CORBA::Object_ptr obj)
      throw();

    /*!
     * @if jp
     * @brief ̾饪֥ȻȤ
     *
     * @else
     * @brief Getting object reference from the given name
     *
     * @endif
     */
    ::CORBA::Object_ptr getReference(const char* name)
      throw();

    /*!
     * @if jp
     * @brief Impl֥Ȥ饪֥ȻȤѴ
     *
     * @else
     * @brief Converting Impl object to object reference
     *
     * @endif
     */
    ::CORBA::Object_ptr toReference(doil::ImplBase* impl)
      throw();

    /*!
     * @if jp
     * @brief Servant֥Ȥ饪֥ȻȤѴ
     *
     * @else
     * @brief Converting Servant object to object reference
     *
     * @endif
     */
    ::CORBA::Object_ptr toReference(doil::ServantBase* servant)
      throw();

    /*!
     * @if jp
     * @brief ORBΥݥ󥿤
     *
     * @else
     * @brief Getting ORB pointer
     *
     * @endif
     */
    doil::ServantBase* toServant(::CORBA::Object_ptr obj)
      throw();

    //------------------------------------------------------------
    // CORBA functions
    //------------------------------------------------------------
    /*!
     * @if jp
     * @brief ORBΥݥ󥿤
     *
     * @else
     * @brief Getting ORB pointer
     *
     * @endif
     */
    ::CORBA::ORB_ptr getORB()
      throw();

    /*!
     * @if jp
     * @brief ǥեPOAΥݥ󥿤
     *
     * @else
     * @brief Getting default POA pointer
     *
     * @endif
     */
    ::PortableServer::POA_ptr getPOA()
      throw();

    /*!
     * @if jp
     * @brief POAManagerΥݥ󥿤
     *
     * @else
     * @brief Getting POAManager pointer
     *
     * @endif
     */
    ::PortableServer::POAManager_ptr getPOAManager()
      throw();

    //------------------------------------------------------------
    // protected functions:
    //------------------------------------------------------------
  protected:
    /*!
     * @if jp
     * @brief CORBA ORB ν
     *
     * ˤͿ줿򸵤ORB롣
     *
     * @return ORB (:true:false)
     *
     * @else
     * @brief CORBA ORB initialization
     *
     * Initialize ORB based on the configuration given by arguments.
     *
     * @return ORB initialization result (Successful:true, Failed:false)
     *
     * @endif
     */
    void initOrb(coil::Properties prop);

    /*!
     * @if jp
     * @brief ORB Υޥɥ饤󥪥ץ
     *
     * ե졼ꤵ줿Ƥ
     * ORB εưץ롣
     *
     * @return ORB ưץ
     *
     * @else
     * @brief Create ORB command options
     *
     * Create ORB launch options from configuration information
     * that has been set.
     *
     * @return ORB launch options
     *
     * @endif
     */
    std::string createORBOptions();
    
    //------------------------------------------------------------
    // private functions:
    //------------------------------------------------------------
  private:
    // private constructor
    CORBAManager(){}
    // private big three
    CORBAManager(const CORBAManager& cmgr);
    CORBAManager& operator=(const CORBAManager& rhs);
    ~CORBAManager() throw() {}

    //------------------------------------------------------------
    // private data:
    //------------------------------------------------------------
  private:
    // static data member for singleton
    static CORBAManager* manager;
    static coil::Mutex mutex;

    // CORBA ORB pointer
    ::CORBA::ORB_ptr m_orb;

    // default POA pointer
    ::PortableServer::POA_ptr m_poa;

    // POAManager pointer
    ::PortableServer::POAManager_ptr m_poaManager;

    // configuration properties
    coil::Properties m_config;


    // Entry class for active object map
    class Entry
    {
    public:
      Entry(ImplBase* impl, CORBAServantBase* servant,
            ::CORBA::Object_ptr objref)
        : impl_(impl), servant_(servant), objref_(objref)
      {
      }
      virtual ~Entry()
      {
      }
      ImplBase* impl_;
      CORBAServantBase* servant_;
      ::CORBA::Object_ptr objref_;
    };

    // Predicate class for active object map
    class EntryPred
    {
    public:
      EntryPred(const char* name)
        : m_name(name)
      {
      }
      EntryPred(Entry* entry)
        : m_name(entry->impl_->name())
      {
      }
      bool operator()(Entry* entry)
      {
        return m_name == entry->impl_->name();
      }
      std::string m_name;
    };
    
    // Active object map
    ObjectManager<const char*, Entry, EntryPred> m_map;



#if 0
    //
    // for unit test
    //
    // Entry class for active object map
    class EntryProxy
    {
    public:
      EntryProxy(ImplBase* impl, CORBAProxyBase* proxy,
            ::CORBA::Object_ptr objref)
        : impl_(impl), proxy_(proxy), objref_(objref)
      {
      }
      virtual ~EntryProxy()
      {
      }
      ImplBase* impl_;
      CORBAProxyBase* proxy_;
      ::CORBA::Object_ptr objref_;
    };

    //
    // for unit test
    //
    // Predicate class for active object map
    class EntryPredProxy
    {
    public:
      EntryPredProxy(const char* name)
        : m_name(name)
      {
      }
      EntryPredProxy(Entry* entry)
        : m_name(entry->impl_->name())
      {
      }
      bool operator()(Entry* entry)
      {
        return m_name == entry->impl_->name();
      }
      std::string m_name;
    };
    
    //
    // for unit test
    //
    // Active object map
    ObjectManager<const char*, EntryProxy, EntryPredProxy> m_map_proxy;
#endif

    // Predicate functor for Factory map
    class FactoryPred
    {
    public:
      FactoryPred(const char* id) : m_id(id) {}
      FactoryPred(ServantFactory* factory) : m_id(factory->id()) {}
      bool operator()(ServantFactory* factory)
      {
        return m_id == factory->id();
      }
      std::string m_id;
    };
    // Servant factory map
    ObjectManager<const char*, ServantFactory, FactoryPred> m_factory;

#if 1
    //
    // for unit test
    //
    // Predicate functor for Factory map
    class ProxyFactoryPred
    {
    public:
      ProxyFactoryPred(const char* id) : m_id(id) {}
      ProxyFactoryPred(ProxyFactory* factory) : m_id(factory->id()) {}
      bool operator()(ProxyFactory* factory)
      {
        return m_id == factory->id();
      }
      std::string m_id;
    };
    //
    // for unit test
    //
    // Servant factory map
    ObjectManager<const char*, ProxyFactory, ProxyFactoryPred> m_factory_proxy;
#endif
  };
};
};

extern "C"
{
  void DoilCORBAInit(coil::Properties& prop);
}
#endif // RTM_CORBA_CORBAMANAGER_H
