// -*- C++ -*-
/*!
 * @file PublisherBase.h
 * @brief Publisher base class
 * @date $Date: 2007-12-31 03:08:06 $
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2006-2010
 *     Noriaki Ando
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id$
 *
 */

#ifndef RTC_PUBLISHERBASE_H
#define RTC_PUBLISHERBASE_H

#include <coil/Properties.h>
#include <coil/Factory.h>
#include <rtm/RTC.h>
#include <rtm/CdrBufferBase.h>
#include <rtm/DataPortStatus.h>

namespace coil
{
  class Properties;
}
namespace RTC
{
  class InPortConsumer;
  class ConnectorListeners;
  class ConnectorInfo;

  /*!
   * @if jp
   *
   * @class PublisherBase
   *
   * @brief Publisher 쥯饹
   * 
   * ǡХߥ󥰤ФưPublisher* δ쥯饹
   * Ƽ Publisher ϤΥ饹Ѿƾܺ٤롣
   *
   * @since 0.4.0
   *
   * @else
   *
   * @class PublisherBase
   *
   * @brief Base class of Publisher.
   *
   * This is a base class of Publisher*. This class manages data send timing.
   * Variation of Publisher* which implements details of Publisher inherits
   * this PublisherBase class.
   *
   * @endif
   */
  class PublisherBase
    : public DataPortStatus
  {
  public:
    DATAPORTSTATUS_ENUM
    /*!
     * @if jp
     *
     * @brief ǥȥ饯
     *
     * @else
     *
     * @brief Destructor
     *
     * @endif
     */
    virtual ~PublisherBase(void){};

    /*!
     * @if jp
     * @brief 
     *
     * InPortConsumerγƼԤ饹ǤϡͿ줿
     * PropertiesɬפʾƳƼԤ init() 
     * ϡOutPortProviderľ太ӡ³ˤ줾ƤФ
     * ǽ롣äơδؿʣƤФ뤳ȤꤷƵ
     * Ҥ٤Ǥ롣
     * 
     * @param prop 
     *
     * @else
     *
     * @brief Initializing configuration
     *
     * This operation would be called to configure in initialization.
     * In the concrete class, configuration should be performed
     * getting appropriate information from the given Properties data.
     * This function might be called right after instantiation and
     * connection sequence respectivly.  Therefore, this function
     * should be implemented assuming multiple call.
     *
     * @param prop Configuration information
     *
     * @endif
     */
    virtual ReturnCode init(coil::Properties& prop) = 0;

    /*!
     * @if jp
     * @brief InPort󥷥塼ޤΥå
     *
     * δؿǤϡ Publisher ˴Ϣդ륳󥷥塼ޤ򥻥åȤ롣
     * 󥷥塼ޥ֥Ȥ̥ݥ󥿤ξ硢INVALID_ARGS֤롣
     * ʳξϡPORT_OK ֤롣
     *
     * @param consumer Consumer ؤΥݥ
     * @return ReturnCode PORT_OK ｪλ
     *                    INVALID_ARGS ͤޤޤƤ
     *
     * @else
     * @brief Store InPort consumer
     *
     * This operation sets a consumer that is associated with this
     * object. If the consumer object is NULL, INVALID_ARGS will be
     * returned.
     *
     * @param consumer A pointer to a consumer object.
     * @return ReturnCode PORT_OK normal return
     *                    INVALID_ARGS given argument has invalid value
     *
     * @endif
     */
    virtual ReturnCode setConsumer(InPortConsumer* consumer) = 0;

    /*!
     * @if jp
     * @brief ХåեΥå
     *
     * δؿǤϡ Publisher ˴ϢդХåե򥻥åȤ롣
     * Хåե֥Ȥ̥ݥ󥿤ξ硢INVALID_ARGS֤롣
     * ʳξϡPORT_OK ֤롣
     *
     * @param buffer CDR buffer ؤΥݥ
     * @return ReturnCode PORT_OK ｪλ
     *                    INVALID_ARGS ͤޤޤƤ
     *
     * @else
     * @brief Setting buffer pointer
     *
     * This operation sets a buffer that is associated with this
     * object. If the buffer object is NULL, INVALID_ARGS will be
     * returned.
     *
     * @param buffer A pointer to a CDR buffer object.
     * @return ReturnCode PORT_OK normal return
     *                    INVALID_ARGS given argument has invalid value
     *
     * @endif
     */
    virtual ReturnCode setBuffer(BufferBase<cdrMemoryStream>* buffer) = 0;

    /*!
     * @if jp
     * @brief ꥹʤꤹ롣
     *
     * Publisher Фƥꥹʥ֥ ConnectorListeners ꤹ롣
     * Ƽꥹʥ֥Ȥޤ ConnectorListeners 򥻥åȤ뤳Ȥǡ
     * Хåեɤ߽񤭡ǡˤΥꥹʤ򥳡뤹롣
     * ConnectorListeners ֥ȤνͭϥݡȤޤ RTObject 
     * Publisher  ConnectorListeners Ϻ뤳ȤϤʤ
     * ConnectorListeners ̥ݥ󥿤ξ INVALID_ARGS ֤
     *
     * @param info ConnectorProfile 벽֥ ConnectorInfo
     * @param listeners ꥹʤ¿ݻ ConnectorListeners ֥
     * @return PORT_OK      ｪλ
     *         INVALID_ARGS ʰ
     * @else
     * @brief Set the listener. 
     *
     * This function sets ConnectorListeners listener object to the
     * Publisher. By setting ConnectorListeners containing various
     * listeners objects, these listeners are called at the time of
     * reading and writing of a buffer, and transmission of data
     * etc. Since the ownership of the ConnectorListeners object is
     * owned by Port or RTObject, the Publisher never deletes the
     * ConnectorListeners object. If the given ConnectorListeners'
     * pointer is NULL, this function returns INVALID_ARGS.
     *
     * @param info ConnectorInfo that is localized object of ConnectorProfile
     * @param listeners ConnectorListeners that holds various listeners
     * @return PORT_OK      Normal return
     *         INVALID_ARGS Invalid arguments
     * @endif
     */
    virtual ReturnCode setListener(ConnectorInfo& info,
                                   ConnectorListeners* listeners) = 0;

    /*!
     * @if jp
     * @brief ǡ񤭹
     *
     * Publisher Фƥǡ񤭹ࡣ󥷥塼ޡꥹŬ
     * ꤵƤʤPublisher ֥Ȥ
     * ʤ硢δؿƤӽФȥ顼
     * PRECONDITION_NOT_MET ֤졢񤭹ϰڹԤʤ
     *
     * 񤭹ߤФơ󥷥塼ޤե֡󥷥塼ޤΥ顼
     * 󥷥塼ޤؤν񤭹ߤॢȤˤϤ줾졢顼
     *  SEND_FULL, SEND_ERROR, SEND_TIMEOUT ֤롣
     *
     * ʳΥ顼ξ硢PORT_ERROR ֤롣
     * 
     *
     * @param data 񤭹ǡ 
     * @param sec ॢȻ
     * @param nsec ॢȻ
     *
     * @return PORT_OK             ｪλ
     *         PRECONDITION_NO_MET consumer, buffer, listenerŬڤ
     *                             ƤʤΥ֥Ȥλ
     *                             ʤ硣
     *         SEND_FULL           褬ե
     *         SEND_TIMEOUT        褬ॢȤ
     *         CONNECTION_LOST     ³Ǥ줿ȤΤ
     *
     * @else
     * @brief Write data 
     *
     * This function writes data into the Publisher. If this function
     * is called without initializing correctly such as a consumer,
     * listeners, etc., error code PRECONDITION_NOT_MET will be
     * returned and no operation of the writing etc. will be
     * performed.
     *
     * When publisher writes data, if the consumer returns
     * full-status, returns error, is returned with timeout, error
     * codes BUFFER_FULL, BUFFER_ERROR and BUFFER_TIMEOUT will be
     * returned respectively.
     *
     * In other cases, PROT_ERROR will be returned.
     *
     * @param data Data to be wrote to the buffer
     * @param sec Timeout time in unit seconds
     * @param nsec Timeout time in unit nano-seconds
     * @return PORT_OK             Normal return
     *         PRECONDITION_NO_MET Precondition does not met. A consumer,
     *                             a buffer, listenes are not set properly.
     *         SEND_FULL           Data was sent but full-status returned
     *         SEND_TIMEOUT        Data was sent but timeout occurred
     *         CONNECTION_LOST     detected that the connection has been lost
     *
     * @endif
     */
    virtual ReturnCode write(const cdrMemoryStream& data,
                             unsigned long sec,
                             unsigned long usec) = 0;

    /*!
     * @if jp
     *
     * @brief ƥֲǧ
     * 
     * Publisher ϥǡݡȤƱ activate/deactivate 롣
     * activate() / deactivate() ؿˤäơƥ־֤󥢥ƥ
     * ־֤ڤؤ롣δؿˤꡢߥƥ־֤󥢥
     * ƥ־֤ǧ뤳ȤǤ롣
     *
     * @return ֳǧ(ƥ־:true󥢥ƥ־:false)
     *
     * @else
     *
     * @brief If publisher is active state
     * 
     * A Publisher can be activated/deactivated synchronized with the
     * data port.  The active state and the non-active state are made
     * transition by the "activate()" and the "deactivate()" functions
     * respectively. This function confirms if the publisher is in
     * active state.
     *
     * @return Result of state confirmation
     *         (Active state:true, Inactive state:false)
     *
     * @endif
     */
    virtual bool isActive() = 0;

    /*!
     * @if jp
     * @brief ƥֲ
     *
     * Publisher 򥢥ƥֲ롣δؿƤӽФȤˤꡢ
     * Publisherġǡ륹åɤư򳫻Ϥ롣
     * ԤƤʤʤɤˤꡢʤ硢顼
     *  PRECONDITION_NOT_MET ֤
     *
     * @return PORT_OK ｪλ
     *         PRECONDITION_NOT_MET ʤ
     *
     * @else
     * @brief activation
     *
     * This function activates the publisher. By calling this
     * function, this publisher starts the thread that pushes data to
     * InPort. If precondition such as initialization process and so
     * on is not met, the error code PRECONDITION_NOT_MET is returned.
     *
     * @return PORT_OK normal return
     *         PRECONDITION_NOT_MET precondition is not met
     *
     * @endif
     */
    virtual ReturnCode activate() = 0;

    /*!
     * @if jp
     * @brief 󥢥ƥֲ
     *
     * Publisher 󥢥ƥֲ롣δؿƤӽФȤˤꡢ
     * Publisherġǡ륹åɤưߤ롣
     * ԤƤʤʤɤˤꡢʤ硢顼
     *  PRECONDITION_NOT_MET ֤
     *
     * @return PORT_OK ｪλ
     *         PRECONDITION_NOT_MET ʤ
     *
     * @else
     * @brief deactivation
     *
     * This function deactivates the publisher. By calling this
     * function, this publisher stops the thread that pushes data to
     * InPort. If precondition such as initialization process and so
     * on is not met, the error code PRECONDITION_NOT_MET is returned.
     *
     * @return PORT_OK normal return
     *         PRECONDITION_NOT_MET precondition is not met
     *
     * @endif
     */
    virtual ReturnCode deactivate() = 0;
  
    /*!
     * @if jp
     *
     * @brief Publisher ˴롣
     *
     *  Publisher ˴롣
     *  Publisher פˤʤä PublisherFactory ƤӽФ롣
     *
     * @else
     *
     * @brief Release the Publisher
     *
     * Release this Publisher.
     * When Publisher becomes unnecessary, this is invoked from
     * PublisherFactory. 
     *
     * @endif
     */
    virtual void release(){}
  };

  typedef coil::GlobalFactory<PublisherBase> PublisherFactory;

#if defined(WIN32) || defined(_WIN32) || defined(__WIN32__) || defined(__NT__)
  EXTERN template class DLL_PLUGIN coil::GlobalFactory<PublisherBase>;
#endif
};
#endif // RTC_PUBLISHERBASE_H
