/***
 *  Software License Agreement: BSD 3-Clause License
 *
 *  Copyright (c) 2016-2022, qbrobotics®
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 *  following conditions are met:
 *
 *  * Redistributions of source code must retain the above copyright notice, this list of conditions and the
 *    following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote
 *    products derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 *  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <qb_softhand_industry_driver/qb_softhand_industry_communication_handler.h>

using namespace qb_softhand_industry_communication_handler;

qbSoftHandIndustryCommunicationHandler::qbSoftHandIndustryCommunicationHandler() 
  : node_handle_(ros::NodeHandle()),
    activate_motors_(node_handle_.advertiseService("/qb_softhand_industry_communication_handler/activate_motors", &qbSoftHandIndustryCommunicationHandler::activateCallback, this)),
    deactivate_motors_(node_handle_.advertiseService("/qb_softhand_industry_communication_handler/deactivate_motors", &qbSoftHandIndustryCommunicationHandler::deactivateCallback, this)),
    set_commands_(node_handle_.advertiseService("/qb_softhand_industry_communication_handler/set_commands", &qbSoftHandIndustryCommunicationHandler::setCommandsCallback, this)),
    set_command_(node_handle_.advertiseService("/qb_softhand_industry_communication_handler/set_command", &qbSoftHandIndustryCommunicationHandler::setCommandCallback, this)),
    get_measurements_(node_handle_.advertiseService("/qb_softhand_industry_communication_handler/get_measurements", &qbSoftHandIndustryCommunicationHandler::getMeasurementsCallback, this)),
    socket_protector_(std::make_unique<std::mutex>()),
    ip_(node_handle_.param<std::string>("ip_address", "192.168.1.110")){
      while(!(init() == 0)){
        ros::Duration(1.0).sleep();
      }
      ROS_INFO_STREAM_NAMED("qbSoftHand Industry Communication Handler", "Connected to device with ip " << ip_ << ".");
    }

qbSoftHandIndustryCommunicationHandler::~qbSoftHandIndustryCommunicationHandler(){
  close(socket_descriptor_);
}

int qbSoftHandIndustryCommunicationHandler::activate(const bool &command, const int &max_repeats) {
  std::string command_prefix = command ? "" : "de";
  bool status = false;
  if(command) {
    sendAndParseResponse("MO=1",max_repeats); // activation command
  } else {
    sendAndParseResponse("MO=0",max_repeats); // deactivation command
  }
  ROS_INFO_STREAM_NAMED("qbSoftHand Industry Communication Handler", "device [" << ip_ << "] motors have been " << command_prefix << "activated!");
}

bool qbSoftHandIndustryCommunicationHandler::activateCallback(qb_softhand_industry_srvs::TriggerRequest &request, qb_softhand_industry_srvs::TriggerResponse &response) {
  ROS_ERROR_STREAM_COND_NAMED(request.max_repeats < 0, "communication_handler", "Device [" << ip_ << "] has request service with non-valid 'max_request' [" << request.max_repeats << "].");
  activate(true, request.max_repeats);
  response.message = "Device [" + ip_ + "] activation.";
  return true;
}

bool qbSoftHandIndustryCommunicationHandler::deactivateCallback(qb_softhand_industry_srvs::TriggerRequest &request, qb_softhand_industry_srvs::TriggerResponse &response) {
  ROS_ERROR_STREAM_COND_NAMED(request.max_repeats < 0, "communication_handler", "Device [" << ip_ << "] has request service with non-valid 'max_request' [" << request.max_repeats << "].");
  activate(false, request.max_repeats);
  response.message = "Device [" + ip_ + "] deactivation.";
  return true;
}

float qbSoftHandIndustryCommunicationHandler::getMeasurement(const std::string &measurement_command, const int &max_repeats) {
  float data = std::numeric_limits<float>::signaling_NaN();
  sendAndParseResponse(measurement_command, max_repeats, data);
  return data;
}

bool qbSoftHandIndustryCommunicationHandler::getMeasurementsCallback(qb_softhand_industry_srvs::GetMeasurementsRequest &request, qb_softhand_industry_srvs::GetMeasurementsResponse &response) {
  ROS_ERROR_STREAM_COND_NAMED(request.max_repeats < 0, "communication_handler", "Device [" << ip_ << "] has request service with non-valid 'max_request' [" << request.max_repeats << "].");
  if(request.get_position) {
    response.position = getMeasurement(QBSOFTHAND_INDUSTRY_GET_POSITION_CMD, request.max_repeats);
  }
  if(request.get_current) {
    response.current = getMeasurement(QBSOFTHAND_INDUSTRY_GET_CURRENT_CMD, request.max_repeats) * 1000; // conversion to [mA]
  }
  if(request.get_velocity) {
    response.velocity = getMeasurement(QBSOFTHAND_INDUSTRY_GET_VELOCITY_CMD, request.max_repeats);
  }

  response.stamp = ros::Time::now();
  response.success = true;
  return true;
}

int qbSoftHandIndustryCommunicationHandler::init() {
  int max_repeats = 3;
  if (initSocket()) {
    return errno;
  }

  auto start = std::chrono::high_resolution_clock::now();
  std::chrono::duration<double> elapsed = std::chrono::high_resolution_clock::now() - start;
  while (elapsed.count() <= 10) {
    if(sendAndParseResponse("SR") == 0) {
      if (getMeasurement("UI[23]", max_repeats) < 108) {
        return QBSOFTHAND_INDUSTRY_ERROR_NOTCOMPATIBLE;
      }

      if (static_cast<int>(getMeasurement("PS", max_repeats)) != 1) {  // program stopped
        sendAndParseResponse("XQ##init", max_repeats);  // resume program
      }

      return 0;
    }
    ros::Duration(0.4).sleep();
    elapsed = std::chrono::high_resolution_clock::now() - start;
    ROS_WARN_STREAM_NAMED("qbSoftHand Industry Communication Handler","Unable to connect to device with IP: " << ip_ << ". Waited for " << elapsed.count() << " s.");
  }

  return QBSOFTHAND_INDUSTRY_ERROR_NOTRESPONDING;
}

int qbSoftHandIndustryCommunicationHandler::initSocket(){
  std::lock_guard<std::mutex> lock(*socket_protector_);
  socket_descriptor_ = socket(AF_INET, SOCK_DGRAM, 0);
  if (socket_descriptor_ < 0) {
    ROS_ERROR_STREAM("UDP socket creation failed.");
    return errno;
  }
  memset(&socket_address_, 0, sizeof(socket_address_));
  address_length_ = sizeof(sockaddr_in);
  socket_address_.sin_family = AF_INET;
  socket_address_.sin_port = htons(5001);  // Elmo Gold Driver only exploit UDP port #5001
  if (inet_pton(AF_INET, ip_.c_str(), &(socket_address_.sin_addr)) != 1) {
    ROS_ERROR_STREAM( "errno has been set to '" << strerror(errno) << "' during 'inet_pton()'.");
    return errno;
  }
  timeval timeout = {0, 100000};  // tv_sec, tv_usec
  if (setsockopt(socket_descriptor_, SOL_SOCKET, SO_RCVTIMEO, &timeout, sizeof(timeout)) < 0) {
    ROS_ERROR_STREAM( "errno has been set to '" << strerror(errno) << "' during 'setsockopt()'.");
    return errno;
  }

  return 0;
}

bool qbSoftHandIndustryCommunicationHandler::isValidCurrent(const int &current){
  return current >= QBSOFTHAND_INDUSTRY_CURR_MIN && current <= QBSOFTHAND_INDUSTRY_CURR_MAX;
}

bool qbSoftHandIndustryCommunicationHandler::isValidPosition(const int &position){
  return position >= QBSOFTHAND_INDUSTRY_POS_MIN && position <= QBSOFTHAND_INDUSTRY_POS_MAX;
}

bool qbSoftHandIndustryCommunicationHandler::isValidVelocity(const int &velocity){
  return velocity >= QBSOFTHAND_INDUSTRY_VEL_MIN && velocity <= QBSOFTHAND_INDUSTRY_VEL_MAX;
}

int qbSoftHandIndustryCommunicationHandler::setCommand(const int &command, const int &max_repeats) {
  if(sendAndParseResponse("SR", max_repeats) != 0) {
    ROS_WARN_STREAM_NAMED("qbSoftHand Industry Communication Handler", "Device not responding");
    return QBSOFTHAND_INDUSTRY_ERROR_NOTRESPONDING;
  }
  if(!isValidPosition(command)){
    return QBSOFTHAND_INDUSTRY_ERROR_BADPARAMS;
  }

  sendAndParseResponse("UI[1]=" + std::to_string(command), max_repeats);
  return 0;
}

int qbSoftHandIndustryCommunicationHandler::setCommands(const int &position, const int &velocity, const int &current, const int &max_repeats) {
  if(sendAndParseResponse("SR", max_repeats) != 0) {
    ROS_WARN_STREAM("Device not responding");
    return QBSOFTHAND_INDUSTRY_ERROR_NOTRESPONDING;
  }

  if(!isValidPosition(position) || !isValidVelocity(velocity) || !isValidCurrent(current)){
    return QBSOFTHAND_INDUSTRY_ERROR_BADPARAMS;
  }

  sendAndParseResponse("UI[2]=" + std::to_string(velocity), max_repeats);
  sendAndParseResponse("UI[3]=" + std::to_string(current), max_repeats);
  sendAndParseResponse("UI[1]=" + std::to_string(position), max_repeats);
  return 0;
}

bool qbSoftHandIndustryCommunicationHandler::setCommandCallback(qb_softhand_industry_srvs::SetCommandRequest &request, qb_softhand_industry_srvs::SetCommandResponse &response) {
  ROS_ERROR_STREAM_COND_NAMED(request.max_repeats < 0, "communication_handler", "Device [" << ip_ << "] has request service with non-valid 'max_request' [" << request.max_repeats << "].");

  if (request.set_commands) {
    response.success = (setCommand(request.position_command, request.max_repeats) == 0);
  }
  return true;
}

bool qbSoftHandIndustryCommunicationHandler::setCommandsCallback(qb_softhand_industry_srvs::SetCommandsRequest &request, qb_softhand_industry_srvs::SetCommandsResponse &response) {
  ROS_ERROR_STREAM_COND_NAMED(request.max_repeats < 0, "communication_handler", "Device [" << ip_ << "] has request service with non-valid 'max_request' [" << request.max_repeats << "].");
  if (request.set_commands) {
    response.success = (setCommands(request.position_command, request.velocity_command, request.current_command, request.max_repeats) == 0);
  }
  return true;
}

ssize_t qbSoftHandIndustryCommunicationHandler::recvfrom(std::string &buffer) {
  buffer.resize(1024);
  // the protector lock should be performed also in the caller (i.e. read and write are related to the same task)
  ssize_t received = ::recvfrom(socket_descriptor_, &buffer[0], buffer.size(), 0, (sockaddr *)&socket_address_, &address_length_);
  buffer.resize(std::max<int>(0, received));
  return received;
}

ssize_t qbSoftHandIndustryCommunicationHandler::sendto(const std::string &buffer) {
  ssize_t sent = ::sendto(socket_descriptor_, buffer.c_str(), buffer.size(), 0, (const sockaddr *)&socket_address_, address_length_);
  return sent;
}

int qbSoftHandIndustryCommunicationHandler::sendAndParseResponse(const std::string &command) {
  float data = 0;
  return sendAndParseResponse(command, data);
}

int qbSoftHandIndustryCommunicationHandler::sendAndParseResponse(const std::string &command, float &parsed_response) {
  std::lock_guard<std::mutex> lock(*socket_protector_);
  if (sendto(command + ";\n\r") > 0) {
    int response_status = QBSOFTHAND_INDUSTRY_ERROR_RECVAGAIN;
    std::string response;
    while (response_status == QBSOFTHAND_INDUSTRY_ERROR_RECVAGAIN) {
      std::string buffer;
      if (recvfrom(buffer) <= 0) {
        return errno;
      }
      response += buffer;
      response_status = parseResponse(command, response, parsed_response);
    }
    return response_status;
  }
  ROS_WARN_STREAM_NAMED("qbSoftHand Industry Communication Handler", "errno has been set to '" << strerror(errno) << "' during 'sendto()'.");
  return errno;
}

int qbSoftHandIndustryCommunicationHandler::sendAndParseResponse(const std::string &command, const int &max_repeats, float &parsed_response){
  int status = EAGAIN;
  int i;
  for (i=0; i<std::max(1, max_repeats); i++) {
    status = sendAndParseResponse(command, parsed_response);
    if (status != EAGAIN) {
      break;
    }
    ROS_DEBUG_STREAM_NAMED("qbSoftHand Industry Communication Handler", "Response not parsed properly. Try to increase the max_repeats parameter.");
  }
  if (i>max_repeats){
     ROS_WARN_STREAM_THROTTLE_NAMED(10, "qbSoftHand Industry Communication Handler", "errno has been set to '" << strerror(status) << "' during 'recvfrom()', after  "<< max_repeats <<" repeats.");
  }
  return status;
}

int qbSoftHandIndustryCommunicationHandler::sendAndParseResponse(const std::string &command, const int &max_repeats) {
  float data = 0;
  return sendAndParseResponse(command, max_repeats, data);
}

int qbSoftHandIndustryCommunicationHandler::parseResponse(const std::string &command, const std::string &response, float &parsed_response){
  std::smatch match_data;
  if (std::regex_search(response, match_data, std::regex(";\n\r\002\\?;")) && match_data.prefix() == command) {
    std::cout << "ERROR: wrong command sent." << std::endl;
    return QBSOFTHAND_INDUSTRY_ERROR_BADPARAMS;
  }
  if (std::regex_search(response, match_data, std::regex(";\n\r;")) && match_data.prefix() == command) {
    return 0;
  }
  if (std::regex_search(response, match_data, std::regex(";\n\r([-]?[[:digit:]]+\\.?[[:digit:]]*(?:e-?[[:digit:]]+)?);")) && match_data.size() > 1 && match_data.prefix() == command) {
    parsed_response = std::stof(match_data[1]);
    return 0;
  }
  ROS_DEBUG_STREAM_NAMED("[qbSoftHand Industry control]", "response not complete.");
  return QBSOFTHAND_INDUSTRY_ERROR_RECVAGAIN;
}
