/***
 *  Software License Agreement: BSD 3-Clause License
 *
 *  Copyright (c) 2018-2019, qbrobotics®
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 *  following conditions are met:
 *
 *  * Redistributions of source code must retain the above copyright notice, this list of conditions and the
 *    following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote
 *    products derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 *  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef QB_SOFTHAND_INDUSTRY_HARDWARE_INTERFACE_H
#define QB_SOFTHAND_INDUSTRY_HARDWARE_INTERFACE_H

// ROS libraries
#include <ros/ros.h>
#include <hardware_interface/robot_hw.h>
#include <hardware_interface/posvel_command_interface.h>
#include <hardware_interface/joint_state_interface.h>
#include <hardware_interface/joint_command_interface.h>
#include <joint_limits_interface/joint_limits.h>
#include <joint_limits_interface/joint_limits_interface.h>
#include <joint_limits_interface/joint_limits_rosparam.h>

#include <controller_manager/controller_manager.h>

// internal libs
#include <qb_softhand_industry_msgs/qb_softhand_industry_msgs.h>
#include <qb_softhand_industry_srvs/qb_softhand_industry_srvs.h>
#include <qb_softhand_industry_hardware_interface/qb_softhand_industry_transmission_interface.h>


namespace qb_softhand_industry_hardware_interface {
/**
 *
 */
class qbSoftHandIndustryHW : public hardware_interface::RobotHW {
  public:
    /**
     *
     */
    qbSoftHandIndustryHW();

    /**
     *
     */
    ~qbSoftHandIndustryHW() override;

    /**
     * The init function is called to initialize the RobotHW from a non-realtime thread. In particular TODO check ...
     * \param root_nh A NodeHandle in the root of the caller namespace.
     * \param robot_hw_nh A NodeHandle in the namespace from which the RobotHW should read its configuration.
     * \return \p true on success.
     */
    bool init(ros::NodeHandle &root_nh, ros::NodeHandle &robot_hw_nh) override;

    /**
     * Read actuator state from the hardware, propagate it to joint states and publish the whole device state to a
     * namespaced \p "~state" topic (each instance should publish on its own topic). TODO check ...
     * \param time The current time.
     * \param period The time passed since the last call to this method, i.e. the control period.
     * \sa getMeasurements(), publish()
     */
    void read(const ros::Time &time, const ros::Duration &period) override;

    /**
     * Enforce joint limits for all registered joint limit interfaces, propagate joint commands to actuators, and send
     * actuator commands to the hardware. TODO check ...
     * \param time The current time.
     * \param period The time passed since the last call to this method, i.e. the control period.
     * \sa qb_device_joint_limits_interface::qbDeviceJointLimitsResources::enforceLimits(), setCommands()
     */
    void write(const ros::Time &time, const ros::Duration &period) override;

    /**
     * Control loop
     * \param e The timer event
     */
    void update(const ros::TimerEvent& e);

  protected:
    hardware_interface::JointStateInterface joint_state_interface_;
    hardware_interface::PositionJointInterface position_joint_interface_;     // to control the device sending position cmds
    
    joint_limits_interface::JointLimits limits_;
    joint_limits_interface::PositionJointSaturationInterface position_joint_saturation_interface_;

 private:
    ros::AsyncSpinner spinner_;
    ros::NodeHandle node_handle_;
    ros::Publisher state_publisher_;
    ros::Duration elapsed_time_;
    std::shared_ptr<controller_manager::ControllerManager> controller_manager_;

    int max_repeats_;
    std::string device_name_;

    double actual_joint_position_;          // variable for reading position - used by joint_state_interface
    double actual_joint_velocity_;          // variable for reading velocity - used by joint_state_interface
    double actual_joint_effort_;            // variable for reading effort   - used by joint_state_interface
    double joint_position_command_;         // variable for sending position command


    double actual_actuator_position_;
    double actual_actuator_velocity_;
    double actual_actuator_effort_;
    double actuator_position_command_;      // variable for sending position command

    qb_softhand_industry_transmission_interface::qbSoftHandIndustryVirtualTransmission transmission_;


    // the transmission interface is used just to convert from actuator values [ticks] to joint states [radians] or [0,1]
    transmission_interface::ActuatorToJointStateInterface actuator_to_joint_state_;
    transmission_interface::JointToActuatorPositionInterface joint_to_actuator_position_;

    transmission_interface::ActuatorData actuator_state_data_;
    transmission_interface::ActuatorData actuator_cmd_data_;
    transmission_interface::JointData joint_state_data_;
    transmission_interface::JointData joint_cmd_data_;


    std::map<std::string, ros::ServiceClient> device_services_;

    /**
     * @brief Get the Measurements from the device, by advertising the server 
     * 
     * @param position 
     * @param velocity 
     * @param current 
     * @return true if the response of the server is true, false otherwise 
     */
    bool getMeasurements(float &position, float &velocity, float &current, ros::Time & stamp);

    /**
     * @brief Initialize the services needed for getting/setting cmds from/to the device
     * 
     */
    void initaliazeServicesAndWait();

    /**
     * @brief publish the state(position, velocity and current) of the qb SoftHand Industry
     * 
     */
    void publish();

    /**
     * @brief set commands to control qb SoftHand Industry
     * 
     */
    bool setCommands(float &position, float &velocity, float &current);

    /**
     * @brief Wait untill all the necessary servers exists
     * 
     */
    void waitForSrvs();
};
}  // namespace qb_softhand_industry_hardware_interface

#endif // QB_SOFTHAND_INDUSTRY_HARDWARE_INTERFACE_H