/***
 *  Software License Agreement: BSD 3-Clause License
 *
 *  Copyright (c) 2018-2022, qbrobotics®
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 *  following conditions are met:
 *
 *  * Redistributions of source code must retain the above copyright notice, this list of conditions and the
 *    following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote
 *    products derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 *  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <qb_softhand_industry_hardware_interface/qb_softhand_industry_hardware_interface.h>

using namespace qb_softhand_industry_hardware_interface;

qbSoftHandIndustryHW::qbSoftHandIndustryHW()
  : spinner_(1) { //TODO: pass this as launch parameter
      spinner_.start();
      initaliazeServicesAndWait();    // initialize all services needed to communicate with the device
}

qbSoftHandIndustryHW::~qbSoftHandIndustryHW() {

}

bool qbSoftHandIndustryHW::getMeasurements(float &position, float &velocity, float &current, ros::Time &stamp) {
  qb_softhand_industry_srvs::GetMeasurements srv;
  srv.request.get_position = true;
  srv.request.get_current = true;
  srv.request.get_velocity = true;
  srv.request.max_repeats = max_repeats_;
  device_services_.at("get_measurements").call(srv);

  position = srv.response.position;
  velocity = srv.response.velocity;
  current = srv.response.current;
  stamp = srv.response.stamp;
  return srv.response.success;
}

bool qbSoftHandIndustryHW::init(ros::NodeHandle &root_nh, ros::NodeHandle &robot_hw_nh) {
  node_handle_ = robot_hw_nh;
  max_repeats_ = robot_hw_nh.param<int>("max_repeats", 3);
  device_name_ = robot_hw_nh.param<std::string>("device_name", "shin");
  std::string joint_name = device_name_+"_synergy_joint";

  // Create joint_state_interface for synergy joint
  hardware_interface::JointStateHandle joint_state_handle_synergy_joint(joint_name, &actual_joint_position_, &actual_joint_velocity_, &actual_joint_effort_);
  joint_state_interface_.registerHandle(joint_state_handle_synergy_joint);

  // Create position joint interface as joint accepts position command.
  hardware_interface::JointHandle joint_position_handle_synergy_joint(joint_state_handle_synergy_joint, &joint_position_command_);
  position_joint_interface_.registerHandle(joint_position_handle_synergy_joint); 

  // Initialize Transmission interface stuff
  actuator_state_data_.position.push_back(&actual_actuator_position_);
  actuator_state_data_.velocity.push_back(&actual_actuator_velocity_);
  actuator_state_data_.effort.push_back(&actual_actuator_effort_);
  actuator_cmd_data_.position.push_back(&actuator_position_command_);

  joint_state_data_.position.push_back(&actual_joint_position_);
  joint_state_data_.velocity.push_back(&actual_joint_velocity_);
  joint_state_data_.effort.push_back(&actual_joint_effort_);
  joint_cmd_data_.position.push_back(&joint_position_command_);

  // Set variables for Transmission interface
  actuator_to_joint_state_.registerHandle(transmission_interface::ActuatorToJointStateHandle(device_name_+"state_transmission",
                                                                                                           &transmission_,
                                                                                                           actuator_state_data_,
                                                                                                           joint_state_data_));
  joint_to_actuator_position_.registerHandle(transmission_interface::JointToActuatorPositionHandle(device_name_+"joint_transmission",
                                                                                                   &transmission_,
                                                                                                   actuator_cmd_data_,
                                                                                                   joint_cmd_data_));

  if(!joint_limits_interface::getJointLimits(joint_name, node_handle_, limits_)){
    ROS_ERROR_STREAM_NAMED("qb SoftHand Industry HW", "Unable to get joint(s) limit(s).");
    return false;
  }
  // Create joint position interface limit
  joint_limits_interface::PositionJointSaturationHandle joint_position_limits_handle_synergy_joint(joint_position_handle_synergy_joint, limits_);
  position_joint_saturation_interface_.registerHandle(joint_position_limits_handle_synergy_joint);


  // register all joint interfaces
  registerInterface(&joint_state_interface_);
  registerInterface(&position_joint_interface_);
  registerInterface(&position_joint_saturation_interface_);

  // Advertise the publisher that update the device state
  state_publisher_ = robot_hw_nh.advertise<qb_softhand_industry_msgs::StateStamped>("state", 1);

  return true;
}

void qbSoftHandIndustryHW::initaliazeServicesAndWait(){
  device_services_["get_measurements"] = node_handle_.serviceClient<qb_softhand_industry_srvs::GetMeasurements>("/qb_softhand_industry_communication_handler/get_measurements", true);
  device_services_["set_command"] = node_handle_.serviceClient<qb_softhand_industry_srvs::SetCommand>("/qb_softhand_industry_communication_handler/set_command", true);
  device_services_["set_commands"] = node_handle_.serviceClient<qb_softhand_industry_srvs::SetCommands>("/qb_softhand_industry_communication_handler/set_commands", true);
  waitForSrvs();
}

void qbSoftHandIndustryHW::publish(){
  qb_softhand_industry_msgs::StateStamped msg;
  // actuator infos
  msg.device_data.actuator.name = device_name_;
  msg.device_data.actuator.position = actual_actuator_position_;
  msg.device_data.actuator.velocity = actual_actuator_velocity_;
  msg.device_data.actuator.effort = actual_actuator_effort_;
  msg.device_data.actuator.command = actuator_position_command_;

  msg.header.stamp = ros::Time::now();
  state_publisher_.publish(msg);
}

void qbSoftHandIndustryHW::read(const ros::Time &time, const ros::Duration &period) {
  float actual_pos, actual_vel, actual_effort;
  ros::Time stamp;
  // get measurements from device
  getMeasurements(actual_pos, actual_vel, actual_effort, stamp);
  actual_actuator_position_ = (double)actual_pos;
  actual_actuator_velocity_ = (double)actual_vel;
  actual_actuator_effort_ = (double)actual_effort;
  // from actuator to joint
  actuator_to_joint_state_.propagate();

  publish();
}

bool qbSoftHandIndustryHW::setCommands(float &position, float &velocity, float &current){
  if(device_services_.at("set_commands")){
    qb_softhand_industry_srvs::SetCommands srv;
    srv.request.set_commands = true;
    srv.request.position_command = position;
    srv.request.velocity_command = velocity;
    srv.request.current_command = current;
    srv.request.max_repeats = max_repeats_;
    device_services_.at("set_commands").call(srv);

    if(!srv.response.success) {
      ROS_ERROR_STREAM_THROTTLE_NAMED(60 ,"qb SoftHand Industry HW", "Cannot send commands to device.");
      return false;
    }
    return true;
  }
  ROS_WARN_STREAM_NAMED("qb SoftHand Industry HW", "Service [set_commands] seems no longer advertised.");
}

void qbSoftHandIndustryHW::waitForSrvs(){
  for(auto &service:device_services_){
    service.second.waitForExistence();
  }
  ROS_INFO_STREAM_NAMED("qb SoftHand Industry HW", "All necessary servers exist in [qb_softhand_industry_communication_handler]");
}

void qbSoftHandIndustryHW::write(const ros::Time &time, const ros::Duration &period) {
  position_joint_saturation_interface_.enforceLimits(period);
  // from joint to actuator
  joint_to_actuator_position_.propagate();

  float pos_cmd = actuator_position_command_;
  float vel_cmd = 3200;
  float curr_cmd = 450;
  setCommands(pos_cmd, vel_cmd, curr_cmd);
}

void qbSoftHandIndustryHW::update(const ros::TimerEvent& e) {
    elapsed_time_ = ros::Duration(e.current_real - e.last_real);
    read(ros::Time::now(), elapsed_time_);
    controller_manager_->update(ros::Time::now(), elapsed_time_);
    write(ros::Time::now(), elapsed_time_);
}
