#ifndef TOPOSENS_ECHO_DRIVER_LIB_UTILS_H
#define TOPOSENS_ECHO_DRIVER_LIB_UTILS_H

#include <iomanip>
#include <sstream>
#include <string>

#include <toposens/sensor_lib.h>

namespace toposens_echo_driver
{
/// @brief Inits the driver on given \p can_device with hard-coded data rate
// NOLINTNEXTLINE (cppcoreguidelines-avoid-magic-numbers)
void InitCanInterface(const std::string& can_device, int data_rate = 1000000);
void DeinitCanInterface();

/// @brief Inits the driver on given \p uart_device with hard-coded data rate
// NOLINTNEXTLINE (cppcoreguidelines-avoid-magic-numbers)
void InitUARTInterface(const std::string& uart_device, int data_rate = 0010002);
void DeinitUARTInterface();

/// @brief Requesting single shot measurement from first sensor in known sensors array
/// @details This is a blocking call!
auto RequestSessionData() -> Sensor_Session_t*;

/// @brief Checks if @p TargetSensor_u16 is known and sets current target accordingly
/// @returns true if successful, false otherwise
/// @todo This would likely make sense to be moved to the Device Driver library itself!
auto SetTargetSensor(uint16_t TargetSensor_u16) -> bool;

///@{
/** Group of functions to log information to ROS console. */

/// @brief Read current configuration from sensor and log to ROS console.
void LogSettings();

/// @brief Get currently known sensors from driver and log to ROS console every @p log_interval
/// seconds
// NOLINTNEXTLINE (cppcoreguidelines-avoid-magic-numbers)
void LogKnownSensors(double log_interval = 3.0);

/// @brief Read all version information (such as bootloader version, app versionetc.) from sensor
/// and log to ROS console
void LogVersions();

/// @brief Log given @p session data to ROS concole.
/// @details Logs all 1D as well as all 3D Points
void LogSessionData(const Sensor_Session_t* session);

///@}

/// @brief Sets log level with the sensor and registers callback
void ConfigureSensorLogMessages(void (*Callback)(uint16_t Sender_u16, uint8_t* ReceivedPayload_pu8),
                                LogLevel_t log_level);

/// @brief Callback handling log messages from sensor
void LogMsgCallback(uint16_t SenderId_u16, uint8_t* ReceivedPayload_pu8);

/// @brief Convert given @p i to hex string
/// @note Don't be tempted to change the parameter type to uint8_t!
/// @todo Potentially move this even to a separate utils header
inline auto toHexString(const uint16_t i) -> std::string
{
  std::stringstream hex_ss;
  hex_ss << "0x" << std::setfill('0') << std::setw(2) << std::uppercase << std::hex << i;
  return hex_ss.str();
};

}  // namespace toposens_echo_driver

#endif  // TOPOSENS_ECHO_DRIVER_LIB_UTILS_H
