// Copyright 2016 Proyectos y Sistemas de Mantenimiento SL (eProsima).
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

grammar IDL;

@header {
    //package com.eprosima.idl.parser.grammar;
    
    import com.eprosima.idl.context.Context;
    import com.eprosima.idl.generator.manager.TemplateManager;
    import com.eprosima.idl.generator.manager.TemplateGroup;
    import com.eprosima.idl.generator.manager.TemplateUtil;
    import com.eprosima.idl.parser.typecode.*;
    import com.eprosima.idl.parser.tree.*;
    import com.eprosima.idl.util.Pair;
    import com.eprosima.idl.parser.strategy.DefaultErrorStrategy;
    import com.eprosima.idl.parser.listener.DefaultErrorListener;
    import com.eprosima.idl.parser.exception.ParseException;
   
    import java.util.Vector;
}

@parser::members {
    private TemplateManager tmanager = null;
    private Context ctx = null;

    public Context getContext_()
    {
        return ctx;
    }
}

@lexer::members{
    Context ctx = null;
    
    public void setContext(Context _ctx)
    {
        ctx = _ctx;
    }
}

specification [Context context, TemplateManager templatemanager, TemplateGroup maintemplates] returns [Specification spec = null]
@init{
    //! Used to catch each definition grammar element in the whole IDL file.
    Pair<Vector<Definition>, TemplateGroup> dtg = null;
    List<Definition> specificationChildren = new ArrayList<Definition>();
    ctx = context;
    tmanager = templatemanager;

    // Set error handler
    DefaultErrorListener listener = new DefaultErrorListener(ctx);
    this.setErrorHandler(DefaultErrorStrategy.INSTANCE);
    // Select listener for errors.
    ((Lexer)this._input.getTokenSource()).removeErrorListeners();
    ((Lexer)this._input.getTokenSource()).addErrorListener(listener);
    this.removeErrorListeners();
    this.addErrorListener(listener);
}
    :   import_decl*
	(
		definition[null]
		{
			dtg=$definition.dtg;
			if (dtg!=null) { 
				if(maintemplates != null) {
					maintemplates.setAttribute("definitions", dtg.second());
				}
                for(int count = 0; count < dtg.first().size(); ++count)
                {
                    ctx.addDefinition(dtg.first().get(count));
                    specificationChildren.add(dtg.first().get(count));
                }
			} 
		}
	)+
	{
        if(getNumberOfSyntaxErrors() == 0)
        {
            $spec = new Specification();
            $spec.setDefinitions(specificationChildren);
        }
	}
    ;

/*!
 * @brief This grammar element represents a global definition: type declaration, interface, module, etc...
 * @return A pair with the object that represents the definition (interface for other objects like module grammar element)
 * and the template group generated by the definition (in fact the template group generated by the element who inherits).
 */	
definition [Vector<Annotation> annotations] returns [Pair<Vector<Definition>, TemplateGroup> dtg = null]
@init {
    // TODO Cambiar esto. No me gusta la forma.
	
    Vector<Definition> vector = new Vector<Definition>();
    Pair<Vector<TypeDeclaration>, TemplateGroup> tdtg = null;
    Pair<ConstDeclaration, TemplateGroup> cdtg = null;
    Pair<com.eprosima.idl.parser.tree.Exception, TemplateGroup> etg = null;
    Pair<Interface, TemplateGroup> itg = null;
    Pair<Module, TemplateGroup> mtg = null;
    Pair<AnnotationDeclaration, TemplateGroup> atg = null;

    if(annotations == null) annotations = new Vector<Annotation>();
}
    :   type_decl[annotations] SEMICOLON { tdtg=$type_decl.returnPair; if(tdtg!=null){ for(TypeDeclaration tydl : tdtg.first()) vector.add(tydl); $dtg = new Pair<Vector<Definition>, TemplateGroup>(vector, tdtg.second());}}  // Type Declaration
    |   const_decl SEMICOLON { cdtg=$const_decl.returnPair; if(cdtg!=null){ vector.add(cdtg.first()); $dtg = new Pair<Vector<Definition>, TemplateGroup>(vector, cdtg.second());}} // Const Declaration
    |   except_decl SEMICOLON { etg=$except_decl.returnPair; if(etg!=null){ vector.add(etg.first()); $dtg = new Pair<Vector<Definition>, TemplateGroup>(vector, etg.second());}} // Exception.
    |   interface_or_forward_decl[annotations] SEMICOLON { itg=$interface_or_forward_decl.itg; if(itg!=null){ vector.add(itg.first()); $dtg = new Pair<Vector<Definition>, TemplateGroup>(vector, itg.second());}} // Interface
    |   module SEMICOLON { mtg=$module.returnPair; if(mtg!=null){ vector.add(mtg.first()); $dtg = new Pair<Vector<Definition>, TemplateGroup>(vector, mtg.second());}} // Module
    |   value SEMICOLON
    |   type_id_decl SEMICOLON
    |   type_prefix_decl SEMICOLON
    |   event SEMICOLON
    |   component SEMICOLON
    |   home_decl SEMICOLON
	|	annotation_decl SEMICOLON { atg=$annotation_decl.returnPair; if(atg!=null){ vector.add(atg.first()); $dtg = new Pair<Vector<Definition>, TemplateGroup>(vector, atg.second());}}
	|   annotation_appl
        {
            annotations.add($annotation_appl.annotation);
        }
        aux_definition[annotations]{$dtg=$aux_definition.dtg;}
    ;

aux_definition [Vector<Annotation> annotations] returns [Pair<Vector<Definition>, TemplateGroup> dtg = null]
    : definition[annotations] {$dtg=$definition.dtg;}
    ;

/*!
 * @brief This grammar expression catches a module definition.
 * @return This grammar expression returns the Module object as DefinitionContainer to be stored.
 * Also the TemplateGroup of module is returned.
 */
module returns [Pair<Module, TemplateGroup> returnPair = null]
@init{
    Module moduleObject = null;
    TemplateGroup moduleTemplates = null;
    TemplateGroup tg = null;
    // Store old namespace.
    String name = null, old_scope = ctx.getScope();
    Token tk = null;
}
    :   ( KW_MODULE )
    {
        tk = _input.LT(1);
    }
	identifier
	{
		name=$identifier.id;
        // Check if the module already was defined.
        moduleObject = ctx.existsModule(ctx.getScope() + "::" + name);

        if(moduleObject != null)
        {
            // Add the module to the context.
            ctx.addModule(moduleObject);
        }
        else
        {
            // Create the Module object.
            moduleObject = new Module(ctx.getScopeFile(), ctx.isInScopedFile(), ctx.getScope(), name, tk); 
        }

		if(ctx.isInScopedFile() || ctx.isScopeLimitToAll()) {
			if(tmanager != null) {
				moduleTemplates = tmanager.createTemplateGroup("module");
				moduleTemplates.setAttribute("ctx", ctx);
				// Set the module object to the TemplateGroup of the module.
				moduleTemplates.setAttribute("module", moduleObject);
			}
		}
		
		// Update to a new namespace.
		if(old_scope.isEmpty())
			ctx.setScope(name);
		else
			ctx.setScope(old_scope + "::" + name);
	}
	// Each definition is stored in the Module and each TemplateGroup is set as attribute in the TemplateGroup of the module.
	LEFT_BRACE
	definition_list[moduleObject]{ tg=$definition_list.dlTemplates; if(moduleTemplates!=null && tg!=null)moduleTemplates.setAttribute("definition_list", tg);}
	RIGHT_BRACE
	{
	    // Set the old namespace.
	    ctx.setScope(old_scope);
	    // Create the returned data.
		$returnPair = new Pair<Module, TemplateGroup>(moduleObject, moduleTemplates);
	}
    ;
	
	
/*!
 * @brief This grammar expression catches a list of definitions.
 * @param dc An object that inherits from DefinitionContainer. This object could store all definitions.
 * @return This grammar expression returns the template group with all template groups of definitions.
 */
definition_list [DefinitionContainer dc] returns [TemplateGroup dlTemplates]
@init{
    Pair<Vector<Definition>, TemplateGroup> dtg = null;
	if(tmanager != null) {
		$dlTemplates = tmanager.createTemplateGroup("definition_list");
	}
}
	:   (
		definition[null]
		{
			dtg=$definition.dtg;
			if(dtg!=null)
            {
                for(int count = 0; count < dtg.first().size(); ++count)
                    dc.add(dtg.first().get(count));

				if($dlTemplates != null && dtg.second() != null)
                {
                    // Set parent
                    dtg.second().setAttribute("parent", dc);
                    // Print template into definitions rule
					$dlTemplates.setAttribute("definitions", dtg.second());
				}
			}
		}
		)+
	;

/*!
 * @brief This grammar expression catches currently a interface declaration.
 * @return This grammar expression returns the interface object and its template group.
 */
interface_or_forward_decl [Vector<Annotation> annotations] returns [Pair<Interface, TemplateGroup> itg = null]
    :   interface_decl[annotations] {$itg = $interface_decl.returnPair;}
    |   forward_decl
    ;

interface_decl [Vector<Annotation> annotations] returns [Pair<Interface, TemplateGroup> returnPair = null]
@init {
    Token tk = null;   
    Interface interfaceObject = null;
    TemplateGroup interfaceTemplates = null;
    TemplateGroup tg = null;
    String name = null, old_scope = ctx.getScope();
}
    :   ( ( KW_ABSTRACT | KW_LOCAL )? ( KW_INTERFACE )
        {
            tk = _input.LT(1);
        }
		identifier
 	    {
			name=$identifier.id;
           // Create the Interface object.
           interfaceObject = ctx.createInterface(name, tk);

           // Add annotations.
           for(Annotation annotation : annotations)
               interfaceObject.addAnnotation(ctx, annotation);

           if(ctx.isInScopedFile() || ctx.isScopeLimitToAll())
           {
			   if(tmanager != null) {
					interfaceTemplates = tmanager.createTemplateGroup("interface");
					interfaceTemplates.setAttribute("ctx", ctx);
					// Set the the interface object to the TemplateGroup of the module.
					interfaceTemplates.setAttribute("interface", interfaceObject);
				}
            }

	       // Update to a new namespace.
           if(old_scope.isEmpty())
	           ctx.setScope(name);
           else
	           ctx.setScope(old_scope + "::" + name);
        }
		( interface_inheritance_spec[interfaceObject] )?
		LEFT_BRACE interface_body[interfaceObject]{ tg=$interface_body.elTemplates; if(interfaceTemplates!=null && tg!=null)interfaceTemplates.setAttribute("export_list", tg);} RIGHT_BRACE )
		{
	       // Set the old namespace.
	       ctx.setScope(old_scope);
           // Create the returned data.
           $returnPair = new Pair<Interface, TemplateGroup>(interfaceObject, interfaceTemplates);
        }
    ;

forward_decl
    :   ( KW_ABSTRACT | KW_LOCAL )? ( KW_INTERFACE ) ID
    ;


/*!
 * @brief This grammar expression catches a list of exports.
 * @param dc An object that inherits from ExportContainer. This object could store all exports.
 * @return This grammar expression returns the template group with all template groups of exports.
 */
interface_body [ExportContainer ec] returns [TemplateGroup elTemplates]
@init{
        Pair<Vector<Export>, TemplateGroup> etg = null;
		if(tmanager != null) {
			$elTemplates = tmanager.createTemplateGroup("export_list");
		}
}
    :   (
		export[null]
		{
			etg=$export.etg;
			if(etg!=null)
            {
                for(int count = 0; count < etg.first().size(); ++count)
                {
				    ec.add(etg.first().get(count));
				    etg.first().get(count).resolve(ctx);
                }

				if($elTemplates != null && etg.second() != null)
                {
                    // Add parent
                    etg.second().setAttribute("parent", ec);
                    // Print template into exports rule
					$elTemplates.setAttribute("exports", etg.second());
				}
			}
		}
		)*
    ;

/*!
 * @brief This grammar element represents an export of a interface: type declaration, operation, etc...
 * @return A pair with the object that represents the export (interface for other objects like operation grammar element)
 * and the template group generated by the export (in fact the template group generated by the element who inherits).
 */
export [Vector<Annotation> annotations] returns [Pair<Vector<Export>, TemplateGroup> etg = null]
@init {
        // TODO Cambiar esto. No me gusta la forma.
        Vector<Export> vector = new Vector<Export>();
        Pair<Vector<TypeDeclaration>, TemplateGroup> tetg = null;
        Pair<ConstDeclaration, TemplateGroup> cetg = null;
        Pair<Operation, TemplateGroup> oetg = null;
        Pair<com.eprosima.idl.parser.tree.Exception, TemplateGroup> eetg = null;
        
        if(annotations == null) annotations = new Vector<Annotation>();
}
    :   type_decl[annotations] SEMICOLON { tetg=$type_decl.returnPair; if(tetg!=null){ for(TypeDeclaration tydl : tetg.first()) vector.add(tydl); $etg = new Pair<Vector<Export>, TemplateGroup>(vector, tetg.second());} }  // Type Declaration
    |   const_decl SEMICOLON { cetg=$const_decl.returnPair; if(cetg!=null){ vector.add(cetg.first()); $etg = new Pair<Vector<Export>, TemplateGroup>(vector, cetg.second());}} // Const Declaration
    |   except_decl SEMICOLON { eetg=$except_decl.returnPair; if(eetg!=null){ vector.add(eetg.first()); $etg = new Pair<Vector<Export>, TemplateGroup>(vector, eetg.second());}}  // Exception
    |   attr_decl SEMICOLON
	{ System.out.println("WARNING (File " + ctx.getFilename() + ", Line " + (_input.LT(1) != null ? _input.LT(1).getLine() - ctx.getCurrentIncludeLine() : "1") + "): Attribute declarations are not supported. Ignoring..."); }
    |   op_decl[annotations] SEMICOLON { oetg=$op_decl.returnPair; if(oetg!=null){ vector.add(oetg.first()); $etg = new Pair<Vector<Export>, TemplateGroup>(vector, oetg.second());}}  // Operation
    |   type_id_decl SEMICOLON
    |   type_prefix_decl SEMICOLON
	|	annotation_appl
        {
            annotations.add($annotation_appl.annotation);
        }
        aux_export[annotations]{$etg=$aux_export.etg;}
    ;

aux_export [Vector<Annotation> annotations] returns [Pair<Vector<Export>, TemplateGroup> etg = null]
    : export[annotations] {$etg=$export.etg;}
    ;

interface_inheritance_spec [Interface interfaceObject]
@init{
        Vector<Pair<String, Token>> iflist = null;
}
    :   COLON scoped_name_list { iflist=$scoped_name_list.retlist; }
    {
        for(Pair<String, Token> pair : iflist)
        {
            Interface base = ctx.getInterface(pair.first());

            if(base != null)
            {
                if(!$interfaceObject.addBase(base))
                    throw new ParseException(pair.second(), " is duplicated.");
            }
            else
            {
	           throw new ParseException(pair.second(), "was not defined previously");
            }
        }
    }
    ;

interface_name
    :   scoped_name
    ;
	
scoped_name_list returns [Vector<Pair<String, Token>> retlist = null]
@init{
   $retlist = new Vector<Pair<String, Token>>();
   Pair<String, Token> pair = null;
}
	:    scoped_name{ pair=$scoped_name.pair; $retlist.add(pair);} (COMA scoped_name{ pair=$scoped_name.pair; $retlist.add(pair);})*
	;

scoped_name returns [Pair<String, Token> pair = null]
@init{
    String literalStr = "";
    Token tk = _input.LT(1);
}
:   ( {literalStr += _input.LT(1).getText();} DOUBLE_COLON )?
	  {literalStr += _input.LT(1).getText();} ID /* identifier */
	( {literalStr += _input.LT(1).getText();} DOUBLE_COLON identifier { literalStr+=$identifier.id; } )*
    {$pair = new Pair<String, Token>(literalStr, tk);}
    ;

value
@init{
    System.out.println("WARNING (File " + ctx.getFilename() + ", Line " + (_input.LT(1) != null ? _input.LT(1).getLine() - ctx.getCurrentIncludeLine() : "1") + "): ValueType declarations are not supported. Ignoring...");
}
    :   ( value_decl | value_abs_decl | value_box_decl | value_forward_decl )
    ;

value_forward_decl
    :   ( KW_ABSTRACT )? KW_VALUETYPE ID
    ;

value_box_decl
    :   KW_VALUETYPE ID type_spec
    ;

value_abs_decl
    :   KW_ABSTRACT KW_VALUETYPE ID value_inheritance_spec LEFT_BRACE export[null]* RIGHT_BRACE
    ;

value_decl
    :   value_header LEFT_BRACE  value_element* RIGHT_BRACE
    ;

value_header
    :   ( KW_CUSTOM )? KW_VALUETYPE ID value_inheritance_spec
    ;

value_inheritance_spec
    :   ( COLON ( KW_TRUNCATABLE )? value_name ( COMA value_name )* )? ( KW_SUPPORTS interface_name ( COMA interface_name )* )?
    ;

value_name
    :   scoped_name
    ;

value_element
    :   ( export[null] |  state_member | init_decl )
    ;

state_member
    :   ( KW_PUBLIC | KW_PRIVATE ) type_spec declarators SEMICOLON
    ;

init_decl
    :   KW_FACTORY ID LEFT_BRACKET ( init_param_decls )? RIGHT_BRACKET ( raises_expr )? SEMICOLON
    ;

init_param_decls
    :   init_param_decl ( COMA init_param_decl )*
    ;

init_param_decl
    :   init_param_attribute param_type_spec simple_declarator
    ;

init_param_attribute
    :   KW_IN
    ;

const_decl returns [Pair<ConstDeclaration, TemplateGroup> returnPair = null]
@init {
    ConstDeclaration constDecl = null;
    TypeCode typecode = null;
    String constName = null, constValue = null;
	TemplateGroup constTemplates = null;
	if(tmanager != null) {
		constTemplates = tmanager.createTemplateGroup("const_decl");
	}
    Token tk = null;
}
    :   KW_CONST const_type { typecode=$const_type.typecode; tk = _input.LT(1);} identifier { constName=$identifier.id; } EQUAL const_exp { constValue=$const_exp.literalStr; }
	{
		if(typecode != null)
        {
			constDecl = new ConstDeclaration(ctx.getScopeFile(), ctx.isInScopedFile(), ctx.getScope(), constName, typecode, constValue, tk);

			if(constTemplates != null)
            {
				constTemplates.setAttribute("ctx", ctx);
				constTemplates.setAttribute("const", constDecl);
			}			

			$returnPair = new Pair<ConstDeclaration, TemplateGroup>(constDecl, constTemplates);
       }
    }
    ;

// TODO Not supported fixed types: Show warning
const_type returns [TypeCode typecode = null]
@init{
    Pair<String, Token> pair = null;
}
    :   integer_type { $typecode = $integer_type.typecode; }
    |   char_type { $typecode = $char_type.typecode; }
    |   wide_char_type { $typecode = $wide_char_type.typecode; }
    |   boolean_type { $typecode = $boolean_type.typecode; }
    |   floating_pt_type { $typecode = $floating_pt_type.typecode; }
    |   string_type { $typecode = $string_type.typecode; }
    |   wide_string_type { $typecode = $wide_string_type.typecode; }
    |   fixed_pt_const_type
    |   scoped_name
	    {
		   pair = $scoped_name.pair;
	       // Find typecode in the global map.
	       $typecode = ctx.getTypeCode(pair.first());
	       
	       if($typecode == null)
	           throw new ParseException(pair.second(), "was not defined previously");
	    }
    |   octet_type { $typecode = $octet_type.typecode; }
    ;

/*   EXPRESSIONS   */
	
const_exp returns [String literalStr = null]
@init{
    String aux = null;
}
    :   or_expr { $literalStr = $or_expr.literalStr; }
    ;

or_expr returns [String literalStr = null]
@init{
    String aux = null;
}
    :   xor_expr { $literalStr = $xor_expr.literalStr; }
	( 
		{$literalStr += _input.LT(1).getText();}
		PIPE
		xor_expr
		{ aux=$xor_expr.literalStr; $literalStr += aux;}
	)*
    ;

xor_expr returns [String literalStr = null]
@init{
    String aux = null;
}
    :   and_expr { $literalStr = $and_expr.literalStr; }
	(
		{$literalStr += _input.LT(1).getText();}
		CARET
		and_expr
		{ aux=$and_expr.literalStr; $literalStr += aux;}
	)*
    ;

and_expr returns [String literalStr = null]
@init{
    String aux = null;
}
    :   shift_expr { $literalStr = $shift_expr.literalStr; }
	(
		{$literalStr += _input.LT(1).getText();}
		AMPERSAND
		shift_expr
		{ aux=$shift_expr.literalStr; $literalStr += aux;}
	)*
    ;

shift_expr returns [String literalStr = null]
@init{
    String aux = null;
}
    :   add_expr { $literalStr = $add_expr.literalStr; }
	(
		{$literalStr += _input.LT(1).getText();}
		( RIGHT_SHIFT | LEFT_SHIFT )
		add_expr
	    { aux=$add_expr.literalStr; $literalStr += aux;}
	)*
    ;

add_expr returns [String literalStr = null]
@init{
    String aux = null;
}
   :   mult_expr { $literalStr = $mult_expr.literalStr; }
	(
		{$literalStr += _input.LT(1).getText();}
		( PLUS | MINUS )
		mult_expr
		{ aux=$mult_expr.literalStr; $literalStr += aux;}
	)*
    ;

mult_expr returns [String literalStr = null]
@init{
    String aux = null;
}
    :   unary_expr  { $literalStr = $unary_expr.literalStr; }
	(
		{$literalStr += _input.LT(1).getText();}
		( '*' | SLASH | PERCENT )
		unary_expr
		{ aux=$unary_expr.literalStr; $literalStr += aux;}
	)*
    ;

unary_expr returns [String literalStr = null]
@init{
    String aux = null;
}
    :   
		{$literalStr = _input.LT(1).getText();}
		( MINUS | PLUS | TILDE )
		primary_expr
		{ aux=$primary_expr.literalStr; $literalStr += aux;}
    |   primary_expr { $literalStr = $primary_expr.literalStr; }
    ;

//unary_operator
//    :   ( MINUS | PLUS | TILDE )
//    ;

primary_expr  returns [String literalStr = null]
@init{
    String aux = null;
}
    :   scoped_name { $literalStr = $scoped_name.pair.first(); }
    |   literal  { $literalStr = $literal.pair.first(); }
    |   {$literalStr = _input.LT(1).getText();}
		LEFT_BRACKET
		const_exp
	   { aux=$const_exp.literalStr; $literalStr += aux; $literalStr += _input.LT(1).getText();}
		RIGHT_BRACKET
    ;

literal returns [Pair<String, Token> pair = null]
@init{
    Token tk = _input.LT(1);
    String literalStr = tk.getText();
}
    :   ( HEX_LITERAL
	| INTEGER_LITERAL
	| STRING_LITERAL
	| WIDE_STRING_LITERAL
	| CHARACTER_LITERAL
	| WIDE_CHARACTER_LITERAL
	| FIXED_PT_LITERAL
	| FLOATING_PT_LITERAL
	| boolean_literal {literalStr = $boolean_literal.literalStr;} )
    {$pair = new Pair<String, Token>(literalStr, tk);}
    ;
	
boolean_literal returns [String literalStr = null]
	:   'TRUE'{ $literalStr = "true";}
	|   'FALSE'{ $literalStr = "false";}
	;

positive_int_const returns [String literalStr = null]
    :   const_exp { $literalStr = $const_exp.literalStr; }
	    {
           // TODO Calcular la expresion
           /*try {
               int value = Integer.parseInt($literalStr);

               if(value < 0)
                   throw new ParseException($literalStr, "expression is not supported. You must use a positive integer.");
           } catch(NumberFormatException e) {
           }*/
       }
    ;

type_decl [Vector<Annotation> annotations] returns [Pair<Vector<TypeDeclaration>, TemplateGroup> returnPair = null]
@init {
    Pair<Vector<TypeCode>, TemplateGroup> ttg = null;
    Vector<TypeDeclaration> vector = null;
    Token tk = null;
}
    :   ( KW_TYPEDEF {tk = _input.LT(1);} type_declarator { ttg=$type_declarator.returnPair; }
    |   struct_type { ttg=$struct_type.returnPair; }
    |   union_type { ttg=$union_type.returnPair; }
    |   enum_type { ttg=$enum_type.returnPair; }
    |   KW_NATIVE { System.out.println("WARNING (File " + ctx.getFilename() + ", Line " + (_input.LT(1) != null ? _input.LT(1).getLine() - ctx.getCurrentIncludeLine() : "1") + "): Native declarations are not supported. Ignoring..."); } simple_declarator
    |   constr_forward_decl )
	{
	    if(ttg!=null)
        {
            vector = new Vector<TypeDeclaration>();

            for(int count = 0; count < ttg.first().size(); ++count)
            {
                String name = null;
                if(ttg.first().get(count) instanceof MemberedTypeCode)
                    name = ((MemberedTypeCode)ttg.first().get(count)).getName();
                else if(ttg.first().get(count) instanceof AliasTypeCode)
                    name = ((AliasTypeCode)ttg.first().get(count)).getName();

                TypeDeclaration typedeclaration = new TypeDeclaration(ctx.getScopeFile(), ctx.isInScopedFile(), ctx.getScope(), name, ttg.first().get(count), tk);

                // Add annotations
                for(Annotation annotation : annotations)
                    typedeclaration.addAnnotation(ctx, annotation);

                // Add type declaration to the map with all typedeclarations.
                ctx.addTypeDeclaration(typedeclaration);

                vector.add(typedeclaration);

                $returnPair = new Pair<Vector<TypeDeclaration>, TemplateGroup>(vector, ttg.second());
            }
        }
	}
    ;

type_declarator returns [Pair<Vector<TypeCode>, TemplateGroup> returnPair = null]
@init {
    Vector<TypeCode> vector = null;
    AliasTypeCode typedefTypecode = null;
    TemplateGroup typedefTemplates =  null;
	if(tmanager != null) {
		typedefTemplates = tmanager.createTemplateGroup("typedef_decl");
	}
}
    :   type_spec declarators
	{
	   if($type_spec.typecode != null)
	   {
           vector = new Vector<TypeCode>();

	       for(int count = 0; count < $declarators.ret.size(); ++count)
	       {
	           typedefTypecode = new AliasTypeCode(ctx.getScope(), $declarators.ret.get(count).first().first());
	           
	           if($declarators.ret.get(count).second() != null)
	           {
	               // Array declaration
	               $declarators.ret.get(count).second().setContentTypeCode($type_spec.typecode);
	               typedefTypecode.setContentTypeCode($declarators.ret.get(count).second());
	           }
	           else
	           {
	               // Simple declaration
	               typedefTypecode.setContentTypeCode($type_spec.typecode);
	           }
	           
			   if(typedefTemplates != null) {
					typedefTemplates.setAttribute("typedefs", typedefTypecode);
			   }

               vector.add(typedefTypecode);
	       }

			if(typedefTemplates != null) {
				typedefTemplates.setAttribute("ctx", ctx);
			}

            $returnPair = new Pair<Vector<TypeCode>, TemplateGroup>(vector, typedefTemplates);
       }
	}
    ;

type_spec returns [TypeCode typecode = null]
    :   simple_type_spec { $typecode=$simple_type_spec.typecode; }
    |   constr_type_spec
    ;

simple_type_spec returns [TypeCode typecode = null]
@init {
    Pair<String, Token> pair = null;
} 
    :   base_type_spec { $typecode=$base_type_spec.typecode; }
    |   template_type_spec { $typecode=$template_type_spec.typecode; }
    |   scoped_name
	    {
		   pair=$scoped_name.pair;
			
	       // Find typecode in the global map.
	       $typecode = ctx.getTypeCode(pair.first());
	       
	       if($typecode == null)
	           throw new ParseException(pair.second(), "was not defined previously");
	    }
    ;

base_type_spec returns [TypeCode typecode = null]
    :   floating_pt_type { $typecode=$floating_pt_type.typecode; }
    |   integer_type { $typecode=$integer_type.typecode; }
    |   char_type { $typecode=$char_type.typecode; }
    |   wide_char_type { $typecode=$wide_char_type.typecode; }
    |   boolean_type { $typecode=$boolean_type.typecode; }
    |   octet_type { $typecode=$octet_type.typecode; }
    |   any_type
    |   object_type
    |   value_base_type
    ;

template_type_spec returns [TypeCode typecode = null]
    :   sequence_type { $typecode=$sequence_type.typecode; }
    |   string_type { $typecode=$string_type.typecode; }
    |   wide_string_type { $typecode=$wide_string_type.typecode; }
    |   fixed_pt_type
    ;

constr_type_spec
    :   struct_type
    |   union_type
    |   enum_type
    ;

declarators returns [Vector<Pair<Pair<String, Token>, ContainerTypeCode>> ret = new Vector<Pair<Pair<String, Token>, ContainerTypeCode>>()]
    :   declarator
	    {
            if($declarator.ret != null)
                $ret.add($declarator.ret);
            else
                throw new ParseException(null, "Cannot parse type declarator");
        }
		( COMA declarator
		{
            if($declarator.ret != null)
                $ret.add($declarator.ret);
            else
                throw new ParseException(null, "Cannot parse type declarator");
        }
		)*
    ;

declarator returns [Pair<Pair<String, Token>, ContainerTypeCode> ret = null]
    :   simple_declarator { $ret=$simple_declarator.ret; }
    |   complex_declarator { $ret=$complex_declarator.ret; }
    ;

simple_declarator returns [Pair<Pair<String, Token>, ContainerTypeCode> ret = null]
@init
{
    Token tk = _input.LT(1);
}
    :  identifier
	   {
           Pair<String, Token> p = new Pair<String, Token>($identifier.id, tk);
	       $ret = new Pair<Pair<String, Token>, ContainerTypeCode>(p, null);
	   }
    ;

complex_declarator returns [Pair<Pair<String, Token>, ContainerTypeCode> ret = null]
    :   array_declarator { $ret=$array_declarator.pair; }
    ;

floating_pt_type returns [TypeCode typecode = null]
    :   ( KW_FLOAT { $typecode = new PrimitiveTypeCode(TypeCode.KIND_FLOAT);}
	| KW_DOUBLE { $typecode = new PrimitiveTypeCode(TypeCode.KIND_DOUBLE);}
	| KW_LONG KW_DOUBLE { $typecode = new PrimitiveTypeCode(TypeCode.KIND_LONGDOUBLE);}
	)
    ;

integer_type returns [TypeCode typecode = null]
    :   signed_int { $typecode = $signed_int.typecode; }
    |   unsigned_int { $typecode = $unsigned_int.typecode; }
    ;

signed_int returns [TypeCode typecode = null]
    :   signed_short_int { $typecode = $signed_short_int.typecode; }
    |   signed_long_int { $typecode = $signed_long_int.typecode; }
    |   signed_longlong_int { $typecode = $signed_longlong_int.typecode; }
    ;

signed_short_int returns [TypeCode typecode]
@init{
	$typecode = new PrimitiveTypeCode(TypeCode.KIND_SHORT);
}
    :   KW_SHORT  
    ;

signed_long_int returns [TypeCode typecode]
@init{
	$typecode = new PrimitiveTypeCode(TypeCode.KIND_LONG);
}
    :   KW_LONG 
    ;

signed_longlong_int returns [TypeCode typecode]
@init{
	$typecode = new PrimitiveTypeCode(TypeCode.KIND_LONGLONG);
}
    :   KW_LONG KW_LONG 
    ;

unsigned_int returns [TypeCode typecode = null]
    :   unsigned_short_int { $typecode = $unsigned_short_int.typecode; }
    |   unsigned_long_int { $typecode = $unsigned_long_int.typecode; }
    |   unsigned_longlong_int { $typecode = $unsigned_longlong_int.typecode; }
    ;

unsigned_short_int returns [TypeCode typecode]
@init{
	$typecode = new PrimitiveTypeCode(TypeCode.KIND_USHORT);
}
    :   KW_UNSIGNED KW_SHORT 
    ;

unsigned_long_int returns [TypeCode typecode]
@init{
	$typecode = new PrimitiveTypeCode(TypeCode.KIND_ULONG);
}
    :   KW_UNSIGNED KW_LONG 
    ;

unsigned_longlong_int returns [TypeCode typecode]
@init{
	$typecode = new PrimitiveTypeCode(TypeCode.KIND_ULONGLONG);
}
    :   KW_UNSIGNED KW_LONG KW_LONG 
    ;

char_type returns [TypeCode typecode]
@init{
	$typecode = new PrimitiveTypeCode(TypeCode.KIND_CHAR);
}
    :   KW_CHAR
    ;

wide_char_type returns [TypeCode typecode]
@init{
	$typecode = new PrimitiveTypeCode(TypeCode.KIND_WCHAR);
}
    :   KW_WCHAR
    ;

boolean_type returns [TypeCode typecode]
@init{
	$typecode = new PrimitiveTypeCode(TypeCode.KIND_BOOLEAN);
}
    :   KW_BOOLEAN
    ;

octet_type returns [TypeCode typecode]
@init{
	$typecode = new PrimitiveTypeCode(TypeCode.KIND_OCTET);
}
    :   KW_OCTET  
    ;

any_type
@init{
    Token tk = _input.LT(1);
}
    :   KW_ANY
    {throw new ParseException(tk, ". Any type is not supported"); }
    ;

object_type
@init{
    Token tk = _input.LT(1);
}
    :   KW_OBJECT
    {throw new ParseException(tk, ". Object type is not supported"); }
    ;

annotation_decl returns [Pair<AnnotationDeclaration, TemplateGroup> returnPair = null]
:
	annotation_def { $returnPair=$annotation_def.returnPair; }
	| annotation_forward_dcl
	;
	
annotation_def returns [Pair<AnnotationDeclaration, TemplateGroup> returnPair = null]
@init
{
	TemplateGroup annotationTemplates = null;
}
    : annotation_header LEFT_BRACE annotation_body[$annotation_header.annotation] RIGHT_BRACE
	{
		if($annotation_header.annotation != null)
        {
			if(ctx.isInScopedFile() || ctx.isScopeLimitToAll())
            {
				if(tmanager != null)
                {
					annotationTemplates = tmanager.createTemplateGroup("annotation");
					annotationTemplates.setAttribute("ctx", ctx);
					// Set the annotation object to the TemplateGroup of the annotation.
					annotationTemplates.setAttribute("annotation", $annotation_header.annotation);
				}
			}
			
			$returnPair = new Pair<AnnotationDeclaration, TemplateGroup>($annotation_header.annotation, annotationTemplates);
		}
    }
	;
	
annotation_header returns [AnnotationDeclaration annotation = null]
@init
{
    Token tk = null;
}
    : KW_AT_ANNOTATION
    {
        tk = _input.LT(1);
    }
	identifier { $annotation = ctx.createAnnotationDeclaration($identifier.id, tk); }
	( annotation_inheritance_spec[$annotation] )?
	;

annotation_inheritance_spec [AnnotationDeclaration annotation]
@init
{
    AnnotationDeclaration inhanno = null;
}
    :
	COLON scoped_name
    {
        if(annotation != null)
        {
            inhanno = ctx.getAnnotationDeclaration($scoped_name.pair.first());

            if(inhanno != null)
            {
                annotation.addMembers(inhanno);
            }
            else
                throw new ParseException($scoped_name.pair.second(), "was not defined previously");
        }
    }
	;
	
annotation_body [AnnotationDeclaration annotation]
:
	( annotation_member[annotation] )*
	;

annotation_member [AnnotationDeclaration annotation]
@init
{
    String literalStr = null;
}
    :
	const_type simple_declarator ( KW_DEFAULT const_exp { literalStr=$const_exp.literalStr; } )? SEMICOLON
	{
		if(!$annotation.addMember(new AnnotationMember($simple_declarator.ret.first().first(), $const_type.typecode, literalStr)))
        {
            throw new ParseException($simple_declarator.ret.first().second(), "was defined previously");
        }
	}
	;

annotation_forward_dcl
:
	KW_AT_ANNOTATION scoped_name
	;
		
struct_type returns [Pair<Vector<TypeCode>, TemplateGroup> returnPair = null]
@init{
    String name = null;
    Vector<TypeCode> vector = null;
    StructTypeCode structTP = null;
    TemplateGroup structTemplates = null;
}
    :   KW_STRUCT
		identifier
	    {
			name=$identifier.id;
	       structTP = ctx.createStructTypeCode(name);
	    }
		LEFT_BRACE member_list[structTP] RIGHT_BRACE
		{
           if(ctx.isInScopedFile() || ctx.isScopeLimitToAll())
           {
				if(tmanager != null) {
				   structTemplates = tmanager.createTemplateGroup("struct_type");
				   structTemplates.setAttribute("ctx", ctx);
				   structTemplates.setAttribute("struct", structTP);
				}
           }
           // Return the returned data.
           vector = new Vector<TypeCode>();
           vector.add(structTP);
           $returnPair = new Pair<Vector<TypeCode>, TemplateGroup>(vector, structTemplates);
	    }
    ;

member_list [StructTypeCode structTP]
    :   (
			member_def
	       {
               if($member_def.ret != null)
               {
                   for(Pair<Pair<String, Token>, Member> pair : $member_def.ret)
                   {
	                   if(!$structTP.addMember(pair.second()))
                           throw new ParseException(pair.first().second(), "was defined previously");
                   }
               }
	       }
		)+
    ;

member_def returns [Vector<Pair<Pair<String, Token>, Member>> ret = null]
    :   member { $ret=$member.ret; }
    |   annotation_appl defret=member_def
        {
            if($defret.ret != null)
            {
                $ret=$defret.ret; 

                for(Pair<Pair<String, Token>, Member> pair : $ret)
                {
                    if(pair.second() != null)
                        pair.second().addAnnotation(ctx, $annotation_appl.annotation);
                }
            }
        }
    ; 

member returns [Vector<Pair<Pair<String, Token>, Member>> ret = new Vector<Pair<Pair<String, Token>, Member>>()]
    :   type_spec declarators SEMICOLON
		{
	       if($type_spec.typecode!=null)
	       {
		       for(int count = 0; count < $declarators.ret.size(); ++count)
		       {
                   Member member = null;

		           if($declarators.ret.get(count).second() != null)
		           {
		               // Array declaration
		               $declarators.ret.get(count).second().setContentTypeCode($type_spec.typecode);
                       member = new Member($declarators.ret.get(count).second(), $declarators.ret.get(count).first().first());
		               
		           }
		           else
		           {
		               // Simple declaration
                       member = new Member($type_spec.typecode, $declarators.ret.get(count).first().first());
		           }

                   $ret.add(new Pair<Pair<String, Token>, Member>($declarators.ret.get(count).first(), member));
		       }
	       }
	   }
    ;

union_type returns [Pair<Vector<TypeCode>, TemplateGroup> returnPair = null]
@init {
    String name = null;
    int line = 0;
    TypeCode dist_type = null;
    Vector<TypeCode> vector = null;
    UnionTypeCode unionTP = null;
    TemplateGroup unionTemplates = null;
}
    :   KW_UNION
	    identifier { name=$identifier.id;}
		KW_SWITCH LEFT_BRACKET switch_type_spec { dist_type=$switch_type_spec.typecode; } RIGHT_BRACKET
		{
            // TODO Check supported types for discriminator: long, enumeration, etc...
	       unionTP = new UnionTypeCode(ctx.getScope(), name, dist_type);
           line= _input.LT(1) != null ? _input.LT(1).getLine() - ctx.getCurrentIncludeLine() : 1;
	    }
		LEFT_BRACE switch_body[unionTP] RIGHT_BRACE
	    {
	       // Calculate default label.
	       TemplateUtil.setUnionDefaultLabel(unionTP, ctx.getScopeFile(), line);

           if(ctx.isInScopedFile() || ctx.isScopeLimitToAll())
           {
				if(tmanager != null) {
                   unionTemplates = tmanager.createTemplateGroup("union_type");
                   unionTemplates.setAttribute("ctx", ctx);
                   unionTemplates.setAttribute("union", unionTP);
				}
           }
           
           // Return the returned data.
           vector = new Vector<TypeCode>();
           vector.add(unionTP);
           $returnPair = new Pair<Vector<TypeCode>, TemplateGroup>(vector, unionTemplates);
        }
    ;

switch_type_spec returns [TypeCode typecode = null]
@init {
    Pair<String, Token> pair = null;
}
    :   integer_type { $typecode=$integer_type.typecode; }
    |   char_type { $typecode=$char_type.typecode; }
    |   boolean_type { $typecode=$boolean_type.typecode; }
    |   enum_type
    |   scoped_name
	    {
		   pair=$scoped_name.pair;
           // Find typecode in the global map.
           $typecode = ctx.getTypeCode(pair.first());
           
           if($typecode == null)
               throw new ParseException(pair.second(), "was not defined previously");
        }
    ;

switch_body [UnionTypeCode unionTP]
@init {
}
    :   case_stmt_list[unionTP]
    ;
	
case_stmt_list [UnionTypeCode unionTP]
@init {
}
	:  (case_stmt[unionTP])+
	;

case_stmt [UnionTypeCode unionTP]
@init
{
    List<String> labels = new ArrayList<String>();
    boolean defaul = false;
}
	:	( KW_CASE const_exp
		{
			labels.add(TemplateUtil.checkUnionLabel(unionTP.getDiscriminator(), $const_exp.literalStr, ctx.getScopeFile(), _input.LT(1) != null ? _input.LT(1).getLine() - ctx.getCurrentIncludeLine() : 1));
		} COLON
		| KW_DEFAULT { defaul = true; } COLON
		)+
		element_spec[labels, defaul] SEMICOLON
	    {
	       if($element_spec.ret != null)
	       {
		       int ret = unionTP.addMember($element_spec.ret.second());

               if(ret == -1)
                   throw new ParseException($element_spec.ret.first().second(), " is already defined.");
               else if(ret == -2)
                   throw new ParseException($element_spec.ret.first().second(), " is also a default attribute. Another was defined previously.");
	       }
	    }
	;

//case_label
//    :   KW_CASE const_exp COLON
//    |   KW_DEFAULT COLON
//    ;

element_spec [List<String> labels, boolean isDefault] returns [Pair<Pair<String, Token>, UnionMember> ret = null]
    :   type_spec declarator
	    {
	        if($type_spec.typecode != null)
	        {
                UnionMember member = null;

	            if($declarator.ret.second() != null)
	            {
	                $declarator.ret.second().setContentTypeCode($type_spec.typecode);
                    member = new UnionMember($declarator.ret.second(), $declarator.ret.first().first(), labels, isDefault);
	            }
	            else
	            {
                    member = new UnionMember($type_spec.typecode, $declarator.ret.first().first(), labels, isDefault);
	            }

                $ret = new Pair<Pair<String, Token>, UnionMember>($declarator.ret.first(), member);
            }
        }
    ;

enum_type returns [Pair<Vector<TypeCode>, TemplateGroup> returnPair = null]
@init{
    String name = null;
    Vector<TypeCode> vector = null;
    EnumTypeCode enumTP = null;
    TemplateGroup enumTemplates = null;
}
    :   KW_ENUM
		identifier { name=$identifier.id; enumTP = new EnumTypeCode(ctx.getScope(), name); }
		LEFT_BRACE  enumerator_list[enumTP] RIGHT_BRACE
	   {
           if(ctx.isInScopedFile() || ctx.isScopeLimitToAll())
           {
				if(tmanager != null) {
                    enumTemplates = tmanager.createTemplateGroup("enum_type");
				    enumTemplates.setAttribute("ctx", ctx);
                    enumTemplates.setAttribute("enum", enumTP);
                }
		   }

           // Return the returned data.
           vector = new Vector<TypeCode>();
           vector.add(enumTP);
           $returnPair = new Pair<Vector<TypeCode>, TemplateGroup>(vector, enumTemplates);
	   }
    ;
	
enumerator_list [EnumTypeCode enumTP]
	:    enumerator[enumTP] (COMA enumerator[enumTP])*
	;

enumerator [EnumTypeCode enumTP]
@init{
    String name = null;
}
    :   identifier{ name=$identifier.id; enumTP.addMember(new EnumMember(name));}
    ;

sequence_type returns [SequenceTypeCode typecode = null]
@init {
    TypeCode type = null;
    String maxsize = null;
}
    :   ( (KW_SEQUENCE) 
		LEFT_ANG_BRACKET simple_type_spec { type=$simple_type_spec.typecode; } COMA positive_int_const { maxsize=$positive_int_const.literalStr; } RIGHT_ANG_BRACKET
    |   (KW_SEQUENCE) 
		LEFT_ANG_BRACKET simple_type_spec { type=$simple_type_spec.typecode; } RIGHT_ANG_BRACKET )
		{
           if(type != null)
           {
               $typecode = new SequenceTypeCode(maxsize);
               $typecode.setContentTypeCode(type);
           }
	    }
    ;
	
string_type returns [TypeCode typecode = null]
@init{
    String maxsize = null;
}
    :   ( KW_STRING LEFT_ANG_BRACKET positive_int_const { maxsize=$positive_int_const.literalStr; } RIGHT_ANG_BRACKET
    |   KW_STRING )
	   {$typecode = new StringTypeCode(TypeCode.KIND_STRING, maxsize);}
    ;

wide_string_type returns [TypeCode typecode = null]
@init
{
    String maxsize = null;
}
    :   ( KW_WSTRING LEFT_ANG_BRACKET positive_int_const { maxsize=$positive_int_const.literalStr; } RIGHT_ANG_BRACKET
    |   KW_WSTRING )
	   {$typecode = new StringTypeCode(TypeCode.KIND_WSTRING, maxsize);}
    ;

array_declarator returns [Pair<Pair<String, Token>, ContainerTypeCode> pair = null]
@init
{
    Token tk = _input.LT(1);
    ArrayTypeCode typecode = new ArrayTypeCode();
}
    :   ID
		(
			fixed_array_size
			{
	           typecode.addDimension($fixed_array_size.literalStr);
	        }
		)+
	    {
            Pair<String, Token> p = new Pair<String, Token>(tk.getText(), tk);
            $pair = new Pair<Pair<String, Token>, ContainerTypeCode>(p, typecode);
	    }
    ;

fixed_array_size returns [String literalStr = null]
    :   LEFT_SQUARE_BRACKET
		positive_int_const { $literalStr=$positive_int_const.literalStr; }
		RIGHT_SQUARE_BRACKET
    ;

attr_decl returns [Vector<Pair<Pair<String, Token>, TypeCode>> ret = null]
    :   readonly_attr_spec
    |   attr_spec { $ret=$attr_spec.ret; }
    ;

except_decl returns [Pair<com.eprosima.idl.parser.tree.Exception, TemplateGroup> returnPair = null]
@init {
    String name = null;
    com.eprosima.idl.parser.tree.Exception exceptionObject = null;
    TemplateGroup exTemplates = null;
    Token tk = null;
}
    :   KW_EXCEPTION
        {
            tk = _input.LT(1);
        }
		identifier { name=$identifier.id; }
	    {
            // Create the Exception object.
            exceptionObject = ctx.createException(name, tk);

            if(ctx.isInScopedFile() || ctx.isScopeLimitToAll())
            {
				if(tmanager != null) {
					exTemplates = tmanager.createTemplateGroup("exception");
					exTemplates.setAttribute("ctx", ctx);
					// Set the the exception object to the TemplateGroup of the module.
					exTemplates.setAttribute("exception", exceptionObject);
				}
            }
            // Its a dependency.
            else
            {
                ctx.addIncludeDependency(ctx.getScopeFile());
            }
	    }
		LEFT_BRACE opt_member_list[exceptionObject] RIGHT_BRACE
	    {
            // Create the returned data.
            $returnPair = new Pair<com.eprosima.idl.parser.tree.Exception, TemplateGroup>(exceptionObject, exTemplates);
        }
    ;
	
opt_member_list [com.eprosima.idl.parser.tree.Exception exceptionObject]
	:  (
	      member
	      {
	          for(int count = 0; count < $member.ret.size(); ++count)
	              $exceptionObject.addMember($member.ret.get(count).second());
	      }
	   )*
	;

 /*!
 * @brief This grammar expression catches a operation.
 * @return This grammar expression returns the operation object and its template group.
 */
op_decl [Vector<Annotation> annotations] returns [Pair<Operation, TemplateGroup> returnPair = null]
@init {
        Operation operationObject = null;
        TemplateGroup operationTemplates = null;
		if(tmanager != null) {
			operationTemplates = tmanager.createTemplateGroup("operation");
		}
        TemplateGroup tpl = null;
        String name = "";
        Token tk = null, tkoneway = null;
        TypeCode retType = null;
        Vector<Pair<String, Token>> exceptions = null;
}
    :   ( op_attribute { tkoneway=$op_attribute.token; } )?
		op_type_spec { retType=$op_type_spec.typecode; }
		{
            tk = _input.LT(1);
            name += tk.getText();
        }
        ID
		{
           // Create the Operation object.
           operationObject = ctx.createOperation(name, tk);

           // Add annotations.
           for(Annotation annotation : annotations)
               operationObject.addAnnotation(ctx, annotation);
		   
		   if(operationTemplates != null)
           {
			   operationTemplates.setAttribute("ctx", ctx);
			   // Set the the interface object to the TemplateGroup of the module.
			   operationTemplates.setAttribute("operation", operationObject);
		   }
           
           // Set return type
           operationObject.setRettype(retType);
           
           // Set oneway
           if(tkoneway != null)
           {
               operationObject.setOneway(true);

               if(retType != null)
               {
                   throw new ParseException(tkoneway, ". Oneway function cannot have a return type.");
               }
           }
        }
		parameter_decls[operationObject] { tpl=$parameter_decls.tpl; }
		( 
		   raises_expr { exceptions=$raises_expr.exlist; }
	       {
	          // Search global exceptions and add them to the operation.
              for(Pair<String, Token> pair : exceptions)
	          {
	             com.eprosima.idl.parser.tree.Exception exception = ctx.getException(pair.first());
	             
	             if(exception != null)
	                operationObject.addException(exception);
	             else
	                operationObject.addUnresolvedException(pair.first());
	          }
	       }
		)?
		( context_expr )?
	    {
			if(operationTemplates != null) {
				// Store the parameter list template group in the operation template group.
				operationTemplates.setAttribute("param_list", tpl);
			}
           // Create the returned data.
           $returnPair = new Pair<Operation, TemplateGroup>(operationObject, operationTemplates);
        }
    ;

op_attribute returns [Token token = null]
@init {
    Token tk = _input.LT(1);
}
    :   KW_ONEWAY { $token = tk;}
    ;

op_type_spec returns [TypeCode typecode = null]
    :   param_type_spec { $typecode=$param_type_spec.typecode; }
    |   KW_VOID
    ;

parameter_decls [Operation operation] returns [TemplateGroup tpl]
@init {
	if(tmanager != null) {
		$tpl = tmanager.createTemplateGroup("param_list");
	}
}
    :   LEFT_BRACKET (param_decl_list[operation, $tpl])? RIGHT_BRACKET
    ;

param_decl_list [Operation operation, TemplateGroup tpl]
@init {
        Pair<Param, TemplateGroup> ptg = null;
}
	:   param_decl { ptg=$param_decl.returnPair; }
		{
			if(ptg!=null) {
				operation.add(ptg.first());
				if(tpl != null) {
					tpl.setAttribute("parameters", ptg.second());
				}
			}
		}
	    (
		COMA param_decl { ptg=$param_decl.returnPair; }
		{
			if(ptg!=null) {
				operation.add(ptg.first());
				if(tpl != null) {
					tpl.setAttribute("parameters", ptg.second());
				}
			}
		}
		)*
	;
	
param_decl returns [Pair<Param, TemplateGroup> returnPair = null]
@init{
        TemplateGroup paramTemplate = null;
		if(tmanager != null) {
			paramTemplate = tmanager.createTemplateGroup("param");
		}
        TypeCode typecode = null;
        String literalStr = _input.LT(1).getText();
}
    :   ('in' | 'out' | 'inout')
		param_type_spec { typecode=$param_type_spec.typecode; }
        simple_declarator
	    {
	        if(typecode != null)
	        {
		        Param param = null;
		        if(literalStr.equals("in"))
		            param = ctx.createParam($simple_declarator.ret.first().first(), typecode, Param.Kind.IN_PARAM);
		        else if(literalStr.equals("out"))
		            param = ctx.createParam($simple_declarator.ret.first().first(), typecode, Param.Kind.OUT_PARAM);
		        else if(literalStr.equals("inout"))
		            param = ctx.createParam($simple_declarator.ret.first().first(), typecode, Param.Kind.INOUT_PARAM);
		            
				if(paramTemplate != null) {
					paramTemplate.setAttribute("parameter", param);
				}
		        $returnPair = new Pair<Param, TemplateGroup>(param, paramTemplate);
	        }
	    }
    ;

//param_attribute
//    :   KW_IN
//    |   KW_OUT
//    |   KW_INOUT
//    ;

raises_expr returns [Vector<Pair<String, Token>> exlist = null]
    :   KW_RAISES LEFT_BRACKET scoped_name_list { $exlist=$scoped_name_list.retlist; } RIGHT_BRACKET
    ;

context_expr
@init {
	System.out.println("WARNING (File " + ctx.getFilename() + ", Line " + (_input.LT(1) != null ? _input.LT(1).getLine() - ctx.getCurrentIncludeLine() : "1") + "): Context declarations are not supported. Ignoring...");
}
    :   KW_CONTEXT LEFT_BRACKET STRING_LITERAL ( COMA STRING_LITERAL )* RIGHT_BRACKET
    ;

param_type_spec returns [TypeCode typecode = null]
@init{
    Pair<String, Token> pair = null;
}
    :   base_type_spec { $typecode=$base_type_spec.typecode; }
    |   string_type { $typecode=$string_type.typecode; }
    |   wide_string_type { $typecode=$wide_string_type.typecode; }
    |   scoped_name
        {
           pair=$scoped_name.pair;
           // Find typecode in the global map.
           $typecode = ctx.getTypeCode(pair.first());
           
           if($typecode == null)
               throw new ParseException(pair.second(), "was not defined previously");
        }
    ;

fixed_pt_type
@init{
    Token tk = _input.LT(1);
}
    :   KW_FIXED LEFT_ANG_BRACKET positive_int_const COMA positive_int_const RIGHT_ANG_BRACKET
    {throw new ParseException(tk, ". Fixed type is not supported");}
    ;

fixed_pt_const_type
@init{
    Token tk = _input.LT(1);
}
    :   KW_FIXED
    {throw new ParseException(tk, ". Fixed type is not supported");}
    ;

value_base_type
@init{
    Token tk = _input.LT(1);
}
    :   KW_VALUEBASE
    {throw new ParseException(tk, ". Value type is not supported");}
    ;

constr_forward_decl
    :   KW_STRUCT ID
    |   KW_UNION ID
    ;

import_decl
    :   KW_IMPORT
	{
		System.out.println("WARNING (File " + ctx.getFilename() + ", Line " + (_input.LT(1) != null ? _input.LT(1).getLine() - ctx.getCurrentIncludeLine() : "1") + "): Import declarations are not supported. Ignoring...");
	}
	imported_scope SEMICOLON
    ;

imported_scope
    :   scoped_name | STRING_LITERAL
    ;

type_id_decl
    :   KW_TYPEID
        {
            System.out.println("WARNING (File " + ctx.getFilename() + ", Line " + (_input.LT(1) != null ? _input.LT(1).getLine() - ctx.getCurrentIncludeLine() : "1") + "): TypeID declarations are not supported. Ignoring...");
        }
		scoped_name STRING_LITERAL
    ;

type_prefix_decl
    :   KW_TYPEPREFIX
	    {
            System.out.println("WARNING (File " + ctx.getFilename() + ", Line " + (_input.LT(1) != null ? _input.LT(1).getLine() - ctx.getCurrentIncludeLine() : "1") + "): TypePrefix declarations are not supported. Ignoring...");
        }
		scoped_name STRING_LITERAL
    ;

readonly_attr_spec
    :   KW_READONLY KW_ATTRIBUTE param_type_spec readonly_attr_declarator
    ;

readonly_attr_declarator
    :   simple_declarator raises_expr
    |   simple_declarator ( COMA simple_declarator )*
    ;

attr_spec returns [Vector<Pair<Pair<String, Token>, TypeCode>> ret = new Vector<Pair<Pair<String, Token>, TypeCode>>()]
@init {
    TypeCode typecode = null;
}
    :   KW_ATTRIBUTE param_type_spec { typecode=$param_type_spec.typecode; } attr_declarator
    	{
    	   if(typecode != null)
           {
               for(int count = 0; count < $attr_declarator.ret.size(); ++count)
               {
                   // attr_declarator always is a simple declarator. Not a complex (array):
                   // Simple declaration
                   $ret.add(new Pair<Pair<String, Token>, TypeCode>($attr_declarator.ret.get(count).first(), typecode));
               }
           }
    	}
    ;

attr_declarator returns [Vector<Pair<Pair<String, Token>, ContainerTypeCode>> ret]
@init {
	$ret = new Vector<Pair<Pair<String, Token>, ContainerTypeCode>>();
}
    :   simple_declarator {$ret.add($simple_declarator.ret);}
		( attr_raises_expr
		| (COMA simple_declarator {$ret.add($simple_declarator.ret);})* 
		)
    ;

attr_raises_expr
    :   get_excep_expr ( set_excep_expr )?
    |   set_excep_expr
    ;

get_excep_expr
    :   KW_GETRAISES exception_list
    ;

set_excep_expr
    :   KW_SETRAISES exception_list
    ;

exception_list
    :   LEFT_BRACKET scoped_name ( COMA scoped_name )* RIGHT_BRACKET
    ;

// Component Stuff
	
component
    :   component_decl
    |   component_forward_decl
    ;

component_forward_decl
    :   KW_COMPONENT
	    {
            System.out.println("WARNING (File " + ctx.getFilename() + ", Line " + (_input.LT(1) != null ? _input.LT(1).getLine() - ctx.getCurrentIncludeLine() : "1") + "): Component declarations are not supported. Ignoring...");
        }
		ID
    ;

component_decl
    :   component_header LEFT_BRACE component_body RIGHT_BRACE
    ;

component_header
    :   KW_COMPONENT ID ( component_inheritance_spec )? ( supported_interface_spec )?
    ;

supported_interface_spec
    :   KW_SUPPORTS scoped_name ( COMA scoped_name )*
    ;

component_inheritance_spec
    :   COLON scoped_name
    ;

component_body
    :   component_export*
    ;

component_export
    :   provides_decl SEMICOLON
    |   uses_decl SEMICOLON
    |   emits_decl SEMICOLON
    |   publishes_decl SEMICOLON
    |   consumes_decl SEMICOLON
    |   attr_decl SEMICOLON
    ;

provides_decl
    :   KW_PROVIDES interface_type ID
    ;

interface_type
    :   scoped_name
    |   KW_OBJECT
    ;

uses_decl
    :   KW_USES ( KW_MULTIPLE )? interface_type ID
    ;

emits_decl
    :   KW_EMITS scoped_name ID
    ;

publishes_decl
    :   KW_PUBLISHES scoped_name ID
    ;

consumes_decl
    :   KW_CONSUMES scoped_name ID
    ;

home_decl
@init {
    System.out.println("WARNING (File " + ctx.getFilename() + ", Line " + (_input.LT(1) != null ? _input.LT(1).getLine() - ctx.getCurrentIncludeLine() : "1") + "): Home declarations are not supported. Ignoring...");
}
    :   home_header home_body
    ;

home_header
    :   KW_HOME ID ( home_inheritance_spec )? ( supported_interface_spec )? KW_MANAGES scoped_name ( primary_key_spec )?
    ;

home_inheritance_spec
    :   COLON scoped_name
    ;

primary_key_spec
    :   KW_PRIMARYKEY scoped_name
    ;

home_body
    :   LEFT_BRACE home_export* RIGHT_BRACE
    ;

home_export
    :   export[null]
    |   factory_decl SEMICOLON
    |   finder_decl SEMICOLON
    ;

factory_decl
    :   KW_FACTORY ID LEFT_BRACKET ( init_param_decls )? RIGHT_BRACKET ( raises_expr )?
    ;

finder_decl
    :   KW_FINDER ID LEFT_BRACKET ( init_param_decls )? RIGHT_BRACKET ( raises_expr )?
    ;

event
@init{
    System.out.println("WARNING (File " + ctx.getFilename() + ", Line " + (_input.LT(1) != null ? _input.LT(1).getLine() - ctx.getCurrentIncludeLine() : "1") + "): Event declarations are not supported. Ignoring...");
}
    :   ( event_decl | event_abs_decl | event_forward_decl)
    ;

event_forward_decl
    :   ( KW_ABSTRACT )? KW_EVENTTYPE ID
    ;

event_abs_decl
    :   KW_ABSTRACT KW_EVENTTYPE ID value_inheritance_spec LEFT_BRACE export[null]* RIGHT_BRACE
    ;

event_decl
    :   event_header LEFT_BRACE value_element* RIGHT_BRACE
    ;

event_header
    :   ( KW_CUSTOM )? KW_EVENTTYPE ID value_inheritance_spec
    ;

annotation_appl returns [Annotation annotation = null]
@init
{
    AnnotationDeclaration anndecl = null;
}
: 
	AT scoped_name
	{
        anndecl = ctx.getAnnotationDeclaration($scoped_name.pair.first());
		if(anndecl == null)
        {
			throw new ParseException($scoped_name.pair.second(), "was not defined previously");
		}

        $annotation = new Annotation(anndecl);
	}
	( LEFT_BRACKET ( annotation_appl_params[$annotation, $scoped_name.pair.second()] )? RIGHT_BRACKET )?
	;
	
// TODO Support several members in annotations.
annotation_appl_params [Annotation annotation, Token tkannot]
    : const_exp
	{
        if(!annotation.addValue($const_exp.literalStr))
            throw new ParseException(tkannot, "not has only one attribute.");
	}
	| annotation_appl_param[annotation] ( COMA annotation_appl_param[annotation] )*
	;
	
annotation_appl_param [Annotation annotation]
@init
{
    Token tk = _input.LT(1);
}
    : identifier EQUAL const_exp
    {
        if(!annotation.addValue($identifier.id, $const_exp.literalStr))
            throw new ParseException(tk, "is not an attribute of annotation " + annotation.getName());
    }
	;
	
identifier returns [String id]
@init {
	$id = _input.LT(1).getText();
}
	:   ID
	;


INTEGER_LITERAL : ('0' | '1'..'9' '0'..'9'*) INTEGER_TYPE_SUFFIX? ;

OCTAL_LITERAL : '0' ('0'..'7')+ INTEGER_TYPE_SUFFIX? ;

HEX_LITERAL : '0' ('x' | 'X') HEX_DIGIT+ INTEGER_TYPE_SUFFIX? ;

fragment
HEX_DIGIT : ( '0'..'9' | 'a'..'f' | 'A'..'F' ) ;

fragment
INTEGER_TYPE_SUFFIX : ('l' | 'L') ;

FLOATING_PT_LITERAL
    :   ('0'..'9')+ '.' ('0'..'9')* EXPONENT? FLOAT_TYPE_SUFFIX?
    |   '.' ('0'..'9')+ EXPONENT? FLOAT_TYPE_SUFFIX?
    |   ('0'..'9')+ EXPONENT FLOAT_TYPE_SUFFIX?
    |   ('0'..'9')+ EXPONENT? FLOAT_TYPE_SUFFIX
    ;

FIXED_PT_LITERAL
    :   FLOATING_PT_LITERAL
    ;

fragment
EXPONENT : ('e' | 'E') (PLUS|MINUS)? ('0'..'9')+ ;

fragment
FLOAT_TYPE_SUFFIX : ('f' | 'F' | 'd' | 'D') ;

WIDE_CHARACTER_LITERAL
    :   'L' CHARACTER_LITERAL
    ;

CHARACTER_LITERAL
    :   '\'' ( ESCAPE_SEQUENCE | ~('\'' | '\\') ) '\''
    ;

WIDE_STRING_LITERAL
    :   'L' STRING_LITERAL
    ;

STRING_LITERAL
    :   '"' ( ESCAPE_SEQUENCE | ~('\\' | '"') )* '"'
    ;

BOOLEAN_LITERAL
    :   'TRUE'
    |   'FALSE'
    ;

fragment
ESCAPE_SEQUENCE
    :   '\\' ('b' | 't' | 'n' | 'f' | 'r' | '\"' | '\'' | '\\')
    |   UNICODE_ESCAPE
    |   OCTAL_ESCAPE
    ;

fragment
OCTAL_ESCAPE
    :   '\\' ('0'..'3') ('0'..'7') ('0'..'7')
    |   '\\' ('0'..'7') ('0'..'7')
    |   '\\' ('0'..'7')
    ;

fragment
UNICODE_ESCAPE
    :   '\\' 'u' HEX_DIGIT HEX_DIGIT HEX_DIGIT HEX_DIGIT
    ;

fragment
LETTER
    :   '\u0024'
    |   '\u0041'..'\u005a'
    |   '\u005f'
    |   '\u0061'..'\u007a'
    |   '\u00c0'..'\u00d6'
    |   '\u00d8'..'\u00f6'
    |   '\u00f8'..'\u00ff'
    |   '\u0100'..'\u1fff'
    |   '\u3040'..'\u318f'
    |   '\u3300'..'\u337f'
    |   '\u3400'..'\u3d2d'
    |   '\u4e00'..'\u9fff'
    |   '\uf900'..'\ufaff'
    ;

fragment
ID_DIGIT
    :   '\u0030'..'\u0039'
    |   '\u0660'..'\u0669'
    |   '\u06f0'..'\u06f9'
    |   '\u0966'..'\u096f'
    |   '\u09e6'..'\u09ef'
    |   '\u0a66'..'\u0a6f'
    |   '\u0ae6'..'\u0aef'
    |   '\u0b66'..'\u0b6f'
    |   '\u0be7'..'\u0bef'
    |   '\u0c66'..'\u0c6f'
    |   '\u0ce6'..'\u0cef'
    |   '\u0d66'..'\u0d6f'
    |   '\u0e50'..'\u0e59'
    |   '\u0ed0'..'\u0ed9'
    |   '\u1040'..'\u1049'
    ;

SEMICOLON:              ';';
COLON:                  ':';
COMA:                   ',';
LEFT_BRACE:             '{';
RIGHT_BRACE:            '}';
LEFT_BRACKET:           '(';
RIGHT_BRACKET:          ')';
LEFT_SQUARE_BRACKET:    '[';
RIGHT_SQUARE_BRACKET:   ']';
TILDE:                  '~';
SLASH:                  '/';
LEFT_ANG_BRACKET:       '<';
RIGHT_ANG_BRACKET:      '>';
STAR:                   '*';
PLUS:                   '+';
MINUS:                  '-';
CARET:                  '^';
AMPERSAND:              '&';
PIPE:                   '|';
EQUAL:                  '=';
PERCENT:                '%';
AT:						'@';

DOUBLE_COLON:           '::';
RIGHT_SHIFT:            '>>';
LEFT_SHIFT:             '<<';

KW_SETRAISES:           'setraises';
KW_OUT:                 'out';
KW_EMITS:               'emits';
KW_STRING:              'string';
KW_SWITCH:              'switch';
KW_PUBLISHES:           'publishes';
KW_TYPEDEF:             'typedef';
KW_USES:                'uses';
KW_PRIMARYKEY:          'primarykey';
KW_CUSTOM:              'custom';
KW_OCTET:               'octet';
KW_SEQUENCE:            'sequence';
KW_IMPORT:              'import';
KW_STRUCT:              'struct';
KW_NATIVE:              'native';
KW_READONLY:            'readonly';
KW_FINDER:              'finder';
KW_RAISES:              'raises';
KW_VOID:                'void';
KW_PRIVATE:             'private';
KW_EVENTTYPE:           'eventtype';
KW_WCHAR:               'wchar';
KW_IN:                  'in';
KW_DEFAULT:             'default';
KW_PUBLIC:              'public';
KW_SHORT:               'short';
KW_LONG:                'long';
KW_ENUM:                'enum';
KW_WSTRING:             'wstring';
KW_CONTEXT:             'context';
KW_HOME:                'home';
KW_FACTORY:             'factory';
KW_EXCEPTION:           'exception';
KW_GETRAISES:           'getraises';
KW_CONST:               'const';
KW_VALUEBASE:           'ValueBase';
KW_VALUETYPE:           'valuetype';
KW_SUPPORTS:            'supports';
KW_MODULE:              'module';
KW_OBJECT:              'Object';
KW_TRUNCATABLE:         'truncatable';
KW_UNSIGNED:            'unsigned';
KW_FIXED:               'fixed';
KW_UNION:               'union';
KW_ONEWAY:              'oneway';
KW_ANY:                 'any';
KW_CHAR:                'char';
KW_CASE:                'case';
KW_FLOAT:               'float';
KW_BOOLEAN:             'boolean';
KW_MULTIPLE:            'multiple';
KW_ABSTRACT:            'abstract';
KW_INOUT:               'inout';
KW_PROVIDES:            'provides';
KW_CONSUMES:            'consumes';
KW_DOUBLE:              'double';
KW_TYPEPREFIX:          'typeprefix';
KW_TYPEID:              'typeid';
KW_ATTRIBUTE:           'attribute';
KW_LOCAL:               'local';
KW_MANAGES:             'manages';
KW_INTERFACE:           'interface';
KW_COMPONENT:           'component';
KW_AT_ANNOTATION:        '@annotation';

ID
    :   LETTER (LETTER|ID_DIGIT)*
    ;

WS
    :   (' ' | '\r' | '\t' | '\u000C' | '\n') -> channel(HIDDEN)
    ;
	
PREPROC_DIRECTIVE
	:
	'#'
	(~'\n')* '\n'
	{ 
		ctx.processPreprocessorLine(new String(getText()), getLine());
		skip();
		//newline();
	}
	;
	
COMMENT
    :   '/*' .*? '*/' -> channel(HIDDEN)
    ;

LINE_COMMENT
    :   '//' ~('\n' | '\r')* '\r'? '\n' -> channel(HIDDEN)
    ;

// [EOF] IDL.g
