// License: Apache 2.0. See LICENSE file in root directory.
// Copyright(c) 2020 Intel Corporation. All Rights Reserved.

#pragma once

#include <cstdint>
#include <vector>
#include <map>
#include <types.h>

#include "calibration.h"
#include "coeffs.h"
#include "frame-data.h"
#include "k-to-dsm.h"


namespace librealsense {
namespace algo {
namespace depth_to_rgb_calibration {

    struct optimization_params
    {
        p_matrix curr_p_mat;
        p_matrix calib_gradients;
        double cost;
        double step_size = 0;
    };

    struct params
    {
        params();

        void set_depth_resolution(size_t width, size_t height, rs2_ambient_light ambient);
        void set_rgb_resolution(size_t width, size_t height);

        double gamma = 0.9;
        double alpha = (double)1 / (double)3;
        double grad_ir_threshold = 3.5; // Ignore pixels with IR gradient less than this (resolution-dependent!)
        int grad_z_threshold = 0; //Ignore pixels with Z grad of less than this
        double grad_z_min = 25; // Ignore pixels with Z grad of less than this
        double grad_z_max = 1000;
        double edge_thresh4_logic_lum = 0.1;

        // enhanced preprocessing params
        double grad_ir_low_th = std::numeric_limits<double>::max();
        double grad_ir_high_th = 2.5;
        double grad_z_low_th = 0;
        double grad_z_high_th = std::numeric_limits<double>::max();

        double max_step_size = 1;
        double min_step_size = 0.00001;
        double control_param = 0.5;
        int max_back_track_iters = 50;
        int max_optimization_iters = 50;
        double min_rgb_mat_delta = 0.00001;
        double min_cost_delta = 1;
        double tau = 0.5;
        double min_weighted_edge_per_section_depth = 0;      // resolution-dependent
        double min_weighted_edge_per_section_rgb = 0;        // resolution-dependent
        size_t num_of_sections_for_edge_distribution_x = 2;
        size_t num_of_sections_for_edge_distribution_y = 2;
        p_matrix normalize_mat;  //% rgbPmatNormalizationMat

        double edge_distribution_min_max_ratio = 0.005;
        double grad_dir_ratio = 10;
        double grad_dir_ratio_prep = 1.5;
        size_t dilation_size = 1;
        double gauss_sigma = 1;
        size_t gause_kernel_size = 5;
        double const move_thresh_pix_val = 20;
        double move_threshold_pix_num = 0;                   // resolution-dependent
        double const move_last_success_thresh_pix_val = 25;
        double move_last_success_thresh_pix_num = 0;         // resolution-dependent

        //smearing
        double max_sub_mm_z = 4;
        double constant_weights = 1000;

        // output validation
        double max_xy_movement_per_calibration[3]; // resolution-dependent = { 10, 2, 2 } * res/hd
        double max_xy_movement_from_origin;        // resolution-dependent = 20 * res/hd
        double const max_scaling_step = 0.020000000000000;
        double const max_K2DSM_iters = 10;
        // TODO: the following should be 0.2% but was increased to 0.5% to account for
        // manual trigger activation
        double max_global_los_scaling_step = 0.004;  // the difference (.5%) between starting and final scale

        // input validation
        double edges_per_direction_ratio_th = 0.004;
        double dir_std_th[N_BASIC_DIRECTIONS] = { 0.09, 0.09, 0.09, 0.09 };
        int minimal_full_directions = 2;
        bool require_orthogonal_valid_dirs = false;
        int saturation_value = 230;
        double saturation_ratio_th = 0.05;
        double pix_per_section_rgb_th = 0.01;
        double pix_per_section_depth_th = 0.01;
        int min_section_with_enough_edges = 2;

        bool use_enhanced_preprocessing = true;
    };
    // svm
    struct decision_params
    {
        double initial_cost;
        bool is_valid;
        double xy_movement;
        double xy_movement_from_origin;
        std::vector< double >  improvement_per_section;
        double min_improvement_per_section;
        double max_improvement_per_section;
        bool is_valid_1;
        double moving_pixels;
        double min_max_ratio_depth;// edgeDistributionMinMaxRatioDepth;
        std::vector< double > distribution_per_section_depth;// edgeWeightDistributionPerSectionDepth;
        double min_max_ratio_rgb;// edgeDistributionMinMaxRatioRgb;
        std::vector< double > distribution_per_section_rgb;// edgeWeightDistributionPerSectionRgb;
        double dir_ratio_1; // dirRatio1;
        double dir_ratio_2;
        std::vector< double > edge_weights_per_dir;
        double new_cost;
    };
    struct svm_features
    {
        double max_over_min_depth;
        double max_over_min_rgb;
        double max_over_min_perp;
        double max_over_min_diag;
        double initial_cost;
        double final_cost;
        double xy_movement;
        double xy_movement_from_origin;
        double positive_improvement_sum;
        double negative_improvement_sum;
    };
    struct svm_model_linear
    {
        svm_model_linear();
        std::vector< double > mu = { 0,0,0,0,0,0,0,0,0,0 };
        std::vector< double > sigma = { 1,1,1,1,1,1,1,1,1,1 };
        std::vector< double > beta = { -0.00598037736275758, -0.419810195765952, -0.0519142921084610, 0.0511261823872767, -3.260600000000000e-04, 0.00134513193142993, -0.0305126906030304, 0, 0.000390629893900041, -0.000589640043298588 };// { 0.036404266074054, 0.217416714219929, -1.448206804928638, -0.426383040511132, 0.482625970398315, -0.341211816013196, 1.476722279284806, -0.570440485581453, 0.174994874978338, -0.217118614894225 };
        double bias = 14.782481183385837;
    };
    struct svm_model_gaussian
    {
        svm_model_gaussian();
        std::vector< double > mu = { 26.451657421181550, 1.834669247710750, 3.491429858876404, 3.786071463994616, 1.267059679244040e+04, 1.739017433204909e+04, 7.950284312071314, 7.037505628020683, 1.669357700781532e+04, -4.000499638952811e+02 };
        std::vector< double > sigma = {64.543314844188930, 0.399466249006394, 5.051691865199965, 4.838800439657138, 3.560637036816310e+03, 4.169466021585641e+03, 3.946730380178832, 2.688331347779885, 9.166971780432328e+03, 1.309539174549818e+03};
        std::vector< double > alpha = { 0.853782362459551, 0.809419984641913, 0.853782362459551, 0.523660624979399, 1.11094229588706, 0.853782362459551, 0.853782362459551, 0.739635480910988, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.858846426133943, 1.11094229588706, 0.853782362459551, 1.11094229588706, 0.853782362459551, 0.565954460907804, 0.410526909886890, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.853782362459551, 1.11094229588706, 0.147472046069076, 0.853782362459551, 1.11094229588706, 0.853782362459551, 0.474919097314781, 1.07387030895150, 1.11094229588706, 0.728324034495017, 0.811599038284503, 0.118602560912953, 1.00766686099462, 0.759793036827252, 0.804341226802145, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.912663846986493, 0.502945760137574, 0.741688370316241, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.720528402669790, 0.853782362459551, 0.853782362459551, 0.626880433540383, 0.853782362459551, 1.11094229588706, 1.11094229588706, 0.828188533980146, 0.853782362459551, 0.853782362459551, 1.08218326015022, 1.11094229588706, 1.11094229588706, 0.874637498781498, 0.853782362459551, 0.853782362459551, 0.638113857711959, 0.567946527216678, 0.756319722985775, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.858929103546975, 0.853782362459551, 0.0551892575127066, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.853782362459551, 1.11094229588706, 1.11094229588706, 1.10609139305017, 0.904762930421056, 1.11094229588706, 0.877257559768069, 1.11094229588706, 0.164605124354074, 1.11094229588706, 1.11094229588706, 1.11094229588706, 1.11094229588706, 1.11094229588706, 0.163757887887552, 0.853782362459551, 0.853782362459551, 1.10487919283239, 0.853782362459551, 0.641471719725152, 1.11094229588706, 0.853782362459551, 0.582518093538500, 0.853782362459551, 0.590370095260419, 0.625770558050165, 0.849648792088831, 0.464867191830818, 0.371399214192577, 0.508643001909889, 0.508719179090522, 0.849956268470741, 0.395653653246930, 0.666814548675369, 0.617065518321757, 0.853782362459551, 0.797324899450533, 0.645341371666331, 0.643837379212830, 0.670038550795845, 0.841968985583819, 1.06151054576426, 0.734320857082393, 0.580669562338322, 0.434135177363688, 1.10848731131424, 0.665617404389425, 0.709773717912651, 0.825756139249980, 1.11094229588706, 0.690353773003999, 1.11094229588706, 0.219534940218192, 0.657932066278636, 0.476475742895029, 0.661076180573250, 0.657638691861222, 0.407781991490912, 0.735521726220470, 0.722224781567731, 0.741663155838191, 0.657750600427332, 0.853782362459551, 0.851047964636070, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.572911397588253, 0.657581429355579, 0.509194534549591, 0.488095502769136, 0.853180064239348, 1.11094229588706, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.574573645023846, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.313804046292503, 0.0818345445662691, 0.608815326606831, 1.11094229588706, 1.11094229588706, 0.813193890719519, 0.807412795095128, 1.11094229588706, 1.11094229588706, 1.11094229588706, 0.489360336829842, 0.932772552136191, 0.853782362459551, 1.00400067721180, 1.11094229588706, 0.853782362459551, 0.626502674030708, 1.11094229588706, 0.853782362459551, 1.11094229588706, 0.853782362459551, 1.11094229588706, 0.853782362459551, 1.11094229588706, 0.853782362459551, 0.853782362459551, 0.472192610210062, 0.0505120252117739, 0.853782362459551, 0.853782362459551, 0.473165559571867, 0.296883608413205, 0.815314527893011, 0.683661864430910, 0.659295345426884, 0.541872778451743, 0.515306135002716, 0.606229468786921, 0.481728358034791, 0.467598697870095, 0.550458139680351, 0.654416225571927, 0.167005758120683, 0.853782362459551, 0.339472108535104, 1.11094229588706, 0.135084020329085, 0.656894107435376, 0.853782362459551, 0.416327424192007, 0.853782362459551, 1.11094229588706, 0.853782362459551, 0.122200722533292, 1.11094229588706, 0.853782362459551, 1.11094229588706, 0.481820439865076, 1.10047904288990, 0.723810502326994, 0.853782362459551, 1.04408869091412, 0.442766463043284, 0.473432211087953, 0.621097330923599, 0.636524721655964, 0.658340995052940, 0.657854873791700, 0.770188713793156, 1.11094229588706, 0.853782362459551, 1.11094229588706, 1.11094229588706, 1.11094229588706, 1.11094229588706, 0.542621644264350, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.820277591673038, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.852910435896307, 0.277144002177960, 1.09737060681344, 0.816340094455241, 1.11094229588706, 0.387365127119839, 0.822911949368766, 0.583552640059492, 1.10698101307889, 0.199605742376812, 0.309716857446239, 1.11094229588706, 0.825553571525066, 0.316140379615421, 0.828377208677995, 0.830376808053154, 0.772655504859029, 1.11094229588706, 0.515252846432175, 0.516918757966514, 0.741182454077847, 0.383763427412591, 1.11094229588706, 0.853782362459551, 1.11026858485958, 0.0978599136880101, 0.156617362810719, 0.853782362459551, 0.290818738206244, 0.853782362459551, 0.853782362459551, 0.853782362459551, 1.10819605976630, 0.386039711890531, 0.290732589038562, 0.853782362459551, 0.674592403169784, 0.360752576373701, 0.853782362459551, 1.11094229588706, 0.853782362459551, 0.853782362459551, 0.936084924636588, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.424951759018112, 1.11094229588706, 0.985693515370822, 0.853782362459551, 0.316412633956148, 0.367991629921835, 0.853782362459551, 0.853782362459551, 0.0914759047151927, 0.953020231941869, 0.853782362459551, 0.616881479123683, 0.853782362459551, 0.853782362459551, 0.673529054889632, 0.540929179359247, 0.657743115458036, 0.621498920245097, 0.587012931684455, 1.11017036408162, 1.10745536715549, 0.659692571990333, 0.608207729800647, 0.212511065146229, 0.296070075276644, 0.853782362459551, 0.349119696103402, 0.841330883963117, 0.581304165917077, 1.11094229588706, 0.572646085149409, 0.798236813225304, 0.853782362459551, 0.853782362459551, 0.853782362459551, 0.655613482190814, 0.853782362459551, 0.346650809612394, 0.845449841690559, 0.742414233118956, 0.497349430092853, 1.02290125480437, 0.531325561408872, 0.408979355805631, 0.806101219345509, 0.393215113631691, 0.621176662488951, 1.11094229588706, 1.11094229588706, 1.11094229588706, 0.951142231548956 };
        double bias = -0.342210526291181;
        double kernel_param_scale = 1;
        std::vector< double > support_vectors_labels = { 1, -1, 1, -1, -1, 1, 1, 1, 1, 1, 1, -1, -1, 1, -1, 1, -1, -1, 1, 1, 1, 1, -1, -1, 1, -1, 1, 1, -1, -1, -1, 1, 1, -1, 1, 1, 1, 1, 1, 1, 1, 1, 1, -1, -1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, -1, -1, 1, 1, 1, -1, -1, -1, -1, 1, 1, 1, -1, 1, 1, 1, 1, -1, 1, -1, 1, 1, 1, 1, 1, 1, 1, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 1, 1, -1, 1, 1, -1, 1, -1, 1, 1, -1, 1, -1, -1, 1, -1, 1, -1, -1, 1, 1, -1, -1, -1, 1, 1, -1, -1, -1, 1, -1, 1, -1, -1, -1, 1, -1, -1, -1, -1, -1, -1, -1, 1, -1, -1, -1, 1, -1, 1, 1, 1, 1, -1, -1, -1, -1, 1, -1, 1, 1, 1, -1, 1, 1, 1, 1, 1, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 1, -1, -1, 1, -1, -1, 1, -1, 1, -1, 1, -1, 1, 1, -1, 1, 1, 1, 1, 1, 1, -1, -1, 1, -1, -1, -1, -1, -1, -1, -1, 1, 1, -1, 1, -1, 1, 1, 1, -1, 1, 1, -1, 1, -1, 1, -1, -1, 1, -1, -1, 1, -1, -1, -1, -1, 1, -1, 1, -1, -1, -1, -1, -1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, -1, 1, -1, -1, 1, 1, -1, -1, 1, -1, -1, -1, 1, 1, -1, -1, 1, 1, -1, -1, -1, 1, -1, -1, -1, 1, -1, 1, 1, 1, -1, -1, -1, 1, -1, -1, 1, -1, -1, 1, -1, 1, 1, 1, -1, -1, -1, 1, -1, 1, 1, 1, 1, -1, 1, 1, 1, 1, 1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 1, 1, 1, 1, -1, -1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, -1, 1, 1, -1, 1, 1, -1, -1, -1, -1 };
        std::vector< double > support_vectors[10] = {
            {-0.0562123043369483, -0.0324974207934130, -0.0455795300828793, -0.0583388591877621, -0.0274359457172453, -0.00286615800048622, -0.0850389364052901, -0.0513925026855241, -0.0631250403340051, -0.0615118136651025, -0.0879080977150691, -0.0251284074787236, -0.0444919493167321, -0.0537704917650131, -0.0771568631874931, 0.00567872580422870, -0.0448018187377292, -0.0330399671536552, -0.111479363523474, -0.133896656168909, -0.135858082969125, -0.126684814651919, -0.157647032033827, -0.151843688868152, -0.116643853844292, -0.151720854659326, -0.143974119155527, -0.112671168982125, -0.143856744663014, -0.100774023471855, -0.132079187607702, -0.152777227682572, -0.104146002431311, -0.151123062310040, -0.104146002431311, -0.188970085737752, -0.128300292246279, -0.301025529869738, -0.304239542177909, -0.305034243161145, -0.299932444643398, -0.309162765273118, -0.286860826025910, -0.300112601286068, -0.295385423649795, -0.293959644085562, -0.299459798400177, -0.307099545823721, -0.308862208872332, -0.296878678382869, -0.305849589458012, -0.307708045125148, 0.348160236158141, 0.505695227718318, 0.485976265962213, 0.417781529492118, 0.281844782481576, 0.578655377229695, 0.153405223155692, 0.413908161974968, 0.266351312228530, 0.366037317121122, 0.320748712014903, 0.344584819965545, 0.292911546948038, 0.203132425686729, 0.408743671952102, 0.351735652350737, 0.574007336286582, 0.463440309669521, 0.350543846953205, -0.380930061392109, -0.380062386221645, -0.380742525503985, -0.379892647099266, -0.380930061392109, -0.380742525503985, -0.380930061392109, -0.380742525503985, -0.380930061392109, -0.380742525503985, -0.380930061392109, -0.380742525503985, -0.380171120008432, -0.381322845769040, -0.380030087870659, -0.380013725018617, -0.380171120008432, -0.381322845769040, -0.380030087870659, -0.380013725018617, -0.380171120008432, -0.380030087870659, -0.380171120008432, -0.380030087870659, -0.380171120008432, -0.380030087870659, 2.14659456464675, 1.88984565945844, 2.67647118955455, 2.14659456464675, 1.88984565945844, 2.67647118955455, 2.14659456464675, 2.67647118955455, 2.14659456464675, 2.14659456464675, 0.111494574553853, 0.0957018330431097, 0.0861782670520956, 0.0444485231406666, 0.0957018330431097, 0.177101136398129, 0.0181361027093097, 0.108156423705904, -0.0534781621222321, 0.0181361027093097, -0.316313830554705, -0.312840904894096, -0.312128615753399, -0.316462110744288, -0.316313830554705, -0.165507874069173, -0.173384153709154, -0.159562868123635, -0.196173028758245, -0.165507874069173, -0.322441539624239, -0.322915422504161, -0.311887124731291, -0.316217742131253, -0.322441539624239, -0.322915422504161, -0.322441539624239, -0.322441539624239, -0.234632594250674, -0.223045597135125, -0.166699686224635, -0.250816057954379, -0.332888826834439, -0.336684399748136, -0.318683645095036, -0.323348098275387, -0.115452058153772, -0.196667821564477, -0.188717782031995, -0.196667821564477, -0.196667821564477, -0.196667821564477, -0.196667821564477, 0.0859630372597497, 0.144063535201371, 0.125557454159761, 0.165367055658050, -0.343522603467787, -0.339435693676029, -0.339590926933874, -0.343364526160185, -0.343522603467787, -0.339435693676029, -0.339590926933874, -0.343364526160185, -0.343522603467787, -0.339590926933874, -0.343364526160185, -0.343522603467787, 0.0839244591927724, 0.00107176645329464, 0.103902882262456, 0.00711852332056431, 0.0839244591927724, 0.00107176645329464, 0.0839244591927724, 0.00107176645329464, 0.0839244591927724, -0.241259035021247, -0.220955211947689, -0.214315153109716, -0.241259035021247, -0.246216945658042, -0.214315153109716, -0.241259035021247, -0.214315153109716, -0.241259035021247, -0.214315153109716, -0.241259035021247, -0.214315153109716, -0.352834155784050, -0.350861068039928, -0.352163553789278, -0.350419255694952, -0.352834155784050, -0.350861068039928, -0.352834155784050, -0.350861068039928, 3.16916030843535, 3.15366685256908, 1.38741378243466, 3.16916030843535, -0.304509281897474, -0.304227759057322, -0.303641035721330, -0.239953153011648, -0.263474284848948, -0.254606369813228, -0.250352095943331, -0.234771892835509, -0.196701270276139, -0.216661193267720, -0.234771892835509, -0.216661193267720, -0.284911885395958, -0.294840307002373, -0.283888209847379, -0.306353742751674, -0.284911885395958, -0.283888209847379, -0.306353742751674, -0.284911885395958, -0.306353742751674, -0.284911885395958, -0.306353742751674, -0.291261546452128, -0.290035563399750, -0.293477411041730, -0.289632551358936, -0.290035563399750, 1.68488891221884, 3.13301127330454, 4.47061304141983, 9.25808846387818, -0.316017435338466, -0.308026776231821, -0.316017435338466, -0.308026776231821, -0.308026776231821, -0.308026776231821, -0.308026776231821, -0.294834486496376, -0.282563195685673, -0.286209874450064, -0.287326283448068, -0.282563195685673, -0.286209874450064, -0.287326283448068, -0.286209874450064, -0.287326283448068, -0.286209874450064, -0.287326283448068, -0.286209874450064, -0.287326283448068, -0.232553859769381, -0.235088585408672, -0.237264348426925, -0.232553859769381, -0.237264348426925, -0.237264348426925, -0.226305330061371, -0.202043331539786, -0.223751406262257, -0.224440598627960, -0.226305330061371, -0.223751406262257, -0.223751406262257, -0.344219019334402, 0.816092873585745, 0.684630776400184, 1.48922878577253, 2.02687227747499, 0.816092873585745, 0.684630776400184, 2.02687227747499, -0.320571147349952, 0.00286172794913721, 0.0612882816690623, 0.166247412244897, 0.542424515955872, 0.00286172794913721, 0.0612882816690623, 0.166247412244897, 0.0612882816690623, 0.0612882816690623, 0.0612882816690623, -0.336391161237969, -0.334917444445115, -0.336391161237969, -0.203414977228364, -0.207884684767060, -0.207479529785634, -0.203414977228364, -0.207884684767060, -0.207479529785634, -0.203414977228364, -0.207884684767060, -0.203414977228364, -0.203414977228364, -0.348450014764882, -0.353504672038965, -0.349281142286866, -0.355600847558802, -0.348450014764882, -0.349281142286866, -0.348450014764882, -0.317463073034788, -0.306655564735309, -0.325507759932246, -0.340107376755598, -0.317463073034788, -0.306655564735309, -0.317463073034788, -0.317463073034788, -0.317463073034788, -0.306021748942754, -0.322826202652428, -0.316867177810017, -0.344534086320147, -0.238293147442116, -0.239399824825181, -0.266513398537719, -0.209962236118425, -0.238293147442116, -0.239399824825181, -0.365663617047502, -0.365663617047502, -0.369958582494668, -0.373376551681660, -0.371281602077414, -0.371281602077414, -0.288508583084618, -0.281952304078295, -0.288508583084618, -0.281952304078295, -0.288508583084618, -0.281952304078295, -0.281952304078295, -0.369644016342871, -0.365477017255581, -0.365571526062232, -0.366578448863611, -0.369644016342871, -0.365477017255581, -0.366578448863611, -0.373469392014868, -0.372385440940071, -0.378193807190106, -0.378193807190106, -0.378193807190106, -0.378193807190106},
            {-1.21987837917425, -1.21987837917425, -1.21987837917425, -1.21987837917425, -1.21987837917425, -1.21987837917425, -1.21987837917425, -1.21987837917425, -1.21987837917425, -1.21987837917425, -1.21987837917425, -1.21987837917425, -1.21987837917425, -1.21987837917425, -1.21987837917425, -1.21987837917425, -1.21987837917425, -1.21987837917425, -1.21438737126708, -1.21438737126708, -1.21438737126708, -1.21438737126708, -1.21438737126708, -1.21438737126708, -1.21438737126708, -1.21438737126708, -1.21438737126708, -1.21438737126708, -1.21438737126708, -1.21438737126708, -1.21438737126708, -1.21438737126708, -1.21438737126708, -1.21438737126708, -1.21438737126708, -1.21438737126708, -1.21438737126708, -1.75854819606953, -1.75854819606953, -1.75854819606953, -1.75854819606953, -1.75854819606953, -1.75854819606953, -1.75854819606953, -1.75854819606953, -1.75854819606953, -1.75854819606953, -1.75854819606953, -1.75854819606953, -1.75854819606953, -1.75854819606953, -1.75854819606953, -1.75292361167551, -1.75292361167551, -1.75292361167551, -1.75292361167551, -1.75292361167551, -1.75292361167551, -1.75292361167551, -1.75292361167551, -1.75292361167551, -1.75292361167551, -1.75292361167551, -1.75292361167551, -1.75292361167551, -1.75292361167551, -1.75292361167551, -1.75292361167551, -1.75292361167551, -1.75292361167551, -1.75292361167551, -1.54465504221194, -1.54465504221194, -1.54465504221194, -1.54465504221194, -1.54465504221194, -1.54465504221194, -1.54465504221194, -1.54465504221194, -1.54465504221194, -1.54465504221194, -1.54465504221194, -1.54465504221194, -1.55422294355045, -1.55422294355045, -1.55422294355045, -1.55422294355045, -1.55422294355045, -1.55422294355045, -1.55422294355045, -1.55422294355045, -1.55422294355045, -1.55422294355045, -1.55422294355045, -1.55422294355045, -1.55422294355045, -1.55422294355045, 0.0753935137262907, 0.0753935137262907, 0.0753935137262907, 0.0753935137262907, 0.0753935137262907, 0.0753935137262907, 0.0753935137262907, 0.0753935137262907, 0.0753935137262907, 0.0753935137262907, -0.349904922225673, -0.349904922225673, -0.349904922225673, -0.349904922225673, -0.349904922225673, -0.361105755406697, -0.361105755406697, -0.361105755406697, -0.361105755406697, -0.361105755406697, -1.08179416439212, -1.08179416439212, -1.08179416439212, -1.08179416439212, -1.08179416439212, -0.808250847685785, -0.808250847685785, -0.808250847685785, -0.808250847685785, -0.808250847685785, -0.655734692966619, -0.655734692966619, -0.655734692966619, -0.655734692966619, -0.655734692966619, -0.655734692966619, -0.655734692966619, -0.655734692966619, -0.673259436879201, -0.673259436879201, -0.673259436879201, -0.673259436879201, 0.400937185066202, 0.400937185066202, 0.400937185066202, 0.400937185066202, 0.0765135259095146, 0.0883986307707854, 0.0883986307707854, 0.0883986307707854, 0.0883986307707854, 0.0883986307707854, 0.0883986307707854, 0.0565645180689857, 0.0565645180689857, 0.0565645180689857, 0.0565645180689857, -0.319942922031094, -0.319942922031094, -0.319942922031094, -0.319942922031094, -0.319942922031094, -0.319942922031094, -0.319942922031094, -0.319942922031094, -0.319942922031094, -0.319942922031094, -0.319942922031094, -0.319942922031094, 0.687663825936970, 0.687663825936970, 0.687663825936970, 0.687663825936970, 0.687663825936970, 0.687663825936970, 0.687663825936970, 0.687663825936970, 0.687663825936970, 0.674502743323060, 0.674502743323060, 0.674502743323060, 0.674502743323060, 0.674502743323060, 0.674502743323060, 0.674502743323060, 0.674502743323060, 0.674502743323060, 0.674502743323060, 0.674502743323060, 0.674502743323060, 0.170803649772231, 0.170803649772231, 0.170803649772231, 0.170803649772231, 0.170803649772231, 0.170803649772231, 0.170803649772231, 0.170803649772231, 0.168632359461486, 0.168632359461486, 0.168632359461486, 0.168632359461486, 0.694066342873892, 0.694066342873892, 0.694066342873892, 0.671739914720242, 0.671739914720242, 0.671739914720242, 0.671739914720242, 1.27959245254742, 1.27959245254742, 1.27959245254742, 1.27959245254742, 1.27959245254742, 1.26773215910237, 1.26773215910237, 1.26773215910237, 1.26773215910237, 1.26773215910237, 1.26773215910237, 1.26773215910237, 1.26773215910237, 1.26773215910237, 1.26773215910237, 1.26773215910237, 0.750603628375997, 0.750603628375997, 0.750603628375997, 0.750603628375997, 0.750603628375997, 0.792591871839333, 0.792591871839333, 0.792591871839333, 0.792591871839333, 1.05052406413540, 1.05052406413540, 1.05052406413540, 1.05052406413540, 1.05052406413540, 1.05052406413540, 1.05052406413540, 1.06305188874812, 1.06305188874812, 1.06305188874812, 1.06305188874812, 1.06305188874812, 1.06305188874812, 1.06305188874812, 1.06305188874812, 1.06305188874812, 1.06305188874812, 1.06305188874812, 1.06305188874812, 1.06305188874812, 1.82660906215719, 1.82660906215719, 1.82660906215719, 1.82660906215719, 1.82660906215719, 1.82660906215719, 1.83897317208506, 1.83897317208506, 1.83897317208506, 1.83897317208506, 1.83897317208506, 1.83897317208506, 1.83897317208506, 0.908065308383070, 0.305100482975081, 0.305100482975081, 0.305100482975081, 0.305100482975081, 0.305100482975081, 0.305100482975081, 0.305100482975081, -1.57631673024680, -1.59206593323724, -1.59206593323724, -1.59206593323724, -1.59206593323724, -1.59206593323724, -1.59206593323724, -1.59206593323724, -1.59206593323724, -1.59206593323724, -1.59206593323724, -1.31619450032808, -1.31619450032808, -1.31619450032808, -1.32035207258010, -1.32035207258010, -1.32035207258010, -1.32035207258010, -1.32035207258010, -1.32035207258010, -1.32035207258010, -1.32035207258010, -1.32035207258010, -1.32035207258010, 1.25108244163170, 1.25108244163170, 1.25108244163170, 1.25108244163170, 1.25108244163170, 1.25108244163170, 1.25108244163170, 1.24621153720812, 1.24621153720812, 1.24621153720812, 1.24621153720812, 1.24621153720812, 1.24621153720812, 1.24621153720812, 1.24621153720812, 1.24621153720812, -0.0449581753031382, -0.0449581753031382, -0.0449581753031382, -0.0449581753031382, -0.00762733064969650, -0.00762733064969650, -0.00762733064969650, -0.00762733064969650, -0.00762733064969650, -0.00762733064969650, -0.00472169089691700, -0.00472169089691700, 0.00756552273551637, 0.00756552273551637, 0.00756552273551637, 0.00756552273551637, 1.08795057015157, 1.08795057015157, 1.08795057015157, 1.08795057015157, 1.08795057015157, 1.08795057015157, 1.08795057015157, 1.03255577362754, 1.03255577362754, 1.03255577362754, 1.03255577362754, 1.03255577362754, 1.03255577362754, 1.03255577362754, -0.0830525352135685, -0.0830525352135685, -0.0535324720554350, -0.0535324720554350, -0.0535324720554350, -0.0535324720554350},
            {-0.455346387028116, -0.468730452447457, -0.458946616111499, -0.476899906663255, -0.464753110819491, -0.472558381909235, -0.458683594732494, -0.443589601993333, -0.456920931618443, -0.455094548337438, -0.444602745035972, -0.468652729674597, -0.452231325554993, -0.458328814416776, -0.445767853180217, -0.451197085825580, -0.454326502287152, -0.465124020329030, -0.260764224383015, -0.266405559173948, -0.237283623548776, -0.271224526747735, -0.320086926706107, -0.249353892709193, -0.241246422896001, -0.245486264273016, -0.302565346668632, -0.283785823994845, -0.282619053543948, -0.328295734278742, -0.192901851389046, -0.254243368687387, -0.246890340825031, -0.280570515333163, -0.241801562353596, -0.271151804063531, -0.265391502730347, -0.452564111538745, -0.457674730325576, -0.449601130600087, -0.459601852368803, -0.454147668587520, -0.448922932387875, -0.458722103240935, -0.452867579367921, -0.456628241998785, -0.451632969573302, -0.447699062579495, -0.462369460437271, -0.459401343485521, -0.466413022427727, -0.465419987935184, -0.397128159199368, -0.384082624902792, -0.388010078392084, -0.371513683409782, -0.379072858252143, -0.382957901342264, -0.393302433992645, -0.378607085351951, -0.366070031486966, -0.390324554627094, -0.358083966986187, -0.405159232035796, -0.408536058439525, -0.402568513401922, -0.404042604084960, -0.403442581306979, -0.390611323255539, -0.388608020455881, -0.388205147105909, -0.459160082893752, -0.455796005159578, -0.462978436464233, -0.445658157583295, -0.459160082893752, -0.462978436464233, -0.459160082893752, -0.462978436464233, -0.459160082893752, -0.462978436464233, -0.459160082893752, -0.462978436464233, -0.435269510046111, -0.445782568768696, -0.450602701433552, -0.434554257670807, -0.435269510046111, -0.445782568768696, -0.450602701433552, -0.434554257670807, -0.435269510046111, -0.450602701433552, -0.435269510046111, -0.450602701433552, -0.435269510046111, -0.450602701433552, 0.490148491024994, 0.343446367142837, 0.468909275761798, 0.490148491024994, 0.343446367142837, 0.468909275761798, 0.490148491024994, 0.468909275761798, 0.490148491024994, 0.490148491024994, 0.305926514756817, 0.233751200663184, 0.199649970507922, 0.272061525735601, 0.233751200663184, 0.680948630272350, 0.587582502956514, 0.734727273333442, 0.474490962034151, 0.587582502956514, 0.381427198255032, 0.368973816482603, 0.473185690834583, 0.385793727480614, 0.381427198255032, 1.48634759853398, 1.18501841085869, 1.33116184443119, 1.26176879539380, 1.48634759853398, 0.0946746382493046, 0.134641577528356, 0.0677739174046016, 1.14681149328697, 0.0946746382493046, 0.134641577528356, 0.0946746382493046, 0.0946746382493046, 0.565677155545849, 0.472752452057500, 0.243639627813908, 11.3120375885850, 0.144738215611596, 0.0946543722512796, 0.0381962154643959, 0.0566104928960270, -0.262241491515252, 0.0541423294345829, -0.00205263232167131, 0.0541423294345829, 0.0541423294345829, 0.0541423294345829, 0.0541423294345829, 1.99287556335306, 2.19898005477759, 1.79741730011656, 2.35324724326977, -0.424539130206231, -0.433289854935123, -0.455959077281552, -0.431026125077710, -0.424539130206231, -0.433289854935123, -0.455959077281552, -0.431026125077710, -0.424539130206231, -0.455959077281552, -0.431026125077710, -0.424539130206231, -0.369164007003499, -0.345404704861329, -0.375796199840768, -0.363529420904128, -0.369164007003499, -0.345404704861329, -0.369164007003499, -0.345404704861329, -0.369164007003499, -0.169002532488381, -0.0940874589394906, -0.129013257628514, -0.169002532488381, -0.160395859374761, -0.129013257628514, -0.169002532488381, -0.129013257628514, -0.169002532488381, -0.129013257628514, -0.169002532488381, -0.129013257628514, -0.0797565032844359, -0.0486139495704364, -0.114988552881947, -0.0382297239152158, -0.0797565032844359, -0.0486139495704364, -0.0797565032844359, -0.0486139495704364, -0.488111489300505, -0.484996041146409, -0.482629700627788, -0.488111489300505, -0.424767055623732, -0.412171508201987, -0.407336425627621, -0.355653273627718, -0.455095223917619, -0.425685840073863, -0.381838385327480, -0.399984964586631, -0.404866358181129, -0.398395540185751, -0.399984964586631, -0.398395540185751, -0.352857181852127, -0.421878327521780, -0.380796047776644, -0.425717543292975, -0.352857181852127, -0.380796047776644, -0.425717543292975, -0.352857181852127, -0.425717543292975, -0.352857181852127, -0.425717543292975, 0.0274677904397428, 0.0379257102791340, 0.0204218173894372, 0.0449387535706461, 0.0379257102791340, 0.400648885183352, 0.335284758670244, 0.232642212956276, 0.312437414197087, -0.288891976410425, -0.298741385904160, -0.288891976410425, -0.298741385904160, -0.298741385904160, -0.298741385904160, -0.298741385904160, -0.162903188437913, -0.243795084453742, -0.237401765918999, -0.222316693389635, -0.243795084453742, -0.237401765918999, -0.222316693389635, -0.237401765918999, -0.222316693389635, -0.237401765918999, -0.222316693389635, -0.237401765918999, -0.222316693389635, -0.262270711030727, -0.252991030710321, -0.269163567602038, -0.262270711030727, -0.269163567602038, -0.269163567602038, -0.320231968125509, -0.320196896169068, -0.329461375374193, -0.320645239981881, -0.320231968125509, -0.329461375374193, -0.329461375374193, -0.291021959523801, -0.130576576715280, -0.213673502951809, -0.210269432406720, -0.218579408813314, -0.130576576715280, -0.213673502951809, -0.218579408813314, -0.0303873363907555, 0.140117833220232, 0.141520771626209, 0.119929403047299, 0.146888048493643, 0.140117833220232, 0.141520771626209, 0.119929403047299, 0.141520771626209, 0.141520771626209, 0.141520771626209, 0.0749759872667360, 0.0414275773972281, 0.0749759872667360, 0.311823609207830, 0.239452003869262, 0.238379998257423, 0.311823609207830, 0.239452003869262, 0.238379998257423, 0.311823609207830, 0.239452003869262, 0.311823609207830, 0.311823609207830, -0.130700977433798, -0.167146193596160, -0.0615386598768579, -0.144889324695907, -0.130700977433798, -0.0615386598768579, -0.130700977433798, -0.401382708655795, -0.401280247512171, -0.359434869405504, -0.453125206740792, -0.401382708655795, -0.401280247512171, -0.401382708655795, -0.401382708655795, -0.401382708655795, -0.199922817196940, 0.120468538533776, 0.0252623460102056, 0.224394100493349, -0.178790540101980, -0.295233752499250, -0.479535261282215, -0.460194978857168, -0.178790540101980, -0.295233752499250, -0.425698191828129, -0.425698191828129, -0.388123069706955, -0.458102179795361, -0.489346851293502, -0.489346851293502, -0.406443557513611, -0.411844850201016, -0.406443557513611, -0.411844850201016, -0.406443557513611, -0.411844850201016, -0.411844850201016, -0.317037308416094, -0.307194671977445, -0.356310292961308, -0.327837839444650, -0.317037308416094, -0.307194671977445, -0.327837839444650, -0.401824072625035, -0.386891377953320, -0.300612907082113, -0.300612907082113, -0.300612907082113, -0.300612907082113},
            {0.0146878036505866, -0.0715201014611666, -0.0154441781599043, -0.0641364210551600, -0.00266091291952416, -0.117182736233026, -0.0735867293423339, -0.0570537062929955, 0.00463733399775743, 0.0256346284929428, -0.0246765362312497, -0.0160655869691669, -0.0508538226494937, -0.0141936417733490, -0.0127266472030019, -0.0730303292553236, -0.0664265944324324, -0.0246765364704428, 1.48186516228254, 1.52037374551689, 1.09405796722225, 1.29245423786242, 1.23252202774826, 0.892241080465828, 1.42196291085630, 1.26696582467700, 1.45490914734366, 1.24285517020238, 0.977756714834265, 1.55284931972475, 1.44779575855285, 1.17290777010029, 1.17290777010029, 1.26696582467700, 1.17290777010029, 1.20470204429025, 1.35858632807581, 0.126876175563901, 0.257400040762596, 0.140519932285093, 0.162304082115068, 0.104331144506133, 0.230656835238993, 0.197141523090582, 0.143568169911863, 0.195074895209414, 0.162888700678495, 0.140519932285093, 0.235646063077548, 0.257762606449042, 0.143643156577770, 0.159942221229570, 2.13779485912842, 2.01611840787698, 1.80084468535214, 1.75315327406719, 1.57942028230350, 1.86443323373208, 1.66689658924460, 1.80497793701620, 1.98167461094824, 1.60156272331805, 1.78494754825716, 1.76110184261468, 2.19538270334008, 1.71282165202887, 1.83804398212289, 1.99889650941261, 2.30810785649804, 1.80879325389964, 1.78494754825716, 0.0146878057116800, 0.0424139932373443, 0.0897147937676463, 0.0551843062345696, 0.0146878057116800, 0.0897147937676463, 0.0146878057116800, 0.0897147937676463, 0.0146878057116800, 0.0897147937676463, 0.0146878057116800, 0.0897147937676463, -0.0438747173255158, -0.0539537632571007, -0.0416065374409651, -0.0322020706585786, -0.0438747173255158, -0.0539537632571007, -0.0416065374409651, -0.0322020706585786, -0.0438747173255158, -0.0416065374409651, -0.0438747173255158, -0.0416065374409651, -0.0438747173255158, -0.0416065374409651, -0.499485899868839, -0.446495101855180, -0.492335989415017, -0.499485899868839, -0.446495101855180, -0.492335989415017, -0.499485899868839, -0.492335989415017, -0.499485899868839, -0.499485899868839, -0.495346379428251, -0.493112187129704, -0.509708636158468, -0.507276715414884, -0.493112187129704, -0.563188707155963, -0.542444595595567, -0.569450891541650, -0.561562402655941, -0.542444595595567, -0.494633631062968, -0.467435901227490, -0.507936630468358, -0.490196086502091, -0.494633631062968, -0.574217584013908, -0.538202250696055, -0.537663264229382, -0.554305844666604, -0.574217584013908, -0.523589731735383, -0.544285831877649, -0.538021602696807, -0.517911723793302, -0.523589731735383, -0.544285831877649, -0.523589731735383, -0.523589731735383, 0.388648661426724, 0.0958765461173931, 0.457536235191985, 2.52416124374966, -0.160589900696309, -0.177760082459930, -0.136847488254109, -0.141402740079129, 8.36238608682430, -0.465685910978104, -0.474452343658915, -0.465685910978104, -0.465685910978104, -0.465685910978104, -0.465685910978104, 1.11196850169663, 0.312872585857810, 0.288448813741144, 1.24285484615520, -0.516361751884633, -0.523802230967843, -0.532464743079345, -0.501969164869399, -0.516361751884633, -0.523802230967843, -0.532464743079345, -0.501969164869399, -0.516361751884633, -0.532464743079345, -0.501969164869399, -0.516361751884633, -0.544912082031247, -0.503445327828003, -0.529687041658847, -0.539134965439934, -0.544912082031247, -0.503445327828003, -0.544912082031247, -0.503445327828003, -0.544912082031247, -0.349432362341764, -0.423499475892707, -0.447843210123798, -0.349432362341764, -0.398637786232781, -0.447843210123798, -0.349432362341764, -0.447843210123798, -0.349432362341764, -0.447843210123798, -0.349432362341764, -0.447843210123798, -0.385323359617264, -0.384779914380513, -0.394577263980320, -0.388063537087760, -0.385323359617264, -0.384779914380513, -0.385323359617264, -0.384779914380513, 2.34702486688598, 4.05937350301211, 3.65392084092929, 2.34702486688598, 1.07063619762551, 0.944670306552827, 1.38751911392757, 1.19154580075536, 0.971241248296592, 1.00598783060296, 1.36072950401619, 0.0951169728004483, 0.115779112618027, 0.137167821487774, 0.0951169728004483, 0.137167821487774, 0.0861426361794709, 0.263995288719754, 0.0603565854650902, 0.108985959231675, 0.0861426361794709, 0.0603565854650902, 0.108985959231675, 0.0861426361794709, 0.108985959231675, 0.0861426361794709, 0.108985959231675, 0.516583152575397, 0.575807514235328, 0.480122738905976, 0.557221045313747, 0.575807514235328, 0.735239725398820, 0.602759112289538, 0.606075485570604, 0.587438934636138, -0.558443506780906, -0.560356518538487, -0.558443506780906, -0.560356518538487, -0.560356518538487, -0.560356518538487, -0.560356518538487, -0.552654193154295, -0.566902213382612, -0.557629980210787, -0.557747237574831, -0.566902213382612, -0.557629980210787, -0.557747237574831, -0.557629980210787, -0.557747237574831, -0.557629980210787, -0.557747237574831, -0.557629980210787, -0.557747237574831, -0.512450842115765, -0.516180978743920, -0.521398499483264, -0.512450842115765, -0.521398499483264, -0.521398499483264, -0.529397098370384, -0.522796956226171, -0.524478028406689, -0.530356909115416, -0.529397098370384, -0.524478028406689, -0.524478028406689, -0.567360584888808, -0.548668214477132, -0.543611888049474, -0.543500251088595, -0.527913842886170, -0.548668214477132, -0.543611888049474, -0.527913842886170, -0.491899495847044, -0.498429827032548, -0.467405353370493, -0.480587170568557, -0.473139408679441, -0.498429827032548, -0.467405353370493, -0.480587170568557, -0.467405353370493, -0.467405353370493, -0.467405353370493, -0.474727674100595, -0.464117706265432, -0.474727674100595, -0.503922240823354, -0.468124063080712, -0.481911379760844, -0.503922240823354, -0.468124063080712, -0.481911379760844, -0.503922240823354, -0.468124063080712, -0.503922240823354, -0.503922240823354, -0.180821754511368, -0.201201001246597, -0.135707133264597, -0.220575637624468, -0.180821754511368, -0.135707133264597, -0.180821754511368, -0.525005475944051, -0.513778466463775, -0.531689241608024, -0.514876674700704, -0.525005475944051, -0.513778466463775, -0.525005475944051, -0.525005475944051, -0.525005475944051, 0.438749011119932, 0.664199307188112, 0.139593823463500, 0.664199277963077, -0.377549732273387, -0.378508279996288, -0.443301159193859, -0.359997042390898, -0.377549732273387, -0.378508279996288, -0.374699993643990, -0.374699993643990, -0.552527738764995, -0.529297869604807, -0.540649756477113, -0.540649756477113, -0.246344538911226, -0.250384714193122, -0.246344538911226, -0.250384714193122, -0.246344538911226, -0.250384714193122, -0.250384714193122, -0.525814870547797, -0.526240155150164, -0.551641502928304, -0.555111024192912, -0.525814870547797, -0.526240155150164, -0.555111024192912, -0.157390593752744, -0.147035345820185, -0.454777845783357, -0.454777845783357, -0.454777845783357, -0.454777845783357},
            {0.746652481987945, 1.26105059165182, 1.05723073915883, -0.506749404680611, 0.593626735473724, 0.0404471064820669, -0.586908921744923, 0.216619551560428, 0.941567458493259, 0.570350583248558, -0.0421640764210375, 0.793993850384826, -0.360756941393818, 1.19821784373011, 0.765698900398831, -0.0821038220422749, 0.920111611243153, 1.11128203089680, 0.262402539232842, 1.40961457512035, 1.52394300203970, 0.236468747229413, 0.215359023962812, 1.80301766858827, -0.215580968676673, 1.59950828546875, 1.32858590837004, -0.175726794119504, 2.20342168186837, 0.137672291237078, 0.236637146695239, -0.121476385097831, -0.230369952383949, 1.67607299504352, 0.154427489554299, -0.162912527275333, 1.04546526638195, -0.302888686519197, 0.923935156769341, 1.55766184100970, 1.12351979012344, 1.55576720988604, -0.242445071735610, 1.95812619286061, -0.0513997174832906, 0.0401130502127249, 1.32379392878224, 0.619799139098112, 0.472650095115247, 0.904372843577969, 0.191766642773318, 1.06499357056068, 1.27139371975480, 0.380139272604670, -0.577744864168958, 0.342920247994823, 0.115990722878870, 1.06158718397182, 2.35291707345733, 1.22926313548044, 1.09932402322577, 1.48929220641682, -0.693320379640610, 1.90135747130601, 0.513422513013761, 0.571554063470129, 0.334299511822442, 2.72206460998923, 0.108139577750639, 1.63198798960715, 0.00248893437970992, 0.747092405022448, -0.193455089024266, -0.0207973656960025, 0.992509393970536, 0.747092405022448, -0.0207973656960025, 0.747092405022448, -0.0207973656960025, 0.747092405022448, -0.0207973656960025, 0.747092405022448, -0.0207973656960025, 0.782446144995354, -0.307117871801016, -0.270987409861955, 1.00970561238718, 0.782446144995354, -0.307117871801016, -0.270987409861955, 1.00970561238718, 0.782446144995354, -0.270987409861955, 0.782446144995354, -0.270987409861955, 0.782446144995354, -0.270987409861955, 0.690011294230760, -0.382125846248701, 0.896664316889851, 0.690011294230760, -0.382125846248701, 0.896664316889851, 0.690011294230760, 0.896664316889851, 0.690011294230760, 0.690011294230760, 2.35331475949226, 0.512110971897181, 1.15525732984078, 2.56691220311994, 0.512110971897181, 2.18300640069863, -0.328959885904351, 0.528212319519730, 2.99212524259019, -0.328959885904351, 2.26087113220827, 0.522210003054581, 1.39277024027806, 2.75354982477407, 2.26087113220827, 2.08211811616256, -0.0194932297676597, 1.29415288013456, 2.51331399919555, 2.08211811616256, 0.173841150448476, -0.836938520165749, -0.116177835781395, 0.412256671051930, 0.173841150448476, -0.836938520165749, 0.173841150448476, 0.173841150448476, -0.169183624646849, -1.12131585002422, -0.416042979703697, -0.110207059932810, 0.685796645059184, -0.687395680518830, -0.0466889206998595, 0.334394133672133, -1.55028427986226, 0.883907097421726, 0.0586298605315750, 0.883907097421726, 0.883907097421726, 0.883907097421726, 0.883907097421726, 0.279572702357546, -1.88650190423760, 0.632381925243329, 0.732198925881166, 0.197104686427017, -0.279385716121005, -0.532218041156286, 0.0643529739735787, 0.197104686427017, -0.279385716121005, -0.532218041156286, 0.0643529739735787, 0.197104686427017, -0.532218041156286, 0.0643529739735787, 0.197104686427017, 0.254700869200498, -0.912479668166473, -0.340760739025025, 0.171503579361775, 0.254700869200498, -0.912479668166473, 0.254700869200498, -0.912479668166473, 0.254700869200498, -1.89355356330251, -1.68156894224614, -1.57478077968220, -1.89355356330251, -1.59428330283521, -1.57478077968220, -1.89355356330251, -1.57478077968220, -1.89355356330251, -1.57478077968220, -1.89355356330251, -1.57478077968220, 0.163240760620832, -0.980696262202460, -0.426603875194079, -0.170607614918133, 0.163240760620832, -0.980696262202460, 0.163240760620832, -0.980696262202460, -1.60149047085309, -1.30306399044543, -1.42029621108536, -1.60149047085309, 0.421574866249710, -0.287728894867627, 1.08741264081401, 0.144112336606881, -0.00174793035657583, -0.484171808545769, 0.643494918656688, -0.568070122870895, -0.696040277853307, 0.0694093462080534, -0.568070122870895, 0.0694093462080534, 0.370638141830822, -0.249954755406262, -0.542076266562879, 0.552835393205272, 0.370638141830822, -0.542076266562879, 0.552835393205272, 0.370638141830822, 0.552835393205272, 0.370638141830822, 0.552835393205272, -1.16349580222016, -1.49178013873109, -1.31896155825835, -0.923147527486940, -1.49178013873109, -1.41195755773155, -1.45112112991909, -1.31141868837375, -1.16864048818101, -0.751241293631580, -0.940358830546292, -0.751241293631580, -0.940358830546292, -0.940358830546292, -0.940358830546292, -0.940358830546292, -0.922091602497640, -1.29422672108368, -1.12606860954099, -1.01736702861656, -1.29422672108368, -1.12606860954099, -1.01736702861656, -1.12606860954099, -1.01736702861656, -1.12606860954099, -1.01736702861656, -1.12606860954099, -1.01736702861656, -0.418835558467606, -0.876543660985182, -0.354298520186538, -0.418835558467606, -0.354298520186538, -0.354298520186538, -0.672166773777051, -0.953966549596707, -1.07839510473839, -0.573309430681438, -0.672166773777051, -1.07839510473839, -1.07839510473839, -0.771191693539798, 0.434625002052529, 0.155593943834686, -0.389453691408209, 0.585379001375486, 0.434625002052529, 0.155593943834686, 0.585379001375486, 0.526561236809189, 0.0625971215628006, -0.565507653474784, -0.470689703429145, 0.0187671837212449, 0.0625971215628006, -0.565507653474784, -0.470689703429145, -0.565507653474784, -0.565507653474784, -0.565507653474784, 0.591127895843202, 0.381829026853781, 0.591127895843202, 0.501692969113144, -0.597934698496828, 0.188777689388023, 0.501692969113144, -0.597934698496828, 0.188777689388023, 0.501692969113144, -0.597934698496828, 0.501692969113144, 0.501692969113144, 1.65583872958623, -0.767690410835019, -0.155182333554967, 0.858877561181861, 1.65583872958623, -0.155182333554967, 1.65583872958623, -0.0767729093105824, -0.954094906192809, -0.768386772795495, -0.0783746240995459, -0.0767729093105824, -0.954094906192809, -0.0767729093105824, -0.0767729093105824, -0.0767729093105824, 3.40431283021840, -0.372278317225116, 1.20243770396007, 2.51135299564400, 1.58684267092178, -1.07785342893219, 0.206855125204832, 0.367806068729651, 1.58684267092178, -1.07785342893219, 0.155450502595880, 0.155450502595880, -0.154571268392329, -0.925836433615535, 0.0570067529765944, 0.0570067529765944, 0.0313628603793186, 0.400553730975719, 0.0313628603793186, 0.400553730975719, 0.0313628603793186, 0.400553730975719, 0.400553730975719, -0.116810567976282, -0.785173237463049, -0.831455115409628, 0.449076364029460, -0.116810567976282, -0.785173237463049, 0.449076364029460, -0.692448213026071, 0.588093414263563, 0.643831717588340, 0.643831717588340, 0.643831717588340, 0.643831717588340},
            {0.231214382539635, 0.759654352692475, 0.573760963020423, 0.284597735940930, 0.681006410683275, 0.693998931720632, 0.467617275145568, 0.717508293319345, 0.639963047968708, 0.674673165643432, 0.658695306541066, 0.764970156295444, 0.727483622692381, 0.720470209620153, 0.650473798871370, 0.640386981900739, 0.666988953244499, 0.615928101981335, 0.857823398292310, 1.21484275530793, 1.33168784724604, 0.432010103901961, 0.312681368088020, 1.28152732859875, 0.687038451835498, 1.28674476079208, 1.21859211132117, 0.697333254072230, 1.24484462994305, 1.21594310979008, -0.376909942688284, 1.32411465400942, 1.30971542181418, 1.32902151360945, 1.23135369376351, 1.22768397174856, 1.26381063772213, 1.25343573305183, 1.17205868707798, 1.25852856145961, 1.13521906260214, 1.04825779960757, 0.492336136647599, 1.21085356042929, -0.0269018907861091, 1.01638218383641, 1.30341168819723, 0.601727017911226, 1.25674194928639, 1.18452421842082, 1.14135978074025, 1.22096473603469, 1.64875529575885, 1.46768210515928, 1.80239785299724, 0.898522929508411, 1.25819644084325, 1.61515818031894, 1.64280710452247, 0.814056084401213, 1.47415400704319, 1.24616374370283, 1.38452828415317, 1.66789368772245, 1.61497549055817, 1.54818832994016, 1.59372070798141, 1.65136729090250, 1.62546516187306, 1.63762152592845, 1.50306021154901, 0.670922404324348, 0.308079928293169, 0.692966500354670, 0.391483425305660, 0.670922404324348, 0.692966500354670, 0.670922404324348, 0.692966500354670, 0.670922404324348, 0.692966500354670, 0.670922404324348, 0.692966500354670, 0.710263004781530, 0.400638056061107, 0.599996617972085, 0.782560838080664, 0.710263004781530, 0.400638056061107, 0.599996617972085, 0.782560838080664, 0.710263004781530, 0.599996617972085, 0.710263004781530, 0.599996617972085, 0.710263004781530, 0.599996617972085, -0.135628062381722, -0.0461330329359392, -0.204131101445074, -0.135628062381722, -0.0461330329359392, -0.204131101445074, -0.135628062381722, -0.204131101445074, -0.135628062381722, -0.135628062381722, 1.95582774061669, 1.77700491312152, 1.61274808078270, 1.71282600016488, 1.77700491312152, 1.82087715651623, 1.67711249411180, 1.95379192087238, 1.87992717154822, 1.67711249411180, 1.52810557244399, 1.21679706731283, 1.47422380390051, 1.46647494743238, 1.52810557244399, 1.36338358386815, 1.22931600027805, 1.00030314271269, 1.36673430145469, 1.36338358386815, -0.0249550737457550, 0.189644498565838, -0.415473174791648, 0.0650702244415299, -0.0249550737457550, 0.189644498565838, -0.0249550737457550, -0.0249550737457550, -0.309819938659154, -1.04053682254237, -0.235847917654961, -0.0181800942330890, -0.0502383997650206, -0.212297802186540, -0.116979185289919, -0.162076855387610, 0.361999640094659, 0.298307899833394, 0.140278914595514, 0.298307899833394, 0.298307899833394, 0.298307899833394, 0.298307899833394, 0.0892883286249521, 0.110087792105655, 0.144701412111872, 0.0825672191699474, -0.0167316923349238, -0.447086466803769, -0.146255453670104, -0.106280123403084, -0.0167316923349238, -0.447086466803769, -0.146255453670104, -0.106280123403084, -0.0167316923349238, -0.146255453670104, -0.106280123403084, -0.0167316923349238, -0.0105239879511963, -0.127146573129401, -0.652884842781702, -0.547496272584674, -0.0105239879511963, -0.127146573129401, -0.0105239879511963, -0.127146573129401, -0.0105239879511963, -1.48092548079475, -1.34808004486064, -1.55733992336978, -1.48092548079475, -1.55532401212504, -1.55733992336978, -1.48092548079475, -1.55733992336978, -1.48092548079475, -1.55733992336978, -1.48092548079475, -1.55733992336978, -0.316077297182987, -0.308690072754117, -0.509923312671711, -0.433073459500191, -0.316077297182987, -0.308690072754117, -0.316077297182987, -0.308690072754117, -1.35536023182711, -1.40097622556656, -1.46655388628126, -1.35536023182711, 0.946528645849249, 0.836730694323816, 0.737607446050449, 0.554257659970582, 0.848830377709846, 0.537410853416373, 0.686575637774894, -0.835848638187543, -0.713352577200754, -0.713536906485012, -0.835848638187543, -0.713536906485012, 0.172741479777548, -0.0822578251683793, 0.171854263349629, 0.270785921591835, 0.172741479777548, 0.171854263349629, 0.270785921591835, 0.172741479777548, 0.270785921591835, 0.172741479777548, 0.270785921591835, -1.66288729312809, -1.70604048836185, -1.68982349304699, -1.73050264732824, -1.70604048836185, -1.72633802343573, -2.01002126034521, -1.67226981764921, -1.68044424757286, -1.12321541595285, -1.48014483339394, -1.12321541595285, -1.48014483339394, -1.48014483339394, -1.48014483339394, -1.48014483339394, -1.61387350405606, -1.42062357246624, -1.59862804352024, -1.57040317813503, -1.42062357246624, -1.59862804352024, -1.57040317813503, -1.59862804352024, -1.57040317813503, -1.59862804352024, -1.57040317813503, -1.59862804352024, -1.57040317813503, -0.972115525436076, -0.963738028816805, -1.10196180446430, -0.972115525436076, -1.10196180446430, -1.10196180446430, -1.11210319414552, -1.15437081247485, -1.43282888800881, -1.30935965271949, -1.11210319414552, -1.43282888800881, -1.43282888800881, -1.47058219550612, -0.00135436726666069, -0.0248136062899834, -0.518701555676455, -0.447042199669429, -0.00135436726666069, -0.0248136062899834, -0.447042199669429, -0.157697453917382, -0.448459919056492, -0.168054089611604, -0.165041113864838, -0.456230089330041, -0.448459919056492, -0.168054089611604, -0.165041113864838, -0.168054089611604, -0.168054089611604, -0.168054089611604, -0.149464235365586, -0.157571913261266, -0.149464235365586, -0.0538584676398656, -0.289785195779601, -0.163392690100400, -0.0538584676398656, -0.289785195779601, -0.163392690100400, -0.0538584676398656, -0.289785195779601, -0.0538584676398656, -0.0538584676398656, 1.07655269606775, 1.05805324983155, 0.796634507753521, 0.598030595085087, 1.07655269606775, 0.796634507753521, 1.07655269606775, -0.688616852028498, -0.460867598641761, -0.531277871963968, -0.612379712208465, -0.688616852028498, -0.460867598641761, -0.688616852028498, -0.688616852028498, -0.688616852028498, 2.62180780512001, 2.51145288404189, 2.58724711830089, 2.22455173094889, 0.876213230672054, 0.691224857968713, -0.0346062459022550, 0.274573220863073, 0.876213230672054, 0.691224857968713, -0.381410668897071, -0.381410668897071, -0.558266302417037, -0.696284903410133, -0.703257796830294, -0.703257796830294, 0.0402150482718513, 0.0561741699628573, 0.0402150482718513, 0.0561741699628573, 0.0402150482718513, 0.0561741699628573, 0.0561741699628573, 0.574791520645250, 0.425960730644230, 0.129577042096975, 0.417413660452146, 0.574791520645250, 0.425960730644230, 0.417413660452146, 0.102643887007419, 0.0839088178233579, -0.0143604729272548, -0.0143604729272548, -0.0143604729272548, -0.0143604729272548},
            {-0.748863403637009, -1.17957946825521, -0.788373516893258, 1.31311455691775, -0.535977856925875, 2.12984416587385, 2.17338417270776, 0.0830943823934754, -0.854878661524892, 0.304456176855422, 0.376996919255485, -0.787024701666123, 1.19558567983422, -0.726019567387843, -0.878296988673828, 0.678682327785494, -0.562736205299652, -0.957636779524893, 0.788661270836098, -0.641074171010679, -1.28495438928320, 0.318137880004040, 0.275808230653592, -0.993443087728918, 1.66143533649873, -0.593936437570384, -0.258908688745897, 1.71025341051400, -1.28033381515686, 1.29523635178962, -0.858120699128343, 0.935011105891908, 1.22088961780812, -1.08344225215906, 1.43870080479111, 0.984887064727023, -0.418781834191209, 1.44973465461495, -0.716019801183593, -0.757727840506846, -0.274559125692989, -1.23279345481362, 1.32581835694487, -1.26225511210061, -0.265459455640849, 1.43857539542119, -0.799201371557448, -0.511338298648682, 0.0605035396131036, -0.195051155802495, 0.242307497599998, -0.354178973306550, -0.534683047467867, 0.998992840022473, 1.24447672808786, 0.958676264777589, 0.182003471774160, -0.476697282711382, -1.06546455318109, -0.321979211443012, -0.633411755163303, -0.472050491019262, 2.06572300745341, -1.09237549503676, 0.332923862190279, 0.824008348775133, 0.663018963282930, -1.63422259557570, 1.11670584415779, -0.734522757614081, 1.34625597642015, -0.691610522539409, 0.759146815962733, 1.03860335326562, -1.16910126168017, -0.691610522539409, 1.03860335326562, -0.691610522539409, 1.03860335326562, -0.691610522539409, 1.03860335326562, -0.691610522539409, 1.03860335326562, -0.618218857025345, 1.45195529451773, 1.02255675319333, -0.841413585544158, -0.618218857025345, 1.45195529451773, 1.02255675319333, -0.841413585544158, -0.618218857025345, 1.02255675319333, -0.618218857025345, 1.02255675319333, -0.618218857025345, 1.02255675319333, -1.02275704467417, 1.16940822572881, -1.20008450432819, -1.02275704467417, 1.16940822572881, -1.20008450432819, -1.02275704467417, -1.20008450432819, -1.02275704467417, -1.02275704467417, -0.720946166365106, 0.757769729259143, 0.550292612648572, -1.19870475962254, 0.757769729259143, -1.10383522928156, 1.41403067949023, 0.922442380696744, -1.32695360065029, 1.41403067949023, -0.769249433708676, 0.495245148351054, 0.390828920338764, -1.09621189814630, -0.769249433708676, -1.18892929254667, 0.669192535521923, -0.271962136862170, -1.37520370795266, -1.18892929254667, -0.916310767588616, 2.06716364456989, -0.285511060726067, -1.07643316113618, -0.916310767588616, 2.06716364456989, -0.916310767588616, -0.916310767588616, -1.17832736805757, -0.584804388220423, -0.121767139371858, -1.03932950629037, -1.40275897367131, 1.06252368034906, 0.712248544344382, -0.180359653507658, 2.44723619184903, -0.994885778756078, 0.420919436749478, -0.994885778756078, -0.994885778756078, -0.994885778756078, -0.994885778756078, -1.54740711352197, 2.57545579485412, -1.10478679401902, -1.39940227952446, -0.834994268932898, 0.268635249640068, 0.986692814671913, -0.217682110250317, -0.834994268932898, 0.268635249640068, 0.986692814671913, -0.217682110250317, -0.834994268932898, 0.986692814671913, -0.217682110250317, -0.834994268932898, -0.329955271533458, 1.94767098723717, 0.0780780075967514, -1.02665011155489, -0.329955271533458, 1.94767098723717, -0.329955271533458, 1.94767098723717, -0.329955271533458, -0.109487411491573, 0.457016554542513, -0.156787403875798, -0.109487411491573, -0.0438357271577242, -0.156787403875798, -0.109487411491573, -0.156787403875798, -0.109487411491573, -0.156787403875798, -0.109487411491573, -0.156787403875798, -0.845603442518522, 1.83875065391607, 0.404512314943238, -0.401860388986373, -0.845603442518522, 1.83875065391607, -0.845603442518522, 1.83875065391607, 0.0726945868792075, 0.215489706634699, -0.502042661353760, 0.0726945868792075, -0.0672343609277610, 0.479925391023872, -1.09594265461945, -0.476435107053653, 0.404416868505632, 1.23647565861439, -0.736571473110903, 0.339413495037852, 1.30490592164706, -0.937430768683648, 0.339413495037852, -0.937430768683648, -0.434751835476868, 0.569042884088908, 1.07641343256800, -0.697844987189229, -0.434751835476868, 1.07641343256800, -0.697844987189229, -0.434751835476868, -0.697844987189229, -0.434751835476868, -0.697844987189229, -0.425652286243003, 1.97687179786747, 0.612587965290393, -1.21476181047567, 1.97687179786747, -0.401597850873819, -0.721725806696591, 0.571337464776273, -1.02290480542505, -0.104042615090100, -0.402537092147174, -0.104042615090100, -0.402537092147174, -0.402537092147174, -0.402537092147174, -0.402537092147174, -1.18933506072477, 0.827846745386210, -0.251027591819956, -0.517707234039079, 0.827846745386210, -0.251027591819956, -0.517707234039079, -0.251027591819956, -0.517707234039079, -0.251027591819956, -0.517707234039079, -0.251027591819956, -0.517707234039079, 0.0611716646763475, 1.67597895722442, -0.626164350092102, 0.0611716646763475, -0.626164350092102, -0.626164350092102, 0.0268979362964186, 1.07276786192454, 0.189058292514292, -1.06990818894304, 0.0268979362964186, 0.189058292514292, 0.189058292514292, -1.29680980338010, -0.724865632039011, 0.806298564322980, 0.281197209011806, -0.948832389337955, -0.724865632039011, 0.806298564322980, -0.948832389337955, -0.807317943302244, -0.803905431112132, 0.927079869381654, 0.728522766047627, -0.427754403417575, -0.803905431112132, 0.927079869381654, 0.728522766047627, 0.927079869381654, 0.927079869381654, 0.927079869381654, -0.798394909568813, 0.683790282835010, -0.798394909568813, -0.769688245685114, 0.986087515111903, -0.422213918941944, -0.769688245685114, 0.986087515111903, -0.422213918941944, -0.769688245685114, 0.986087515111903, -0.769688245685114, -0.769688245685114, -1.02651219749346, 2.11079209038143, 1.11110350072513, -1.03256930091492, -1.02651219749346, 1.11110350072513, -1.02651219749346, -0.734044558879847, 1.63477920038599, 1.30089837945035, -1.18552155267844, -0.734044558879847, 1.63477920038599, -0.734044558879847, -0.734044558879847, -0.734044558879847, -1.33260964780639, 2.13549797789185, -0.0690600458881005, -0.852526933793239, -1.30867917198782, 1.96076092247263, -0.838293332762362, -0.774049060519241, -1.30867917198782, 1.96076092247263, -0.491907095407903, -0.491907095407903, -0.838254066822840, 0.925599362234278, -0.820782898464703, -0.820782898464703, -0.778736930425014, -0.438678066974198, -0.778736930425014, -0.438678066974198, -0.778736930425014, -0.438678066974198, -0.438678066974198, -0.732695381197886, 0.503720792010490, 1.21397542954061, -0.823786199155863, -0.732695381197886, 0.503720792010490, -0.823786199155863, 1.65078834679289, -0.424743491175061, -1.01280161877701, -1.01280161877701, -1.01280161877701, -1.01280161877701},
            {-0.759870342387174, -1.39220317548960, -0.817875009275250, 1.10189533390064, -0.447333773783130, 1.10198402035639, 1.10185489287682, 0.461524137822443, -0.915510992641892, 0.786504657777828, 0.893001482310819, -0.815894818091386, 1.10211492156507, -0.726333381774462, -0.949891361448735, 1.10197373272752, -0.486617616728089, -1.06636984657578, 1.10199537222272, -0.601625325673062, -1.54690374931343, 0.806590721020676, 0.744446702497179, -1.11893696948924, 1.10207093308302, -0.532422574763945, -0.0405694423067069, 1.10212840190634, -1.54012029968644, 1.10227597616870, -0.920270617348791, 1.10195812391131, 1.10202942782173, -1.25106444579178, 1.10194535306168, 1.10195883340296, -0.275278679553388, 1.10200424086830, -0.711652758009244, -0.772884192903130, -0.0635457971422700, -1.47032645354978, 1.10216919767598, -1.51357901538931, -0.0501866015676457, 1.10210002224050, -0.833771346975815, -0.411160519586955, 0.428358596086389, 0.0531795910775631, 0.695264386853659, -0.180435432678142, -0.445432868290702, 1.10197302323588, 1.10198827730626, 1.10196309035283, 0.606732245542107, -0.360304158404802, -1.22467144524865, -0.133163068545537, -0.590376158253512, -0.353482218856020, 1.10201240002223, -1.26417931018082, 0.828297972674534, 1.10198189188145, 1.10198969628956, -2.05966325038592, 1.10197054001511, -0.738816887285065, 1.10193861289105, -0.675817576582367, 1.10198437510221, 1.10201062629311, -1.37682015499489, -0.675817576582367, 1.10201062629311, -0.675817576582367, 1.10201062629311, -0.675817576582367, 1.10201062629311, -0.675817576582367, 1.10201062629311, -0.568071514633758, 1.10201559273463, 1.10200459561412, -0.895742959029344, -0.568071514633758, 1.10201559273463, 1.10200459561412, -0.895742959029344, -0.568071514633758, 1.10200459561412, -0.568071514633758, 1.10200459561412, -0.568071514633758, 1.10200459561412, -1.16197269294447, 1.10189675288393, -1.42230655128171, -1.16197269294447, 1.10189675288393, -1.42230655128171, -1.16197269294447, -1.42230655128171, -1.16197269294447, -1.16197269294447, -0.718885138475212, 1.10198756781462, 1.10198472984803, -1.42028095263250, 1.10198756781462, -1.28100330820848, 1.10203971545060, 1.10199856493513, -1.60856265292400, 1.10203971545060, -0.789799005861464, 1.06660131722672, 0.913308197455560, -1.26981152092579, -0.789799005861464, -1.40592962167746, 1.10199572696855, -0.0597331664098119, -1.67939847622920, -1.40592962167746, -1.00569926482802, 1.10144196873888, -0.0796242968229336, -1.24077442239110, -1.00569926482802, 1.10144196873888, -1.00569926482802, -1.00569926482802, -1.39036497132138, -0.519015843249042, 0.160767613762035, -1.18630266915407, -1.71985227089834, 1.10210782664861, 1.10200211239336, 0.0747481371146373, 1.10183963880644, -1.12105497942533, 0.957483985299388, -1.12105497942533, -1.12105497942533, -1.12105497942533, -1.12105497942533, -1.93220972767404, 1.10189923610470, -1.28240029725936, -1.71492431929844, -0.886318781496105, 0.733916072742071, 1.10199359849361, 0.0199551615348945, -0.886318781496105, 0.733916072742071, 1.10199359849361, 0.0199551615348945, -0.886318781496105, 1.10199359849361, 0.0199551615348945, -0.886318781496105, -0.144872696043313, 1.10276162320036, 0.454159614537416, -1.16768809158493, -0.144872696043313, 1.10276162320036, -0.144872696043313, 1.10276162320036, -0.144872696043313, 0.178795441739964, 1.01047804481081, 0.109354479010717, 0.178795441739964, 0.275178462859847, 0.109354479010717, 0.178795441739964, 0.109354479010717, 0.178795441739964, 0.109354479010717, 0.178795441739964, 0.109354479010717, -0.901894074226874, 1.10259524740938, 0.933396743919169, -0.250436361688960, -0.901894074226874, 1.10259524740938, -0.901894074226874, 1.10259524740938, 0.446256232347284, 0.655893276436248, -0.397513625149896, 0.446256232347284, 0.240827005837577, 1.04411043204990, -1.26941624539254, -0.359919259186870, 0.933256619319092, 1.10216387648864, -0.741824599745185, 0.837825381242260, 1.10199182476449, -1.03670546873960, 0.837825381242260, -1.03670546873960, -0.298724185739840, 1.10198260137310, 1.10197337798170, -0.684970373561028, -0.298724185739840, 1.10197337798170, -0.684970373561028, -0.298724185739840, -0.684970373561028, -0.298724185739840, -0.684970373561028, -0.285365167538127, 1.10228200684769, 1.10199679120601, -1.44385425600168, 1.10228200684769, -0.250050930352294, -0.720029725873051, 1.10200353137665, -1.16218962001523, 0.186788929369132, -0.251429827366210, 0.186788929369132, -0.251429827366210, -0.251429827366210, -0.251429827366210, -0.251429827366210, -1.40652532860070, 1.10199998391842, -0.0289992299173617, -0.420510732616103, 1.10199998391842, -0.0289992299173617, -0.420510732616103, -0.0289992299173617, -0.420510732616103, -0.0289992299173617, -0.420510732616103, -0.0289992299173617, -0.420510732616103, 0.429339468286923, 1.10202729934679, -0.579736266784927, 0.429339468286923, -0.579736266784927, -0.579736266784927, 0.379022320755696, 1.10202197815945, 0.617089404589778, -1.23119513224677, 0.379022320755696, 0.617089404589778, 0.617089404589778, -1.56430864362620, -0.724639293096830, 1.10200495035994, 0.752358243841239, -1.05344415039683, -0.724639293096830, 1.10200495035994, -1.05344415039683, -0.845687259169652, -0.840677361284643, 1.10202339714274, 1.10198934154373, -0.288451278825126, -0.840677361284643, 1.10202339714274, 1.10198934154373, 1.10202339714274, 1.10202339714274, 1.10202339714274, -0.832587382791625, 1.10202552561768, -0.832587382791625, -0.790443224275993, 1.10199005103538, -0.280317311850059, -0.790443224275993, 1.10199005103538, -0.280317311850059, -0.790443224275993, 1.10199005103538, -0.790443224275993, -0.790443224275993, -1.16748562040647, 1.10229548718896, 1.10199785544348, -1.17637803395106, -1.16748562040647, 1.10199785544348, -1.16748562040647, -0.738114845301391, 1.10223518039906, 1.10203865121313, -1.40092673002246, -0.738114845301391, 1.10223518039906, -0.738114845301391, -0.738114845301391, -0.738114845301391, -1.61686627708891, 1.10200246713918, 0.238146723772065, -0.912058428919757, -1.58173402451026, 1.10223092344918, -0.891162126217227, -0.796845322143280, -1.58173402451026, 1.10223092344918, -0.382633634232164, -0.382633634232164, -0.891104480020993, 1.10202871833008, -0.865455115409289, -0.865455115409289, -0.803727568481969, -0.304488273244501, -0.803727568481969, -0.304488273244501, -0.803727568481969, -0.304488273244501, -0.304488273244501, -0.736134121998792, 1.07904438171350, 1.10198579408550, -0.869864251243085, -0.736134121998792, 1.07904438171350, -0.869864251243085, 1.10218090428814, -0.284030968497906, -1.14735716503777, -1.14735716503777, -1.14735716503777, -1.14735716503777},
            {0.00807748085553650, 0.625591585904198, -0.0411498578784241, -0.263186039816633, 0.813608410124607, -0.884603743626652, -0.763924344263941, 2.12463252818764, 0.533894140907224, -0.441318853539060, 2.18148873231194, 1.12016986811597, 1.43509908735797, 0.865145331074714, 0.253014001405086, 1.27152484444923, 0.946541485428841, -1.01447459841414, 1.21076152605558, 0.398497014096557, -0.132628489135916, 0.607275789298017, 0.836472847942274, 0.561921475058992, -0.0835119421687813, 0.118938492536001, 1.37376817422043, 0.370638211267020, -0.786434035366673, 0.913132654264581, -0.0567923769877430, 1.57477043269614, 0.254172627612780, 0.716501983234525, 0.373233295343833, 1.35604106540605, 0.0770310806118117, 0.874225568610481, 0.0128954312848458, -0.680530316068411, -0.289096292447115, -0.897384319643869, 0.457350665940084, -1.04533383404431, -0.398968487608656, -0.322870523380811, -0.297872278746290, -0.116961894123421, 0.784620596578893, -0.129038835890747, 0.423875998898999, 0.215792228973284, 0.0887641417797961, -0.219119253744783, 0.997888782008803, -0.0419422320723215, 0.722014086695218, 0.765881021464601, -0.911586545784448, -0.437460317198248, 0.573056899869385, -0.712547751784097, -0.393482910253674, -0.632145953537235, 0.600079650981565, 0.380330999436682, 0.785947114925546, -1.34298758071757, 0.368245535225754, -0.310945600701237, -0.345457981781413, 0.116496599382934, 0.526488563539253, 0.487541422400223, -0.742799443715990, 0.116496599382934, 0.487541422400223, 0.116496599382934, 0.487541422400223, 0.116496599382934, 0.487541422400223, 0.116496599382934, 0.487541422400223, -0.0688609028420118, 0.0112243931555806, 0.711233834721939, -0.173609724501656, -0.0688609028420118, 0.0112243931555806, 0.711233834721939, -0.173609724501656, -0.0688609028420118, 0.711233834721939, -0.0688609028420118, 0.711233834721939, -0.0688609028420118, 0.711233834721939, -1.27172365303134, -0.159416553457405, -1.38276632592370, -1.27172365303134, -0.159416553457405, -1.38276632592370, -1.27172365303134, -1.38276632592370, -1.27172365303134, -1.27172365303134, 3.58611825743335, 4.49302088355422, 2.05572303207701, 2.22707315252822, 4.49302088355422, 1.62064589436142, 3.26011092681418, 1.54795669451330, -0.582328968236560, 3.26011092681418, -0.352318229289400, 1.09205304830638, -0.214913640888278, -1.54042111920021, -0.352318229289400, 0.127566188316502, 3.75283302080943, -0.718962595083334, -1.43205822216795, 0.127566188316502, 0.173578648262692, -0.0374440863391778, -0.539019933931473, -0.342767765927891, 0.173578648262692, -0.0374440863391778, 0.173578648262692, 0.173578648262692, -0.116181238289476, -0.580254818524919, 0.0190141196689126, 0.250835877881501, -0.724008094756817, 0.136068391114962, -0.474621686811020, -1.06408704506836, -0.337966753855241, -0.753820454721026, 0.784362792659932, -0.753820454721026, -0.753820454721026, -0.753820454721026, -0.753820454721026, 0.120341286752893, -0.326993788126011, -1.08730569694824, -1.21311651286900, 0.197658173476916, 0.622621727369985, 0.696072833676134, 0.511687422674543, 0.197658173476916, 0.622621727369985, 0.696072833676134, 0.511687422674543, 0.197658173476916, 0.696072833676134, 0.511687422674543, 0.197658173476916, -0.148564819488414, -0.453757253445657, -0.324054077736041, -1.03321790209750, -0.148564819488414, -0.453757253445657, -0.148564819488414, -0.453757253445657, -0.148564819488414, -0.293310001625041, 0.409573847107187, -0.438984023583743, -0.293310001625041, -0.184442815625268, -0.438984023583743, -0.293310001625041, -0.438984023583743, -0.293310001625041, -0.438984023583743, -0.293310001625041, -0.438984023583743, -0.312821710129675, -0.371664389327065, -0.0713438102549139, -0.136098615109566, -0.312821710129675, -0.371664389327065, -0.312821710129675, -0.371664389327065, -0.312820964415860, -0.653454938866988, -0.404550901229066, -0.312820964415860, 1.08851559502827, 3.51133892829183, 0.132021082648365, 1.74262507290399, 2.08302681076696, 1.76487887763757, 1.10738471128532, 0.524552690475055, 0.0793670824031241, -0.689986393367027, 0.524552690475055, -0.689986393367027, 0.444424249606728, 0.999411103497213, 1.03749769089353, 0.351791040302253, 0.444424249606728, 1.03749769089353, 0.351791040302253, 0.444424249606728, 0.351791040302253, 0.444424249606728, 0.351791040302253, -0.644901920998551, -1.24178718498217, -0.622354518082879, -1.30473059771449, -1.24178718498217, -0.214949754743042, -1.13577959840251, -0.343239270120173, -0.929170051424126, -0.661701574887934, -0.969045286671415, -0.661701574887934, -0.969045286671415, -0.969045286671415, -0.969045286671415, -0.969045286671415, -1.39330938763849, -0.645425518627350, -0.815596239414594, -0.852732148227106, -0.645425518627350, -0.815596239414594, -0.852732148227106, -0.815596239414594, -0.852732148227106, -0.815596239414594, -0.852732148227106, -0.815596239414594, -0.852732148227106, -0.574410552824837, -0.494077573310061, -1.01297177201547, -0.574410552824837, -1.01297177201547, -1.01297177201547, -0.427832086538871, -1.13317055882430, -0.445738699321611, -1.32641547607236, -0.427832086538871, -0.445738699321611, -0.445738699321611, -0.872606912829185, -0.965547782347693, -1.02345970417320, -0.0129770030021522, -1.01231820712174, -0.965547782347693, -1.02345970417320, -1.01231820712174, -0.984493653862425, -0.174554863497120, 0.491643487566963, 0.00344595888002130, -0.857843057655234, -0.174554863497120, 0.491643487566963, 0.00344595888002130, 0.491643487566963, 0.491643487566963, 0.491643487566963, -1.08767083039131, -0.909821973836650, -1.08767083039131, -0.423484255404758, 0.268569591589664, -0.462321137429687, -0.423484255404758, 0.268569591589664, -0.462321137429687, -0.423484255404758, 0.268569591589664, -0.423484255404758, -0.423484255404758, -0.175634763632040, 0.520376480048978, 0.934084655737956, -0.503894256503587, -0.175634763632040, 0.934084655737956, -0.175634763632040, -0.850827382082055, -0.833197003002566, -0.370549653703921, -0.809503864075149, -0.850827382082055, -0.833197003002566, -0.850827382082055, -0.850827382082055, -0.850827382082055, 0.274370179766286, 0.966470367547794, 1.98882525877863, 0.163273948642405, -0.694900860470129, -0.232569480768028, -0.477888019852038, -0.438743584143621, -0.694900860470129, -0.232569480768028, -0.471195984074645, -0.471195984074645, -0.334774353012471, 0.444707833917585, -0.998300918007323, -0.998300918007323, 0.333759254129109, -0.0851540039897996, 0.333759254129109, -0.0851540039897996, 0.333759254129109, -0.0851540039897996, -0.0851540039897996, 1.47716885183278, 1.83109783830305, 0.829066418329942, 0.435020159214084, 1.47716885183278, 1.83109783830305, 0.435020159214084, 0.232843934132196, -0.207989581583821, -0.471219207733461, -0.471219207733461, -0.471219207733461, -0.471219207733461},
            {0.305489115308679, 0.305489115308679, 0.182567836410798, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, -0.301237233034322, 0.305489115308679, -0.305735801966496, 0.305489115308679, 0.305489115308679, 0.305489115308679, -0.0930184492086217, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, -0.989359812055965, -0.805150547655912, 0.305489115308679, 0.305489115308679, 0.305489115308679, -4.10061639537301, -3.35876505001599, -5.48530480040745, 0.305489115308679, -3.56988080801826, -0.316696539838224, 0.305489115308679, -1.93047792003247, 0.305489115308679, 0.305489115308679, -4.85862770737041, -0.419727513118965, -2.10863726429585, -0.913624230513248, -1.58685211347130, 0.305489115308679, -0.0874047810555463, -0.564396873909662, -0.200474630243951, 0.305489115308679, -0.233470241410002, -1.90120317474321, -0.669819402936397, -0.673989710894159, -1.75204582506311, -0.339695362227079, 0.305489115308679, -0.718369681115585, -1.54602153786089, -1.71903982028676, -2.35924874141170, -0.576499603886324, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, -0.559620567202887, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, -4.16236693148931, -0.0381809689644984, -6.85920622263371, -2.69151857326129, 0.305489115308679, 0.305489115308679, 0.305489115308679, -0.440661642492699, 0.0230531958135220, 0.0170373229069763, 0.305489115308679, 0.0170373229069763, 0.0170373229069763, 0.0170373229069763, 0.0170373229069763, 0.305489115308679, -4.13188447664887, 0.212768982129600, 0.0818080184554714, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, -0.00276448554629029, 0.305489115308679, 0.0474895057748349, -0.00276448554629029, -0.0162391021694579, 0.0474895057748349, -0.00276448554629029, 0.0474895057748349, -0.00276448554629029, 0.0474895057748349, -0.00276448554629029, 0.0474895057748349, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, -0.513927459867575, -1.37689242322558, 0.305489115308679, -0.513927459867575, 0.305489115308679, -0.00476870452842700, 0.305489115308679, 0.305489115308679, 0.305489115308679, -1.29465314491025, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, -0.195727873083928, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, -1.46496908865993, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, -0.457657432511492, 0.305489115308679, 0.305489115308679, 0.0452121865258278, 0.305489115308679, 0.305489115308679, 0.0452121865258278, 0.305489115308679, 0.0452121865258278, 0.305489115308679, 0.0452121865258278, 0.305489115308679, 0.0452121865258278, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, -0.617504587073939, 0.0200521923177877, 0.305489115308679, -0.592446664644113, -0.617504587073939, 0.305489115308679, 0.305489115308679, 0.290972073172454, -7.00299189350891, -1.91041293884145, 0.305489115308679, 0.305489115308679, -7.00299189350891, -1.91041293884145, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.133014493274124, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.0806302313159857, 0.305489115308679, 0.305489115308679, 0.0806302313159857, 0.305489115308679, 0.305489115308679, 0.305489115308679, -0.133949511921237, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.138549929701295, 0.305489115308679, 0.305489115308679, 0.126175148140415, -0.840015284174348, -0.930821973506928, 0.305489115308679, 0.126175148140415, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679, 0.305489115308679}
        };
    };
    // Data that's passed to a callback at each optimization iteration
    enum data_type
    {
        k_to_dsm_data,
        cycle_data,
        iteration_data,
        general_data
    };

    struct k_to_dsm_inputs
    {
        rs2_intrinsics_double old_k;
        rs2_intrinsics_double new_k;
        z_frame_data z;
        std::vector<double3> new_vertices;
        rs2_dsm_params_double previous_dsm_params;
        algo_calibration_registers new_dsm_regs;
    };

    struct convert_norm_vertices_to_los_data
    {
        double3 laser_incident;
        std::vector<double3> fovex_indicent_direction;
        std::vector<double3> mirror_normal_direction;
        std::vector<double> ang_x;
        std::vector<double> ang_y;
        std::vector<double> dsm_x_corr;
        std::vector<double> dsm_y_corr;
        std::vector<double> dsm_x;
        std::vector<double> dsm_y;
    };

    struct k2dsm_data_params
    {
        k_to_dsm_inputs inputs;
        algo_calibration_registers dsm_regs_orig;
        std::vector < uint8_t> relevant_pixels_image_rot;
        convert_norm_vertices_to_los_data first_norm_vertices_to_los_data;
        pre_process_data dsm_pre_process_data;

        std::vector < double> errL2;
        std::vector < std::vector < double>> sg_mat;
        std::vector < double> sg_mat_tag_x_sg_mat;
        std::vector <double> sg_mat_tag_x_err_l2;
        std::vector <double> quad_coef;
        double2 focal_scaling;
        double2 opt_scaling_1;
        double2 opt_scaling;
        double2 new_los_scaling;
        rs2_dsm_params_double dsm_params_cand;
        algo_calibration_registers dsm_regs_cand;
        convert_norm_vertices_to_los_data second_norm_vertices_to_los_data;
        std::vector< double2> los_orig;
        std::vector< double2> dsm;
        std::vector < double3> vertices;
        rs2_intrinsics_double k_depth;
    };
    
    struct iteration_data_params
    {
        size_t iteration;
        optimization_params params;
        calib c;
        std::vector< double2 > uvmap;
        std::vector< double > d_vals;
        std::vector< double > d_vals_x;
        std::vector< double > d_vals_y;
        std::vector < double2> xy;
        std::vector<double > rc;
        coeffs<k_matrix> coeffs_k;
        coeffs<rotation_in_angles> coeffs_r;
        coeffs < p_matrix> coeffs_p;
        p_matrix unit_grad;
        p_matrix normalized_grads;
        p_matrix grads_norm;
        double grads_norma;
        double cost;
        int back_tracking_line_search_iters;
        double t;
        optimization_params next_params;
    };

    struct cycle_data_params
    {
        size_t cycle;
        optimization_params new_params;
        calib new_calib;
        rs2_intrinsics_double new_k_depth;
        algo_calibration_registers new_dsm_regs;
        rs2_dsm_params_double new_dsm_params;
        std::vector<double3> new_vertices;
        calib optimaized_calib_candidate;
    };

    struct data_collect
    {
        data_type type;

        cycle_data_params cycle_data_p;
        k2dsm_data_params k2dsm_data_p;
        iteration_data_params iteration_data_p;

    };

    struct input_validity_data
    {
        bool edges_dir_spread;
        bool not_saturated;
        bool depth_spatial_spread;
        bool rgb_spatial_spread;
        bool is_movement_from_last_success;
    };

    class optimizer
    {
    public:
#pragma pack(push, 1)
        struct settings
        {
            bool is_manual_trigger = false;
            double hum_temp = 0.;
            rs2_ambient_light ambient = RS2_AMBIENT_LIGHT_NO_AMBIENT;
            int receiver_gain = 0;  // aka APD

            std::string to_string() const;
        };
#pragma pack(pop)

        optimizer( settings const &, bool debug_mode = false );

        void set_yuy_data( std::vector< yuy_t > && yuy_data, 
            std::vector< yuy_t > && prev_yuy_data,
            std::vector< yuy_t > && last_successful_yuy_data,
            calib const & calibration);

        void set_ir_data( std::vector< ir_t > && ir_data, size_t width, size_t height );
        void set_z_data( std::vector< z_t > && z_data,
                         rs2_intrinsics_double const & depth_intrinsics,
                         rs2_dsm_params const & dms_params, algo_calibration_info const & cal_info,
                         algo_calibration_registers const & cal_regs, float depth_units );

        // Write dumps of all the pertinent data from the above to a directory of choice, so that
        // a reproduction can be made
        void write_data_to( std::string const & directory );

        // (optional) Return whether the scene passed in is valid and can go thru optimization
        bool is_scene_valid(input_validity_data* data = nullptr);

        // This function checks for specific cases in which we expect poor
        // performance of the AC2 algorithm.
        // Covered Cases(By priority) :
        // 1. Enough Edges in RGB image(Lights off bug fix)
        // 2. Enough Edges in enough locations(3 / 4 quarters)
        // 3. Enough Edges in enough directions(2 / 4 directions)
        // 4. Large enough STD of edges in the chosen direction(weights will be
        //    normalized by direction)  (Normalize by weights is done in a seperate
        //    function)
        // 5. Verify there is movement in RGB between this scene and the previous
        // 6. Check for saturation in the depth
        //    one in which we converged
        bool input_validity_checks(input_validity_data* data = nullptr);

        // Optimize the calibration, optionally calling the callback after each iteration
        size_t optimize( std::function< void( data_collect const & data ) >
                             iteration_callback = nullptr );

        // (optional) Return whether the results of calibration are valid:
        //      1. If pixel movement is acceptable
        //      2. Movement from the camera's factory settings is within thresholds
        //      3. No worsening of the fit in any image section
        bool is_valid_results();

        calib const & get_calibration() const;
        rs2_dsm_params const & get_dsm_params() const;

        double get_cost() const;
        double calc_correction_in_pixels( std::vector< double2 > const & old_uvmap,
                                          std::vector< double2 > const & new_uvmap ) const;
        double calc_correction_in_pixels( calib const & from_calibration, calib const & to_calibration) const;
        double calc_correction_in_pixels(calib const & to_calibration) const { return calc_correction_in_pixels( _original_calibration, to_calibration); }

        // for debugging/unit-testing
        z_frame_data    const & get_z_data() const   { return _z; }
        yuy2_frame_data const & get_yuy_data() const { return _yuy; }
        ir_frame_data   const & get_ir_data() const  { return _ir; }
        decision_params const& get_decision_params() const { return _decision_params; };
        std::vector< double >const& get_extracted_features() const { return _extracted_features; };
        params const & get_params() const { return _params; }
        void set_cycle_data(const std::vector<double3>& vertices,
            const rs2_intrinsics_double& k_depth,
            const p_matrix& p_mat,
            const algo_calibration_registers& dsm_regs_cand,
            const rs2_dsm_params_double& dsm_params_cand);

        void set_final_data(const std::vector<double3>& vertices,
            const p_matrix& p_mat,
            const p_matrix& p_mat_opt = p_matrix());
    private:

        void adjust_params_to_manual_mode();
        void adjust_params_to_auto_mode();
        void adjust_params_to_apd_gain();

        // 1 cycle of optimization
        size_t optimize_p(const optimization_params& params_curr,
            const std::vector<double3>& new_vertices,
            optimization_params& params_new,
            calib& optimaized_calibration,
            calib& new_rgb_calib_for_k_to_dsm,
            rs2_intrinsics_double& new_z_k,
            std::function<void(data_collect const&data)> cb,
            data_collect& data);

        calib decompose_p_mat(p_matrix p);
        rs2_intrinsics_double get_new_z_intrinsics_from_new_calib(const rs2_intrinsics_double& orig, const calib & new_c, const calib & orig_c);
        std::vector<direction> get_direction( std::vector<double> gradient_x, std::vector<double> gradient_y );
        std::vector<direction> get_direction2(std::vector<double> gradient_x, std::vector<double> gradient_y);
        std::vector<uint16_t> get_closest_edges( const z_frame_data& z_data, ir_frame_data const & ir_data, size_t width, size_t height );
        std::vector<double> blur_edges( std::vector<double> const & edges, size_t image_width, size_t image_height );
        std::vector< byte > get_luminance_from_yuy2( std::vector< yuy_t > const & yuy2_imagh );

        std::vector<uint8_t> get_logic_edges( std::vector<double> const & edges );
        std::vector <double3> subedges2vertices(z_frame_data& z_data, const rs2_intrinsics_double& intrin, double depth_units);
        
        optimization_params back_tracking_line_search( optimization_params const & opt_params,
                                                       const std::vector<double3>& new_vertices,
                                                       data_collect * data = nullptr ) const;
       
        // input validation
        bool is_movement_in_images( movement_inputs_for_frame const & prev,
                                    movement_inputs_for_frame const & curr,
                                    movement_result_data * result_data,
                                    double const move_thresh_pix_val,
                                    double const move_threshold_pix_num,
                                    size_t width, size_t height );

        bool is_edge_distributed( z_frame_data & z_data, yuy2_frame_data & yuy_data );
        void section_per_pixel( frame_data const &, size_t section_w, size_t section_h, byte * section_map );
        void sum_per_section(std::vector< double >& sum_weights_per_section, std::vector< byte > const& section_map, std::vector< double > const& weights, size_t num_of_sections);
        std::vector<uint8_t> images_dilation(std::vector<uint8_t> const &logic_edges, size_t width, size_t height);
        void gaussian_filter(std::vector<uint8_t> const& lum_frame,
            std::vector<uint8_t> const& prev_lum_frame,
            std::vector<double>& yuy_diff,
            std::vector<double>& gaussian_filtered_image,
            size_t width, size_t height);

        // svm
        bool valid_by_svm(svm_model model);
        void collect_decision_params(z_frame_data& z_data, yuy2_frame_data& yuy_data);// , decision_params& decision_params);

        // output validation
        void clip_pixel_movement( size_t iteration_number = 0 );
        std::vector< double > cost_per_section_diff( calib const & old_calib, calib const & new_calib );
        void clip_ac_scaling( rs2_dsm_params_double const & ac_data_orig,
                              rs2_dsm_params_double & ac_data_new ) const;

    private:
        settings const _settings;
        params _params;
        decision_params _decision_params;
        svm_features _svm_features;
        std::vector< double > _extracted_features;
        svm_model_linear _svm_model_linear;
        svm_model_gaussian _svm_model_gaussian;
        yuy2_frame_data _yuy;
        ir_frame_data _ir;
        z_frame_data _z;
        calib _original_calibration;         // starting state of auto-calibration
        rs2_dsm_params _original_dsm_params;
        calib _final_calibration;         // starting state of auto-calibration
        calib _optimaized_calibration;         // the last result of optimize_p that gives the optimal cost but not used for calibration
        rs2_dsm_params _final_dsm_params;
        calib _factory_calibration;          // factory default calibration of the camera
        optimization_params _params_curr;  // last-known setting

        //cycle data from bin files
        bool get_cycle_data_from_bin = false;
        bool get_final_data_from_bin = false;
        rs2_intrinsics_double _k_dapth_from_bin;
        p_matrix _p_mat_from_bin;
        p_matrix _p_mat_from_bin_opt;
        std::vector<double3> _vertices_from_bin;
        algo_calibration_registers _dsm_regs_cand_from_bin;
        rs2_dsm_params_double _dsm_params_cand_from_bin;

        std::shared_ptr<k_to_DSM> _k_to_DSM;
        bool _debug_mode;
    };

}  // librealsense::algo::depth_to_rgb_calibration
}  // librealsense::algo
}  // librealsense
