// Copyright (c) 2009, Willow Garage, Inc.
// All rights reserved.
//
// Software License Agreement (BSD License 2.0)
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//
//  * Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
//  * Redistributions in binary form must reproduce the above
//    copyright notice, this list of conditions and the following
//    disclaimer in the documentation and/or other materials provided
//    with the distribution.
//  * Neither the name of the Willow Garage nor the names of its
//    contributors may be used to endorse or promote products derived
//    from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
// FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
// COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
// BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
// ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.

// Original version: Kevin Watts <watts@willowgarage.com>

#include <algorithm>
#include <limits>
#include <random>

#include "control_toolbox/dither.hpp"

namespace control_toolbox
{
Dither::Dither()
: amplitude_(0), has_saved_value_(false) {}

double Dither::update()
{
  if (has_saved_value_) {
    has_saved_value_ = false;
    return saved_value_;
  }

  // Generates gaussian random noise using the polar method.
  double v1, v2, r;
  // uniform distribution on the interval [-1.0, 1.0]
  std::uniform_real_distribution<double> distribution(
    -1.0, std::nextafter(1.0, std::numeric_limits<double>::max()));
  for (int i = 0; i < 100; ++i) {
    v1 = distribution(generator_);
    v2 = distribution(generator_);
    r = v1 * v1 + v2 * v2;
    if (r <= 1.0) {
      break;
    }
  }
  r = std::min(r, 1.0);

  double f = sqrt(-2.0 * log(r) / r);
  double current = amplitude_ * f * v1;
  saved_value_ = amplitude_ * f * v2;
  has_saved_value_ = true;

  return current;
}

}  // namespace control_toolbox
