// BSD 3-Clause License
//
// Copyright (c) 2022, Woven Planet. All rights reserved.
// Copyright (c) 2020-2022, Toyota Research Institute. All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// * Redistributions of source code must retain the above copyright notice, this
//   list of conditions and the following disclaimer.
//
// * Redistributions in binary form must reproduce the above copyright notice,
//   this list of conditions and the following disclaimer in the documentation
//   and/or other materials provided with the distribution.
//
// * Neither the name of the copyright holder nor the names of its
//   contributors may be used to endorse or promote products derived from
//   this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
// OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#pragma once

#include <optional>
#include <ostream>
#include <unordered_map>

#include <maliput/api/type_specific_identifier.h>

#include "maliput_malidrive/common/macros.h"
#include "maliput_malidrive/xodr/connection.h"

namespace malidrive {
namespace xodr {

/// Holds a junction description of a XODR road.
/// For example:
/// @code{.xml}
///   <OpenDRIVE>
///       ...
///    <junction id="406" name="junction406">
///        <connection id="0" incomingRoad="166" connectingRoad="409" contactPoint="start">
///            <laneLink from="-2" to="-2"/>
///            <laneLink from="-3" to="-3"/>
///            <laneLink from="-4" to="-4"/>
///        </connection>
///    </junction>
///       ...
///   </OpenDRIVE>
/// @endcode
struct Junction {
  using Id = maliput::api::TypeSpecificIdentifier<struct Junction>;

  /// Convenient constants that hold the tag names in the XODR junction description.
  static constexpr const char* kJunctionTag = "junction";
  static constexpr const char* kId = "id";
  static constexpr const char* kName = "name";
  static constexpr const char* kType = "type";

  /// Enum junction types.
  enum Type { kDefault = 0, kVirtual };

  /// Matches Type with a string.
  /// @param type Is a string.
  /// @returns A Type that matches with `type`.
  /// @throw maliput::common::assertion_error When `type` doesn't match with a Type.
  static Type str_to_type(const std::string& type);

  /// Matches string with a Type.
  /// @param type Is a Type.
  /// @returns A string that matches with `type`.
  static std::string type_to_str(Type type);

  /// Equality operator.
  bool operator==(const Junction& other) const;
  /// Inequality operator.
  bool operator!=(const Junction& other) const { return !(*this == other); }

  /// Junction's id.
  Id id{"None"};
  /// Junction's name.
  std::optional<std::string> name{std::nullopt};
  /// Type of the junction, required for "virtual" junctions only.
  std::optional<Type> type{Type::kDefault};
  /// Connections within the junction.
  std::unordered_map<Connection::Id, Connection> connections{};
};

/// Streams a string representation of @p junction into @p out. Returns
/// @p out. This method is provided for the purposes of debugging or
/// text-logging. It is not intended for serialization.
std::ostream& operator<<(std::ostream& out, const Junction& junction);

}  // namespace xodr
}  // namespace malidrive
