// Copyright (C) 2022 Intel Corporation
// SPDX-License-Identifier: Apache-2.0
//

#include <gtest/gtest.h>

#include "openvino/op/gru_sequence.hpp"
#include "base_reference_test.hpp"

using namespace reference_tests;
using namespace ov;

namespace {
struct GRUSequenceParams {
    template <class T>
    GRUSequenceParams(
        const size_t batchSize, const size_t inputSize, const size_t hiddenSize, const size_t seqLength,
        const float clip, const bool linear_before_reset, const op::RecurrentSequenceDirection& gru_direction,
        const element::Type_t& iType,
        const std::vector<T>& XValues, const std::vector<T>& H_tValues, const std::vector<int64_t>& seqLengthsValues,
        const std::vector<T>& WValues, const std::vector<T>& RValues, const std::vector<T>& BValues,
        const std::vector<T>& YValues, const std::vector<T>& HoValues,
        const std::string& testcaseName = "") :
        batchSize(batchSize), inputSize(inputSize), hiddenSize(hiddenSize), seqLength(seqLength),
        clip(clip), linear_before_reset(linear_before_reset), gru_direction(gru_direction), iType(iType), oType(iType),
        testcaseName(testcaseName) {
            numDirections = (gru_direction == op::RecurrentSequenceDirection::BIDIRECTIONAL) ? 2 : 1;

            Shape XShape = Shape{batchSize, seqLength, inputSize};
            Shape H_tShape = Shape{batchSize, numDirections, hiddenSize};
            Shape seqLengthsShape = Shape{batchSize};
            Shape WShape = Shape{numDirections, 3 * hiddenSize, inputSize};
            Shape RShape = Shape{numDirections, 3 * hiddenSize, hiddenSize};
            Shape YShape = Shape{batchSize, numDirections, seqLength, hiddenSize};
            Shape HoShape = Shape{batchSize, numDirections, hiddenSize};

            X = reference_tests::Tensor(XShape, iType, XValues);
            H_t = reference_tests::Tensor(H_tShape, iType, H_tValues);
            sequence_lengths = reference_tests::Tensor(seqLengthsShape, element::Type_t::i64, seqLengthsValues);
            W = reference_tests::Tensor(WShape, iType, WValues);
            R = reference_tests::Tensor(RShape, iType, RValues);
            Y = reference_tests::Tensor(YShape, oType, YValues);
            Ho = reference_tests::Tensor(HoShape, oType, HoValues);

            if (linear_before_reset == true) {
                Shape BShape = Shape{numDirections, 4 * hiddenSize};
                B = reference_tests::Tensor(BShape, iType, BValues);
            } else {
                Shape BShape = Shape{numDirections, 3 * hiddenSize};
                B = reference_tests::Tensor(BShape, iType, BValues);
            }
        }

    size_t batchSize;
    size_t inputSize;
    size_t hiddenSize;
    size_t seqLength;
    size_t numDirections;
    float clip;
    bool linear_before_reset;
    op::RecurrentSequenceDirection gru_direction;
    element::Type_t iType;
    element::Type_t oType;

    reference_tests::Tensor X;
    reference_tests::Tensor H_t;
    reference_tests::Tensor sequence_lengths;
    reference_tests::Tensor W;
    reference_tests::Tensor R;
    reference_tests::Tensor B;
    reference_tests::Tensor Y;
    reference_tests::Tensor Ho;
    std::string testcaseName;
};

class ReferenceGRUSequenceTest : public testing::TestWithParam<GRUSequenceParams>, public CommonReferenceTest {
public:
    void SetUp() override {
        auto params = GetParam();
        function = CreateFunction(params);
        inputData = {params.X.data, params.H_t.data, params.sequence_lengths.data, params.W.data, params.R.data, params.B.data};
        refOutData = {params.Y.data, params.Ho.data};
    }

    static std::string getTestCaseName(const testing::TestParamInfo<GRUSequenceParams>& obj) {
        auto param = obj.param;
        std::ostringstream result;
        result << "iType=" << param.iType << "_";
        result << "xShape=" << param.X.shape << "_";
        result << "htShape=" << param.H_t.shape << "_";
        result << "stShape=" << param.sequence_lengths.shape << "_";
        result << "wShape=" << param.W.shape << "_";
        result << "rShape=" << param.R.shape << "_";
        result << "bShape=" << param.B.shape << "_";
        result << "YShape=" << param.Y.shape << "_";
        result << "hoShape=" << param.Ho.shape << "_";
        result << "clip=" << param.clip << "_";
        result << "linear_before_reset=" << param.linear_before_reset << "_";
        result << "GRUdirection=" << param.gru_direction;
        if (!param.testcaseName.empty())
            result << "_" << param.testcaseName;

        return result.str();
    }

private:
    static std::shared_ptr<Model> CreateFunction(const GRUSequenceParams& params) {
        const auto X = std::make_shared<op::v0::Parameter>(params.X.type, params.X.shape);
        const auto H_t = std::make_shared<op::v0::Parameter>(params.H_t.type, params.H_t.shape);
        const auto sequence_lengths = std::make_shared<op::v0::Parameter>(params.sequence_lengths.type, params.sequence_lengths.shape);
        const auto W = std::make_shared<op::v0::Parameter>(params.W.type, params.W.shape);
        const auto R = std::make_shared<op::v0::Parameter>(params.R.type, params.R.shape);
        const auto B = std::make_shared<op::v0::Parameter>(params.B.type, params.B.shape);

        const auto gru_sequence =
            std::make_shared<op::v5::GRUSequence>(X,
                                                  H_t,
                                                  sequence_lengths,
                                                  W,
                                                  R,
                                                  B,
                                                  params.hiddenSize,
                                                  params.gru_direction,
                                                  std::vector<std::string>{"sigmoid", "tanh"},
                                                  std::vector<float>{},
                                                  std::vector<float>{},
                                                  params.clip,
                                                  params.linear_before_reset);

        auto function = std::make_shared<Model>(gru_sequence->outputs(), ParameterVector{X, H_t, sequence_lengths, W, R, B});
        return function;
    }
};

TEST_P(ReferenceGRUSequenceTest, CompareWithRefs) {
    Exec();
}

template <element::Type_t ET>
std::vector<GRUSequenceParams> generateParams() {
    using T = typename element_type_traits<ET>::value_type;

    std::vector<GRUSequenceParams> params {
        GRUSequenceParams(
            5, 10, 10, 10,
            0.7f, false, op::RecurrentSequenceDirection::FORWARD,
            ET,
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 7.95465,
                2.37637, 6.20771, 1.08107, 7.38138, 5.23577, 7.88133, 5.20653, 3.42101, 8.48523, 5.96192,
                1.63073, 5.25228, 7.68488, 2.7276, 5.1788, 3.07327, 5.57423, 2.87711, 1.44374, 5.66976,
                2.55051, 4.56682, 1.96629, 5.58829, 1.91922, 3.59846, 3.08583, 9.70901, 3.50487, 3.1026,
                1.82401, 6.1306, 4.76134, 4.31059, 8.31695, 3.60784, 7.45652, 6.51653, 4.84219, 7.76686,
                4.85031, 4.85544, 4.25714, 2.38005, 9.43471, 9.24769, 8.03763, 6.54696, 1.32399, 6.88891,
                2.16793, 3.64924, 4.24733, 3.47181, 1.66572, 2.36923, 2.45457, 9.44841, 4.34021, 1.45016,
                7.6686, 3.68812, 2.83922, 9.83581, 9.03719, 7.83414, 6.86009, 1.35715, 8.32489, 7.86316,
                5.09754, 5.78644, 1.98402, 2.31429, 5.5791, 2.94085, 9.24799, 5.15937, 2.19041, 7.87817,
                2.9146, 1.66833, 1.85877, 2.45985, 4.20817, 1.85777, 2.28937, 9.37912, 6.18926, 8.55681,
                6.60963, 3.92066, 7.5521, 5.70463, 7.6313, 2.48866, 7.18352, 4.8413, 7.55702, 7.80702,
                4.5785, 9.3268, 2.83159, 1.07202, 9.33716, 3.6506, 2.50256, 1.21691, 5.06801, 8.27505,
                4.31539, 6.48286, 1.31363, 4.1912, 1.70668, 7.23867, 1.11441, 5.13591, 9.65186, 4.00767,
                5.24875, 1.94852, 5.52768, 8.97121, 5.8094, 3.53329, 4.19126, 9.06652, 3.1734, 1.21496,
                9.69154, 4.86971, 4.1166, 6.19361, 2.13874, 9.55039, 3.8225, 9.57548, 2.96554, 3.2383,
                8.77422, 3.11741, 8.3359, 5.89508, 2.72134, 6.29956, 1.43323, 1.14286, 1.4474, 4.59474,
                6.19214, 8.80766, 8.07546, 3.29232, 1.74029, 2.4198, 2.88544, 4.75644, 4.12921, 7.29896,
                7.27759, 1.67252, 1.32823, 8.1046, 9.10476, 1.04197, 3.37783, 5.2064, 4.23835, 3.16196,
                1.20852, 5.78501, 2.17175, 6.05313, 2.51048, 4.78967, 7.16219, 3.4651, 1.09, 2.9788,
                1.28761, 9.41385, 8.03883, 5.65835, 1.14816, 3.6892, 5.86648, 8.73895, 2.66603, 1.75192,
                1.39845, 4.99427, 1.17387, 1.60329, 8.30594, 6.72662, 7.95565, 7.35114, 3.1439, 1.39976,
                3.53095, 8.78581, 1.65811, 6.94299, 2.68641, 5.70058, 9.13491, 5.27637, 8.6232, 8.54902,
                2.25352, 5.86274, 5.20377, 2.96815, 4.96745, 5.3225, 3.99956, 1.08021, 5.54918, 7.05833,
                1.49913, 2.41822, 6.44593, 3.87301, 9.01465, 8.11336, 2.95749, 2.80188, 7.12396, 2.40595,
                5.59325, 9.89258, 2.30223, 1.4347, 9.09158, 7.43797, 3.79295, 4.53646, 1.72705, 4.16909,
                1.00912, 6.62167, 2.80244, 6.626, 3.89307, 1.42586, 7.51028, 7.83327, 4.65674, 7.33902,
                6.26823, 9.72608, 3.73491, 3.8238, 3.03815, 7.05101, 8.0103, 5.61396, 6.53738, 1.41095,
                5.0149, 9.71211, 4.23604, 5.98629, 4.70219, 9.69442, 2.82752, 9.93544, 6.9328, 8.2817,
                5.12336, 8.98577, 5.80541, 6.19552, 9.25748, 3.82732, 7.53525, 8.24712, 5.32057, 5.38817,
                8.57269, 5.99975, 3.42893, 5.38068, 3.48261, 3.02851, 6.82079, 9.2902, 2.80427, 8.91868,
                5.19227, 7.52482, 3.72584, 5.40107, 2.83307, 1.79755, 2.49121, 5.52697, 8.08823, 9.13242,
                2.97572, 7.64318, 3.32023, 6.07788, 2.19187, 4.34879, 1.7457, 5.55154, 7.24966, 5.1128,
                4.25147, 8.34407, 1.4123, 4.49045, 5.12671, 7.62159, 9.18673, 3.49665, 8.35992, 6.90684,
                1.10152, 7.61818, 6.43145, 7.12017, 6.25564, 6.16169, 4.24916, 9.6283, 9.88249, 4.48422,
                8.52562, 9.83928, 6.26818, 7.03839, 1.77631, 9.92305, 8.0155, 9.94928, 6.88321, 1.33685,
                7.4718, 7.19305, 6.47932, 1.9559, 3.52616, 7.98593, 9.0115, 5.59539, 7.44137, 1.70001,
                6.53774, 8.54023, 7.26405, 5.99553, 8.75071, 7.70789, 3.38094, 9.99792, 6.16359, 6.75153,
                5.4073, 9.00437, 8.87059, 8.63011, 6.82951, 6.27021, 3.53425, 9.92489, 8.19695, 5.51473,
                7.95084, 2.11852, 9.28916, 1.40353, 3.05744, 8.58238, 3.75014, 5.35889, 6.85048, 2.29549,
                3.75218, 8.98228, 8.98158, 5.63695, 3.40379, 8.92309, 5.48185, 4.00095, 9.05227, 2.84035,
                8.37644, 8.54954, 5.70516, 2.45744, 9.54079, 1.53504, 8.9785, 6.1691, 4.40962, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 10},
            std::vector<int64_t>{10, 10, 10, 10, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 7.95465,
                2.37637, 6.20771, 1.08107, 7.38138, 5.23577, 7.88133, 5.20653, 3.42101, 8.48523, 5.96192,
                1.63073, 5.25228, 7.68488, 2.7276, 5.1788, 3.07327, 5.57423, 2.87711, 1.44374, 5.66976,
                2.55051, 4.56682, 1.96629, 5.58829, 1.91922, 3.59846, 3.08583, 9.70901, 3.50487, 3.1026,
                1.82401, 6.1306, 4.76134, 4.31059, 8.31695, 3.60784, 7.45652, 6.51653, 4.84219, 7.76686,
                4.85031, 4.85544, 4.25714, 2.38005, 9.43471, 9.24769, 8.03763, 6.54696, 1.32399, 6.88891,
                2.16793, 3.64924, 4.24733, 3.47181, 1.66572, 2.36923, 2.45457, 9.44841, 4.34021, 1.45016,
                7.6686, 3.68812, 2.83922, 9.83581, 9.03719, 7.83414, 6.86009, 1.35715, 8.32489, 7.86316,
                5.09754, 5.78644, 1.98402, 2.31429, 5.5791, 2.94085, 9.24799, 5.15937, 2.19041, 7.87817,
                2.9146, 1.66833, 1.85877, 2.45985, 4.20817, 1.85777, 2.28937, 9.37912, 6.18926, 8.55681,
                6.60963, 3.92066, 7.5521, 5.70463, 7.6313, 2.48866, 7.18352, 4.8413, 7.55702, 7.80702,
                4.5785, 9.3268, 2.83159, 1.07202, 9.33716, 3.6506, 2.50256, 1.21691, 5.06801, 8.27505,
                4.31539, 6.48286, 1.31363, 4.1912, 1.70668, 7.23867, 1.11441, 5.13591, 9.65186, 4.00767,
                5.24875, 1.94852, 5.52768, 8.97121, 5.8094, 3.53329, 4.19126, 9.06652, 3.1734, 1.21496,
                9.69154, 4.86971, 4.1166, 6.19361, 2.13874, 9.55039, 3.8225, 9.57548, 2.96554, 3.2383,
                8.77422, 3.11741, 8.3359, 5.89508, 2.72134, 6.29956, 1.43323, 1.14286, 1.4474, 4.59474,
                6.19214, 8.80766, 8.07546, 3.29232, 1.74029, 2.4198, 2.88544, 4.75644, 4.12921, 7.29896,
                7.27759, 1.67252, 1.32823, 8.1046, 9.10476, 1.04197, 3.37783, 5.2064, 4.23835, 3.16196,
                1.20852, 5.78501, 2.17175, 6.05313, 2.51048, 4.78967, 7.16219, 3.4651, 1.09, 2.9788,
                1.28761, 9.41385, 8.03883, 5.65835, 1.14816, 3.6892, 5.86648, 8.73895, 2.66603, 1.75192,
                1.39845, 4.99427, 1.17387, 1.60329, 8.30594, 6.72662, 7.95565, 7.35114, 3.1439, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 7.95465,
                2.37637, 6.20771, 1.08107, 7.38138, 5.23577, 7.88133, 5.20653, 3.42101, 8.48523, 5.96192,
                1.63073, 5.25228, 7.68488, 2.7276, 5.1788, 3.07327, 5.57423, 2.87711, 1.44374, 5.66976,
                2.55051, 4.56682, 1.96629, 5.58829, 1.91922, 3.59846, 3.08583, 9.70901, 3.50487, 3.1026,
                1.82401, 6.1306, 4.76134, 4.31059, 8.31695, 3.60784, 7.45652, 6.51653, 4.84219, 7.76686,
                4.85031, 4.85544, 4.25714, 2.38005, 9.43471, 9.24769, 8.03763, 6.54696, 1.32399, 6.88891,
                2.16793, 3.64924, 4.24733, 3.47181, 1.66572, 2.36923, 2.45457, 9.44841, 4.34021, 1.45016,
                7.6686, 3.68812, 2.83922, 9.83581, 9.03719, 7.83414, 6.86009, 1.35715, 8.32489, 7.86316,
                5.09754, 5.78644, 1.98402, 2.31429, 5.5791, 2.94085, 9.24799, 5.15937, 2.19041, 7.87817,
                2.9146, 1.66833, 1.85877, 2.45985, 4.20817, 1.85777, 2.28937, 9.37912, 6.18926, 8.55681,
                6.60963, 3.92066, 7.5521, 5.70463, 7.6313, 2.48866, 7.18352, 4.8413, 7.55702, 7.80702,
                4.5785, 9.3268, 2.83159, 1.07202, 9.33716, 3.6506, 2.50256, 1.21691, 5.06801, 8.27505,
                4.31539, 6.48286, 1.31363, 4.1912, 1.70668, 7.23867, 1.11441, 5.13591, 9.65186, 4.00767,
                5.24875, 1.94852, 5.52768, 8.97121, 5.8094, 3.53329, 4.19126, 9.06652, 3.1734, 1.21496,
                9.69154, 4.86971, 4.1166, 6.19361, 2.13874, 9.55039, 3.8225, 9.57548, 2.96554, 3.2383,
                8.77422, 3.11741, 8.3359, 5.89508, 2.72134, 6.29956, 1.43323, 1.14286, 1.4474, 4.59474,
                6.19214, 8.80766, 8.07546, 3.29232, 1.74029, 2.4198, 2.88544, 4.75644, 4.12921, 7.29896,
                7.27759, 1.67252, 1.32823, 8.1046, 9.10476, 1.04197, 3.37783, 5.2064, 4.23835, 3.16196,
                1.20852, 5.78501, 2.17175, 6.05313, 2.51048, 4.78967, 7.16219, 3.4651, 1.09, 2.9788,
                1.28761, 9.41385, 8.03883, 5.65835, 1.14816, 3.6892, 5.86648, 8.73895, 2.66603, 1.75192,
                1.39845, 4.99427, 1.17387, 1.60329, 8.30594, 6.72662, 7.95565, 7.35114, 3.1439, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 10},
            std::vector<T>{
                0.868724, 6.86548, 6.47683, 1.63923, 6.87664, 2.28849, 3.25364, 3.2015, 4.89637, 6.49477,
                0.781008, 4.78797, 4.52828, 1.29585, 4.79543, 1.72968, 2.37458, 2.33974, 3.47223, 4.54026,
                0.722396, 3.3998, 3.22628, 1.06641, 3.40478, 1.35629, 1.7872, 1.76392, 2.52064, 3.23428,
                0.683233, 2.47224, 2.3563, 0.913096, 2.47557, 1.10679, 1.39472, 1.37917, 1.8848, 2.36164,
                0.657064, 1.85246, 1.77498, 0.810656, 1.85468, 0.94008, 1.13247, 1.12208, 1.45993, 1.77856,
                0.639579, 1.43833, 1.38656, 0.742207, 1.43981, 0.828687, 0.957241, 0.950296, 1.17605, 1.38895,
                0.627896, 1.16161, 1.12702, 0.69647, 1.1626, 0.754255, 0.840153, 0.835513, 0.986357, 1.12861,
                0.620089, 0.976709, 0.953597, 0.66591, 0.977373, 0.704521, 0.761917, 0.758816, 0.859608, 0.954663,
                0.614872, 0.853162, 0.837718, 0.645489, 0.853605, 0.671289, 0.70964, 0.707568, 0.774916, 0.838431,
                0.611387, 0.770609, 0.76029, 0.631845, 0.770905, 0.649083, 0.674709, 0.673325, 0.718326, 0.760766,
                5.95818, 2.75265, 4.02319, 3.53551, 2.24933, 4.08252, 6.36501, 3.61821, 3.45881, 6.51635,
                4.18172, 2.03983, 2.88879, 2.56292, 1.70351, 2.92843, 4.45356, 2.61818, 2.51167, 4.55468,
                2.99471, 1.56352, 2.13079, 1.91305, 1.3388, 2.15728, 3.17635, 1.94997, 1.87881, 3.24392,
                2.20156, 1.24526, 1.6243, 1.47881, 1.09511, 1.642, 2.32293, 1.50349, 1.45593, 2.36808,
                1.67159, 1.03261, 1.28588, 1.18866, 0.932274, 1.2977, 1.75269, 1.20515, 1.17337, 1.78286,
                1.31748, 0.890512, 1.05974, 0.994786, 0.823471, 1.06765, 1.37166, 1.0058, 0.984569, 1.39182,
                1.08086, 0.795566, 0.908644, 0.86524, 0.75077, 0.913924, 1.11707, 0.872601, 0.858414, 1.13054,
                0.922752, 0.732124, 0.807681, 0.778679, 0.702192, 0.81121, 0.946946, 0.783598, 0.774118, 0.955946,
                0.817109, 0.689733, 0.740219, 0.720841, 0.669733, 0.742577, 0.833275, 0.724127, 0.717793, 0.839288,
                0.746518, 0.661408, 0.695142, 0.682194, 0.648044, 0.696717, 0.75732, 0.684389, 0.680157, 0.761339,
                5.54972, 5.17435, 5.69626, 1.4268, 3.98473, 6.07069, 5.85496, 5.8577, 2.51076, 1.22499,
                3.90879, 3.65797, 4.00671, 1.15391, 2.86309, 4.2569, 4.11275, 4.11458, 1.8782, 1.01906,
                2.81234, 2.64475, 2.87777, 0.971563, 2.11362, 3.04494, 2.94862, 2.94985, 1.45553, 0.881462,
                2.07971, 1.96773, 2.12343, 0.849723, 1.61283, 2.23513, 2.17077, 2.17159, 1.1731, 0.789519,
                1.59017, 1.51535, 1.61938, 0.768311, 1.27821, 1.69402, 1.65102, 1.65157, 0.984388, 0.728083,
                1.26307, 1.21307, 1.28259, 0.713913, 1.05462, 1.33246, 1.30373, 1.30409, 0.858293, 0.687033,
                1.0445, 1.0111, 1.05755, 0.677564, 0.905221, 1.09087, 1.07167, 1.07192, 0.774037, 0.659604,
                0.898462, 0.876139, 0.907177, 0.653277, 0.805394, 0.929444, 0.916614, 0.916777, 0.717739, 0.641276,
                0.800878, 0.785962, 0.806701, 0.637048, 0.738691, 0.821579, 0.813007, 0.813116, 0.680121, 0.629029,
                0.735673, 0.725707, 0.739564, 0.626204, 0.694121, 0.749506, 0.743778, 0.743851, 0.654985, 0.620846,
                4.90107, 4.43524, 4.90784, 3.34509, 2.05673, 2.61047, 1.72339, 5.57933, 3.34961, 1.07422,
                3.47537, 3.16411, 3.4799, 2.43568, 1.57482, 1.94482, 1.35209, 3.92858, 2.4387, 0.918317,
                2.52274, 2.31475, 2.52576, 1.82803, 1.25281, 1.50004, 1.10398, 2.82557, 1.83005, 0.814145,
                1.8862, 1.74723, 1.88822, 1.422, 1.03765, 1.20285, 0.938205, 2.08854, 1.42335, 0.744538,
                1.46087, 1.36801, 1.46222, 1.1507, 0.893882, 1.00426, 0.827433, 1.59608, 1.1516, 0.698028,
                1.17667, 1.11463, 1.17758, 0.969422, 0.797818, 0.871573, 0.753417, 1.26702, 0.970024, 0.66695,
                0.986775, 0.945316, 0.987378, 0.848292, 0.733629, 0.782911, 0.703961, 1.04714, 0.848694, 0.646185,
                0.859888, 0.832185, 0.86029, 0.767355, 0.690738, 0.723668, 0.670915, 0.900223, 0.767624, 0.632309,
                0.775103, 0.756592, 0.775372, 0.713274, 0.662079, 0.684083, 0.648834, 0.802055, 0.713453, 0.623038,
                0.718451, 0.706082, 0.718631, 0.677138, 0.64293, 0.657632, 0.634079, 0.73646, 0.677257, 0.616843,
                4.62145, 4.84158, 2.66378, 3.55164, 2.20451, 1.30991, 3.69058, 1.44707, 6.30131, 6.88241,
                3.28853, 3.43562, 1.98044, 2.5737, 1.67356, 1.0758, 2.66654, 1.16745, 4.411, 4.79928,
                2.39789, 2.49618, 1.52384, 1.92025, 1.31879, 0.919376, 1.98228, 0.980615, 3.14791, 3.40736,
                1.80278, 1.86845, 1.21875, 1.48362, 1.08174, 0.814853, 1.52507, 0.855772, 2.30393, 2.47729,
                1.40513, 1.44901, 1.01489, 1.19188, 0.92334, 0.745011, 1.21957, 0.772353, 1.74, 1.85583,
                1.13943, 1.16875, 0.878674, 0.996934, 0.817501, 0.698344, 1.01544, 0.716613, 1.36318, 1.44058,
                0.961889, 0.981481, 0.787656, 0.866675, 0.746781, 0.667161, 0.879041, 0.679369, 1.1114, 1.16312,
                0.843259, 0.85635, 0.726839, 0.779639, 0.699526, 0.646326, 0.787901, 0.654483, 0.943158, 0.977716,
                0.763992, 0.772739, 0.686201, 0.721482, 0.667952, 0.632404, 0.727003, 0.637854, 0.830743, 0.853834,
                0.711027, 0.716871, 0.659048, 0.682622, 0.646854, 0.623101, 0.686311, 0.626743, 0.755629, 0.771058},
            std::vector<T>{
                0.611387, 0.770609, 0.76029, 0.631845, 0.770905, 0.649083, 0.674709, 0.673325, 0.718326, 0.760766,
                0.746518, 0.661408, 0.695142, 0.682194, 0.648044, 0.696717, 0.75732, 0.684389, 0.680157, 0.761339,
                0.735673, 0.725707, 0.739564, 0.626204, 0.694121, 0.749506, 0.743778, 0.743851, 0.654985, 0.620846,
                0.718451, 0.706082, 0.718631, 0.677138, 0.64293, 0.657632, 0.634079, 0.73646, 0.677257, 0.616843,
                0.711027, 0.716871, 0.659048, 0.682622, 0.646854, 0.623101, 0.686311, 0.626743, 0.755629, 0.771058}),
        GRUSequenceParams(
            5, 10, 10, 10,
            0.7f, false, op::RecurrentSequenceDirection::REVERSE,
            ET,
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 7.95465,
                2.37637, 6.20771, 1.08107, 7.38138, 5.23577, 7.88133, 5.20653, 3.42101, 8.48523, 5.96192,
                1.63073, 5.25228, 7.68488, 2.7276, 5.1788, 3.07327, 5.57423, 2.87711, 1.44374, 5.66976,
                2.55051, 4.56682, 1.96629, 5.58829, 1.91922, 3.59846, 3.08583, 9.70901, 3.50487, 3.1026,
                1.82401, 6.1306, 4.76134, 4.31059, 8.31695, 3.60784, 7.45652, 6.51653, 4.84219, 7.76686,
                4.85031, 4.85544, 4.25714, 2.38005, 9.43471, 9.24769, 8.03763, 6.54696, 1.32399, 6.88891,
                2.16793, 3.64924, 4.24733, 3.47181, 1.66572, 2.36923, 2.45457, 9.44841, 4.34021, 1.45016,
                7.6686, 3.68812, 2.83922, 9.83581, 9.03719, 7.83414, 6.86009, 1.35715, 8.32489, 7.86316,
                5.09754, 5.78644, 1.98402, 2.31429, 5.5791, 2.94085, 9.24799, 5.15937, 2.19041, 7.87817,
                2.9146, 1.66833, 1.85877, 2.45985, 4.20817, 1.85777, 2.28937, 9.37912, 6.18926, 8.55681,
                6.60963, 3.92066, 7.5521, 5.70463, 7.6313, 2.48866, 7.18352, 4.8413, 7.55702, 7.80702,
                4.5785, 9.3268, 2.83159, 1.07202, 9.33716, 3.6506, 2.50256, 1.21691, 5.06801, 8.27505,
                4.31539, 6.48286, 1.31363, 4.1912, 1.70668, 7.23867, 1.11441, 5.13591, 9.65186, 4.00767,
                5.24875, 1.94852, 5.52768, 8.97121, 5.8094, 3.53329, 4.19126, 9.06652, 3.1734, 1.21496,
                9.69154, 4.86971, 4.1166, 6.19361, 2.13874, 9.55039, 3.8225, 9.57548, 2.96554, 3.2383,
                8.77422, 3.11741, 8.3359, 5.89508, 2.72134, 6.29956, 1.43323, 1.14286, 1.4474, 4.59474,
                6.19214, 8.80766, 8.07546, 3.29232, 1.74029, 2.4198, 2.88544, 4.75644, 4.12921, 7.29896,
                7.27759, 1.67252, 1.32823, 8.1046, 9.10476, 1.04197, 3.37783, 5.2064, 4.23835, 3.16196,
                1.20852, 5.78501, 2.17175, 6.05313, 2.51048, 4.78967, 7.16219, 3.4651, 1.09, 2.9788,
                1.28761, 9.41385, 8.03883, 5.65835, 1.14816, 3.6892, 5.86648, 8.73895, 2.66603, 1.75192,
                1.39845, 4.99427, 1.17387, 1.60329, 8.30594, 6.72662, 7.95565, 7.35114, 3.1439, 1.39976,
                3.53095, 8.78581, 1.65811, 6.94299, 2.68641, 5.70058, 9.13491, 5.27637, 8.6232, 8.54902,
                2.25352, 5.86274, 5.20377, 2.96815, 4.96745, 5.3225, 3.99956, 1.08021, 5.54918, 7.05833,
                1.49913, 2.41822, 6.44593, 3.87301, 9.01465, 8.11336, 2.95749, 2.80188, 7.12396, 2.40595,
                5.59325, 9.89258, 2.30223, 1.4347, 9.09158, 7.43797, 3.79295, 4.53646, 1.72705, 4.16909,
                1.00912, 6.62167, 2.80244, 6.626, 3.89307, 1.42586, 7.51028, 7.83327, 4.65674, 7.33902,
                6.26823, 9.72608, 3.73491, 3.8238, 3.03815, 7.05101, 8.0103, 5.61396, 6.53738, 1.41095,
                5.0149, 9.71211, 4.23604, 5.98629, 4.70219, 9.69442, 2.82752, 9.93544, 6.9328, 8.2817,
                5.12336, 8.98577, 5.80541, 6.19552, 9.25748, 3.82732, 7.53525, 8.24712, 5.32057, 5.38817,
                8.57269, 5.99975, 3.42893, 5.38068, 3.48261, 3.02851, 6.82079, 9.2902, 2.80427, 8.91868,
                5.19227, 7.52482, 3.72584, 5.40107, 2.83307, 1.79755, 2.49121, 5.52697, 8.08823, 9.13242,
                2.97572, 7.64318, 3.32023, 6.07788, 2.19187, 4.34879, 1.7457, 5.55154, 7.24966, 5.1128,
                4.25147, 8.34407, 1.4123, 4.49045, 5.12671, 7.62159, 9.18673, 3.49665, 8.35992, 6.90684,
                1.10152, 7.61818, 6.43145, 7.12017, 6.25564, 6.16169, 4.24916, 9.6283, 9.88249, 4.48422,
                8.52562, 9.83928, 6.26818, 7.03839, 1.77631, 9.92305, 8.0155, 9.94928, 6.88321, 1.33685,
                7.4718, 7.19305, 6.47932, 1.9559, 3.52616, 7.98593, 9.0115, 5.59539, 7.44137, 1.70001,
                6.53774, 8.54023, 7.26405, 5.99553, 8.75071, 7.70789, 3.38094, 9.99792, 6.16359, 6.75153,
                5.4073, 9.00437, 8.87059, 8.63011, 6.82951, 6.27021, 3.53425, 9.92489, 8.19695, 5.51473,
                7.95084, 2.11852, 9.28916, 1.40353, 3.05744, 8.58238, 3.75014, 5.35889, 6.85048, 2.29549,
                3.75218, 8.98228, 8.98158, 5.63695, 3.40379, 8.92309, 5.48185, 4.00095, 9.05227, 2.84035,
                8.37644, 8.54954, 5.70516, 2.45744, 9.54079, 1.53504, 8.9785, 6.1691, 4.40962, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 10},
            std::vector<int64_t>{10, 10, 10, 10, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 7.95465,
                2.37637, 6.20771, 1.08107, 7.38138, 5.23577, 7.88133, 5.20653, 3.42101, 8.48523, 5.96192,
                1.63073, 5.25228, 7.68488, 2.7276, 5.1788, 3.07327, 5.57423, 2.87711, 1.44374, 5.66976,
                2.55051, 4.56682, 1.96629, 5.58829, 1.91922, 3.59846, 3.08583, 9.70901, 3.50487, 3.1026,
                1.82401, 6.1306, 4.76134, 4.31059, 8.31695, 3.60784, 7.45652, 6.51653, 4.84219, 7.76686,
                4.85031, 4.85544, 4.25714, 2.38005, 9.43471, 9.24769, 8.03763, 6.54696, 1.32399, 6.88891,
                2.16793, 3.64924, 4.24733, 3.47181, 1.66572, 2.36923, 2.45457, 9.44841, 4.34021, 1.45016,
                7.6686, 3.68812, 2.83922, 9.83581, 9.03719, 7.83414, 6.86009, 1.35715, 8.32489, 7.86316,
                5.09754, 5.78644, 1.98402, 2.31429, 5.5791, 2.94085, 9.24799, 5.15937, 2.19041, 7.87817,
                2.9146, 1.66833, 1.85877, 2.45985, 4.20817, 1.85777, 2.28937, 9.37912, 6.18926, 8.55681,
                6.60963, 3.92066, 7.5521, 5.70463, 7.6313, 2.48866, 7.18352, 4.8413, 7.55702, 7.80702,
                4.5785, 9.3268, 2.83159, 1.07202, 9.33716, 3.6506, 2.50256, 1.21691, 5.06801, 8.27505,
                4.31539, 6.48286, 1.31363, 4.1912, 1.70668, 7.23867, 1.11441, 5.13591, 9.65186, 4.00767,
                5.24875, 1.94852, 5.52768, 8.97121, 5.8094, 3.53329, 4.19126, 9.06652, 3.1734, 1.21496,
                9.69154, 4.86971, 4.1166, 6.19361, 2.13874, 9.55039, 3.8225, 9.57548, 2.96554, 3.2383,
                8.77422, 3.11741, 8.3359, 5.89508, 2.72134, 6.29956, 1.43323, 1.14286, 1.4474, 4.59474,
                6.19214, 8.80766, 8.07546, 3.29232, 1.74029, 2.4198, 2.88544, 4.75644, 4.12921, 7.29896,
                7.27759, 1.67252, 1.32823, 8.1046, 9.10476, 1.04197, 3.37783, 5.2064, 4.23835, 3.16196,
                1.20852, 5.78501, 2.17175, 6.05313, 2.51048, 4.78967, 7.16219, 3.4651, 1.09, 2.9788,
                1.28761, 9.41385, 8.03883, 5.65835, 1.14816, 3.6892, 5.86648, 8.73895, 2.66603, 1.75192,
                1.39845, 4.99427, 1.17387, 1.60329, 8.30594, 6.72662, 7.95565, 7.35114, 3.1439, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 7.95465,
                2.37637, 6.20771, 1.08107, 7.38138, 5.23577, 7.88133, 5.20653, 3.42101, 8.48523, 5.96192,
                1.63073, 5.25228, 7.68488, 2.7276, 5.1788, 3.07327, 5.57423, 2.87711, 1.44374, 5.66976,
                2.55051, 4.56682, 1.96629, 5.58829, 1.91922, 3.59846, 3.08583, 9.70901, 3.50487, 3.1026,
                1.82401, 6.1306, 4.76134, 4.31059, 8.31695, 3.60784, 7.45652, 6.51653, 4.84219, 7.76686,
                4.85031, 4.85544, 4.25714, 2.38005, 9.43471, 9.24769, 8.03763, 6.54696, 1.32399, 6.88891,
                2.16793, 3.64924, 4.24733, 3.47181, 1.66572, 2.36923, 2.45457, 9.44841, 4.34021, 1.45016,
                7.6686, 3.68812, 2.83922, 9.83581, 9.03719, 7.83414, 6.86009, 1.35715, 8.32489, 7.86316,
                5.09754, 5.78644, 1.98402, 2.31429, 5.5791, 2.94085, 9.24799, 5.15937, 2.19041, 7.87817,
                2.9146, 1.66833, 1.85877, 2.45985, 4.20817, 1.85777, 2.28937, 9.37912, 6.18926, 8.55681,
                6.60963, 3.92066, 7.5521, 5.70463, 7.6313, 2.48866, 7.18352, 4.8413, 7.55702, 7.80702,
                4.5785, 9.3268, 2.83159, 1.07202, 9.33716, 3.6506, 2.50256, 1.21691, 5.06801, 8.27505,
                4.31539, 6.48286, 1.31363, 4.1912, 1.70668, 7.23867, 1.11441, 5.13591, 9.65186, 4.00767,
                5.24875, 1.94852, 5.52768, 8.97121, 5.8094, 3.53329, 4.19126, 9.06652, 3.1734, 1.21496,
                9.69154, 4.86971, 4.1166, 6.19361, 2.13874, 9.55039, 3.8225, 9.57548, 2.96554, 3.2383,
                8.77422, 3.11741, 8.3359, 5.89508, 2.72134, 6.29956, 1.43323, 1.14286, 1.4474, 4.59474,
                6.19214, 8.80766, 8.07546, 3.29232, 1.74029, 2.4198, 2.88544, 4.75644, 4.12921, 7.29896,
                7.27759, 1.67252, 1.32823, 8.1046, 9.10476, 1.04197, 3.37783, 5.2064, 4.23835, 3.16196,
                1.20852, 5.78501, 2.17175, 6.05313, 2.51048, 4.78967, 7.16219, 3.4651, 1.09, 2.9788,
                1.28761, 9.41385, 8.03883, 5.65835, 1.14816, 3.6892, 5.86648, 8.73895, 2.66603, 1.75192,
                1.39845, 4.99427, 1.17387, 1.60329, 8.30594, 6.72662, 7.95565, 7.35114, 3.1439, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 10},
            std::vector<T>{
                0.611387, 0.770609, 0.76029, 0.631845, 0.770905, 0.649083, 0.674709, 0.673325, 0.718326, 0.760766,
                0.614872, 0.853162, 0.837718, 0.645489, 0.853605, 0.671289, 0.70964, 0.707568, 0.774916, 0.838431,
                0.620089, 0.976709, 0.953597, 0.66591, 0.977373, 0.704521, 0.761917, 0.758816, 0.859608, 0.954663,
                0.627896, 1.16161, 1.12702, 0.69647, 1.1626, 0.754255, 0.840153, 0.835513, 0.986357, 1.12861,
                0.639579, 1.43833, 1.38656, 0.742207, 1.43981, 0.828687, 0.957241, 0.950296, 1.17605, 1.38895,
                0.657064, 1.85246, 1.77498, 0.810656, 1.85468, 0.94008, 1.13247, 1.12208, 1.45993, 1.77856,
                0.683233, 2.47224, 2.3563, 0.913096, 2.47557, 1.10679, 1.39472, 1.37917, 1.8848, 2.36164,
                0.722396, 3.3998, 3.22628, 1.06641, 3.40478, 1.35629, 1.7872, 1.76392, 2.52064, 3.23428,
                0.781008, 4.78797, 4.52828, 1.29585, 4.79543, 1.72968, 2.37458, 2.33974, 3.47223, 4.54026,
                0.868724, 6.86548, 6.47683, 1.63923, 6.87664, 2.28849, 3.25364, 3.2015, 4.89637, 6.49477,
                0.746518, 0.661408, 0.695142, 0.682194, 0.648044, 0.696717, 0.75732, 0.684389, 0.680157, 0.761339,
                0.817109, 0.689733, 0.740219, 0.720841, 0.669733, 0.742577, 0.833275, 0.724127, 0.717793, 0.839288,
                0.922752, 0.732124, 0.807681, 0.778679, 0.702192, 0.81121, 0.946946, 0.783598, 0.774118, 0.955946,
                1.08086, 0.795566, 0.908644, 0.86524, 0.75077, 0.913924, 1.11707, 0.872601, 0.858414, 1.13054,
                1.31748, 0.890512, 1.05974, 0.994786, 0.823471, 1.06765, 1.37166, 1.0058, 0.984569, 1.39182,
                1.67159, 1.03261, 1.28588, 1.18866, 0.932274, 1.2977, 1.75269, 1.20515, 1.17337, 1.78286,
                2.20156, 1.24526, 1.6243, 1.47881, 1.09511, 1.642, 2.32293, 1.50349, 1.45593, 2.36808,
                2.99471, 1.56352, 2.13079, 1.91305, 1.3388, 2.15728, 3.17635, 1.94997, 1.87881, 3.24392,
                4.18172, 2.03983, 2.88879, 2.56292, 1.70351, 2.92843, 4.45356, 2.61818, 2.51167, 4.55468,
                5.95818, 2.75265, 4.02319, 3.53551, 2.24933, 4.08252, 6.36501, 3.61821, 3.45881, 6.51635,
                0.735673, 0.725707, 0.739564, 0.626204, 0.694121, 0.749506, 0.743778, 0.743851, 0.654985, 0.620846,
                0.800878, 0.785962, 0.806701, 0.637048, 0.738691, 0.821579, 0.813007, 0.813116, 0.680121, 0.629029,
                0.898462, 0.876139, 0.907177, 0.653277, 0.805394, 0.929444, 0.916614, 0.916777, 0.717739, 0.641276,
                1.0445, 1.0111, 1.05755, 0.677564, 0.905221, 1.09087, 1.07167, 1.07192, 0.774037, 0.659604,
                1.26307, 1.21307, 1.28259, 0.713913, 1.05462, 1.33246, 1.30373, 1.30409, 0.858293, 0.687033,
                1.59017, 1.51535, 1.61938, 0.768311, 1.27821, 1.69402, 1.65102, 1.65157, 0.984388, 0.728083,
                2.07971, 1.96773, 2.12343, 0.849723, 1.61283, 2.23513, 2.17077, 2.17159, 1.1731, 0.789519,
                2.81234, 2.64475, 2.87777, 0.971563, 2.11362, 3.04494, 2.94862, 2.94985, 1.45553, 0.881462,
                3.90879, 3.65797, 4.00671, 1.15391, 2.86309, 4.2569, 4.11275, 4.11458, 1.8782, 1.01906,
                5.54972, 5.17435, 5.69626, 1.4268, 3.98473, 6.07069, 5.85496, 5.8577, 2.51076, 1.22499,
                0.718451, 0.706082, 0.718631, 0.677138, 0.64293, 0.657632, 0.634079, 0.73646, 0.677257, 0.616843,
                0.775103, 0.756592, 0.775372, 0.713274, 0.662079, 0.684083, 0.648834, 0.802055, 0.713453, 0.623038,
                0.859888, 0.832185, 0.86029, 0.767355, 0.690738, 0.723668, 0.670915, 0.900223, 0.767624, 0.632309,
                0.986775, 0.945316, 0.987378, 0.848292, 0.733629, 0.782911, 0.703961, 1.04714, 0.848694, 0.646185,
                1.17667, 1.11463, 1.17758, 0.969422, 0.797818, 0.871573, 0.753417, 1.26702, 0.970024, 0.66695,
                1.46087, 1.36801, 1.46222, 1.1507, 0.893882, 1.00426, 0.827433, 1.59608, 1.1516, 0.698028,
                1.8862, 1.74723, 1.88822, 1.422, 1.03765, 1.20285, 0.938205, 2.08854, 1.42335, 0.744538,
                2.52274, 2.31475, 2.52576, 1.82803, 1.25281, 1.50004, 1.10398, 2.82557, 1.83005, 0.814145,
                3.47537, 3.16411, 3.4799, 2.43568, 1.57482, 1.94482, 1.35209, 3.92858, 2.4387, 0.918317,
                4.90107, 4.43524, 4.90784, 3.34509, 2.05673, 2.61047, 1.72339, 5.57933, 3.34961, 1.07422,
                0.711027, 0.716871, 0.659048, 0.682622, 0.646854, 0.623101, 0.686311, 0.626743, 0.755629, 0.771058,
                0.763992, 0.772739, 0.686201, 0.721482, 0.667952, 0.632404, 0.727003, 0.637854, 0.830743, 0.853834,
                0.843259, 0.85635, 0.726839, 0.779639, 0.699526, 0.646326, 0.787901, 0.654483, 0.943158, 0.977716,
                0.961889, 0.981481, 0.787656, 0.866675, 0.746781, 0.667161, 0.879041, 0.679369, 1.1114, 1.16312,
                1.13943, 1.16875, 0.878674, 0.996934, 0.817501, 0.698344, 1.01544, 0.716613, 1.36318, 1.44058,
                1.40513, 1.44901, 1.01489, 1.19188, 0.92334, 0.745011, 1.21957, 0.772353, 1.74, 1.85583,
                1.80278, 1.86845, 1.21875, 1.48362, 1.08174, 0.814853, 1.52507, 0.855772, 2.30393, 2.47729,
                2.39789, 2.49618, 1.52384, 1.92025, 1.31879, 0.919376, 1.98228, 0.980615, 3.14791, 3.40736,
                3.28853, 3.43562, 1.98044, 2.5737, 1.67356, 1.0758, 2.66654, 1.16745, 4.411, 4.79928,
                4.62145, 4.84158, 2.66378, 3.55164, 2.20451, 1.30991, 3.69058, 1.44707, 6.30131, 6.88241},
            std::vector<T>{
                0.611387, 0.770609, 0.76029, 0.631845, 0.770905, 0.649083, 0.674709, 0.673325, 0.718326, 0.760766,
                0.746518, 0.661408, 0.695142, 0.682194, 0.648044, 0.696717, 0.75732, 0.684389, 0.680157, 0.761339,
                0.735673, 0.725707, 0.739564, 0.626204, 0.694121, 0.749506, 0.743778, 0.743851, 0.654985, 0.620846,
                0.718451, 0.706082, 0.718631, 0.677138, 0.64293, 0.657632, 0.634079, 0.73646, 0.677257, 0.616843,
                0.711027, 0.716871, 0.659048, 0.682622, 0.646854, 0.623101, 0.686311, 0.626743, 0.755629, 0.771058}),
        GRUSequenceParams(
            5, 10, 10, 5,
            0.f, true, op::RecurrentSequenceDirection::BIDIRECTIONAL,
            ET,
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 7.95465,
                2.37637, 6.20771, 1.08107, 7.38138, 5.23577, 7.88133, 5.20653, 3.42101, 8.48523, 5.96192,
                1.63073, 5.25228, 7.68488, 2.7276, 5.1788, 3.07327, 5.57423, 2.87711, 1.44374, 5.66976,
                2.55051, 4.56682, 1.96629, 5.58829, 1.91922, 3.59846, 3.08583, 9.70901, 3.50487, 3.1026,
                1.82401, 6.1306, 4.76134, 4.31059, 8.31695, 3.60784, 7.45652, 6.51653, 4.84219, 7.76686,
                4.85031, 4.85544, 4.25714, 2.38005, 9.43471, 9.24769, 8.03763, 6.54696, 1.32399, 6.88891,
                2.16793, 3.64924, 4.24733, 3.47181, 1.66572, 2.36923, 2.45457, 9.44841, 4.34021, 1.45016,
                7.6686, 3.68812, 2.83922, 9.83581, 9.03719, 7.83414, 6.86009, 1.35715, 8.32489, 7.86316,
                5.09754, 5.78644, 1.98402, 2.31429, 5.5791, 2.94085, 9.24799, 5.15937, 2.19041, 7.87817,
                2.9146, 1.66833, 1.85877, 2.45985, 4.20817, 1.85777, 2.28937, 9.37912, 6.18926, 8.55681,
                6.60963, 3.92066, 7.5521, 5.70463, 7.6313, 2.48866, 7.18352, 4.8413, 7.55702, 7.80702,
                4.5785, 9.3268, 2.83159, 1.07202, 9.33716, 3.6506, 2.50256, 1.21691, 5.06801, 8.27505,
                4.31539, 6.48286, 1.31363, 4.1912, 1.70668, 7.23867, 1.11441, 5.13591, 9.65186, 4.00767,
                5.24875, 1.94852, 5.52768, 8.97121, 5.8094, 3.53329, 4.19126, 9.06652, 3.1734, 1.21496,
                9.69154, 4.86971, 4.1166, 6.19361, 2.13874, 9.55039, 3.8225, 9.57548, 2.96554, 3.2383,
                8.77422, 3.11741, 8.3359, 5.89508, 2.72134, 6.29956, 1.43323, 1.14286, 1.4474, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 10},
            std::vector<int64_t>{5, 5, 5, 5, 5},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 7.95465,
                2.37637, 6.20771, 1.08107, 7.38138, 5.23577, 7.88133, 5.20653, 3.42101, 8.48523, 5.96192,
                1.63073, 5.25228, 7.68488, 2.7276, 5.1788, 3.07327, 5.57423, 2.87711, 1.44374, 5.66976,
                2.55051, 4.56682, 1.96629, 5.58829, 1.91922, 3.59846, 3.08583, 9.70901, 3.50487, 3.1026,
                1.82401, 6.1306, 4.76134, 4.31059, 8.31695, 3.60784, 7.45652, 6.51653, 4.84219, 7.76686,
                4.85031, 4.85544, 4.25714, 2.38005, 9.43471, 9.24769, 8.03763, 6.54696, 1.32399, 6.88891,
                2.16793, 3.64924, 4.24733, 3.47181, 1.66572, 2.36923, 2.45457, 9.44841, 4.34021, 1.45016,
                7.6686, 3.68812, 2.83922, 9.83581, 9.03719, 7.83414, 6.86009, 1.35715, 8.32489, 7.86316,
                5.09754, 5.78644, 1.98402, 2.31429, 5.5791, 2.94085, 9.24799, 5.15937, 2.19041, 7.87817,
                2.9146, 1.66833, 1.85877, 2.45985, 4.20817, 1.85777, 2.28937, 9.37912, 6.18926, 8.55681,
                6.60963, 3.92066, 7.5521, 5.70463, 7.6313, 2.48866, 7.18352, 4.8413, 7.55702, 7.80702,
                4.5785, 9.3268, 2.83159, 1.07202, 9.33716, 3.6506, 2.50256, 1.21691, 5.06801, 8.27505,
                4.31539, 6.48286, 1.31363, 4.1912, 1.70668, 7.23867, 1.11441, 5.13591, 9.65186, 4.00767,
                5.24875, 1.94852, 5.52768, 8.97121, 5.8094, 3.53329, 4.19126, 9.06652, 3.1734, 1.21496,
                9.69154, 4.86971, 4.1166, 6.19361, 2.13874, 9.55039, 3.8225, 9.57548, 2.96554, 3.2383,
                8.77422, 3.11741, 8.3359, 5.89508, 2.72134, 6.29956, 1.43323, 1.14286, 1.4474, 4.59474,
                6.19214, 8.80766, 8.07546, 3.29232, 1.74029, 2.4198, 2.88544, 4.75644, 4.12921, 7.29896,
                7.27759, 1.67252, 1.32823, 8.1046, 9.10476, 1.04197, 3.37783, 5.2064, 4.23835, 3.16196,
                1.20852, 5.78501, 2.17175, 6.05313, 2.51048, 4.78967, 7.16219, 3.4651, 1.09, 2.9788,
                1.28761, 9.41385, 8.03883, 5.65835, 1.14816, 3.6892, 5.86648, 8.73895, 2.66603, 1.75192,
                1.39845, 4.99427, 1.17387, 1.60329, 8.30594, 6.72662, 7.95565, 7.35114, 3.1439, 1.39976,
                3.53095, 8.78581, 1.65811, 6.94299, 2.68641, 5.70058, 9.13491, 5.27637, 8.6232, 8.54902,
                2.25352, 5.86274, 5.20377, 2.96815, 4.96745, 5.3225, 3.99956, 1.08021, 5.54918, 7.05833,
                1.49913, 2.41822, 6.44593, 3.87301, 9.01465, 8.11336, 2.95749, 2.80188, 7.12396, 2.40595,
                5.59325, 9.89258, 2.30223, 1.4347, 9.09158, 7.43797, 3.79295, 4.53646, 1.72705, 4.16909,
                1.00912, 6.62167, 2.80244, 6.626, 3.89307, 1.42586, 7.51028, 7.83327, 4.65674, 7.33902,
                6.26823, 9.72608, 3.73491, 3.8238, 3.03815, 7.05101, 8.0103, 5.61396, 6.53738, 1.41095,
                5.0149, 9.71211, 4.23604, 5.98629, 4.70219, 9.69442, 2.82752, 9.93544, 6.9328, 8.2817,
                5.12336, 8.98577, 5.80541, 6.19552, 9.25748, 3.82732, 7.53525, 8.24712, 5.32057, 5.38817,
                8.57269, 5.99975, 3.42893, 5.38068, 3.48261, 3.02851, 6.82079, 9.2902, 2.80427, 8.91868,
                5.19227, 7.52482, 3.72584, 5.40107, 2.83307, 1.79755, 2.49121, 5.52697, 8.08823, 9.13242,
                2.97572, 7.64318, 3.32023, 6.07788, 2.19187, 4.34879, 1.7457, 5.55154, 7.24966, 5.1128,
                4.25147, 8.34407, 1.4123, 4.49045, 5.12671, 7.62159, 9.18673, 3.49665, 8.35992, 6.90684,
                1.10152, 7.61818, 6.43145, 7.12017, 6.25564, 6.16169, 4.24916, 9.6283, 9.88249, 4.48422,
                8.52562, 9.83928, 6.26818, 7.03839, 1.77631, 9.92305, 8.0155, 9.94928, 6.88321, 1.33685,
                7.4718, 7.19305, 6.47932, 1.9559, 3.52616, 7.98593, 9.0115, 5.59539, 7.44137, 1.70001,
                6.53774, 8.54023, 7.26405, 5.99553, 8.75071, 7.70789, 3.38094, 9.99792, 6.16359, 6.75153,
                5.4073, 9.00437, 8.87059, 8.63011, 6.82951, 6.27021, 3.53425, 9.92489, 8.19695, 5.51473,
                7.95084, 2.11852, 9.28916, 1.40353, 3.05744, 8.58238, 3.75014, 5.35889, 6.85048, 2.29549,
                3.75218, 8.98228, 8.98158, 5.63695, 3.40379, 8.92309, 5.48185, 4.00095, 9.05227, 2.84035,
                8.37644, 8.54954, 5.70516, 2.45744, 9.54079, 1.53504, 8.9785, 6.1691, 4.40962, 6.76815,
                2.15026, 4.66033, 4.20769, 2.3604, 8.70826, 1.91807, 9.57809, 2.50132, 4.95464, 2.44375,
                8.91345, 5.30605, 4.59238, 3.47923, 1.46477, 2.24843, 8.00775, 2.60248, 6.25095, 9.35311,
                1.96897, 5.0738, 2.40393, 3.7379, 7.87712, 6.6423, 7.39986, 2.74115, 1.61588, 8.9169,
                3.87864, 6.44032, 9.59689, 4.74213, 2.76302, 2.49426, 6.62326, 1.50664, 3.95698, 9.82371,
                7.94438, 2.88399, 6.26754, 4.95809, 6.52199, 3.87974, 2.0071, 3.81713, 9.94544, 2.49197,
                5.16628, 6.71965, 1.33379, 3.27947, 8.83191, 2.25668, 1.16264, 7.20743, 8.8397, 4.19957,
                4.57119, 8.76621, 9.39031, 5.05683, 9.77492, 7.28143, 7.78249, 6.19997, 2.21146, 8.26929,
                1.94202, 4.0196, 6.07928, 6.6033, 5.12291, 8.3157, 3.23435, 4.80489, 8.2505, 5.93311,
                3.22774, 1.09695, 3.22228, 3.24883, 6.54187, 3.3574, 5.71848, 4.4223, 1.32481, 8.59246,
                5.0497, 9.78833, 1.51703, 2.4678, 2.64335, 5.87163, 3.45429, 5.43941, 7.60549, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 7.95465,
                2.37637, 6.20771, 1.08107, 7.38138, 5.23577, 7.88133, 5.20653, 3.42101, 8.48523, 5.96192,
                1.63073, 5.25228, 7.68488, 2.7276, 5.1788, 3.07327, 5.57423, 2.87711, 1.44374, 5.66976,
                2.55051, 4.56682, 1.96629, 5.58829, 1.91922, 3.59846, 3.08583, 9.70901, 3.50487, 3.1026,
                1.82401, 6.1306, 4.76134, 4.31059, 8.31695, 3.60784, 7.45652, 6.51653, 4.84219, 7.76686,
                4.85031, 4.85544, 4.25714, 2.38005, 9.43471, 9.24769, 8.03763, 6.54696, 1.32399, 6.88891,
                2.16793, 3.64924, 4.24733, 3.47181, 1.66572, 2.36923, 2.45457, 9.44841, 4.34021, 1.45016,
                7.6686, 3.68812, 2.83922, 9.83581, 9.03719, 7.83414, 6.86009, 1.35715, 8.32489, 7.86316,
                5.09754, 5.78644, 1.98402, 2.31429, 5.5791, 2.94085, 9.24799, 5.15937, 2.19041, 7.87817,
                2.9146, 1.66833, 1.85877, 2.45985, 4.20817, 1.85777, 2.28937, 9.37912, 6.18926, 8.55681,
                6.60963, 3.92066, 7.5521, 5.70463, 7.6313, 2.48866, 7.18352, 4.8413, 7.55702, 7.80702,
                4.5785, 9.3268, 2.83159, 1.07202, 9.33716, 3.6506, 2.50256, 1.21691, 5.06801, 8.27505,
                4.31539, 6.48286, 1.31363, 4.1912, 1.70668, 7.23867, 1.11441, 5.13591, 9.65186, 4.00767,
                5.24875, 1.94852, 5.52768, 8.97121, 5.8094, 3.53329, 4.19126, 9.06652, 3.1734, 1.21496,
                9.69154, 4.86971, 4.1166, 6.19361, 2.13874, 9.55039, 3.8225, 9.57548, 2.96554, 3.2383,
                8.77422, 3.11741, 8.3359, 5.89508, 2.72134, 6.29956, 1.43323, 1.14286, 1.4474, 4.59474,
                6.19214, 8.80766, 8.07546, 3.29232, 1.74029, 2.4198, 2.88544, 4.75644, 4.12921, 7.29896,
                7.27759, 1.67252, 1.32823, 8.1046, 9.10476, 1.04197, 3.37783, 5.2064, 4.23835, 3.16196,
                1.20852, 5.78501, 2.17175, 6.05313, 2.51048, 4.78967, 7.16219, 3.4651, 1.09, 2.9788,
                1.28761, 9.41385, 8.03883, 5.65835, 1.14816, 3.6892, 5.86648, 8.73895, 2.66603, 1.75192,
                1.39845, 4.99427, 1.17387, 1.60329, 8.30594, 6.72662, 7.95565, 7.35114, 3.1439, 1.39976,
                3.53095, 8.78581, 1.65811, 6.94299, 2.68641, 5.70058, 9.13491, 5.27637, 8.6232, 8.54902,
                2.25352, 5.86274, 5.20377, 2.96815, 4.96745, 5.3225, 3.99956, 1.08021, 5.54918, 7.05833,
                1.49913, 2.41822, 6.44593, 3.87301, 9.01465, 8.11336, 2.95749, 2.80188, 7.12396, 2.40595,
                5.59325, 9.89258, 2.30223, 1.4347, 9.09158, 7.43797, 3.79295, 4.53646, 1.72705, 4.16909,
                1.00912, 6.62167, 2.80244, 6.626, 3.89307, 1.42586, 7.51028, 7.83327, 4.65674, 7.33902,
                6.26823, 9.72608, 3.73491, 3.8238, 3.03815, 7.05101, 8.0103, 5.61396, 6.53738, 1.41095,
                5.0149, 9.71211, 4.23604, 5.98629, 4.70219, 9.69442, 2.82752, 9.93544, 6.9328, 8.2817,
                5.12336, 8.98577, 5.80541, 6.19552, 9.25748, 3.82732, 7.53525, 8.24712, 5.32057, 5.38817,
                8.57269, 5.99975, 3.42893, 5.38068, 3.48261, 3.02851, 6.82079, 9.2902, 2.80427, 8.91868,
                5.19227, 7.52482, 3.72584, 5.40107, 2.83307, 1.79755, 2.49121, 5.52697, 8.08823, 9.13242,
                2.97572, 7.64318, 3.32023, 6.07788, 2.19187, 4.34879, 1.7457, 5.55154, 7.24966, 5.1128,
                4.25147, 8.34407, 1.4123, 4.49045, 5.12671, 7.62159, 9.18673, 3.49665, 8.35992, 6.90684,
                1.10152, 7.61818, 6.43145, 7.12017, 6.25564, 6.16169, 4.24916, 9.6283, 9.88249, 4.48422,
                8.52562, 9.83928, 6.26818, 7.03839, 1.77631, 9.92305, 8.0155, 9.94928, 6.88321, 1.33685,
                7.4718, 7.19305, 6.47932, 1.9559, 3.52616, 7.98593, 9.0115, 5.59539, 7.44137, 1.70001,
                6.53774, 8.54023, 7.26405, 5.99553, 8.75071, 7.70789, 3.38094, 9.99792, 6.16359, 6.75153,
                5.4073, 9.00437, 8.87059, 8.63011, 6.82951, 6.27021, 3.53425, 9.92489, 8.19695, 5.51473,
                7.95084, 2.11852, 9.28916, 1.40353, 3.05744, 8.58238, 3.75014, 5.35889, 6.85048, 2.29549,
                3.75218, 8.98228, 8.98158, 5.63695, 3.40379, 8.92309, 5.48185, 4.00095, 9.05227, 2.84035,
                8.37644, 8.54954, 5.70516, 2.45744, 9.54079, 1.53504, 8.9785, 6.1691, 4.40962, 6.76815,
                2.15026, 4.66033, 4.20769, 2.3604, 8.70826, 1.91807, 9.57809, 2.50132, 4.95464, 2.44375,
                8.91345, 5.30605, 4.59238, 3.47923, 1.46477, 2.24843, 8.00775, 2.60248, 6.25095, 9.35311,
                1.96897, 5.0738, 2.40393, 3.7379, 7.87712, 6.6423, 7.39986, 2.74115, 1.61588, 8.9169,
                3.87864, 6.44032, 9.59689, 4.74213, 2.76302, 2.49426, 6.62326, 1.50664, 3.95698, 9.82371,
                7.94438, 2.88399, 6.26754, 4.95809, 6.52199, 3.87974, 2.0071, 3.81713, 9.94544, 2.49197,
                5.16628, 6.71965, 1.33379, 3.27947, 8.83191, 2.25668, 1.16264, 7.20743, 8.8397, 4.19957,
                4.57119, 8.76621, 9.39031, 5.05683, 9.77492, 7.28143, 7.78249, 6.19997, 2.21146, 8.26929,
                1.94202, 4.0196, 6.07928, 6.6033, 5.12291, 8.3157, 3.23435, 4.80489, 8.2505, 5.93311,
                3.22774, 1.09695, 3.22228, 3.24883, 6.54187, 3.3574, 5.71848, 4.4223, 1.32481, 8.59246,
                5.0497, 9.78833, 1.51703, 2.4678, 2.64335, 5.87163, 3.45429, 5.43941, 7.60549, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 10,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 10,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 10,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 10,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 10}),
    };
    return params;
}

template <element::Type_t ET>
std::vector<GRUSequenceParams> generateParamsBF16() {
    using T = typename element_type_traits<ET>::value_type;

    std::vector<GRUSequenceParams> params {
        GRUSequenceParams(
            5, 10, 10, 10,
            0.7f, false, op::RecurrentSequenceDirection::FORWARD,
            ET,
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 2.15625,
                5.71875, 1.17188, 1.75, 7.125, 9.25, 2.90625, 9.1875, 3.375, 3.6875, 5.4375,
                6.25, 1.47656, 6.0625, 6.15625, 6.5, 2.3125, 9.625, 6.3125, 3.34375, 7.3125,
                3.07812, 1.92188, 5.8125, 4.71875, 9.5, 7.25, 5.4375, 4.71875, 5.875, 1.45312,
                7.875, 5.8125, 1.40625, 6.96875, 2.25, 5.625, 8.125, 9.5, 1.26562, 6.25,
                8.9375, 9.125, 5.875, 2.23438, 5.03125, 2.25, 9, 8.25, 4.375, 4.5625,
                5.84375, 2.48438, 6.875, 9.375, 4.25, 4.125, 6.125, 7.75, 6.75, 7.53125,
                2.125, 8.9375, 7.1875, 6.625, 6.8125, 7.75, 4.1875, 4.125, 7.875, 3.42188,
                4.1875, 9.0625, 7.75, 4.84375, 8.875, 9.625, 1.10156, 6.96875, 5.46875, 6.59375,
                1.66406, 2.03125, 8.0625, 9.5, 1.57812, 5.0625, 4.1875, 6.1875, 9.5, 4.6875,
                4.40625, 3.125, 7.875, 9.125, 7.9375, 6.15625, 3.71875, 1.02344, 7.9375, 6.5625,
                2.375, 3.9375, 6.1875, 5.75, 1.07812, 9, 7.375, 4.1875, 5.25, 9.125,
                7.875, 6.625, 5.1875, 1.14062, 3.40625, 9.375, 8.5, 7.1875, 5.9375, 10,
                1.625, 2.54688, 5.25, 2.21875, 7.6875, 9.375, 2.71875, 7.25, 5.1875, 1.59375,
                3.0625, 7.8125, 5.5625, 7.78125, 2.875, 9.25, 1.4375, 7.375, 5.65625, 2.125,
                2.54688, 1.17188, 4.5625, 1.23438, 1.96875, 1.25, 5.5625, 3.21875, 1.92188, 8.75,
                3.59375, 5.84375, 3.07812, 5.96875, 9.6875, 8.5625, 3.5, 2.125, 3.09375, 3.5,
                1.82031, 6.25, 6.125, 9.75, 4.75, 6.0625, 4.3125, 1.16406, 8.3125, 8.1875,
                3.59375, 3.09375, 7.4375, 8.25, 6.5, 4.5, 4.8125, 8.75, 7.75, 7.71875,
                4.84375, 6, 4.84375, 2.21875, 4.25, 1.53906, 2.375, 2.09375, 9.375, 1.39844,
                9.25, 1.96875, 8, 3.03125, 6.5625, 7.40625, 1.32031, 6.03125, 6.875, 1.10938,
                2.15625, 1.64062, 3.65625, 9.6875, 4.25, 6.125, 3.46875, 2.82812, 1.66406, 3.26562,
                2.375, 7.6875, 2.45312, 2.75, 9.4375, 6.21875, 4.3125, 9.75, 1.45312, 8.625,
                7.65625, 3.15625, 3.6875, 5.4375, 2.84375, 6.5625, 9.8125, 8.4375, 9, 2.40625,
                7.8125, 1.16406, 6.875, 1.625, 1.35938, 5.375, 8.3125, 6.4375, 7.875, 6.125,
                5.09375, 3.84375, 5.78125, 9.875, 1.98438, 6.1875, 2.3125, 4.40625, 5.5625, 5.9375,
                2.9375, 7.6875, 9.25, 7, 5.15625, 3.375, 2.1875, 1.59375, 7.875, 4.3125,
                2.90625, 6.65625, 1.67188, 2.89062, 1.85938, 7.75, 2.45312, 1.59375, 4.1875, 3.34375,
                1.85938, 8.25, 2.28125, 2.73438, 9.375, 6.75, 6.1875, 5.71875, 8.5, 9.3125,
                6.625, 3.375, 3.90625, 1.59375, 7.5625, 7.625, 5.6875, 7.9375, 7.625, 9.125,
                2.48438, 9.375, 7.1875, 1.125, 4.8125, 3.09375, 7.5625, 6.5625, 7.8125, 9.5,
                4.5625, 9.5, 9.3125, 6, 2.82812, 9.25, 1.07031, 6.75, 9.3125, 4.5,
                3.65625, 5.375, 2.5, 6.4375, 1.21875, 5.9375, 5.0625, 9.3125, 8.25, 9.25,
                4.3125, 4.5625, 6.46875, 9.625, 1.3125, 2.5625, 4.1875, 2.125, 1.70312, 2.21875,
                7.25, 5.5625, 1.10938, 1.1875, 5.125, 9.5, 9.625, 8.4375, 4, 1.13281,
                5.25, 2.57812, 1.94531, 3.98438, 5.5, 2.17188, 9, 8.25, 5.8125, 4.09375,
                3.53125, 9.4375, 4.1875, 6.25, 9.0625, 8.875, 3.17188, 8.625, 1.21875, 9.125,
                9.6875, 5.125, 4.875, 5.90625, 4.125, 8.125, 6.1875, 3.5625, 2.125, 5.40625,
                9.5, 6.375, 3.8125, 1.14062, 9.5625, 6.3125, 2.96875, 4.875, 3.23438, 8.25,
                8.75, 3.84375, 3.125, 9, 8.3125, 6.1875, 5.875, 2.65625, 2.71875, 8.0625,
                6.3125, 6.5, 1.42969, 1.48438, 1.14062, 4.78125, 1.44531, 7.125, 4.59375, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 10},
            std::vector<int64_t>{10, 10, 10, 10, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 2.15625,
                5.71875, 1.17188, 1.75, 7.125, 9.25, 2.90625, 9.1875, 3.375, 3.6875, 5.4375,
                6.25, 1.47656, 6.0625, 6.15625, 6.5, 2.3125, 9.625, 6.3125, 3.34375, 7.3125,
                3.07812, 1.92188, 5.8125, 4.71875, 9.5, 7.25, 5.4375, 4.71875, 5.875, 1.45312,
                7.875, 5.8125, 1.40625, 6.96875, 2.25, 5.625, 8.125, 9.5, 1.26562, 6.25,
                8.9375, 9.125, 5.875, 2.23438, 5.03125, 2.25, 9, 8.25, 4.375, 4.5625,
                5.84375, 2.48438, 6.875, 9.375, 4.25, 4.125, 6.125, 7.75, 6.75, 7.53125,
                2.125, 8.9375, 7.1875, 6.625, 6.8125, 7.75, 4.1875, 4.125, 7.875, 3.42188,
                4.1875, 9.0625, 7.75, 4.84375, 8.875, 9.625, 1.10156, 6.96875, 5.46875, 6.59375,
                1.66406, 2.03125, 8.0625, 9.5, 1.57812, 5.0625, 4.1875, 6.1875, 9.5, 4.6875,
                4.40625, 3.125, 7.875, 9.125, 7.9375, 6.15625, 3.71875, 1.02344, 7.9375, 6.5625,
                2.375, 3.9375, 6.1875, 5.75, 1.07812, 9, 7.375, 4.1875, 5.25, 9.125,
                7.875, 6.625, 5.1875, 1.14062, 3.40625, 9.375, 8.5, 7.1875, 5.9375, 10,
                1.625, 2.54688, 5.25, 2.21875, 7.6875, 9.375, 2.71875, 7.25, 5.1875, 1.59375,
                3.0625, 7.8125, 5.5625, 7.78125, 2.875, 9.25, 1.4375, 7.375, 5.65625, 2.125,
                2.54688, 1.17188, 4.5625, 1.23438, 1.96875, 1.25, 5.5625, 3.21875, 1.92188, 8.75,
                3.59375, 5.84375, 3.07812, 5.96875, 9.6875, 8.5625, 3.5, 2.125, 3.09375, 3.5,
                1.82031, 6.25, 6.125, 9.75, 4.75, 6.0625, 4.3125, 1.16406, 8.3125, 8.1875,
                3.59375, 3.09375, 7.4375, 8.25, 6.5, 4.5, 4.8125, 8.75, 7.75, 7.71875,
                4.84375, 6, 4.84375, 2.21875, 4.25, 1.53906, 2.375, 2.09375, 9.375, 1.39844,
                9.25, 1.96875, 8, 3.03125, 6.5625, 7.40625, 1.32031, 6.03125, 6.875, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 2.15625,
                5.71875, 1.17188, 1.75, 7.125, 9.25, 2.90625, 9.1875, 3.375, 3.6875, 5.4375,
                6.25, 1.47656, 6.0625, 6.15625, 6.5, 2.3125, 9.625, 6.3125, 3.34375, 7.3125,
                3.07812, 1.92188, 5.8125, 4.71875, 9.5, 7.25, 5.4375, 4.71875, 5.875, 1.45312,
                7.875, 5.8125, 1.40625, 6.96875, 2.25, 5.625, 8.125, 9.5, 1.26562, 6.25,
                8.9375, 9.125, 5.875, 2.23438, 5.03125, 2.25, 9, 8.25, 4.375, 4.5625,
                5.84375, 2.48438, 6.875, 9.375, 4.25, 4.125, 6.125, 7.75, 6.75, 7.53125,
                2.125, 8.9375, 7.1875, 6.625, 6.8125, 7.75, 4.1875, 4.125, 7.875, 3.42188,
                4.1875, 9.0625, 7.75, 4.84375, 8.875, 9.625, 1.10156, 6.96875, 5.46875, 6.59375,
                1.66406, 2.03125, 8.0625, 9.5, 1.57812, 5.0625, 4.1875, 6.1875, 9.5, 4.6875,
                4.40625, 3.125, 7.875, 9.125, 7.9375, 6.15625, 3.71875, 1.02344, 7.9375, 6.5625,
                2.375, 3.9375, 6.1875, 5.75, 1.07812, 9, 7.375, 4.1875, 5.25, 9.125,
                7.875, 6.625, 5.1875, 1.14062, 3.40625, 9.375, 8.5, 7.1875, 5.9375, 10,
                1.625, 2.54688, 5.25, 2.21875, 7.6875, 9.375, 2.71875, 7.25, 5.1875, 1.59375,
                3.0625, 7.8125, 5.5625, 7.78125, 2.875, 9.25, 1.4375, 7.375, 5.65625, 2.125,
                2.54688, 1.17188, 4.5625, 1.23438, 1.96875, 1.25, 5.5625, 3.21875, 1.92188, 8.75,
                3.59375, 5.84375, 3.07812, 5.96875, 9.6875, 8.5625, 3.5, 2.125, 3.09375, 3.5,
                1.82031, 6.25, 6.125, 9.75, 4.75, 6.0625, 4.3125, 1.16406, 8.3125, 8.1875,
                3.59375, 3.09375, 7.4375, 8.25, 6.5, 4.5, 4.8125, 8.75, 7.75, 7.71875,
                4.84375, 6, 4.84375, 2.21875, 4.25, 1.53906, 2.375, 2.09375, 9.375, 1.39844,
                9.25, 1.96875, 8, 3.03125, 6.5625, 7.40625, 1.32031, 6.03125, 6.875, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 10},
            std::vector<T>{
                0.867188, 3.375, 6.875, 5.1875, 6.4375, 0.867188, 1.64062, 2.65625, 6.875, 1.75,
                0.777344, 2.4375, 4.75, 3.65625, 4.5, 0.777344, 1.29688, 1.96875, 4.75, 1.375,
                0.714844, 1.82812, 3.375, 2.625, 3.1875, 0.714844, 1.0625, 1.51562, 3.375, 1.11719,
                0.675781, 1.42188, 2.4375, 1.95312, 2.3125, 0.675781, 0.910156, 1.21875, 2.4375, 0.945312,
                0.648438, 1.14844, 1.82812, 1.5, 1.75, 0.648438, 0.808594, 1.01562, 1.82812, 0.832031,
                0.632812, 0.964844, 1.42188, 1.20312, 1.375, 0.632812, 0.738281, 0.878906, 1.42188, 0.753906,
                0.621094, 0.839844, 1.14844, 1, 1.11719, 0.621094, 0.691406, 0.785156, 1.14844, 0.699219,
                0.613281, 0.761719, 0.964844, 0.867188, 0.945312, 0.613281, 0.660156, 0.722656, 0.964844, 0.664062,
                0.609375, 0.707031, 0.839844, 0.777344, 0.832031, 0.609375, 0.640625, 0.679688, 0.839844, 0.640625,
                0.605469, 0.671875, 0.761719, 0.714844, 0.753906, 0.605469, 0.625, 0.652344, 0.761719, 0.625,
                2.28125, 1.42188, 3.25, 1.98438, 3.1875, 2.9375, 4.875, 3.25, 6.4375, 4.09375,
                1.71875, 1.14844, 2.34375, 1.53125, 2.3125, 2.15625, 3.4375, 2.34375, 4.5, 2.9375,
                1.34375, 0.964844, 1.76562, 1.21875, 1.75, 1.64062, 2.46875, 1.76562, 3.1875, 2.15625,
                1.09375, 0.839844, 1.375, 1.01562, 1.375, 1.29688, 1.84375, 1.375, 2.3125, 1.64062,
                0.929688, 0.761719, 1.11719, 0.878906, 1.11719, 1.0625, 1.4375, 1.11719, 1.75, 1.29688,
                0.816406, 0.707031, 0.945312, 0.785156, 0.945312, 0.910156, 1.15625, 0.945312, 1.375, 1.0625,
                0.746094, 0.671875, 0.832031, 0.722656, 0.832031, 0.808594, 0.972656, 0.832031, 1.11719, 0.910156,
                0.695312, 0.648438, 0.753906, 0.679688, 0.753906, 0.738281, 0.847656, 0.753906, 0.945312, 0.808594,
                0.664062, 0.632812, 0.699219, 0.652344, 0.699219, 0.691406, 0.761719, 0.699219, 0.832031, 0.738281,
                0.640625, 0.621094, 0.664062, 0.632812, 0.664062, 0.660156, 0.707031, 0.664062, 0.753906, 0.691406,
                5.9375, 3.375, 2.71875, 4.9375, 4, 2.09375, 3.53125, 6.125, 2.21875, 1.03125,
                4.15625, 2.4375, 2, 3.5, 2.875, 1.59375, 2.53125, 4.25, 1.6875, 0.886719,
                2.96875, 1.82812, 1.53125, 2.53125, 2.125, 1.26562, 1.89062, 3.03125, 1.32812, 0.792969,
                2.1875, 1.42188, 1.21875, 1.89062, 1.625, 1.04688, 1.46875, 2.21875, 1.08594, 0.730469,
                1.65625, 1.14844, 1.01562, 1.46875, 1.28125, 0.898438, 1.17969, 1.6875, 0.925781, 0.6875,
                1.3125, 0.964844, 0.878906, 1.17969, 1.05469, 0.800781, 0.988281, 1.32812, 0.816406, 0.65625,
                1.07812, 0.839844, 0.785156, 0.988281, 0.902344, 0.730469, 0.855469, 1.08594, 0.746094, 0.636719,
                0.917969, 0.761719, 0.722656, 0.855469, 0.800781, 0.6875, 0.769531, 0.925781, 0.695312, 0.625,
                0.808594, 0.707031, 0.679688, 0.769531, 0.730469, 0.65625, 0.714844, 0.816406, 0.664062, 0.617188,
                0.738281, 0.671875, 0.652344, 0.714844, 0.6875, 0.636719, 0.675781, 0.746094, 0.640625, 0.609375,
                4.0625, 4.875, 6.375, 3.375, 3.625, 4.1875, 3.4375, 1.70312, 6.5, 2.0625,
                2.90625, 3.4375, 4.4375, 2.4375, 2.59375, 3, 2.46875, 1.34375, 4.5, 1.57812,
                2.125, 2.46875, 3.15625, 1.82812, 1.9375, 2.1875, 1.84375, 1.09375, 3.1875, 1.25,
                1.625, 1.84375, 2.28125, 1.42188, 1.5, 1.65625, 1.4375, 0.929688, 2.3125, 1.03125,
                1.28125, 1.4375, 1.71875, 1.14844, 1.20312, 1.3125, 1.15625, 0.816406, 1.75, 0.886719,
                1.05469, 1.15625, 1.34375, 0.964844, 1, 1.07812, 0.972656, 0.746094, 1.375, 0.792969,
                0.902344, 0.972656, 1.09375, 0.839844, 0.867188, 0.917969, 0.847656, 0.695312, 1.11719, 0.730469,
                0.800781, 0.847656, 0.929688, 0.761719, 0.777344, 0.808594, 0.761719, 0.664062, 0.945312, 0.6875,
                0.730469, 0.761719, 0.816406, 0.707031, 0.714844, 0.738281, 0.707031, 0.640625, 0.832031, 0.65625,
                0.6875, 0.707031, 0.746094, 0.671875, 0.675781, 0.691406, 0.671875, 0.625, 0.753906, 0.636719,
                5.53125, 5.65625, 5.125, 6.65625, 5.6875, 2.71875, 1.42188, 5, 3.96875, 6.875,
                3.875, 3.96875, 3.625, 4.625, 4, 2, 1.14844, 3.53125, 2.84375, 4.75,
                2.78125, 2.84375, 2.59375, 3.28125, 2.875, 1.53125, 0.964844, 2.53125, 2.09375, 3.375,
                2.0625, 2.09375, 1.9375, 2.375, 2.125, 1.21875, 0.839844, 1.89062, 1.59375, 2.4375,
                1.57812, 1.59375, 1.5, 1.78125, 1.625, 1.01562, 0.761719, 1.46875, 1.26562, 1.82812,
                1.25, 1.26562, 1.20312, 1.39062, 1.28125, 0.878906, 0.707031, 1.17969, 1.04688, 1.42188,
                1.03125, 1.04688, 1, 1.125, 1.05469, 0.785156, 0.671875, 0.988281, 0.898438, 1.14844,
                0.886719, 0.898438, 0.867188, 0.949219, 0.902344, 0.722656, 0.648438, 0.855469, 0.800781, 0.964844,
                0.792969, 0.800781, 0.777344, 0.832031, 0.800781, 0.679688, 0.632812, 0.769531, 0.730469, 0.839844,
                0.730469, 0.730469, 0.714844, 0.753906, 0.730469, 0.652344, 0.621094, 0.714844, 0.6875, 0.761719},
            std::vector<T>{
                0.605469, 0.671875, 0.761719, 0.714844, 0.753906, 0.605469, 0.625, 0.652344, 0.761719, 0.625,
                0.640625, 0.621094, 0.664062, 0.632812, 0.664062, 0.660156, 0.707031, 0.664062, 0.753906, 0.691406,
                0.738281, 0.671875, 0.652344, 0.714844, 0.6875, 0.636719, 0.675781, 0.746094, 0.640625, 0.609375,
                0.6875, 0.707031, 0.746094, 0.671875, 0.675781, 0.691406, 0.671875, 0.625, 0.753906, 0.636719,
                0.730469, 0.730469, 0.714844, 0.753906, 0.730469, 0.652344, 0.621094, 0.714844, 0.6875, 0.761719}),
        GRUSequenceParams(
            5, 10, 10, 10,
            0.7f, false, op::RecurrentSequenceDirection::REVERSE,
            ET,
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 2.15625,
                5.71875, 1.17188, 1.75, 7.125, 9.25, 2.90625, 9.1875, 3.375, 3.6875, 5.4375,
                6.25, 1.47656, 6.0625, 6.15625, 6.5, 2.3125, 9.625, 6.3125, 3.34375, 7.3125,
                3.07812, 1.92188, 5.8125, 4.71875, 9.5, 7.25, 5.4375, 4.71875, 5.875, 1.45312,
                7.875, 5.8125, 1.40625, 6.96875, 2.25, 5.625, 8.125, 9.5, 1.26562, 6.25,
                8.9375, 9.125, 5.875, 2.23438, 5.03125, 2.25, 9, 8.25, 4.375, 4.5625,
                5.84375, 2.48438, 6.875, 9.375, 4.25, 4.125, 6.125, 7.75, 6.75, 7.53125,
                2.125, 8.9375, 7.1875, 6.625, 6.8125, 7.75, 4.1875, 4.125, 7.875, 3.42188,
                4.1875, 9.0625, 7.75, 4.84375, 8.875, 9.625, 1.10156, 6.96875, 5.46875, 6.59375,
                1.66406, 2.03125, 8.0625, 9.5, 1.57812, 5.0625, 4.1875, 6.1875, 9.5, 4.6875,
                4.40625, 3.125, 7.875, 9.125, 7.9375, 6.15625, 3.71875, 1.02344, 7.9375, 6.5625,
                2.375, 3.9375, 6.1875, 5.75, 1.07812, 9, 7.375, 4.1875, 5.25, 9.125,
                7.875, 6.625, 5.1875, 1.14062, 3.40625, 9.375, 8.5, 7.1875, 5.9375, 10,
                1.625, 2.54688, 5.25, 2.21875, 7.6875, 9.375, 2.71875, 7.25, 5.1875, 1.59375,
                3.0625, 7.8125, 5.5625, 7.78125, 2.875, 9.25, 1.4375, 7.375, 5.65625, 2.125,
                2.54688, 1.17188, 4.5625, 1.23438, 1.96875, 1.25, 5.5625, 3.21875, 1.92188, 8.75,
                3.59375, 5.84375, 3.07812, 5.96875, 9.6875, 8.5625, 3.5, 2.125, 3.09375, 3.5,
                1.82031, 6.25, 6.125, 9.75, 4.75, 6.0625, 4.3125, 1.16406, 8.3125, 8.1875,
                3.59375, 3.09375, 7.4375, 8.25, 6.5, 4.5, 4.8125, 8.75, 7.75, 7.71875,
                4.84375, 6, 4.84375, 2.21875, 4.25, 1.53906, 2.375, 2.09375, 9.375, 1.39844,
                9.25, 1.96875, 8, 3.03125, 6.5625, 7.40625, 1.32031, 6.03125, 6.875, 1.10938,
                2.15625, 1.64062, 3.65625, 9.6875, 4.25, 6.125, 3.46875, 2.82812, 1.66406, 3.26562,
                2.375, 7.6875, 2.45312, 2.75, 9.4375, 6.21875, 4.3125, 9.75, 1.45312, 8.625,
                7.65625, 3.15625, 3.6875, 5.4375, 2.84375, 6.5625, 9.8125, 8.4375, 9, 2.40625,
                7.8125, 1.16406, 6.875, 1.625, 1.35938, 5.375, 8.3125, 6.4375, 7.875, 6.125,
                5.09375, 3.84375, 5.78125, 9.875, 1.98438, 6.1875, 2.3125, 4.40625, 5.5625, 5.9375,
                2.9375, 7.6875, 9.25, 7, 5.15625, 3.375, 2.1875, 1.59375, 7.875, 4.3125,
                2.90625, 6.65625, 1.67188, 2.89062, 1.85938, 7.75, 2.45312, 1.59375, 4.1875, 3.34375,
                1.85938, 8.25, 2.28125, 2.73438, 9.375, 6.75, 6.1875, 5.71875, 8.5, 9.3125,
                6.625, 3.375, 3.90625, 1.59375, 7.5625, 7.625, 5.6875, 7.9375, 7.625, 9.125,
                2.48438, 9.375, 7.1875, 1.125, 4.8125, 3.09375, 7.5625, 6.5625, 7.8125, 9.5,
                4.5625, 9.5, 9.3125, 6, 2.82812, 9.25, 1.07031, 6.75, 9.3125, 4.5,
                3.65625, 5.375, 2.5, 6.4375, 1.21875, 5.9375, 5.0625, 9.3125, 8.25, 9.25,
                4.3125, 4.5625, 6.46875, 9.625, 1.3125, 2.5625, 4.1875, 2.125, 1.70312, 2.21875,
                7.25, 5.5625, 1.10938, 1.1875, 5.125, 9.5, 9.625, 8.4375, 4, 1.13281,
                5.25, 2.57812, 1.94531, 3.98438, 5.5, 2.17188, 9, 8.25, 5.8125, 4.09375,
                3.53125, 9.4375, 4.1875, 6.25, 9.0625, 8.875, 3.17188, 8.625, 1.21875, 9.125,
                9.6875, 5.125, 4.875, 5.90625, 4.125, 8.125, 6.1875, 3.5625, 2.125, 5.40625,
                9.5, 6.375, 3.8125, 1.14062, 9.5625, 6.3125, 2.96875, 4.875, 3.23438, 8.25,
                8.75, 3.84375, 3.125, 9, 8.3125, 6.1875, 5.875, 2.65625, 2.71875, 8.0625,
                6.3125, 6.5, 1.42969, 1.48438, 1.14062, 4.78125, 1.44531, 7.125, 4.59375, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 10},
            std::vector<int64_t>{10, 10, 10, 10, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 2.15625,
                5.71875, 1.17188, 1.75, 7.125, 9.25, 2.90625, 9.1875, 3.375, 3.6875, 5.4375,
                6.25, 1.47656, 6.0625, 6.15625, 6.5, 2.3125, 9.625, 6.3125, 3.34375, 7.3125,
                3.07812, 1.92188, 5.8125, 4.71875, 9.5, 7.25, 5.4375, 4.71875, 5.875, 1.45312,
                7.875, 5.8125, 1.40625, 6.96875, 2.25, 5.625, 8.125, 9.5, 1.26562, 6.25,
                8.9375, 9.125, 5.875, 2.23438, 5.03125, 2.25, 9, 8.25, 4.375, 4.5625,
                5.84375, 2.48438, 6.875, 9.375, 4.25, 4.125, 6.125, 7.75, 6.75, 7.53125,
                2.125, 8.9375, 7.1875, 6.625, 6.8125, 7.75, 4.1875, 4.125, 7.875, 3.42188,
                4.1875, 9.0625, 7.75, 4.84375, 8.875, 9.625, 1.10156, 6.96875, 5.46875, 6.59375,
                1.66406, 2.03125, 8.0625, 9.5, 1.57812, 5.0625, 4.1875, 6.1875, 9.5, 4.6875,
                4.40625, 3.125, 7.875, 9.125, 7.9375, 6.15625, 3.71875, 1.02344, 7.9375, 6.5625,
                2.375, 3.9375, 6.1875, 5.75, 1.07812, 9, 7.375, 4.1875, 5.25, 9.125,
                7.875, 6.625, 5.1875, 1.14062, 3.40625, 9.375, 8.5, 7.1875, 5.9375, 10,
                1.625, 2.54688, 5.25, 2.21875, 7.6875, 9.375, 2.71875, 7.25, 5.1875, 1.59375,
                3.0625, 7.8125, 5.5625, 7.78125, 2.875, 9.25, 1.4375, 7.375, 5.65625, 2.125,
                2.54688, 1.17188, 4.5625, 1.23438, 1.96875, 1.25, 5.5625, 3.21875, 1.92188, 8.75,
                3.59375, 5.84375, 3.07812, 5.96875, 9.6875, 8.5625, 3.5, 2.125, 3.09375, 3.5,
                1.82031, 6.25, 6.125, 9.75, 4.75, 6.0625, 4.3125, 1.16406, 8.3125, 8.1875,
                3.59375, 3.09375, 7.4375, 8.25, 6.5, 4.5, 4.8125, 8.75, 7.75, 7.71875,
                4.84375, 6, 4.84375, 2.21875, 4.25, 1.53906, 2.375, 2.09375, 9.375, 1.39844,
                9.25, 1.96875, 8, 3.03125, 6.5625, 7.40625, 1.32031, 6.03125, 6.875, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 2.15625,
                5.71875, 1.17188, 1.75, 7.125, 9.25, 2.90625, 9.1875, 3.375, 3.6875, 5.4375,
                6.25, 1.47656, 6.0625, 6.15625, 6.5, 2.3125, 9.625, 6.3125, 3.34375, 7.3125,
                3.07812, 1.92188, 5.8125, 4.71875, 9.5, 7.25, 5.4375, 4.71875, 5.875, 1.45312,
                7.875, 5.8125, 1.40625, 6.96875, 2.25, 5.625, 8.125, 9.5, 1.26562, 6.25,
                8.9375, 9.125, 5.875, 2.23438, 5.03125, 2.25, 9, 8.25, 4.375, 4.5625,
                5.84375, 2.48438, 6.875, 9.375, 4.25, 4.125, 6.125, 7.75, 6.75, 7.53125,
                2.125, 8.9375, 7.1875, 6.625, 6.8125, 7.75, 4.1875, 4.125, 7.875, 3.42188,
                4.1875, 9.0625, 7.75, 4.84375, 8.875, 9.625, 1.10156, 6.96875, 5.46875, 6.59375,
                1.66406, 2.03125, 8.0625, 9.5, 1.57812, 5.0625, 4.1875, 6.1875, 9.5, 4.6875,
                4.40625, 3.125, 7.875, 9.125, 7.9375, 6.15625, 3.71875, 1.02344, 7.9375, 6.5625,
                2.375, 3.9375, 6.1875, 5.75, 1.07812, 9, 7.375, 4.1875, 5.25, 9.125,
                7.875, 6.625, 5.1875, 1.14062, 3.40625, 9.375, 8.5, 7.1875, 5.9375, 10,
                1.625, 2.54688, 5.25, 2.21875, 7.6875, 9.375, 2.71875, 7.25, 5.1875, 1.59375,
                3.0625, 7.8125, 5.5625, 7.78125, 2.875, 9.25, 1.4375, 7.375, 5.65625, 2.125,
                2.54688, 1.17188, 4.5625, 1.23438, 1.96875, 1.25, 5.5625, 3.21875, 1.92188, 8.75,
                3.59375, 5.84375, 3.07812, 5.96875, 9.6875, 8.5625, 3.5, 2.125, 3.09375, 3.5,
                1.82031, 6.25, 6.125, 9.75, 4.75, 6.0625, 4.3125, 1.16406, 8.3125, 8.1875,
                3.59375, 3.09375, 7.4375, 8.25, 6.5, 4.5, 4.8125, 8.75, 7.75, 7.71875,
                4.84375, 6, 4.84375, 2.21875, 4.25, 1.53906, 2.375, 2.09375, 9.375, 1.39844,
                9.25, 1.96875, 8, 3.03125, 6.5625, 7.40625, 1.32031, 6.03125, 6.875, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 10},
            std::vector<T>{
                0.605469, 0.671875, 0.761719, 0.714844, 0.753906, 0.605469, 0.625, 0.652344, 0.761719, 0.625,
                0.609375, 0.707031, 0.839844, 0.777344, 0.832031, 0.609375, 0.640625, 0.679688, 0.839844, 0.640625,
                0.613281, 0.761719, 0.964844, 0.867188, 0.945312, 0.613281, 0.660156, 0.722656, 0.964844, 0.664062,
                0.621094, 0.839844, 1.14844, 1, 1.11719, 0.621094, 0.691406, 0.785156, 1.14844, 0.699219,
                0.632812, 0.964844, 1.42188, 1.20312, 1.375, 0.632812, 0.738281, 0.878906, 1.42188, 0.753906,
                0.648438, 1.14844, 1.82812, 1.5, 1.75, 0.648438, 0.808594, 1.01562, 1.82812, 0.832031,
                0.675781, 1.42188, 2.4375, 1.95312, 2.3125, 0.675781, 0.910156, 1.21875, 2.4375, 0.945312,
                0.714844, 1.82812, 3.375, 2.625, 3.1875, 0.714844, 1.0625, 1.51562, 3.375, 1.11719,
                0.777344, 2.4375, 4.75, 3.65625, 4.5, 0.777344, 1.29688, 1.96875, 4.75, 1.375,
                0.867188, 3.375, 6.875, 5.1875, 6.4375, 0.867188, 1.64062, 2.65625, 6.875, 1.75,
                0.640625, 0.621094, 0.664062, 0.632812, 0.664062, 0.660156, 0.707031, 0.664062, 0.753906, 0.691406,
                0.664062, 0.632812, 0.699219, 0.652344, 0.699219, 0.691406, 0.761719, 0.699219, 0.832031, 0.738281,
                0.695312, 0.648438, 0.753906, 0.679688, 0.753906, 0.738281, 0.847656, 0.753906, 0.945312, 0.808594,
                0.746094, 0.671875, 0.832031, 0.722656, 0.832031, 0.808594, 0.972656, 0.832031, 1.11719, 0.910156,
                0.816406, 0.707031, 0.945312, 0.785156, 0.945312, 0.910156, 1.15625, 0.945312, 1.375, 1.0625,
                0.929688, 0.761719, 1.11719, 0.878906, 1.11719, 1.0625, 1.4375, 1.11719, 1.75, 1.29688,
                1.09375, 0.839844, 1.375, 1.01562, 1.375, 1.29688, 1.84375, 1.375, 2.3125, 1.64062,
                1.34375, 0.964844, 1.76562, 1.21875, 1.75, 1.64062, 2.46875, 1.76562, 3.1875, 2.15625,
                1.71875, 1.14844, 2.34375, 1.53125, 2.3125, 2.15625, 3.4375, 2.34375, 4.5, 2.9375,
                2.28125, 1.42188, 3.25, 1.98438, 3.1875, 2.9375, 4.875, 3.25, 6.4375, 4.09375,
                0.738281, 0.671875, 0.652344, 0.714844, 0.6875, 0.636719, 0.675781, 0.746094, 0.640625, 0.609375,
                0.808594, 0.707031, 0.679688, 0.769531, 0.730469, 0.65625, 0.714844, 0.816406, 0.664062, 0.617188,
                0.917969, 0.761719, 0.722656, 0.855469, 0.800781, 0.6875, 0.769531, 0.925781, 0.695312, 0.625,
                1.07812, 0.839844, 0.785156, 0.988281, 0.902344, 0.730469, 0.855469, 1.08594, 0.746094, 0.636719,
                1.3125, 0.964844, 0.878906, 1.17969, 1.05469, 0.800781, 0.988281, 1.32812, 0.816406, 0.65625,
                1.65625, 1.14844, 1.01562, 1.46875, 1.28125, 0.898438, 1.17969, 1.6875, 0.925781, 0.6875,
                2.1875, 1.42188, 1.21875, 1.89062, 1.625, 1.04688, 1.46875, 2.21875, 1.08594, 0.730469,
                2.96875, 1.82812, 1.53125, 2.53125, 2.125, 1.26562, 1.89062, 3.03125, 1.32812, 0.792969,
                4.15625, 2.4375, 2, 3.5, 2.875, 1.59375, 2.53125, 4.25, 1.6875, 0.886719,
                5.9375, 3.375, 2.71875, 4.9375, 4, 2.09375, 3.53125, 6.125, 2.21875, 1.03125,
                0.6875, 0.707031, 0.746094, 0.671875, 0.675781, 0.691406, 0.671875, 0.625, 0.753906, 0.636719,
                0.730469, 0.761719, 0.816406, 0.707031, 0.714844, 0.738281, 0.707031, 0.640625, 0.832031, 0.65625,
                0.800781, 0.847656, 0.929688, 0.761719, 0.777344, 0.808594, 0.761719, 0.664062, 0.945312, 0.6875,
                0.902344, 0.972656, 1.09375, 0.839844, 0.867188, 0.917969, 0.847656, 0.695312, 1.11719, 0.730469,
                1.05469, 1.15625, 1.34375, 0.964844, 1, 1.07812, 0.972656, 0.746094, 1.375, 0.792969,
                1.28125, 1.4375, 1.71875, 1.14844, 1.20312, 1.3125, 1.15625, 0.816406, 1.75, 0.886719,
                1.625, 1.84375, 2.28125, 1.42188, 1.5, 1.65625, 1.4375, 0.929688, 2.3125, 1.03125,
                2.125, 2.46875, 3.15625, 1.82812, 1.9375, 2.1875, 1.84375, 1.09375, 3.1875, 1.25,
                2.90625, 3.4375, 4.4375, 2.4375, 2.59375, 3, 2.46875, 1.34375, 4.5, 1.57812,
                4.0625, 4.875, 6.375, 3.375, 3.625, 4.1875, 3.4375, 1.70312, 6.5, 2.0625,
                0.730469, 0.730469, 0.714844, 0.753906, 0.730469, 0.652344, 0.621094, 0.714844, 0.6875, 0.761719,
                0.792969, 0.800781, 0.777344, 0.832031, 0.800781, 0.679688, 0.632812, 0.769531, 0.730469, 0.839844,
                0.886719, 0.898438, 0.867188, 0.949219, 0.902344, 0.722656, 0.648438, 0.855469, 0.800781, 0.964844,
                1.03125, 1.04688, 1, 1.125, 1.05469, 0.785156, 0.671875, 0.988281, 0.898438, 1.14844,
                1.25, 1.26562, 1.20312, 1.39062, 1.28125, 0.878906, 0.707031, 1.17969, 1.04688, 1.42188,
                1.57812, 1.59375, 1.5, 1.78125, 1.625, 1.01562, 0.761719, 1.46875, 1.26562, 1.82812,
                2.0625, 2.09375, 1.9375, 2.375, 2.125, 1.21875, 0.839844, 1.89062, 1.59375, 2.4375,
                2.78125, 2.84375, 2.59375, 3.28125, 2.875, 1.53125, 0.964844, 2.53125, 2.09375, 3.375,
                3.875, 3.96875, 3.625, 4.625, 4, 2, 1.14844, 3.53125, 2.84375, 4.75,
                5.53125, 5.65625, 5.125, 6.65625, 5.6875, 2.71875, 1.42188, 5, 3.96875, 6.875},
            std::vector<T>{
                0.605469, 0.671875, 0.761719, 0.714844, 0.753906, 0.605469, 0.625, 0.652344, 0.761719, 0.625,
                0.640625, 0.621094, 0.664062, 0.632812, 0.664062, 0.660156, 0.707031, 0.664062, 0.753906, 0.691406,
                0.738281, 0.671875, 0.652344, 0.714844, 0.6875, 0.636719, 0.675781, 0.746094, 0.640625, 0.609375,
                0.6875, 0.707031, 0.746094, 0.671875, 0.675781, 0.691406, 0.671875, 0.625, 0.753906, 0.636719,
                0.730469, 0.730469, 0.714844, 0.753906, 0.730469, 0.652344, 0.621094, 0.714844, 0.6875, 0.761719}),
        GRUSequenceParams(
            5, 10, 10, 5,
            0.f, true, op::RecurrentSequenceDirection::BIDIRECTIONAL,
            ET,
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 2.15625,
                5.71875, 1.17188, 1.75, 7.125, 9.25, 2.90625, 9.1875, 3.375, 3.6875, 5.4375,
                6.25, 1.47656, 6.0625, 6.15625, 6.5, 2.3125, 9.625, 6.3125, 3.34375, 7.3125,
                3.07812, 1.92188, 5.8125, 4.71875, 9.5, 7.25, 5.4375, 4.71875, 5.875, 1.45312,
                7.875, 5.8125, 1.40625, 6.96875, 2.25, 5.625, 8.125, 9.5, 1.26562, 6.25,
                8.9375, 9.125, 5.875, 2.23438, 5.03125, 2.25, 9, 8.25, 4.375, 4.5625,
                5.84375, 2.48438, 6.875, 9.375, 4.25, 4.125, 6.125, 7.75, 6.75, 7.53125,
                2.125, 8.9375, 7.1875, 6.625, 6.8125, 7.75, 4.1875, 4.125, 7.875, 3.42188,
                4.1875, 9.0625, 7.75, 4.84375, 8.875, 9.625, 1.10156, 6.96875, 5.46875, 6.59375,
                1.66406, 2.03125, 8.0625, 9.5, 1.57812, 5.0625, 4.1875, 6.1875, 9.5, 4.6875,
                4.40625, 3.125, 7.875, 9.125, 7.9375, 6.15625, 3.71875, 1.02344, 7.9375, 6.5625,
                2.375, 3.9375, 6.1875, 5.75, 1.07812, 9, 7.375, 4.1875, 5.25, 9.125,
                7.875, 6.625, 5.1875, 1.14062, 3.40625, 9.375, 8.5, 7.1875, 5.9375, 10,
                1.625, 2.54688, 5.25, 2.21875, 7.6875, 9.375, 2.71875, 7.25, 5.1875, 1.59375,
                3.0625, 7.8125, 5.5625, 7.78125, 2.875, 9.25, 1.4375, 7.375, 5.65625, 2.125,
                2.54688, 1.17188, 4.5625, 1.23438, 1.96875, 1.25, 5.5625, 3.21875, 1.92188, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 10},
            std::vector<int64_t>{5, 5, 5, 5, 5},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 2.15625,
                5.71875, 1.17188, 1.75, 7.125, 9.25, 2.90625, 9.1875, 3.375, 3.6875, 5.4375,
                6.25, 1.47656, 6.0625, 6.15625, 6.5, 2.3125, 9.625, 6.3125, 3.34375, 7.3125,
                3.07812, 1.92188, 5.8125, 4.71875, 9.5, 7.25, 5.4375, 4.71875, 5.875, 1.45312,
                7.875, 5.8125, 1.40625, 6.96875, 2.25, 5.625, 8.125, 9.5, 1.26562, 6.25,
                8.9375, 9.125, 5.875, 2.23438, 5.03125, 2.25, 9, 8.25, 4.375, 4.5625,
                5.84375, 2.48438, 6.875, 9.375, 4.25, 4.125, 6.125, 7.75, 6.75, 7.53125,
                2.125, 8.9375, 7.1875, 6.625, 6.8125, 7.75, 4.1875, 4.125, 7.875, 3.42188,
                4.1875, 9.0625, 7.75, 4.84375, 8.875, 9.625, 1.10156, 6.96875, 5.46875, 6.59375,
                1.66406, 2.03125, 8.0625, 9.5, 1.57812, 5.0625, 4.1875, 6.1875, 9.5, 4.6875,
                4.40625, 3.125, 7.875, 9.125, 7.9375, 6.15625, 3.71875, 1.02344, 7.9375, 6.5625,
                2.375, 3.9375, 6.1875, 5.75, 1.07812, 9, 7.375, 4.1875, 5.25, 9.125,
                7.875, 6.625, 5.1875, 1.14062, 3.40625, 9.375, 8.5, 7.1875, 5.9375, 10,
                1.625, 2.54688, 5.25, 2.21875, 7.6875, 9.375, 2.71875, 7.25, 5.1875, 1.59375,
                3.0625, 7.8125, 5.5625, 7.78125, 2.875, 9.25, 1.4375, 7.375, 5.65625, 2.125,
                2.54688, 1.17188, 4.5625, 1.23438, 1.96875, 1.25, 5.5625, 3.21875, 1.92188, 8.75,
                3.59375, 5.84375, 3.07812, 5.96875, 9.6875, 8.5625, 3.5, 2.125, 3.09375, 3.5,
                1.82031, 6.25, 6.125, 9.75, 4.75, 6.0625, 4.3125, 1.16406, 8.3125, 8.1875,
                3.59375, 3.09375, 7.4375, 8.25, 6.5, 4.5, 4.8125, 8.75, 7.75, 7.71875,
                4.84375, 6, 4.84375, 2.21875, 4.25, 1.53906, 2.375, 2.09375, 9.375, 1.39844,
                9.25, 1.96875, 8, 3.03125, 6.5625, 7.40625, 1.32031, 6.03125, 6.875, 1.10938,
                2.15625, 1.64062, 3.65625, 9.6875, 4.25, 6.125, 3.46875, 2.82812, 1.66406, 3.26562,
                2.375, 7.6875, 2.45312, 2.75, 9.4375, 6.21875, 4.3125, 9.75, 1.45312, 8.625,
                7.65625, 3.15625, 3.6875, 5.4375, 2.84375, 6.5625, 9.8125, 8.4375, 9, 2.40625,
                7.8125, 1.16406, 6.875, 1.625, 1.35938, 5.375, 8.3125, 6.4375, 7.875, 6.125,
                5.09375, 3.84375, 5.78125, 9.875, 1.98438, 6.1875, 2.3125, 4.40625, 5.5625, 5.9375,
                2.9375, 7.6875, 9.25, 7, 5.15625, 3.375, 2.1875, 1.59375, 7.875, 4.3125,
                2.90625, 6.65625, 1.67188, 2.89062, 1.85938, 7.75, 2.45312, 1.59375, 4.1875, 3.34375,
                1.85938, 8.25, 2.28125, 2.73438, 9.375, 6.75, 6.1875, 5.71875, 8.5, 9.3125,
                6.625, 3.375, 3.90625, 1.59375, 7.5625, 7.625, 5.6875, 7.9375, 7.625, 9.125,
                2.48438, 9.375, 7.1875, 1.125, 4.8125, 3.09375, 7.5625, 6.5625, 7.8125, 9.5,
                4.5625, 9.5, 9.3125, 6, 2.82812, 9.25, 1.07031, 6.75, 9.3125, 4.5,
                3.65625, 5.375, 2.5, 6.4375, 1.21875, 5.9375, 5.0625, 9.3125, 8.25, 9.25,
                4.3125, 4.5625, 6.46875, 9.625, 1.3125, 2.5625, 4.1875, 2.125, 1.70312, 2.21875,
                7.25, 5.5625, 1.10938, 1.1875, 5.125, 9.5, 9.625, 8.4375, 4, 1.13281,
                5.25, 2.57812, 1.94531, 3.98438, 5.5, 2.17188, 9, 8.25, 5.8125, 4.09375,
                3.53125, 9.4375, 4.1875, 6.25, 9.0625, 8.875, 3.17188, 8.625, 1.21875, 9.125,
                9.6875, 5.125, 4.875, 5.90625, 4.125, 8.125, 6.1875, 3.5625, 2.125, 5.40625,
                9.5, 6.375, 3.8125, 1.14062, 9.5625, 6.3125, 2.96875, 4.875, 3.23438, 8.25,
                8.75, 3.84375, 3.125, 9, 8.3125, 6.1875, 5.875, 2.65625, 2.71875, 8.0625,
                6.3125, 6.5, 1.42969, 1.48438, 1.14062, 4.78125, 1.44531, 7.125, 4.59375, 9.25,
                6.1875, 1, 8.75, 9.75, 8.0625, 4.375, 3.28125, 9.75, 1.73438, 6.4375,
                2.40625, 8.4375, 2.875, 6.1875, 4.75, 6.625, 4.125, 3.5625, 7.3125, 6.28125,
                7.25, 7.75, 1.67188, 8.75, 1.32812, 7.78125, 8.125, 7.28125, 9.125, 8.75,
                1.03906, 3.90625, 3.375, 7.03125, 5.1875, 5.0625, 4.25, 4.4375, 3.15625, 4.6875,
                1.20312, 4.625, 5.78125, 3.84375, 2.15625, 6.59375, 6.0625, 4.875, 2.5, 9.75,
                4.78125, 7.09375, 7.15625, 2.78125, 3.46875, 4.8125, 1.09375, 4.0625, 2.96875, 8.125,
                1.28125, 8.875, 9.375, 9.125, 8, 6.9375, 5.65625, 3.4375, 1.14062, 3.26562,
                3.6875, 8.6875, 5.875, 5.75, 8.75, 8.25, 2.65625, 6.125, 1.75, 7.59375,
                1.39844, 5.65625, 5, 7.9375, 1.17188, 6.125, 1.60156, 5.1875, 8.25, 4.0625,
                6.71875, 1.60938, 7.9375, 4.375, 7.34375, 1.71875, 3.14062, 9.875, 1.39844, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 2.15625,
                5.71875, 1.17188, 1.75, 7.125, 9.25, 2.90625, 9.1875, 3.375, 3.6875, 5.4375,
                6.25, 1.47656, 6.0625, 6.15625, 6.5, 2.3125, 9.625, 6.3125, 3.34375, 7.3125,
                3.07812, 1.92188, 5.8125, 4.71875, 9.5, 7.25, 5.4375, 4.71875, 5.875, 1.45312,
                7.875, 5.8125, 1.40625, 6.96875, 2.25, 5.625, 8.125, 9.5, 1.26562, 6.25,
                8.9375, 9.125, 5.875, 2.23438, 5.03125, 2.25, 9, 8.25, 4.375, 4.5625,
                5.84375, 2.48438, 6.875, 9.375, 4.25, 4.125, 6.125, 7.75, 6.75, 7.53125,
                2.125, 8.9375, 7.1875, 6.625, 6.8125, 7.75, 4.1875, 4.125, 7.875, 3.42188,
                4.1875, 9.0625, 7.75, 4.84375, 8.875, 9.625, 1.10156, 6.96875, 5.46875, 6.59375,
                1.66406, 2.03125, 8.0625, 9.5, 1.57812, 5.0625, 4.1875, 6.1875, 9.5, 4.6875,
                4.40625, 3.125, 7.875, 9.125, 7.9375, 6.15625, 3.71875, 1.02344, 7.9375, 6.5625,
                2.375, 3.9375, 6.1875, 5.75, 1.07812, 9, 7.375, 4.1875, 5.25, 9.125,
                7.875, 6.625, 5.1875, 1.14062, 3.40625, 9.375, 8.5, 7.1875, 5.9375, 10,
                1.625, 2.54688, 5.25, 2.21875, 7.6875, 9.375, 2.71875, 7.25, 5.1875, 1.59375,
                3.0625, 7.8125, 5.5625, 7.78125, 2.875, 9.25, 1.4375, 7.375, 5.65625, 2.125,
                2.54688, 1.17188, 4.5625, 1.23438, 1.96875, 1.25, 5.5625, 3.21875, 1.92188, 8.75,
                3.59375, 5.84375, 3.07812, 5.96875, 9.6875, 8.5625, 3.5, 2.125, 3.09375, 3.5,
                1.82031, 6.25, 6.125, 9.75, 4.75, 6.0625, 4.3125, 1.16406, 8.3125, 8.1875,
                3.59375, 3.09375, 7.4375, 8.25, 6.5, 4.5, 4.8125, 8.75, 7.75, 7.71875,
                4.84375, 6, 4.84375, 2.21875, 4.25, 1.53906, 2.375, 2.09375, 9.375, 1.39844,
                9.25, 1.96875, 8, 3.03125, 6.5625, 7.40625, 1.32031, 6.03125, 6.875, 1.10938,
                2.15625, 1.64062, 3.65625, 9.6875, 4.25, 6.125, 3.46875, 2.82812, 1.66406, 3.26562,
                2.375, 7.6875, 2.45312, 2.75, 9.4375, 6.21875, 4.3125, 9.75, 1.45312, 8.625,
                7.65625, 3.15625, 3.6875, 5.4375, 2.84375, 6.5625, 9.8125, 8.4375, 9, 2.40625,
                7.8125, 1.16406, 6.875, 1.625, 1.35938, 5.375, 8.3125, 6.4375, 7.875, 6.125,
                5.09375, 3.84375, 5.78125, 9.875, 1.98438, 6.1875, 2.3125, 4.40625, 5.5625, 5.9375,
                2.9375, 7.6875, 9.25, 7, 5.15625, 3.375, 2.1875, 1.59375, 7.875, 4.3125,
                2.90625, 6.65625, 1.67188, 2.89062, 1.85938, 7.75, 2.45312, 1.59375, 4.1875, 3.34375,
                1.85938, 8.25, 2.28125, 2.73438, 9.375, 6.75, 6.1875, 5.71875, 8.5, 9.3125,
                6.625, 3.375, 3.90625, 1.59375, 7.5625, 7.625, 5.6875, 7.9375, 7.625, 9.125,
                2.48438, 9.375, 7.1875, 1.125, 4.8125, 3.09375, 7.5625, 6.5625, 7.8125, 9.5,
                4.5625, 9.5, 9.3125, 6, 2.82812, 9.25, 1.07031, 6.75, 9.3125, 4.5,
                3.65625, 5.375, 2.5, 6.4375, 1.21875, 5.9375, 5.0625, 9.3125, 8.25, 9.25,
                4.3125, 4.5625, 6.46875, 9.625, 1.3125, 2.5625, 4.1875, 2.125, 1.70312, 2.21875,
                7.25, 5.5625, 1.10938, 1.1875, 5.125, 9.5, 9.625, 8.4375, 4, 1.13281,
                5.25, 2.57812, 1.94531, 3.98438, 5.5, 2.17188, 9, 8.25, 5.8125, 4.09375,
                3.53125, 9.4375, 4.1875, 6.25, 9.0625, 8.875, 3.17188, 8.625, 1.21875, 9.125,
                9.6875, 5.125, 4.875, 5.90625, 4.125, 8.125, 6.1875, 3.5625, 2.125, 5.40625,
                9.5, 6.375, 3.8125, 1.14062, 9.5625, 6.3125, 2.96875, 4.875, 3.23438, 8.25,
                8.75, 3.84375, 3.125, 9, 8.3125, 6.1875, 5.875, 2.65625, 2.71875, 8.0625,
                6.3125, 6.5, 1.42969, 1.48438, 1.14062, 4.78125, 1.44531, 7.125, 4.59375, 9.25,
                6.1875, 1, 8.75, 9.75, 8.0625, 4.375, 3.28125, 9.75, 1.73438, 6.4375,
                2.40625, 8.4375, 2.875, 6.1875, 4.75, 6.625, 4.125, 3.5625, 7.3125, 6.28125,
                7.25, 7.75, 1.67188, 8.75, 1.32812, 7.78125, 8.125, 7.28125, 9.125, 8.75,
                1.03906, 3.90625, 3.375, 7.03125, 5.1875, 5.0625, 4.25, 4.4375, 3.15625, 4.6875,
                1.20312, 4.625, 5.78125, 3.84375, 2.15625, 6.59375, 6.0625, 4.875, 2.5, 9.75,
                4.78125, 7.09375, 7.15625, 2.78125, 3.46875, 4.8125, 1.09375, 4.0625, 2.96875, 8.125,
                1.28125, 8.875, 9.375, 9.125, 8, 6.9375, 5.65625, 3.4375, 1.14062, 3.26562,
                3.6875, 8.6875, 5.875, 5.75, 8.75, 8.25, 2.65625, 6.125, 1.75, 7.59375,
                1.39844, 5.65625, 5, 7.9375, 1.17188, 6.125, 1.60156, 5.1875, 8.25, 4.0625,
                6.71875, 1.60938, 7.9375, 4.375, 7.34375, 1.71875, 3.14062, 9.875, 1.39844, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 10,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 10,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 10,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 10,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 10}),
    };
    return params;
}

std::vector<GRUSequenceParams> generateCombinedParams() {
    const std::vector<std::vector<GRUSequenceParams>> generatedParams {
        generateParams<element::Type_t::f64>(),
        generateParams<element::Type_t::f32>(),
        generateParams<element::Type_t::f16>(),
        generateParamsBF16<element::Type_t::bf16>(),
    };
    std::vector<GRUSequenceParams> combinedParams;

    for (const auto& params : generatedParams) {
        combinedParams.insert(combinedParams.end(), params.begin(), params.end());
    }
    return combinedParams;
}

INSTANTIATE_TEST_SUITE_P(smoke_GRUSequence_With_Hardcoded_Refs, ReferenceGRUSequenceTest,
    testing::ValuesIn(generateCombinedParams()), ReferenceGRUSequenceTest::getTestCaseName);

} // namespace