// Copyright (C) 2022 Intel Corporation
// SPDX-License-Identifier: Apache-2.0
//

#include <gtest/gtest.h>

#include "openvino/op/lstm_sequence.hpp"
#include "base_reference_test.hpp"

using namespace reference_tests;
using namespace ov;

namespace {
struct LSTMSequenceParams {
    template <class T>
    LSTMSequenceParams(
        const size_t batchSize, const size_t inputSize, const size_t hiddenSize, const size_t seqLength,
        const float clip, const op::RecurrentSequenceDirection& lstm_direction,
        const element::Type_t& iType,
        const std::vector<T>& XValues, const std::vector<T>& H_tValues, const std::vector<T>& C_tValues,
        const std::vector<int64_t>& S_tValues,
        const std::vector<T>& WValues, const std::vector<T>& RValues, const std::vector<T>& BValues,
        const std::vector<T>& YValues, const std::vector<T>& HoValues, const std::vector<T>& CoValues,
        const std::string& testcaseName = "") :
        batchSize(batchSize), inputSize(inputSize), hiddenSize(hiddenSize), seqLength(seqLength),
        clip(clip), lstm_direction(lstm_direction), iType(iType), oType(iType),
        testcaseName(testcaseName) {
            numDirections = (lstm_direction == op::RecurrentSequenceDirection::BIDIRECTIONAL) ? 2 : 1;

            Shape XShape = Shape{batchSize, seqLength, inputSize};
            Shape H_tShape = Shape{batchSize, numDirections, hiddenSize};
            Shape C_tShape = Shape{batchSize, numDirections, hiddenSize};
            Shape S_tShape = Shape{batchSize};
            Shape WShape = Shape{numDirections, 4 * hiddenSize, inputSize};
            Shape RShape = Shape{numDirections, 4 * hiddenSize, hiddenSize};
            Shape BShape = Shape{numDirections, 4 * hiddenSize};
            Shape YShape = Shape{batchSize, numDirections, seqLength, hiddenSize};
            Shape HoShape = Shape{batchSize, numDirections, hiddenSize};
            Shape CoShape = Shape{batchSize, numDirections, hiddenSize};

            X = reference_tests::Tensor(XShape, iType, XValues);
            H_t = reference_tests::Tensor(H_tShape, iType, H_tValues);
            C_t = reference_tests::Tensor(C_tShape, iType, C_tValues);
            S_t = reference_tests::Tensor(S_tShape, element::Type_t::i64, S_tValues);
            W = reference_tests::Tensor(WShape, iType, WValues);
            R = reference_tests::Tensor(RShape, iType, RValues);
            B = reference_tests::Tensor(BShape, iType, BValues);
            Y = reference_tests::Tensor(YShape, oType, YValues);
            Ho = reference_tests::Tensor(HoShape, oType, HoValues);
            Co = reference_tests::Tensor(CoShape, oType, CoValues);
        }

    size_t batchSize;
    size_t inputSize;
    size_t hiddenSize;
    size_t seqLength;
    size_t numDirections;
    float clip;
    op::RecurrentSequenceDirection lstm_direction;
    element::Type_t iType;
    element::Type_t oType;

    reference_tests::Tensor X;
    reference_tests::Tensor H_t;
    reference_tests::Tensor C_t;
    reference_tests::Tensor S_t;
    reference_tests::Tensor W;
    reference_tests::Tensor R;
    reference_tests::Tensor B;
    reference_tests::Tensor Y;
    reference_tests::Tensor Ho;
    reference_tests::Tensor Co;
    std::string testcaseName;
};

struct LSTMSequenceV1Params {
    template <class T>
    LSTMSequenceV1Params(
        const size_t batchSize, const size_t inputSize, const size_t hiddenSize, const size_t seqLength,
        const float clip, const bool input_forget, const op::RecurrentSequenceDirection& lstm_direction,
        const element::Type_t& iType,
        const std::vector<T>& XValues, const std::vector<T>& H_tValues, const std::vector<T>& C_tValues,
        const std::vector<int64_t>& S_tValues,
        const std::vector<T>& WValues, const std::vector<T>& RValues, const std::vector<T>& BValues, const std::vector<T>& PValues,
        const std::vector<T>& YValues, const std::vector<T>& HoValues, const std::vector<T>& CoValues,
        const std::string& testcaseName = "") :
        batchSize(batchSize), inputSize(inputSize), hiddenSize(hiddenSize), seqLength(seqLength),
        clip(clip), input_forget(input_forget), lstm_direction(lstm_direction), iType(iType), oType(iType),
        testcaseName(testcaseName) {
            numDirections = (lstm_direction == op::RecurrentSequenceDirection::BIDIRECTIONAL) ? 2 : 1;

            Shape XShape = Shape{batchSize, seqLength, inputSize};
            Shape H_tShape = Shape{batchSize, numDirections, hiddenSize};
            Shape C_tShape = Shape{batchSize, numDirections, hiddenSize};
            Shape S_tShape = Shape{batchSize};
            Shape WShape = Shape{numDirections, 4 * hiddenSize, inputSize};
            Shape RShape = Shape{numDirections, 4 * hiddenSize, hiddenSize};
            Shape BShape = Shape{numDirections, 4 * hiddenSize};
            Shape PShape = Shape{numDirections, 3 * hiddenSize};
            Shape YShape = Shape{batchSize, numDirections, seqLength, hiddenSize};
            Shape HoShape = Shape{batchSize, numDirections, hiddenSize};
            Shape CoShape = Shape{batchSize, numDirections, hiddenSize};

            X = reference_tests::Tensor(XShape, iType, XValues);
            H_t = reference_tests::Tensor(H_tShape, iType, H_tValues);
            C_t = reference_tests::Tensor(C_tShape, iType, C_tValues);
            S_t = reference_tests::Tensor(S_tShape, element::Type_t::i64, S_tValues);
            W = reference_tests::Tensor(WShape, iType, WValues);
            R = reference_tests::Tensor(RShape, iType, RValues);
            B = reference_tests::Tensor(BShape, iType, BValues);
            P = reference_tests::Tensor(PShape, iType, PValues);
            Y = reference_tests::Tensor(YShape, oType, YValues);
            Ho = reference_tests::Tensor(HoShape, oType, HoValues);
            Co = reference_tests::Tensor(CoShape, oType, CoValues);
        }

    size_t batchSize;
    size_t inputSize;
    size_t hiddenSize;
    size_t seqLength;
    size_t numDirections;
    float clip;
    bool input_forget;
    op::RecurrentSequenceDirection lstm_direction;
    element::Type_t iType;
    element::Type_t oType;

    reference_tests::Tensor X;
    reference_tests::Tensor H_t;
    reference_tests::Tensor C_t;
    reference_tests::Tensor S_t;
    reference_tests::Tensor W;
    reference_tests::Tensor R;
    reference_tests::Tensor B;
    reference_tests::Tensor P;
    reference_tests::Tensor Y;
    reference_tests::Tensor Ho;
    reference_tests::Tensor Co;
    std::string testcaseName;
};

class ReferenceLSTMSequenceTest : public testing::TestWithParam<LSTMSequenceParams>, public CommonReferenceTest {
public:
    void SetUp() override {
        auto params = GetParam();
        function = CreateFunction(params);
        inputData = {params.X.data, params.H_t.data, params.C_t.data, params.S_t.data, params.W.data, params.R.data, params.B.data};
        refOutData = {params.Y.data, params.Ho.data, params.Co.data};
    }

    static std::string getTestCaseName(const testing::TestParamInfo<LSTMSequenceParams>& obj) {
        auto param = obj.param;
        std::ostringstream result;
        result << "iType=" << param.iType << "_";
        result << "xShape=" << param.X.shape << "_";
        result << "htShape=" << param.H_t.shape << "_";
        result << "ctShape=" << param.C_t.shape << "_";
        result << "stShape=" << param.S_t.shape << "_";
        result << "wShape=" << param.W.shape << "_";
        result << "rShape=" << param.R.shape << "_";
        result << "bShape=" << param.B.shape << "_";
        result << "YShape=" << param.Y.shape << "_";
        result << "hoShape=" << param.Ho.shape << "_";
        result << "coShape=" << param.Co.shape << "_";
        result << "clip=" << param.clip << "_";
        result << "LSTMdirection=" << param.lstm_direction;
        if (!param.testcaseName.empty())
            result << "_" << param.testcaseName;

        return result.str();
    }

private:
    static std::shared_ptr<Model> CreateFunction(const LSTMSequenceParams& params) {
        const auto X = std::make_shared<op::v0::Parameter>(params.X.type, params.X.shape);
        const auto H_t = std::make_shared<op::v0::Parameter>(params.H_t.type, params.H_t.shape);
        const auto C_t = std::make_shared<op::v0::Parameter>(params.C_t.type, params.C_t.shape);
        const auto S_t = std::make_shared<op::v0::Parameter>(params.S_t.type, params.S_t.shape);
        const auto W = std::make_shared<op::v0::Parameter>(params.W.type, params.W.shape);
        const auto R = std::make_shared<op::v0::Parameter>(params.R.type, params.R.shape);
        const auto B = std::make_shared<op::v0::Parameter>(params.B.type, params.B.shape);

        const auto lstm_sequence =
            std::make_shared<op::v5::LSTMSequence>(X,
                                                   H_t,
                                                   C_t,
                                                   S_t,
                                                   W,
                                                   R,
                                                   B,
                                                   params.hiddenSize,
                                                   params.lstm_direction,
                                                   std::vector<float>{},
                                                   std::vector<float>{},
                                                   std::vector<std::string>{"sigmoid", "tanh", "tanh"},
                                                   params.clip);

        auto function = std::make_shared<Model>(lstm_sequence->outputs(), ParameterVector{X, H_t, C_t, S_t, W, R, B});
        return function;
    }
};

class ReferenceLSTMSequenceV1Test : public testing::TestWithParam<LSTMSequenceV1Params>, public CommonReferenceTest {
public:
    void SetUp() override {
        auto params = GetParam();
        function = CreateFunction(params);
        inputData = {params.X.data, params.H_t.data, params.C_t.data, params.S_t.data, params.W.data, params.R.data, params.B.data, params.P.data};
        refOutData = {params.Y.data, params.Ho.data, params.Co.data};
    }

    static std::string getTestCaseName(const testing::TestParamInfo<LSTMSequenceV1Params>& obj) {
        auto param = obj.param;
        std::ostringstream result;
        result << "iType=" << param.iType << "_";
        result << "xShape=" << param.X.shape << "_";
        result << "htShape=" << param.H_t.shape << "_";
        result << "ctShape=" << param.C_t.shape << "_";
        result << "stShape=" << param.S_t.shape << "_";
        result << "wShape=" << param.W.shape << "_";
        result << "rShape=" << param.R.shape << "_";
        result << "bShape=" << param.B.shape << "_";
        result << "pShape=" << param.P.shape << "_";
        result << "YShape=" << param.Y.shape << "_";
        result << "hoShape=" << param.Ho.shape << "_";
        result << "coShape=" << param.Co.shape << "_";
        result << "clip=" << param.clip << "_";
        result << "input_forget=" << param.input_forget << "_";
        result << "LSTMdirection=" << param.lstm_direction;
        if (!param.testcaseName.empty())
            result << "_" << param.testcaseName;

        return result.str();
    }

private:
    static std::shared_ptr<Model> CreateFunction(const LSTMSequenceV1Params& params) {
        const auto X = std::make_shared<op::v0::Parameter>(params.X.type, params.X.shape);
        const auto H_t = std::make_shared<op::v0::Parameter>(params.H_t.type, params.H_t.shape);
        const auto C_t = std::make_shared<op::v0::Parameter>(params.C_t.type, params.C_t.shape);
        const auto S_t = std::make_shared<op::v0::Parameter>(params.S_t.type, params.S_t.shape);
        const auto W = std::make_shared<op::v0::Parameter>(params.W.type, params.W.shape);
        const auto R = std::make_shared<op::v0::Parameter>(params.R.type, params.R.shape);
        const auto B = std::make_shared<op::v0::Parameter>(params.B.type, params.B.shape);
        const auto P = std::make_shared<op::v0::Parameter>(params.P.type, params.P.shape);

        const auto lstm_sequence =
            std::make_shared<op::v0::LSTMSequence>(X,
                                                   H_t,
                                                   C_t,
                                                   S_t,
                                                   W,
                                                   R,
                                                   B,
                                                   P,
                                                   params.hiddenSize,
                                                   params.lstm_direction,
                                                   ov::op::LSTMWeightsFormat::FICO,
                                                   std::vector<float>{},
                                                   std::vector<float>{},
                                                   std::vector<std::string>{"sigmoid", "tanh", "tanh"},
                                                   params.clip,
                                                   params.input_forget);

        auto function = std::make_shared<Model>(lstm_sequence->outputs(), ParameterVector{X, H_t, C_t, S_t, W, R, B, P});
        return function;
    }
};

TEST_P(ReferenceLSTMSequenceTest, CompareWithRefs) {
    Exec();
}

TEST_P(ReferenceLSTMSequenceV1Test, CompareWithRefs) {
    Exec();
}

template <element::Type_t ET>
std::vector<LSTMSequenceParams> generateParams() {
    using T = typename element_type_traits<ET>::value_type;

    std::vector<LSTMSequenceParams> params {
        LSTMSequenceParams(
            5, 10, 10, 10,
            0.7f, op::RecurrentSequenceDirection::FORWARD,
            ET,
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 7.95465,
                2.37637, 6.20771, 1.08107, 7.38138, 5.23577, 7.88133, 5.20653, 3.42101, 8.48523, 5.96192,
                1.63073, 5.25228, 7.68488, 2.7276, 5.1788, 3.07327, 5.57423, 2.87711, 1.44374, 5.66976,
                2.55051, 4.56682, 1.96629, 5.58829, 1.91922, 3.59846, 3.08583, 9.70901, 3.50487, 3.1026,
                1.82401, 6.1306, 4.76134, 4.31059, 8.31695, 3.60784, 7.45652, 6.51653, 4.84219, 7.76686,
                4.85031, 4.85544, 4.25714, 2.38005, 9.43471, 9.24769, 8.03763, 6.54696, 1.32399, 6.88891,
                2.16793, 3.64924, 4.24733, 3.47181, 1.66572, 2.36923, 2.45457, 9.44841, 4.34021, 1.45016,
                7.6686, 3.68812, 2.83922, 9.83581, 9.03719, 7.83414, 6.86009, 1.35715, 8.32489, 7.86316,
                5.09754, 5.78644, 1.98402, 2.31429, 5.5791, 2.94085, 9.24799, 5.15937, 2.19041, 7.87817,
                2.9146, 1.66833, 1.85877, 2.45985, 4.20817, 1.85777, 2.28937, 9.37912, 6.18926, 8.55681,
                6.60963, 3.92066, 7.5521, 5.70463, 7.6313, 2.48866, 7.18352, 4.8413, 7.55702, 7.80702,
                4.5785, 9.3268, 2.83159, 1.07202, 9.33716, 3.6506, 2.50256, 1.21691, 5.06801, 8.27505,
                4.31539, 6.48286, 1.31363, 4.1912, 1.70668, 7.23867, 1.11441, 5.13591, 9.65186, 4.00767,
                5.24875, 1.94852, 5.52768, 8.97121, 5.8094, 3.53329, 4.19126, 9.06652, 3.1734, 1.21496,
                9.69154, 4.86971, 4.1166, 6.19361, 2.13874, 9.55039, 3.8225, 9.57548, 2.96554, 3.2383,
                8.77422, 3.11741, 8.3359, 5.89508, 2.72134, 6.29956, 1.43323, 1.14286, 1.4474, 4.59474,
                6.19214, 8.80766, 8.07546, 3.29232, 1.74029, 2.4198, 2.88544, 4.75644, 4.12921, 7.29896,
                7.27759, 1.67252, 1.32823, 8.1046, 9.10476, 1.04197, 3.37783, 5.2064, 4.23835, 3.16196,
                1.20852, 5.78501, 2.17175, 6.05313, 2.51048, 4.78967, 7.16219, 3.4651, 1.09, 2.9788,
                1.28761, 9.41385, 8.03883, 5.65835, 1.14816, 3.6892, 5.86648, 8.73895, 2.66603, 1.75192,
                1.39845, 4.99427, 1.17387, 1.60329, 8.30594, 6.72662, 7.95565, 7.35114, 3.1439, 1.39976,
                3.53095, 8.78581, 1.65811, 6.94299, 2.68641, 5.70058, 9.13491, 5.27637, 8.6232, 8.54902,
                2.25352, 5.86274, 5.20377, 2.96815, 4.96745, 5.3225, 3.99956, 1.08021, 5.54918, 7.05833,
                1.49913, 2.41822, 6.44593, 3.87301, 9.01465, 8.11336, 2.95749, 2.80188, 7.12396, 2.40595,
                5.59325, 9.89258, 2.30223, 1.4347, 9.09158, 7.43797, 3.79295, 4.53646, 1.72705, 4.16909,
                1.00912, 6.62167, 2.80244, 6.626, 3.89307, 1.42586, 7.51028, 7.83327, 4.65674, 7.33902,
                6.26823, 9.72608, 3.73491, 3.8238, 3.03815, 7.05101, 8.0103, 5.61396, 6.53738, 1.41095,
                5.0149, 9.71211, 4.23604, 5.98629, 4.70219, 9.69442, 2.82752, 9.93544, 6.9328, 8.2817,
                5.12336, 8.98577, 5.80541, 6.19552, 9.25748, 3.82732, 7.53525, 8.24712, 5.32057, 5.38817,
                8.57269, 5.99975, 3.42893, 5.38068, 3.48261, 3.02851, 6.82079, 9.2902, 2.80427, 8.91868,
                5.19227, 7.52482, 3.72584, 5.40107, 2.83307, 1.79755, 2.49121, 5.52697, 8.08823, 9.13242,
                2.97572, 7.64318, 3.32023, 6.07788, 2.19187, 4.34879, 1.7457, 5.55154, 7.24966, 5.1128,
                4.25147, 8.34407, 1.4123, 4.49045, 5.12671, 7.62159, 9.18673, 3.49665, 8.35992, 6.90684,
                1.10152, 7.61818, 6.43145, 7.12017, 6.25564, 6.16169, 4.24916, 9.6283, 9.88249, 4.48422,
                8.52562, 9.83928, 6.26818, 7.03839, 1.77631, 9.92305, 8.0155, 9.94928, 6.88321, 1.33685,
                7.4718, 7.19305, 6.47932, 1.9559, 3.52616, 7.98593, 9.0115, 5.59539, 7.44137, 1.70001,
                6.53774, 8.54023, 7.26405, 5.99553, 8.75071, 7.70789, 3.38094, 9.99792, 6.16359, 6.75153,
                5.4073, 9.00437, 8.87059, 8.63011, 6.82951, 6.27021, 3.53425, 9.92489, 8.19695, 5.51473,
                7.95084, 2.11852, 9.28916, 1.40353, 3.05744, 8.58238, 3.75014, 5.35889, 6.85048, 2.29549,
                3.75218, 8.98228, 8.98158, 5.63695, 3.40379, 8.92309, 5.48185, 4.00095, 9.05227, 2.84035,
                8.37644, 8.54954, 5.70516, 2.45744, 9.54079, 1.53504, 8.9785, 6.1691, 4.40962, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 10},
            std::vector<int64_t>{10, 10, 10, 10, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 7.95465,
                2.37637, 6.20771, 1.08107, 7.38138, 5.23577, 7.88133, 5.20653, 3.42101, 8.48523, 5.96192,
                1.63073, 5.25228, 7.68488, 2.7276, 5.1788, 3.07327, 5.57423, 2.87711, 1.44374, 5.66976,
                2.55051, 4.56682, 1.96629, 5.58829, 1.91922, 3.59846, 3.08583, 9.70901, 3.50487, 3.1026,
                1.82401, 6.1306, 4.76134, 4.31059, 8.31695, 3.60784, 7.45652, 6.51653, 4.84219, 7.76686,
                4.85031, 4.85544, 4.25714, 2.38005, 9.43471, 9.24769, 8.03763, 6.54696, 1.32399, 6.88891,
                2.16793, 3.64924, 4.24733, 3.47181, 1.66572, 2.36923, 2.45457, 9.44841, 4.34021, 1.45016,
                7.6686, 3.68812, 2.83922, 9.83581, 9.03719, 7.83414, 6.86009, 1.35715, 8.32489, 7.86316,
                5.09754, 5.78644, 1.98402, 2.31429, 5.5791, 2.94085, 9.24799, 5.15937, 2.19041, 7.87817,
                2.9146, 1.66833, 1.85877, 2.45985, 4.20817, 1.85777, 2.28937, 9.37912, 6.18926, 8.55681,
                6.60963, 3.92066, 7.5521, 5.70463, 7.6313, 2.48866, 7.18352, 4.8413, 7.55702, 7.80702,
                4.5785, 9.3268, 2.83159, 1.07202, 9.33716, 3.6506, 2.50256, 1.21691, 5.06801, 8.27505,
                4.31539, 6.48286, 1.31363, 4.1912, 1.70668, 7.23867, 1.11441, 5.13591, 9.65186, 4.00767,
                5.24875, 1.94852, 5.52768, 8.97121, 5.8094, 3.53329, 4.19126, 9.06652, 3.1734, 1.21496,
                9.69154, 4.86971, 4.1166, 6.19361, 2.13874, 9.55039, 3.8225, 9.57548, 2.96554, 3.2383,
                8.77422, 3.11741, 8.3359, 5.89508, 2.72134, 6.29956, 1.43323, 1.14286, 1.4474, 4.59474,
                6.19214, 8.80766, 8.07546, 3.29232, 1.74029, 2.4198, 2.88544, 4.75644, 4.12921, 7.29896,
                7.27759, 1.67252, 1.32823, 8.1046, 9.10476, 1.04197, 3.37783, 5.2064, 4.23835, 3.16196,
                1.20852, 5.78501, 2.17175, 6.05313, 2.51048, 4.78967, 7.16219, 3.4651, 1.09, 2.9788,
                1.28761, 9.41385, 8.03883, 5.65835, 1.14816, 3.6892, 5.86648, 8.73895, 2.66603, 1.75192,
                1.39845, 4.99427, 1.17387, 1.60329, 8.30594, 6.72662, 7.95565, 7.35114, 3.1439, 1.39976,
                3.53095, 8.78581, 1.65811, 6.94299, 2.68641, 5.70058, 9.13491, 5.27637, 8.6232, 8.54902,
                2.25352, 5.86274, 5.20377, 2.96815, 4.96745, 5.3225, 3.99956, 1.08021, 5.54918, 7.05833,
                1.49913, 2.41822, 6.44593, 3.87301, 9.01465, 8.11336, 2.95749, 2.80188, 7.12396, 2.40595,
                5.59325, 9.89258, 2.30223, 1.4347, 9.09158, 7.43797, 3.79295, 4.53646, 1.72705, 4.16909,
                1.00912, 6.62167, 2.80244, 6.626, 3.89307, 1.42586, 7.51028, 7.83327, 4.65674, 7.33902,
                6.26823, 9.72608, 3.73491, 3.8238, 3.03815, 7.05101, 8.0103, 5.61396, 6.53738, 1.41095,
                5.0149, 9.71211, 4.23604, 5.98629, 4.70219, 9.69442, 2.82752, 9.93544, 6.9328, 8.2817,
                5.12336, 8.98577, 5.80541, 6.19552, 9.25748, 3.82732, 7.53525, 8.24712, 5.32057, 5.38817,
                8.57269, 5.99975, 3.42893, 5.38068, 3.48261, 3.02851, 6.82079, 9.2902, 2.80427, 8.91868,
                5.19227, 7.52482, 3.72584, 5.40107, 2.83307, 1.79755, 2.49121, 5.52697, 8.08823, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 7.95465,
                2.37637, 6.20771, 1.08107, 7.38138, 5.23577, 7.88133, 5.20653, 3.42101, 8.48523, 5.96192,
                1.63073, 5.25228, 7.68488, 2.7276, 5.1788, 3.07327, 5.57423, 2.87711, 1.44374, 5.66976,
                2.55051, 4.56682, 1.96629, 5.58829, 1.91922, 3.59846, 3.08583, 9.70901, 3.50487, 3.1026,
                1.82401, 6.1306, 4.76134, 4.31059, 8.31695, 3.60784, 7.45652, 6.51653, 4.84219, 7.76686,
                4.85031, 4.85544, 4.25714, 2.38005, 9.43471, 9.24769, 8.03763, 6.54696, 1.32399, 6.88891,
                2.16793, 3.64924, 4.24733, 3.47181, 1.66572, 2.36923, 2.45457, 9.44841, 4.34021, 1.45016,
                7.6686, 3.68812, 2.83922, 9.83581, 9.03719, 7.83414, 6.86009, 1.35715, 8.32489, 7.86316,
                5.09754, 5.78644, 1.98402, 2.31429, 5.5791, 2.94085, 9.24799, 5.15937, 2.19041, 7.87817,
                2.9146, 1.66833, 1.85877, 2.45985, 4.20817, 1.85777, 2.28937, 9.37912, 6.18926, 8.55681,
                6.60963, 3.92066, 7.5521, 5.70463, 7.6313, 2.48866, 7.18352, 4.8413, 7.55702, 7.80702,
                4.5785, 9.3268, 2.83159, 1.07202, 9.33716, 3.6506, 2.50256, 1.21691, 5.06801, 8.27505,
                4.31539, 6.48286, 1.31363, 4.1912, 1.70668, 7.23867, 1.11441, 5.13591, 9.65186, 4.00767,
                5.24875, 1.94852, 5.52768, 8.97121, 5.8094, 3.53329, 4.19126, 9.06652, 3.1734, 1.21496,
                9.69154, 4.86971, 4.1166, 6.19361, 2.13874, 9.55039, 3.8225, 9.57548, 2.96554, 3.2383,
                8.77422, 3.11741, 8.3359, 5.89508, 2.72134, 6.29956, 1.43323, 1.14286, 1.4474, 4.59474,
                6.19214, 8.80766, 8.07546, 3.29232, 1.74029, 2.4198, 2.88544, 4.75644, 4.12921, 7.29896,
                7.27759, 1.67252, 1.32823, 8.1046, 9.10476, 1.04197, 3.37783, 5.2064, 4.23835, 3.16196,
                1.20852, 5.78501, 2.17175, 6.05313, 2.51048, 4.78967, 7.16219, 3.4651, 1.09, 2.9788,
                1.28761, 9.41385, 8.03883, 5.65835, 1.14816, 3.6892, 5.86648, 8.73895, 2.66603, 1.75192,
                1.39845, 4.99427, 1.17387, 1.60329, 8.30594, 6.72662, 7.95565, 7.35114, 3.1439, 1.39976,
                3.53095, 8.78581, 1.65811, 6.94299, 2.68641, 5.70058, 9.13491, 5.27637, 8.6232, 8.54902,
                2.25352, 5.86274, 5.20377, 2.96815, 4.96745, 5.3225, 3.99956, 1.08021, 5.54918, 7.05833,
                1.49913, 2.41822, 6.44593, 3.87301, 9.01465, 8.11336, 2.95749, 2.80188, 7.12396, 2.40595,
                5.59325, 9.89258, 2.30223, 1.4347, 9.09158, 7.43797, 3.79295, 4.53646, 1.72705, 4.16909,
                1.00912, 6.62167, 2.80244, 6.626, 3.89307, 1.42586, 7.51028, 7.83327, 4.65674, 7.33902,
                6.26823, 9.72608, 3.73491, 3.8238, 3.03815, 7.05101, 8.0103, 5.61396, 6.53738, 1.41095,
                5.0149, 9.71211, 4.23604, 5.98629, 4.70219, 9.69442, 2.82752, 9.93544, 6.9328, 8.2817,
                5.12336, 8.98577, 5.80541, 6.19552, 9.25748, 3.82732, 7.53525, 8.24712, 5.32057, 5.38817,
                8.57269, 5.99975, 3.42893, 5.38068, 3.48261, 3.02851, 6.82079, 9.2902, 2.80427, 8.91868,
                5.19227, 7.52482, 3.72584, 5.40107, 2.83307, 1.79755, 2.49121, 5.52697, 8.08823, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 10},
            std::vector<T>{
                0.528016, 0.668187, 0.668186, 0.635471, 0.668187, 0.659096, 0.666861, 0.666715, 0.668138, 0.668186,
                0.53964, 0.668141, 0.668109, 0.619255, 0.668141, 0.647193, 0.662341, 0.661921, 0.667534, 0.66811,
                0.54692, 0.667558, 0.667297, 0.604361, 0.667564, 0.631676, 0.652518, 0.651781, 0.664541, 0.667311,
                0.551576, 0.664629, 0.663703, 0.592106, 0.664652, 0.615579, 0.638092, 0.637163, 0.656733, 0.663751,
                0.554596, 0.656917, 0.655047, 0.582718, 0.656967, 0.601233, 0.621878, 0.620939, 0.643723, 0.65514,
                0.556574, 0.643984, 0.641397, 0.575854, 0.644055, 0.589658, 0.606642, 0.605821, 0.627796, 0.641522,
                0.557878, 0.628081, 0.625301, 0.570987, 0.628158, 0.580903, 0.593915, 0.593262, 0.611954, 0.625433,
                0.558742, 0.612216, 0.609684, 0.567605, 0.612287, 0.574556, 0.584071, 0.583581, 0.598219, 0.609803,
                0.559316, 0.598435, 0.596364, 0.565285, 0.598493, 0.57008, 0.576828, 0.576475, 0.587333, 0.596461,
                0.559698, 0.587499, 0.58592, 0.563707, 0.587544, 0.56698, 0.571671, 0.571423, 0.579197, 0.585993,
                0.668182, 0.66458, 0.667903, 0.667432, 0.658361, 0.667935, 0.668185, 0.667547, 0.667307, 0.668186,
                0.66803, 0.656815, 0.666091, 0.664171, 0.646084, 0.666251, 0.668096, 0.66459, 0.663738, 0.668113,
                0.666772, 0.643839, 0.66026, 0.655973, 0.630413, 0.660667, 0.667203, 0.656835, 0.655116, 0.667328,
                0.662084, 0.627922, 0.649014, 0.642661, 0.614386, 0.649671, 0.663395, 0.643868, 0.64149, 0.663807,
                0.652065, 0.61207, 0.633798, 0.626647, 0.600233, 0.634582, 0.654454, 0.627954, 0.625399, 0.65525,
                0.637519, 0.598314, 0.617618, 0.610903, 0.588883, 0.618381, 0.640604, 0.612099, 0.609772, 0.641672,
                0.621298, 0.587406, 0.602959, 0.597357, 0.580333, 0.603611, 0.624467, 0.598338, 0.596436, 0.625592,
                0.606134, 0.57925, 0.591004, 0.586675, 0.57415, 0.591515, 0.608935, 0.587425, 0.585974, 0.609946,
                0.593511, 0.573381, 0.581898, 0.578717, 0.569797, 0.582278, 0.595758, 0.579264, 0.578207, 0.596577,
                0.583768, 0.569262, 0.575267, 0.573003, 0.566785, 0.575539, 0.58546, 0.57339, 0.572642, 0.586082,
                0.668174, 0.668159, 0.668178, 0.618792, 0.66788, 0.668183, 0.66818, 0.66818, 0.662345, 0.595566,
                0.667915, 0.667737, 0.667963, 0.603963, 0.665981, 0.668052, 0.668006, 0.668007, 0.652525, 0.585315,
                0.66615, 0.665341, 0.6664, 0.591792, 0.659985, 0.666907, 0.666636, 0.66664, 0.638101, 0.577728,
                0.660409, 0.658471, 0.661057, 0.582484, 0.648575, 0.662479, 0.661698, 0.661709, 0.621887, 0.572305,
                0.649254, 0.646247, 0.650314, 0.575687, 0.633281, 0.652764, 0.651396, 0.651414, 0.60665, 0.568515,
                0.634083, 0.630598, 0.635357, 0.57087, 0.617117, 0.638404, 0.636684, 0.636707, 0.593922, 0.565907,
                0.617895, 0.614559, 0.619142, 0.567524, 0.602533, 0.622196, 0.62046, 0.620482, 0.584076, 0.564129,
                0.603195, 0.600379, 0.604265, 0.56523, 0.59067, 0.606921, 0.605404, 0.605423, 0.576832, 0.562925,
                0.591189, 0.588995, 0.592029, 0.56367, 0.581651, 0.594139, 0.59293, 0.592946, 0.571674, 0.562114,
                0.582036, 0.580415, 0.582661, 0.562616, 0.57509, 0.584239, 0.583333, 0.583345, 0.568079, 0.561569,
                0.668139, 0.668063, 0.668139, 0.667082, 0.653793, 0.663397, 0.640434, 0.668175, 0.667092, 0.571849,
                0.667538, 0.666978, 0.667544, 0.663011, 0.639734, 0.654459, 0.624289, 0.667925, 0.663042, 0.5682,
                0.664556, 0.66269, 0.664578, 0.653734, 0.623561, 0.640611, 0.608777, 0.666203, 0.653791, 0.565691,
                0.656765, 0.653146, 0.65681, 0.639656, 0.608128, 0.624474, 0.59563, 0.660545, 0.639731, 0.563983,
                0.643768, 0.638894, 0.643833, 0.62348, 0.595107, 0.608942, 0.585363, 0.649473, 0.623558, 0.562827,
                0.627845, 0.622696, 0.627915, 0.608056, 0.584968, 0.595763, 0.577763, 0.634345, 0.608125, 0.562048,
                0.611999, 0.607362, 0.612063, 0.595049, 0.577477, 0.585464, 0.572329, 0.61815, 0.595104, 0.561524,
                0.598256, 0.594491, 0.598309, 0.584924, 0.572127, 0.577836, 0.568532, 0.603413, 0.584966, 0.561173,
                0.587362, 0.584504, 0.587403, 0.577445, 0.568392, 0.572381, 0.565918, 0.591359, 0.577475, 0.560938,
                0.579218, 0.577141, 0.579248, 0.572105, 0.565823, 0.568568, 0.564137, 0.582163, 0.572127, 0.560781,
                0.668102, 0.668132, 0.66388, 0.667456, 0.657447, 0.606385, 0.667634, 0.620685, 0.668185, 0.668187,
                0.667244, 0.667485, 0.655394, 0.664256, 0.644744, 0.59371, 0.664921, 0.6056, 0.668088, 0.668142,
                0.663529, 0.664358, 0.641868, 0.656146, 0.628916, 0.583917, 0.65754, 0.593086, 0.667146, 0.667567,
                0.654712, 0.656356, 0.625799, 0.642901, 0.612988, 0.576717, 0.644878, 0.583449, 0.66321, 0.664664,
                0.640947, 0.643193, 0.610134, 0.626905, 0.599072, 0.571593, 0.629065, 0.57638, 0.654104, 0.656992,
                0.624826, 0.62722, 0.59673, 0.611138, 0.587988, 0.568023, 0.613126, 0.571356, 0.640142, 0.644091,
                0.609258, 0.611426, 0.586197, 0.59755, 0.579676, 0.56557, 0.599186, 0.567859, 0.623984, 0.628198,
                0.596018, 0.597785, 0.578369, 0.586822, 0.573683, 0.563901, 0.588076, 0.565458, 0.608505, 0.612324,
                0.585658, 0.587002, 0.572757, 0.578824, 0.569471, 0.562771, 0.57974, 0.563825, 0.59541, 0.598524,
                0.577977, 0.578955, 0.568828, 0.573079, 0.566562, 0.562011, 0.573728, 0.56272, 0.585197, 0.587567},
            std::vector<T>{
                0.559698, 0.587499, 0.58592, 0.563707, 0.587544, 0.56698, 0.571671, 0.571423, 0.579197, 0.585993,
                0.583768, 0.569262, 0.575267, 0.573003, 0.566785, 0.575539, 0.58546, 0.57339, 0.572642, 0.586082,
                0.582036, 0.580415, 0.582661, 0.562616, 0.57509, 0.584239, 0.583333, 0.583345, 0.568079, 0.561569,
                0.579218, 0.577141, 0.579248, 0.572105, 0.565823, 0.568568, 0.564137, 0.582163, 0.572127, 0.560781,
                0.577977, 0.578955, 0.568828, 0.573079, 0.566562, 0.562011, 0.573728, 0.56272, 0.585197, 0.587567},
            std::vector<T>{
                1.2132, 1.37242, 1.3621, 1.23365, 1.37271, 1.25089, 1.27652, 1.27513, 1.32014, 1.36258,
                1.34833, 1.26322, 1.29695, 1.284, 1.24985, 1.29853, 1.35913, 1.2862, 1.28197, 1.36315,
                1.33748, 1.32752, 1.34137, 1.22801, 1.29593, 1.35132, 1.34559, 1.34566, 1.25679, 1.22266,
                1.32026, 1.30789, 1.32044, 1.27895, 1.24474, 1.25944, 1.23589, 1.33827, 1.27907, 1.21865,
                1.31284, 1.31868, 1.26086, 1.28443, 1.24866, 1.22491, 1.28812, 1.22855, 1.35744, 1.37287}),
        LSTMSequenceParams(
            5, 10, 10, 10,
            0.7f, op::RecurrentSequenceDirection::REVERSE,
            ET,
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 7.95465,
                2.37637, 6.20771, 1.08107, 7.38138, 5.23577, 7.88133, 5.20653, 3.42101, 8.48523, 5.96192,
                1.63073, 5.25228, 7.68488, 2.7276, 5.1788, 3.07327, 5.57423, 2.87711, 1.44374, 5.66976,
                2.55051, 4.56682, 1.96629, 5.58829, 1.91922, 3.59846, 3.08583, 9.70901, 3.50487, 3.1026,
                1.82401, 6.1306, 4.76134, 4.31059, 8.31695, 3.60784, 7.45652, 6.51653, 4.84219, 7.76686,
                4.85031, 4.85544, 4.25714, 2.38005, 9.43471, 9.24769, 8.03763, 6.54696, 1.32399, 6.88891,
                2.16793, 3.64924, 4.24733, 3.47181, 1.66572, 2.36923, 2.45457, 9.44841, 4.34021, 1.45016,
                7.6686, 3.68812, 2.83922, 9.83581, 9.03719, 7.83414, 6.86009, 1.35715, 8.32489, 7.86316,
                5.09754, 5.78644, 1.98402, 2.31429, 5.5791, 2.94085, 9.24799, 5.15937, 2.19041, 7.87817,
                2.9146, 1.66833, 1.85877, 2.45985, 4.20817, 1.85777, 2.28937, 9.37912, 6.18926, 8.55681,
                6.60963, 3.92066, 7.5521, 5.70463, 7.6313, 2.48866, 7.18352, 4.8413, 7.55702, 7.80702,
                4.5785, 9.3268, 2.83159, 1.07202, 9.33716, 3.6506, 2.50256, 1.21691, 5.06801, 8.27505,
                4.31539, 6.48286, 1.31363, 4.1912, 1.70668, 7.23867, 1.11441, 5.13591, 9.65186, 4.00767,
                5.24875, 1.94852, 5.52768, 8.97121, 5.8094, 3.53329, 4.19126, 9.06652, 3.1734, 1.21496,
                9.69154, 4.86971, 4.1166, 6.19361, 2.13874, 9.55039, 3.8225, 9.57548, 2.96554, 3.2383,
                8.77422, 3.11741, 8.3359, 5.89508, 2.72134, 6.29956, 1.43323, 1.14286, 1.4474, 4.59474,
                6.19214, 8.80766, 8.07546, 3.29232, 1.74029, 2.4198, 2.88544, 4.75644, 4.12921, 7.29896,
                7.27759, 1.67252, 1.32823, 8.1046, 9.10476, 1.04197, 3.37783, 5.2064, 4.23835, 3.16196,
                1.20852, 5.78501, 2.17175, 6.05313, 2.51048, 4.78967, 7.16219, 3.4651, 1.09, 2.9788,
                1.28761, 9.41385, 8.03883, 5.65835, 1.14816, 3.6892, 5.86648, 8.73895, 2.66603, 1.75192,
                1.39845, 4.99427, 1.17387, 1.60329, 8.30594, 6.72662, 7.95565, 7.35114, 3.1439, 1.39976,
                3.53095, 8.78581, 1.65811, 6.94299, 2.68641, 5.70058, 9.13491, 5.27637, 8.6232, 8.54902,
                2.25352, 5.86274, 5.20377, 2.96815, 4.96745, 5.3225, 3.99956, 1.08021, 5.54918, 7.05833,
                1.49913, 2.41822, 6.44593, 3.87301, 9.01465, 8.11336, 2.95749, 2.80188, 7.12396, 2.40595,
                5.59325, 9.89258, 2.30223, 1.4347, 9.09158, 7.43797, 3.79295, 4.53646, 1.72705, 4.16909,
                1.00912, 6.62167, 2.80244, 6.626, 3.89307, 1.42586, 7.51028, 7.83327, 4.65674, 7.33902,
                6.26823, 9.72608, 3.73491, 3.8238, 3.03815, 7.05101, 8.0103, 5.61396, 6.53738, 1.41095,
                5.0149, 9.71211, 4.23604, 5.98629, 4.70219, 9.69442, 2.82752, 9.93544, 6.9328, 8.2817,
                5.12336, 8.98577, 5.80541, 6.19552, 9.25748, 3.82732, 7.53525, 8.24712, 5.32057, 5.38817,
                8.57269, 5.99975, 3.42893, 5.38068, 3.48261, 3.02851, 6.82079, 9.2902, 2.80427, 8.91868,
                5.19227, 7.52482, 3.72584, 5.40107, 2.83307, 1.79755, 2.49121, 5.52697, 8.08823, 9.13242,
                2.97572, 7.64318, 3.32023, 6.07788, 2.19187, 4.34879, 1.7457, 5.55154, 7.24966, 5.1128,
                4.25147, 8.34407, 1.4123, 4.49045, 5.12671, 7.62159, 9.18673, 3.49665, 8.35992, 6.90684,
                1.10152, 7.61818, 6.43145, 7.12017, 6.25564, 6.16169, 4.24916, 9.6283, 9.88249, 4.48422,
                8.52562, 9.83928, 6.26818, 7.03839, 1.77631, 9.92305, 8.0155, 9.94928, 6.88321, 1.33685,
                7.4718, 7.19305, 6.47932, 1.9559, 3.52616, 7.98593, 9.0115, 5.59539, 7.44137, 1.70001,
                6.53774, 8.54023, 7.26405, 5.99553, 8.75071, 7.70789, 3.38094, 9.99792, 6.16359, 6.75153,
                5.4073, 9.00437, 8.87059, 8.63011, 6.82951, 6.27021, 3.53425, 9.92489, 8.19695, 5.51473,
                7.95084, 2.11852, 9.28916, 1.40353, 3.05744, 8.58238, 3.75014, 5.35889, 6.85048, 2.29549,
                3.75218, 8.98228, 8.98158, 5.63695, 3.40379, 8.92309, 5.48185, 4.00095, 9.05227, 2.84035,
                8.37644, 8.54954, 5.70516, 2.45744, 9.54079, 1.53504, 8.9785, 6.1691, 4.40962, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 10},
            std::vector<int64_t>{10, 10, 10, 10, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 7.95465,
                2.37637, 6.20771, 1.08107, 7.38138, 5.23577, 7.88133, 5.20653, 3.42101, 8.48523, 5.96192,
                1.63073, 5.25228, 7.68488, 2.7276, 5.1788, 3.07327, 5.57423, 2.87711, 1.44374, 5.66976,
                2.55051, 4.56682, 1.96629, 5.58829, 1.91922, 3.59846, 3.08583, 9.70901, 3.50487, 3.1026,
                1.82401, 6.1306, 4.76134, 4.31059, 8.31695, 3.60784, 7.45652, 6.51653, 4.84219, 7.76686,
                4.85031, 4.85544, 4.25714, 2.38005, 9.43471, 9.24769, 8.03763, 6.54696, 1.32399, 6.88891,
                2.16793, 3.64924, 4.24733, 3.47181, 1.66572, 2.36923, 2.45457, 9.44841, 4.34021, 1.45016,
                7.6686, 3.68812, 2.83922, 9.83581, 9.03719, 7.83414, 6.86009, 1.35715, 8.32489, 7.86316,
                5.09754, 5.78644, 1.98402, 2.31429, 5.5791, 2.94085, 9.24799, 5.15937, 2.19041, 7.87817,
                2.9146, 1.66833, 1.85877, 2.45985, 4.20817, 1.85777, 2.28937, 9.37912, 6.18926, 8.55681,
                6.60963, 3.92066, 7.5521, 5.70463, 7.6313, 2.48866, 7.18352, 4.8413, 7.55702, 7.80702,
                4.5785, 9.3268, 2.83159, 1.07202, 9.33716, 3.6506, 2.50256, 1.21691, 5.06801, 8.27505,
                4.31539, 6.48286, 1.31363, 4.1912, 1.70668, 7.23867, 1.11441, 5.13591, 9.65186, 4.00767,
                5.24875, 1.94852, 5.52768, 8.97121, 5.8094, 3.53329, 4.19126, 9.06652, 3.1734, 1.21496,
                9.69154, 4.86971, 4.1166, 6.19361, 2.13874, 9.55039, 3.8225, 9.57548, 2.96554, 3.2383,
                8.77422, 3.11741, 8.3359, 5.89508, 2.72134, 6.29956, 1.43323, 1.14286, 1.4474, 4.59474,
                6.19214, 8.80766, 8.07546, 3.29232, 1.74029, 2.4198, 2.88544, 4.75644, 4.12921, 7.29896,
                7.27759, 1.67252, 1.32823, 8.1046, 9.10476, 1.04197, 3.37783, 5.2064, 4.23835, 3.16196,
                1.20852, 5.78501, 2.17175, 6.05313, 2.51048, 4.78967, 7.16219, 3.4651, 1.09, 2.9788,
                1.28761, 9.41385, 8.03883, 5.65835, 1.14816, 3.6892, 5.86648, 8.73895, 2.66603, 1.75192,
                1.39845, 4.99427, 1.17387, 1.60329, 8.30594, 6.72662, 7.95565, 7.35114, 3.1439, 1.39976,
                3.53095, 8.78581, 1.65811, 6.94299, 2.68641, 5.70058, 9.13491, 5.27637, 8.6232, 8.54902,
                2.25352, 5.86274, 5.20377, 2.96815, 4.96745, 5.3225, 3.99956, 1.08021, 5.54918, 7.05833,
                1.49913, 2.41822, 6.44593, 3.87301, 9.01465, 8.11336, 2.95749, 2.80188, 7.12396, 2.40595,
                5.59325, 9.89258, 2.30223, 1.4347, 9.09158, 7.43797, 3.79295, 4.53646, 1.72705, 4.16909,
                1.00912, 6.62167, 2.80244, 6.626, 3.89307, 1.42586, 7.51028, 7.83327, 4.65674, 7.33902,
                6.26823, 9.72608, 3.73491, 3.8238, 3.03815, 7.05101, 8.0103, 5.61396, 6.53738, 1.41095,
                5.0149, 9.71211, 4.23604, 5.98629, 4.70219, 9.69442, 2.82752, 9.93544, 6.9328, 8.2817,
                5.12336, 8.98577, 5.80541, 6.19552, 9.25748, 3.82732, 7.53525, 8.24712, 5.32057, 5.38817,
                8.57269, 5.99975, 3.42893, 5.38068, 3.48261, 3.02851, 6.82079, 9.2902, 2.80427, 8.91868,
                5.19227, 7.52482, 3.72584, 5.40107, 2.83307, 1.79755, 2.49121, 5.52697, 8.08823, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 7.95465,
                2.37637, 6.20771, 1.08107, 7.38138, 5.23577, 7.88133, 5.20653, 3.42101, 8.48523, 5.96192,
                1.63073, 5.25228, 7.68488, 2.7276, 5.1788, 3.07327, 5.57423, 2.87711, 1.44374, 5.66976,
                2.55051, 4.56682, 1.96629, 5.58829, 1.91922, 3.59846, 3.08583, 9.70901, 3.50487, 3.1026,
                1.82401, 6.1306, 4.76134, 4.31059, 8.31695, 3.60784, 7.45652, 6.51653, 4.84219, 7.76686,
                4.85031, 4.85544, 4.25714, 2.38005, 9.43471, 9.24769, 8.03763, 6.54696, 1.32399, 6.88891,
                2.16793, 3.64924, 4.24733, 3.47181, 1.66572, 2.36923, 2.45457, 9.44841, 4.34021, 1.45016,
                7.6686, 3.68812, 2.83922, 9.83581, 9.03719, 7.83414, 6.86009, 1.35715, 8.32489, 7.86316,
                5.09754, 5.78644, 1.98402, 2.31429, 5.5791, 2.94085, 9.24799, 5.15937, 2.19041, 7.87817,
                2.9146, 1.66833, 1.85877, 2.45985, 4.20817, 1.85777, 2.28937, 9.37912, 6.18926, 8.55681,
                6.60963, 3.92066, 7.5521, 5.70463, 7.6313, 2.48866, 7.18352, 4.8413, 7.55702, 7.80702,
                4.5785, 9.3268, 2.83159, 1.07202, 9.33716, 3.6506, 2.50256, 1.21691, 5.06801, 8.27505,
                4.31539, 6.48286, 1.31363, 4.1912, 1.70668, 7.23867, 1.11441, 5.13591, 9.65186, 4.00767,
                5.24875, 1.94852, 5.52768, 8.97121, 5.8094, 3.53329, 4.19126, 9.06652, 3.1734, 1.21496,
                9.69154, 4.86971, 4.1166, 6.19361, 2.13874, 9.55039, 3.8225, 9.57548, 2.96554, 3.2383,
                8.77422, 3.11741, 8.3359, 5.89508, 2.72134, 6.29956, 1.43323, 1.14286, 1.4474, 4.59474,
                6.19214, 8.80766, 8.07546, 3.29232, 1.74029, 2.4198, 2.88544, 4.75644, 4.12921, 7.29896,
                7.27759, 1.67252, 1.32823, 8.1046, 9.10476, 1.04197, 3.37783, 5.2064, 4.23835, 3.16196,
                1.20852, 5.78501, 2.17175, 6.05313, 2.51048, 4.78967, 7.16219, 3.4651, 1.09, 2.9788,
                1.28761, 9.41385, 8.03883, 5.65835, 1.14816, 3.6892, 5.86648, 8.73895, 2.66603, 1.75192,
                1.39845, 4.99427, 1.17387, 1.60329, 8.30594, 6.72662, 7.95565, 7.35114, 3.1439, 1.39976,
                3.53095, 8.78581, 1.65811, 6.94299, 2.68641, 5.70058, 9.13491, 5.27637, 8.6232, 8.54902,
                2.25352, 5.86274, 5.20377, 2.96815, 4.96745, 5.3225, 3.99956, 1.08021, 5.54918, 7.05833,
                1.49913, 2.41822, 6.44593, 3.87301, 9.01465, 8.11336, 2.95749, 2.80188, 7.12396, 2.40595,
                5.59325, 9.89258, 2.30223, 1.4347, 9.09158, 7.43797, 3.79295, 4.53646, 1.72705, 4.16909,
                1.00912, 6.62167, 2.80244, 6.626, 3.89307, 1.42586, 7.51028, 7.83327, 4.65674, 7.33902,
                6.26823, 9.72608, 3.73491, 3.8238, 3.03815, 7.05101, 8.0103, 5.61396, 6.53738, 1.41095,
                5.0149, 9.71211, 4.23604, 5.98629, 4.70219, 9.69442, 2.82752, 9.93544, 6.9328, 8.2817,
                5.12336, 8.98577, 5.80541, 6.19552, 9.25748, 3.82732, 7.53525, 8.24712, 5.32057, 5.38817,
                8.57269, 5.99975, 3.42893, 5.38068, 3.48261, 3.02851, 6.82079, 9.2902, 2.80427, 8.91868,
                5.19227, 7.52482, 3.72584, 5.40107, 2.83307, 1.79755, 2.49121, 5.52697, 8.08823, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 10},
            std::vector<T>{
                0.559698, 0.587499, 0.58592, 0.563707, 0.587544, 0.56698, 0.571671, 0.571423, 0.579197, 0.585993,
                0.559316, 0.598435, 0.596364, 0.565285, 0.598493, 0.57008, 0.576828, 0.576475, 0.587333, 0.596461,
                0.558742, 0.612216, 0.609684, 0.567605, 0.612287, 0.574556, 0.584071, 0.583581, 0.598219, 0.609803,
                0.557878, 0.628081, 0.625301, 0.570987, 0.628158, 0.580903, 0.593915, 0.593262, 0.611954, 0.625433,
                0.556574, 0.643984, 0.641397, 0.575854, 0.644055, 0.589658, 0.606642, 0.605821, 0.627796, 0.641522,
                0.554596, 0.656917, 0.655047, 0.582718, 0.656967, 0.601233, 0.621878, 0.620939, 0.643723, 0.65514,
                0.551576, 0.664629, 0.663703, 0.592106, 0.664652, 0.615579, 0.638092, 0.637163, 0.656733, 0.663751,
                0.54692, 0.667558, 0.667297, 0.604361, 0.667564, 0.631676, 0.652518, 0.651781, 0.664541, 0.667311,
                0.53964, 0.668141, 0.668109, 0.619255, 0.668141, 0.647193, 0.662341, 0.661921, 0.667534, 0.66811,
                0.528016, 0.668187, 0.668186, 0.635471, 0.668187, 0.659096, 0.666861, 0.666715, 0.668138, 0.668186,
                0.583768, 0.569262, 0.575267, 0.573003, 0.566785, 0.575539, 0.58546, 0.57339, 0.572642, 0.586082,
                0.593511, 0.573381, 0.581898, 0.578717, 0.569797, 0.582278, 0.595758, 0.579264, 0.578207, 0.596577,
                0.606134, 0.57925, 0.591004, 0.586675, 0.57415, 0.591515, 0.608935, 0.587425, 0.585974, 0.609946,
                0.621298, 0.587406, 0.602959, 0.597357, 0.580333, 0.603611, 0.624467, 0.598338, 0.596436, 0.625592,
                0.637519, 0.598314, 0.617618, 0.610903, 0.588883, 0.618381, 0.640604, 0.612099, 0.609772, 0.641672,
                0.652065, 0.61207, 0.633798, 0.626647, 0.600233, 0.634582, 0.654454, 0.627954, 0.625399, 0.65525,
                0.662084, 0.627922, 0.649014, 0.642661, 0.614386, 0.649671, 0.663395, 0.643868, 0.64149, 0.663807,
                0.666772, 0.643839, 0.66026, 0.655973, 0.630413, 0.660667, 0.667203, 0.656835, 0.655116, 0.667328,
                0.66803, 0.656815, 0.666091, 0.664171, 0.646084, 0.666251, 0.668096, 0.66459, 0.663738, 0.668113,
                0.668182, 0.66458, 0.667903, 0.667432, 0.658361, 0.667935, 0.668185, 0.667547, 0.667307, 0.668186,
                0.582036, 0.580415, 0.582661, 0.562616, 0.57509, 0.584239, 0.583333, 0.583345, 0.568079, 0.561569,
                0.591189, 0.588995, 0.592029, 0.56367, 0.581651, 0.594139, 0.59293, 0.592946, 0.571674, 0.562114,
                0.603195, 0.600379, 0.604265, 0.56523, 0.59067, 0.606921, 0.605404, 0.605423, 0.576832, 0.562925,
                0.617895, 0.614559, 0.619142, 0.567524, 0.602533, 0.622196, 0.62046, 0.620482, 0.584076, 0.564129,
                0.634083, 0.630598, 0.635357, 0.57087, 0.617117, 0.638404, 0.636684, 0.636707, 0.593922, 0.565907,
                0.649254, 0.646247, 0.650314, 0.575687, 0.633281, 0.652764, 0.651396, 0.651414, 0.60665, 0.568515,
                0.660409, 0.658471, 0.661057, 0.582484, 0.648575, 0.662479, 0.661698, 0.661709, 0.621887, 0.572305,
                0.66615, 0.665341, 0.6664, 0.591792, 0.659985, 0.666907, 0.666636, 0.66664, 0.638101, 0.577728,
                0.667915, 0.667737, 0.667963, 0.603963, 0.665981, 0.668052, 0.668006, 0.668007, 0.652525, 0.585315,
                0.668174, 0.668159, 0.668178, 0.618792, 0.66788, 0.668183, 0.66818, 0.66818, 0.662345, 0.595566,
                0.579218, 0.577141, 0.579248, 0.572105, 0.565823, 0.568568, 0.564137, 0.582163, 0.572127, 0.560781,
                0.587362, 0.584504, 0.587403, 0.577445, 0.568392, 0.572381, 0.565918, 0.591359, 0.577475, 0.560938,
                0.598256, 0.594491, 0.598309, 0.584924, 0.572127, 0.577836, 0.568532, 0.603413, 0.584966, 0.561173,
                0.611999, 0.607362, 0.612063, 0.595049, 0.577477, 0.585464, 0.572329, 0.61815, 0.595104, 0.561524,
                0.627845, 0.622696, 0.627915, 0.608056, 0.584968, 0.595763, 0.577763, 0.634345, 0.608125, 0.562048,
                0.643768, 0.638894, 0.643833, 0.62348, 0.595107, 0.608942, 0.585363, 0.649473, 0.623558, 0.562827,
                0.656765, 0.653146, 0.65681, 0.639656, 0.608128, 0.624474, 0.59563, 0.660545, 0.639731, 0.563983,
                0.664556, 0.66269, 0.664578, 0.653734, 0.623561, 0.640611, 0.608777, 0.666203, 0.653791, 0.565691,
                0.667538, 0.666978, 0.667544, 0.663011, 0.639734, 0.654459, 0.624289, 0.667925, 0.663042, 0.5682,
                0.668139, 0.668063, 0.668139, 0.667082, 0.653793, 0.663397, 0.640434, 0.668175, 0.667092, 0.571849,
                0.577977, 0.578955, 0.568828, 0.573079, 0.566562, 0.562011, 0.573728, 0.56272, 0.585197, 0.587567,
                0.585658, 0.587002, 0.572757, 0.578824, 0.569471, 0.562771, 0.57974, 0.563825, 0.59541, 0.598524,
                0.596018, 0.597785, 0.578369, 0.586822, 0.573683, 0.563901, 0.588076, 0.565458, 0.608505, 0.612324,
                0.609258, 0.611426, 0.586197, 0.59755, 0.579676, 0.56557, 0.599186, 0.567859, 0.623984, 0.628198,
                0.624826, 0.62722, 0.59673, 0.611138, 0.587988, 0.568023, 0.613126, 0.571356, 0.640142, 0.644091,
                0.640947, 0.643193, 0.610134, 0.626905, 0.599072, 0.571593, 0.629065, 0.57638, 0.654104, 0.656992,
                0.654712, 0.656356, 0.625799, 0.642901, 0.612988, 0.576717, 0.644878, 0.583449, 0.66321, 0.664664,
                0.663529, 0.664358, 0.641868, 0.656146, 0.628916, 0.583917, 0.65754, 0.593086, 0.667146, 0.667567,
                0.667244, 0.667485, 0.655394, 0.664256, 0.644744, 0.59371, 0.664921, 0.6056, 0.668088, 0.668142,
                0.668102, 0.668132, 0.66388, 0.667456, 0.657447, 0.606385, 0.667634, 0.620685, 0.668185, 0.668187},
            std::vector<T>{
                0.559698, 0.587499, 0.58592, 0.563707, 0.587544, 0.56698, 0.571671, 0.571423, 0.579197, 0.585993,
                0.583768, 0.569262, 0.575267, 0.573003, 0.566785, 0.575539, 0.58546, 0.57339, 0.572642, 0.586082,
                0.582036, 0.580415, 0.582661, 0.562616, 0.57509, 0.584239, 0.583333, 0.583345, 0.568079, 0.561569,
                0.579218, 0.577141, 0.579248, 0.572105, 0.565823, 0.568568, 0.564137, 0.582163, 0.572127, 0.560781,
                0.577977, 0.578955, 0.568828, 0.573079, 0.566562, 0.562011, 0.573728, 0.56272, 0.585197, 0.587567},
            std::vector<T>{
                1.2132, 1.37242, 1.3621, 1.23365, 1.37271, 1.25089, 1.27652, 1.27513, 1.32014, 1.36258,
                1.34833, 1.26322, 1.29695, 1.284, 1.24985, 1.29853, 1.35913, 1.2862, 1.28197, 1.36315,
                1.33748, 1.32752, 1.34137, 1.22801, 1.29593, 1.35132, 1.34559, 1.34566, 1.25679, 1.22266,
                1.32026, 1.30789, 1.32044, 1.27895, 1.24474, 1.25944, 1.23589, 1.33827, 1.27907, 1.21865,
                1.31284, 1.31868, 1.26086, 1.28443, 1.24866, 1.22491, 1.28812, 1.22855, 1.35744, 1.37287}),
        LSTMSequenceParams(
            5, 10, 10, 10,
            0.f, op::RecurrentSequenceDirection::BIDIRECTIONAL,
            ET,
            std::vector<T>{
                -5, 3.90155, -3.69293, -4.60241, 3.26436, 0.320779, 4.5631, -0.380737, 4.23738, -1.26079,
                -3.45027, 3.92344, -4.7321, -2.08498, -1.01256, 3.07289, 1.27094, 4.07925, 0.563973, 3.39919,
                -4.49512, 3.06235, 4.30816, -1.3597, 1.90948, -3.70685, 3.32686, -1.81646, 2.37202, 0.967696,
                -1.36293, 2.94971, 1.98481, 3.21879, 3.99466, 3.31899, 4.09958, 4.74914, 1.5612, 3.1199,
                -3.97256, -2.62472, -1.20655, 1.99236, -0.150548, -0.847778, 1.38662, 2.5929, 0.575229, 1.93455,
                4.13259, -4.75387, 3.37378, -1.16795, -2.01229, -4.93703, -0.624426, 2.37937, -1.2424, -0.0675053,
                -4.8596, -2.50563, -1.52865, 2.96687, 4.38399, -3.99478, 2.35403, 4.76437, 2.05886, 4.51765,
                4.27848, -0.899691, -1.61905, 2.69108, -1.65352, -4.17313, 3.98558, 2.87335, 2.17466, -0.560098,
                2.50892, 1.57898, -1.39303, -1.21677, 1.49632, -4.44363, -4.21741, -4.08097, 3.05105, -4.34693,
                -2.75681, -3.47867, 1.80591, -3.74041, -2.42642, -0.620324, -3.90332, -1.85434, -1.49268, 2.61143,
                -4.01149, 4.3146, -1.73921, -0.733643, -2.29319, 2.8811, -0.46676, -4.16764, -4.29623, -1.70251,
                -3.07895, 3.37865, -4.38315, 0.542347, 0.706112, -0.378897, 4.48829, -1.55226, 3.84449, -0.689599,
                4.9448, 3.00503, 4.41634, -1.2052, -0.945752, -2.13112, -0.494659, 2.65111, 0.798992, -1.21877,
                1.22393, 2.39173, 4.13383, 0.321036, 0.178213, -3.97413, -3.7833, 1.43697, -0.540356, -4.23675,
                -3.47431, -4.20492, 3.79371, -3.38526, 4.22951, -4.96421, 3.02602, -1.96467, -2.55246, 2.68211,
                2.03482, 3.86176, 4.13142, -2.74689, -0.785165, -3.34138, -2.89165, 3.39898, 4.1545, 1.81591,
                -0.907431, -2.77872, 3.353, 2.76737, 2.37346, 3.58702, 4.72511, -0.900406, 0.97014, 2.38442,
                -2.3329, 2.63133, 4.30503, 3.74689, -3.83837, 4.67289, -1.97013, 1.09016, -3.32243, 3.58426,
                2.26392, -4.34882, 0.180985, 0.954856, -0.420779, 0.922418, -1.80545, 3.19832, -4.07261, -3.2313,
                -2.79288, -0.259067, 4.17452, 4.91536, 1.04214, 4.20558, -1.33364, -4.92198, 1.55017, -3.31947,
                -0.0354189, 2.90804, 3.17961, -1.33237, -3.71257, -3.02883, -4.79104, 1.33544, -0.213819, -2.87557,
                -2.34522, -3.11141, 3.8981, 2.16143, -4.31427, -2.54362, 4.20355, -2.95095, -0.341185, 2.63071,
                0.323663, 0.510221, -2.64869, -2.24021, -4.83154, -0.775629, -4.53002, -4.408, -1.81016, -0.569383,
                -3.31418, -1.98672, -0.505968, -0.269187, 2.04221, 2.0064, 0.936495, 4.72935, 1.62178, 3.77815,
                -0.856647, 2.98075, -3.62022, -2.38874, -2.73203, 0.32452, 1.94716, -2.56023, -3.04901, 2.90142,
                -0.946198, -0.649191, 1.26384, 1.78817, -0.685465, -2.04822, 4.97285, -3.02725, 0.579597, -0.268873,
                -4.11244, -1.8748, 0.986787, -4.50772, -1.32017, -3.71768, -4.74007, 2.32116, -2.36806, 2.1148,
                1.77435, 3.20644, 2.91006, 1.0576, 0.040031, -0.693013, -3.99159, -0.134746, -0.996052, 0.871714,
                -0.558411, -2.66077, 1.34212, -2.70768, -2.48549, 4.78571, -0.533973, -4.19112, -3.80203, -0.372218,
                2.78486, -1.01506, 1.24482, 2.98445, -0.054502, -4.6262, 2.19206, 1.93343, -1.84538, 3.36576,
                -4.59602, 1.29602, -4.78253, 2.41768, -1.03034, 0.195816, -0.357983, 1.74115, 3.50352, -1.8395,
                -1.84175, -3.3264, 1.84545, 3.18232, -3.17197, -4.65516, 0.099389, 2.78295, 1.16561, 4.40189,
                -2.72039, 2.82907, -0.886233, 1.94296, -2.49201, 0.0732732, -2.1904, -2.81447, 4.23764, 2.78202,
                -2.40743, -2.72875, 2.93405, -3.2296, -2.05866, 3.17168, -0.537993, 3.49845, 4.39583, -3.02873,
                4.25922, -0.988387, -0.319124, -2.75173, 2.13604, 1.40662, 4.13165, -0.0455708, -2.61541, 1.80269,
                1.47976, 2.68583, 1.17727, 2.07096, 1.70281, -0.164106, 4.14009, -4.72156, 4.03386, -0.540462,
                0.445455, -4.80477, 3.62873, 1.34778, -0.00214738, -1.15734, -4.95873, 2.35589, -2.48324, 4.71877,
                -4.07758, -0.347755, 1.31999, -1.13152, -3.42124, -0.213052, 0.381509, 4.51867, 1.8423, -0.422192,
                -0.373053, -3.20722, 4.84911, 4.98759, 1.53754, 3.11106, -1.74677, -4.38349, -3.607, -2.97331,
                -4.59993, 2.04576, -3.01871, -2.4555, 3.51632, 1.57353, 3.06249, -1.84659, 3.07473, 3.85288,
                1.85531, 2.96839, 3.75157, -1.82426, 3.49344, 3.04875, 0.71311, -3.9376, 2.54977, 2.07647,
                -3.20233, 3.50968, 3.24609, 4.12168, 3.55772, -2.85005, -1.62315, -4.60564, -2.6492, -3.75128,
                -2.22564, -0.612792, 3.59931, -1.77079, 4.87547, 4.65284, 1.9526, 1.70281, -4.63869, 2.85457,
                0.288541, 1.72943, -0.736331, -4.60023, 1.45358, 0.686768, 2.04933, 2.03743, -2.74135, 4.84682,
                -3.81892, -2.94531, 4.53125, 4.34582, -1.61202, 4.303, -3.96169, -3.74095, -0.587597, 1.23201,
                1.07801, -3.29982, -1.67119, -3.5092, 4.58204, -4.3539, -3.8294, -0.393197, -1.27755, 4.51992,
                0.799879, -3.65947, -4.4236, 1.55178, 2.10673, -0.952616, 1.60253, 3.44643, 4.88377, 4.55128,
                2.33666, -2.0583, 2.05126, 0.00725121, -1.64708, 0.236305, 3.96222, 4.37011, -0.541235, 2.26144,
                3.06963, -3.04004, 0.46284, -3.50942, -3.12654, -1.16702, -4.42878, 1.78286, 3.87648, -1.56109,
                -2.42598, -3.80421, -0.981378, -1.63223, -4.18565, 1.30126, 0.627183, -4.24035, -4.25213, 5},
            std::vector<T>{
                -5, 0.0972095, -3.16646, 3.08769, 4.37785, -4.70394, 2.84984, -3.36123, -0.661491, 0.742674,
                -2.82977, 3.30679, -2.0046, -1.39169, 3.69357, -4.87822, 1.75331, -0.187543, 4.77267, -2.89035,
                -2.86541, 4.43989, -1.73837, 4.86129, -0.626666, 4.65384, -2.42069, 2.59966, -3.07773, -2.17051,
                2.69538, -4.91818, -1.47175, -2.66442, -1.14894, -3.90875, -0.92652, -0.946207, 2.70024, 3.4187,
                -1.06892, -0.336707, -3.95977, -3.09231, 1.72681, -0.253682, 3.01543, -2.52343, 2.41461, 1.73951,
                1.55908, -2.88157, 0.895772, 2.83656, -0.0438242, -3.9921, 3.78771, 1.16734, 1.89495, 4.71639,
                -0.788449, -4.285, -3.09649, -2.73812, -4.50129, -3.22373, 0.570888, 4.57888, 4.49166, -0.0809164,
                -3.61267, -2.40318, 0.749356, 4.85626, -3.93755, -0.299189, -4.53363, -2.2065, 0.998577, -4.88086,
                -4.95235, -4.83644, -0.938169, -3.2432, -3.98636, -0.724817, 1.77318, 0.215689, -1.7268, 0.475434,
                2.16854, -0.953869, 0.724856, 3.98419, -2.91751, 3.49012, -2.84873, -3.42912, -1.53297, 5},
            std::vector<T>{
                -5, -2.4224, 0.545224, 2.07068, -0.174741, 4.84283, 0.469625, -1.36484, -2.49666, 1.16077,
                -2.47798, 1.30443, 4.90657, -4.86712, 2.60958, -3.6434, 4.98959, 2.83648, 3.12445, 1.24863,
                -3.56188, 1.18263, 4.00651, -4.49285, -3.35389, 3.84102, -0.692219, 4.58475, -2.85036, -2.88143,
                3.93052, 2.39646, -2.43019, -1.74484, 1.43614, 4.831, 1.03788, 0.298718, -0.686032, -0.304259,
                -2.28994, -3.89952, -1.95116, -4.86558, 1.41202, -4.93245, -2.05543, 2.04177, -4.60741, -1.50938,
                3.88719, 1.83697, -4.37124, 1.66286, 1.39415, 3.97862, -3.03259, -0.518552, 0.380681, 4.66685,
                4.75549, -1.81382, -4.49654, -2.51648, 4.04178, -4.34012, -0.0679991, -2.20335, 2.7089, 3.73041,
                4.24063, 0.901985, 3.52631, 4.06229, -1.28391, -4.74323, -3.84049, 3.54115, -3.26689, -2.21037,
                -2.83436, -3.75905, -3.71528, 3.58726, 4.65168, -2.66566, -1.87998, 1.95093, 3.66591, -3.2642,
                4.71373, -1.11216, -0.158811, 3.29267, -4.30107, -0.782312, -3.3017, 2.70345, 4.31324, 5},
            std::vector<int64_t>{10, 10, 10, 10, 10},
            std::vector<T>{
                -5, 1.38574, -3.40488, 1.59498, -1.40479, -0.774824, 0.776861, 4.44599, 4.65645, -0.93456,
                -2.73502, 3.51855, 4.47696, 2.91625, -0.68445, 4.43112, -4.38335, 3.58399, -3.5086, 0.863423,
                -0.279341, 2.84091, 3.83352, -2.69488, 2.11292, -4.86518, 1.38034, 1.30028, -1.84802, 4.05115,
                1.67509, 4.59603, 2.46173, 1.70552, -2.21902, 4.04495, 1.18181, -2.28113, -4.71689, -3.04267,
                0.500298, -2.41982, 3.09957, -2.96136, 3.4832, -4.11859, -3.75488, -3.0709, 1.24288, -1.92552,
                1.90494, 3.41518, 1.08902, 2.44889, 3.9638, 3.99164, -3.69309, -2.63514, 3.04926, 3.2098,
                -0.405537, 3.22924, 4.65977, 2.24535, -1.10242, -3.97405, 3.67443, 2.69645, 3.40125, -3.05413,
                2.16447, 3.45673, -0.00307435, -4.73115, 3.10638, 2.40497, 4.16993, -1.29241, -2.06006, 0.470756,
                0.781379, -2.77629, 0.291053, -0.862092, 0.863825, -2.31703, 4.10805, 2.96997, -4.90259, 4.54135,
                0.149612, 3.08602, -2.13883, -3.62432, -2.40104, -2.07502, -4.83612, 3.99756, 2.82135, -0.395894,
                4.39348, 0.466077, 0.766771, 2.01238, 2.06508, 0.772249, 1.9867, -0.996579, 2.48071, 4.49585,
                0.453517, -4.17035, 3.7106, 3.36282, 3.23191, -0.640585, 1.38488, -2.17163, 0.652199, -0.158743,
                4.6801, -0.926323, -2.0441, 4.05972, 1.53695, -1.47499, -3.02639, -2.46746, 2.02894, 3.84978,
                3.51725, 4.88996, -1.55536, -2.95742, 2.67589, -0.955491, -2.58142, -1.12036, -1.63169, -3.3286,
                0.909692, -4.98742, 2.3819, -3.80322, 2.35674, -2.01539, -1.54806, -0.301092, 4.52245, 0.48912,
                0.349015, -4.47653, -3.93302, -4.539, 4.10146, -0.21952, -4.79841, 4.41982, 1.43739, 1.89926,
                2.67474, 2.62216, -0.913088, -4.2081, 3.61542, -3.64835, 1.87344, 0.211608, 3.65209, -4.61966,
                3.15605, 0.0621908, -4.82205, -1.12782, 1.66538, 2.10551, -0.21663, -1.87744, 3.8985, -1.33132,
                2.94648, -4.89908, 1.08219, -1.82947, 1.3488, -0.839479, 0.424618, 0.108287, 3.44509, 1.66413,
                1.10641, 1.12783, 2.57802, 0.222069, 0.469913, 1.64298, 4.05762, 2.89772, 0.158175, 2.14327,
                0.0127615, 4.38472, 1.37834, 3.20256, -4.69572, 3.79924, -0.31191, 3.94988, -2.49268, -3.9734,
                -1.95347, -1.11895, 2.22331, -4.87892, -1.7993, -3.93962, 0.940251, 1.50669, 4.12202, 1.14476,
                -3.22335, -4.6922, 0.274547, 0.786363, 4.96083, -3.72688, -0.46743, 1.34387, 3.52815, 0.378364,
                0.989814, -4.81342, -4.99949, -2.57458, -4.8473, 3.61751, -0.291682, -4.45674, -3.86048, 4.55838,
                -0.558553, 4.22434, 0.485826, 3.02256, -4.18217, -0.970154, -2.29033, -3.17236, -3.24192, -1.19776,
                -1.10376, 2.74642, 2.92208, -0.253459, -0.727743, -2.6816, 2.05987, 0.821414, 2.55421, 1.73239,
                1.07334, -0.51151, -4.08995, 4.54727, 0.354294, 1.32332, -0.95764, -0.890992, -2.10241, -2.70027,
                -3.80798, -4.92614, 1.72164, 4.90619, 0.68811, -2.84043, 0.228619, 3.06368, 1.88669, 4.06624,
                -0.317901, 4.32894, 3.31458, -3.78791, -4.13413, -0.720673, -4.2847, 2.5743, -2.0607, -4.7294,
                -1.1922, -4.25132, 2.51276, -2.46105, 3.63289, -2.9029, -3.89352, -4.56467, 4.42607, 4.82693,
                -2.69463, 4.51783, -0.907243, -2.78669, 4.18723, 2.34626, 3.26836, -2.76921, 4.75613, -1.25286,
                -1.87767, 3.64592, -3.43697, -3.65555, -2.30538, 2.84627, -2.97604, 1.01567, -1.38899, -3.59646,
                1.65457, 1.07598, 0.107017, -4.40483, -3.71999, 4.20983, 0.778437, 4.03383, 1.31396, 3.58678,
                4.77689, 0.578005, -2.15946, -4.90681, 1.4671, 2.04727, -4.43527, -1.89183, 3.54344, 3.0036,
                3.81351, -0.304046, 4.08728, -1.53035, 0.84424, 3.17114, 2.83768, -0.247921, -1.93513, -4.63661,
                -1.12005, 1.45256, -1.15546, 1.60745, -3.76519, 3.94909, 1.43606, -3.87999, 1.78168, 0.370893,
                2.32649, -0.945226, 3.86596, -4.42396, 0.844734, -2.1054, 0.0782569, 2.11735, 3.4132, 1.88877,
                -3.23712, 2.91202, 2.16514, 2.07883, 1.37829, -3.31629, 2.28908, 3.98293, -0.77331, -0.164568,
                -3.97808, -1.30991, 3.64372, 4.19263, 3.90947, -1.78475, 3.94366, 4.18793, -4.11379, 3.91526,
                1.33598, -0.0962256, 4.72887, -1.53884, 3.28632, -1.22307, 1.04336, -1.78493, 4.46535, 2.2445,
                3.79802, -1.44113, 2.9843, -3.40275, 4.31126, 1.63422, -4.32046, 4.21314, 4.49346, 2.26706,
                -4.63838, -2.81771, -3.23518, -0.494282, 1.80663, -0.345847, -4.71775, -4.64164, -1.17283, 1.11853,
                -1.94206, 3.73188, 3.61315, 0.36183, -2.94746, 2.7314, -4.61468, 3.86121, 0.19202, 1.18747,
                4.02233, -3.69848, 1.33929, 2.94477, 1.80769, -2.99931, -4.77907, 0.0433281, 2.8262, -4.34346,
                0.171945, -2.70003, -1.17702, -3.16753, -0.665832, -0.493026, 3.40444, -2.0408, 3.98367, -3.76943,
                3.89665, 4.78787, 2.40015, -1.24947, 0.991205, 2.93423, -4.86143, 3.17028, -2.26548, 1.03663,
                4.61205, 4.75579, 4.99013, 4.23479, -4.45891, -2.65092, -3.90674, -3.53842, 0.178697, 2.23177,
                -4.97903, -2.10417, -3.41522, -1.52118, -4.1973, 0.196365, -3.07567, 2.6714, 0.0329277, 1.86999,
                -3.85877, -1.00351, -1.47763, -1.39762, -1.54178, 3.08322, -3.18606, 2.445, -3.89596, -1.96158,
                0.0700858, 0.518499, -3.63671, -0.617976, 0.341244, 2.04532, 3.65188, 2.35683, -0.89304, 0.0257487,
                -1.54115, -0.860534, 2.32265, -3.83482, -2.42048, 2.69591, 4.02649, -1.87481, 4.63578, 0.771746,
                4.82279, -1.68132, 2.46822, 4.48127, -2.72111, -0.355886, 0.266078, -4.89841, -0.850352, 2.77856,
                0.429384, 3.58205, 3.09831, 4.72246, -2.40146, -2.7575, 4.35845, -1.53977, -2.47573, 1.727,
                1.54918, -3.72806, -1.09464, 0.902036, -3.65559, -2.55986, -3.64498, -2.5337, -4.78799, 2.46949,
                -3.26495, -2.32184, 0.154838, -4.15034, -3.54186, 2.08248, 3.73156, -1.08542, 3.89507, 4.29175,
                4.46461, -4.04293, 2.00413, 1.75029, -2.7672, -0.584663, 3.17988, 3.9949, -1.16367, 2.6852,
                0.136113, 3.24949, 1.48249, -1.75121, 2.93909, -3.09163, 0.826285, 2.60739, -1.63573, 3.22668,
                0.00300903, 0.575604, -0.267615, -4.62369, -3.98882, 3.31384, -3.84641, -0.164339, -1.9838, 3.45511,
                -1.04323, 3.56878, 3.3694, -4.81446, 1.85392, 1.59741, 4.2085, 1.54342, -1.25854, -0.0477599,
                4.83944, 3.41224, -4.87642, 3.34542, 4.34374, 1.68048, -4.17476, -3.99847, -1.0523, -4.77302,
                -3.79443, -1.23287, -2.90961, -3.20999, 1.85511, 2.06811, -2.20368, -1.81411, -2.62059, 1.88005,
                -3.17691, -2.90064, -3.11477, 1.7461, -3.02032, -4.01402, -4.87155, -2.7408, 2.13552, 3.60602,
                -3.68662, 3.94441, 1.75357, 3.18497, -2.67031, 3.28189, 0.764719, 1.62342, -1.16762, -4.59845,
                0.336623, 2.26649, 3.89547, 2.07511, 2.93817, -1.03081, 3.30557, 0.72491, 3.1453, 0.769547,
                -2.92608, 0.13144, -0.0691925, -4.07191, 4.01141, -3.50862, -0.749251, -3.21837, 1.06989, 3.97227,
                1.54475, 4.78368, -3.18741, 4.10281, 3.61094, -2.81927, 4.0446, -0.688241, -0.195357, -1.89505,
                1.3421, 3.13891, -0.625937, -3.07901, 0.676794, -4.31063, 0.110105, 0.219899, 3.43871, 3.34377,
                3.90145, -2.36654, 3.87301, -3.27531, 2.66387, -4.84057, 4.20739, -0.915956, 2.33027, 2.63642,
                1.31649, -3.45589, 3.61696, 1.39562, 1.50066, 0.953228, 2.33703, -1.74775, 1.90994, 0.158689,
                2.00179, 1.97409, -0.447395, 3.93314, 0.379663, -3.20044, -4.63811, 3.33694, -2.92811, -0.334244,
                -1.82335, -1.99341, -2.38105, -3.42019, 4.49144, 4.50581, 0.146444, -4.01907, 2.76861, 2.54542,
                -4.88271, 4.25518, 3.71932, -2.67843, -4.61096, 2.49717, 3.22838, 2.91755, 2.81098, -0.212338,
                -4.09407, 3.46949, -0.264652, 0.653515, 0.510898, 1.74902, -0.615688, 0.225622, 3.35548, 2.3847,
                0.542832, -1.47433, 2.86275, -3.14377, -3.10022, 4.98831, 1.61969, 0.994205, 3.34565, -2.12893,
                -2.35919, 0.905238, -0.847957, -3.33345, -0.448516, 1.31451, 3.56578, -4.07352, -0.694577, -3.23525,
                1.98038, 4.05222, 0.15182, 2.11475, -4.91253, -2.83372, -0.800172, -0.797281, 4.91775, 0.71457,
                2.64354, -4.37525, 3.93947, 4.29284, 0.376963, 0.483439, -1.20476, 1.54934, -1.60458, -2.03934,
                3.17352, 1.84459, -0.57479, 2.83865, 4.82469, -0.00921004, -3.30157, -3.63386, -3.94658, 1.96984,
                4.15967, 3.75591, -0.538282, -1.32521, 0.128334, -0.57577, 0.96914, 1.10537, 2.45218, 2.8899,
                1.61422, 1.46003, 2.45143, 1.49513, 0.761625, -3.07383, 0.554756, 4.26704, 0.702351, 5},
            std::vector<T>{
                -5, -2.76745, -3.36272, 0.786832, -0.536305, -2.6808, 2.75696, -2.38186, -2.97924, -1.8377,
                -1.8783, -2.34573, 1.51361, -0.429618, -0.165277, -4.11843, 2.8142, 2.19643, 4.21874, 1.96632,
                -1.04831, -3.02755, 3.21475, -0.624601, 2.22647, 0.583331, -3.05985, -2.92969, 4.40203, 4.96686,
                -2.67395, 0.58967, 1.75651, 3.90978, -2.32869, -0.251364, -0.076078, -2.76866, 3.98696, -3.89213,
                -0.84402, -0.84834, -3.69578, -2.08657, -3.91955, 4.22513, -3.88648, -1.39718, -0.026567, 1.30207,
                2.68671, 1.60689, 3.78493, -1.1378, -1.50906, 4.42047, 1.11537, 3.98267, -0.826257, 0.65135,
                -3.50771, 3.93109, 1.36732, -2.45974, -1.47327, 0.772411, 2.65195, 3.03698, -0.592716, -0.558972,
                3.40876, 3.26508, -3.91006, 0.147352, 4.54504, 0.695234, 4.9661, -0.64909, 1.45945, 3.82606,
                -4.26746, 2.80674, 3.86749, -0.115914, -2.02211, -3.06485, 2.15541, -0.715792, 0.966579, -1.59997,
                -4.06261, 0.592467, -0.94068, -2.1352, 2.87121, 4.68889, -4.91329, 0.299997, -1.02383, 1.62808,
                3.77503, -3.10425, -3.31658, 2.88163, -4.02717, 2.43578, 2.97519, -2.72692, -2.94946, -4.62235,
                0.938072, -3.56148, 4.96502, -4.73707, 3.46557, -0.84019, 4.05851, 0.971805, 3.67174, 0.272782,
                3.11246, -3.17549, 0.703277, -1.83248, -2.66066, 1.93464, 0.505384, 2.92792, 4.47033, 0.221046,
                0.757135, 2.87319, -2.95227, -3.87443, 3.22101, -3.01482, -3.17032, 1.28237, -4.21317, -0.735796,
                -3.95619, 2.21906, 1.35044, 2.38934, 3.73366, -2.37152, -1.21153, 2.32514, -2.52731, -1.27155,
                -3.50355, -3.01653, -2.55113, 1.53067, -0.114533, 0.0749606, -1.90288, -4.09761, -2.89037, -2.07762,
                -4.15337, 4.51941, 4.96817, 1.81584, -2.77298, -3.77737, 0.511126, -3.95519, 0.312086, 2.07223,
                0.682299, -1.93266, -2.17914, -0.747721, -1.33989, -4.1225, 2.73361, 1.20164, 2.0721, 0.616006,
                -2.14654, -1.09562, 2.69654, -1.29804, -3.06921, 0.918753, -0.960092, -2.76928, -4.81197, -0.228397,
                2.9628, -3.62481, -0.877274, -0.636959, -2.77519, 1.92274, -2.71535, 0.823748, 0.0571949, -2.80894,
                -2.93044, 2.44277, 3.72529, -1.39665, 0.419583, -0.735395, 0.141612, 4.91315, -1.85428, 3.59253,
                4.63525, -4.308, -4.04851, -4.52805, 3.46887, -3.40928, 1.49532, -4.29342, 0.864934, -4.68825,
                -4.24461, 4.62154, 0.258795, -1.59565, -4.96134, -1.42749, -1.41203, -4.28422, -1.34112, 0.776945,
                -4.04485, -3.96018, 0.580818, -4.41352, 1.14779, -4.36729, 3.8481, -1.66048, -0.950351, 3.57762,
                -2.33159, 1.01822, 1.35339, 4.7311, -4.59032, 3.43969, -0.238646, -1.39467, -0.329268, 3.23937,
                -0.284994, -3.59913, -2.26108, -0.378409, -3.475, -0.160139, -0.838942, 0.527827, -1.51829, -2.46392,
                1.89068, -4.46563, -2.60946, 3.53656, -2.20977, -0.851908, -2.42698, -2.03249, -0.938796, 2.09651,
                -0.967279, -2.06681, -4.58985, 3.24781, -2.63077, 4.56093, 3.17359, -2.34826, 1.58482, -4.12192,
                -2.55048, -3.4803, 0.631562, -4.64767, 4.07145, -4.89086, -3.90578, 0.794496, 3.11944, 0.274388,
                -0.802882, -0.72629, 2.55014, -2.86997, 4.11855, 1.10845, 0.782459, -4.91899, 2.37826, -1.77523,
                -3.23982, -1.69375, -0.837184, 1.1934, -1.10572, -1.22483, 4.71149, 2.96609, -0.676543, -3.31718,
                -3.692, -2.9575, -3.52942, 0.411185, -2.41244, -0.401702, -1.8739, -0.132347, 1.13978, -4.45176,
                0.169302, 3.5423, 3.94282, 1.60127, -4.07808, 2.34236, -0.781839, -2.74102, 4.48042, -4.34059,
                -1.30121, -0.93276, 1.11985, -3.90853, -0.286499, -1.01405, -2.77668, 0.288616, -4.16595, -2.3096,
                -2.44363, 1.24341, 3.73431, -0.00587227, -4.89328, 4.75827, -4.9217, 3.8995, 2.46474, 3.20115,
                1.39757, 3.46744, 3.32388, -1.17844, 2.83398, -1.88155, -3.75103, -2.30867, -1.46387, 2.3986,
                0.291228, -0.924614, -0.466735, -2.32718, -1.59271, 3.8981, -3.37128, 1.54383, 3.21023, -4.46146,
                3.65466, 2.02678, 4.93931, -4.48006, -4.70468, -0.0110495, -4.34515, 2.66921, -2.4087, 3.08708,
                4.72305, 3.14726, -2.3531, 0.719721, -2.08273, -4.30752, -3.36451, 0.7731, 0.323962, -2.62083,
                0.261967, -3.56867, 2.00299, 0.649599, 1.86126, 1.56565, 3.74213, 0.474407, -0.946149, -0.476581,
                -2.64906, -1.82586, -3.98376, -4.55005, 2.38571, -3.62565, -0.934082, -1.81757, -1.77204, -3.26355,
                4.35017, 3.13568, 0.702371, -3.24041, 0.0219689, 2.41334, -1.23636, 3.9868, 4.02405, -1.44978,
                0.281016, 4.02714, -3.745, -0.925299, 0.628337, 4.63236, -3.27555, 4.35052, 3.93893, -1.18177,
                3.12756, 2.48858, -3.78877, 3.23341, -2.12903, -3.78299, -2.97581, -0.814401, -4.27898, 0.582629,
                1.62196, 2.45058, 1.58468, 3.9512, 4.54651, -0.556351, 2.948, 1.81753, -3.51936, -0.504748,
                -3.25792, 3.13386, -0.620701, 4.71377, 3.99684, -3.2282, 3.46386, -2.72125, -1.49549, -3.83636,
                -4.7897, 0.78226, 2.3058, -4.82898, -2.61577, -4.87581, 2.31266, 1.67058, -0.312257, -0.545391,
                -3.86117, -4.92605, -0.722609, -0.144248, -4.35181, 0.483898, -3.27626, 0.600822, -0.846488, 1.54249,
                4.67917, 0.773491, 0.335107, 0.391722, -1.57747, 3.65133, -0.816687, -3.66164, 0.3431, -2.17813,
                -4.99742, -4.13003, -4.34429, 3.15146, 0.477193, -3.18597, 4.0696, -3.04081, -0.751688, -2.3623,
                2.81943, 1.98781, 1.14407, 2.95467, 0.0443827, -1.67868, -2.85497, -3.04549, 0.0620073, 0.179388,
                -3.74709, 4.28623, 3.73272, 2.019, 1.88438, -1.20996, -2.34781, 0.800072, 4.22213, 3.82745,
                0.660494, -0.233822, -3.85401, -1.10944, 1.61249, -2.75312, -0.620572, -2.36705, -4.35225, -4.28065,
                -0.369473, -0.675064, 4.98128, -3.40084, -0.158063, -1.81529, 0.0704487, -4.2162, 2.33253, 0.775176,
                4.70129, -0.44329, -0.804775, -4.39479, -0.857478, 0.173731, 2.117, -1.91569, 0.661488, -3.72608,
                3.21041, -0.309079, -1.20312, -1.04395, -3.59398, 0.6059, 2.61891, 4.45666, -4.75106, 3.21227,
                1.7304, -2.32413, 3.76143, 3.02153, -3.54336, -0.34727, 4.84026, -0.198416, -3.46508, -0.201522,
                3.50927, -1.84184, -2.48124, -0.669658, -4.94919, -3.52275, 1.42435, 3.78537, -3.23707, -4.68775,
                2.3794, -4.965, 3.11321, -1.14157, 3.72124, -1.05859, -0.206402, 0.79939, 4.87222, 0.225406,
                4.48362, -2.90358, 1.55703, 1.98258, 2.35008, 3.94249, 2.61202, -3.53563, 1.82492, -2.68085,
                -0.527339, 3.24015, 3.29355, 1.34272, -1.10936, -2.71789, 3.91075, -2.14601, -0.396423, -3.01396,
                0.340825, -4.43331, -4.90419, 3.48833, -3.20166, 2.02417, -1.30311, 1.58947, -3.03143, 3.37555,
                -1.96399, 1.02333, -3.52225, -1.53211, 0.12215, -1.34119, -0.846173, -0.635141, 4.67665, -0.405799,
                3.8426, 0.339217, -1.40732, 2.20983, 0.573309, 0.977476, -1.76888, -1.02265, 2.32694, 0.677431,
                -4.85236, 3.18685, -2.53947, 1.13715, 4.96326, -0.793396, 1.19392, 2.5294, 2.75801, -4.3221,
                3.15646, -1.48667, -4.94554, -3.23913, 4.02479, -2.08067, 0.183081, -4.38464, -2.46221, -4.73904,
                -0.697342, -4.85434, 4.90753, 2.70221, -1.31093, -4.60081, -3.58929, 3.71782, -1.75827, 0.163215,
                -3.75722, 1.62284, -1.24612, 1.14538, 2.96385, -4.10806, -0.186223, 2.97828, -3.48557, 0.912886,
                -2.47933, -0.141892, 1.38328, -1.35453, 0.0855794, -0.420718, 2.81809, -1.58504, -3.24831, -3.06183,
                -1.00012, -3.06554, 1.15852, -4.66011, -1.68974, 2.15275, -4.81303, -1.0497, 4.24721, 0.752416,
                -2.9178, 2.80425, -4.72424, -4.04212, 1.70999, 4.54444, 2.7805, -0.316397, -2.46779, 3.17646,
                0.187879, 1.74938, 3.01256, -3.66057, 2.37244, 2.91338, 2.67634, -2.33103, -1.07969, 1.37713,
                -3.45046, 1.41599, -2.58735, 1.66767, 0.978738, -1.99731, -1.185, 4.02593, -3.86151, 3.34423,
                3.45627, -3.98229, 4.22739, -3.05438, -4.33193, 3.05151, 4.80124, 4.1618, -3.6186, -4.59032,
                4.59652, -1.2784, -1.30989, -3.37153, 2.88972, -4.98978, 0.866851, 2.13723, -1.54307, -2.96314,
                -0.727502, 1.31068, 0.444759, 1.69446, 2.77479, -1.25197, -0.59404, -3.28619, -2.14284, 4.55701,
                -1.99698, -4.1932, 1.98522, 0.330638, 4.14927, -1.58213, -1.19538, -2.64269, -4.77377, -4.50282,
                2.11742, 1.85531, -1.0953, -4.23724, 0.736428, -3.5851, 1.48482, 0.141458, -1.86905, 3.5453,
                -2.82172, -3.05134, -3.75254, 4.95023, -1.05753, 1.91518, -3.26179, 4.54684, -4.10982, -1.89519,
                1.14695, 1.47464, 1.4862, -4.79501, -4.77577, -2.36627, 0.19994, 3.35062, 1.23547, 5},
            std::vector<T>{
                -5, -4.90212, 2.04852, -4.06222, -2.05556, 1.71858, 4.89697, -3.57399, 3.22732, -1.36818,
                -3.74683, -4.60178, 0.196569, 1.58981, -3.29433, -2.98955, 2.02674, 1.8465, -1.85874, -2.1893,
                4.83577, 1.90244, 2.07352, 0.598257, 3.49708, 1.4686, 0.754768, 1.94554, 3.96111, 0.133566,
                -3.82646, 4.88763, 0.229708, -2.03485, 3.4875, 1.79253, -0.38706, -2.65237, 0.100763, -2.15537,
                1.71114, 4.87242, -0.449133, 3.7298, -4.50384, 0.0434988, -4.31846, -4.79365, -2.04677, -4.72319,
                1.47757, 1.06105, 3.82314, 3.14707, 4.1567, 1.89765, 4.28057, 3.15959, 1.97148, 3.77296,
                2.90827, 3.08369, 4.84089, -1.83596, 2.32062, -4.18992, 1.58745, 0.212731, -3.36909, -1.486,
                2.07869, -1.65644, 1.39901, 1.31554, 1.7938, 4.02885, -1.31798, 0.48152, -0.964317, 5},
            std::vector<T>{
                -0.999909, -5.07626e-11, 4.13326e-13, 5.835e-05, -1.30828e-35, 0.761594, -3.42026e-22, -1.13159e-30, -4.18379e-13, 0.821244,
                -0.999909, -5.07626e-11, 0.49691, 0.000728936, 6.90693e-27, 0.963522, -9.60537e-09, -0.997578, -0.0213497, 5.66025e-14,
                -0.999928, 2.69134e-08, 4.40231e-06, -6.96162e-19, 0.511536, 0.287868, -0.762594, -0.997594, -0.9476, 0.76483,
                -0.999465, 1.57167e-10, 0.49691, -3.00368e-17, -1.50091e-08, 0.785706, -0.368554, -0.997594, -0.981176, -0.761575,
                -7.99627e-07, 9.21166e-05, 0.332835, 4.10511e-31, -9.42096e-13, 0.378887, -0.00455473, 2.21606e-11, -0.997432, 1.16333e-05,
                -8.98809e-32, 3.03724e-15, 0.25859, -0.000176827, -0.954949, 2.55493e-17, -0.983461, 9.09049e-13, -0.980796, 0.784393,
                -0.000793236, 2.55449e-20, 5.38752e-09, -9.86419e-35, -1.94761e-19, 1.83397e-12, -0.988398, 3.78249e-10, -0.997432, 0.771463,
                -3.52882e-15, 1.21005e-16, 0.000182383, -4.34274e-08, -0.594177, 1.33054e-06, -0.920454, 0.76128, -0.999382, 0.000389586,
                -9.19733e-06, 4.3138e-07, 1.41793e-21, -0.761594, 0.305402, 0.000580937, -0.531783, -1.00171e-05, 5.83801e-30, 0.0243704,
                -1.23107e-30, 0.00725055, 0.76182, 1.27929e-05, 1.58846e-07, -0.680477, -5.73051e-15, 6.33457e-20, 0.000288991, 0.00312473,
                0.761596, -0.000445729, 2.03617e-09, 5.11049e-30, 0.964028, -1.18563e-22, 3.80413e-32, -1.97169e-18, -0.759971, -1.27976e-07,
                0.74886, -0.000445729, 0.805263, -1.49589e-10, 0.641535, -1.41323e-07, 1.47308e-24, -4.77803e-13, -5.96046e-08, -1.95598e-10,
                -0.581045, -0.00103914, 0.805263, 1.64052e-14, 0.00323384, 1.15601e-13, 6.4162e-14, 2.77607e-22, 0.0497418, 5.30151e-18,
                -0.329421, -2.57703e-06, 0.963854, -1.01966e-13, -1.18368e-08, 6.02313e-09, 0.114898, 9.46065e-13, 0.761594, 2.13439e-08,
                0.523866, 0.761529, 0.760565, -0.00461076, 0.00150048, -0.00174691, -0.436961, 5.48788e-11, 2.41781e-14, 2.19602e-08,
                -6.14432e-17, 8.04105e-14, -3.89646e-18, -0.00379894, 3.63214e-16, -0.616305, 0.964023, 5.81763e-07, 1.09818e-06, 2.19604e-08,
                -0.87856, -0.354106, -0.715297, -1.65026e-20, 6.05625e-17, 2.48927e-05, 1.96127e-05, 1.41168e-23, 5.4023e-07, 0.761544,
                -3.06545e-19, -7.18136e-08, -1.8746e-07, -1.47835e-07, 1.34948e-07, 0.761594, 0.999987, 3.23361e-26, 1.28856e-06, 0.761618,
                -0.892197, 1.36596e-19, 6.65404e-09, -2.64782e-09, 0.766419, -0.998266, 0.000204802, 1.00874e-08, -4.23892e-11, 2.38972e-09,
                -0.901095, 0.00314948, 0.156432, -3.46747e-21, 0.766474, -4.19696e-07, 0.00359662, 0.985378, 1.26476e-26, 1.36917e-06,
                -8.30729e-05, 1.09966e-09, 3.39125e-24, -2.54402e-09, -0.0224086, 0.999078, -8.56764e-17, 2.17593e-09, -5.3338e-18, -2.09897e-25,
                -0.731555, 1.0058e-09, 1.80811e-06, -7.35442e-08, 5.03322e-18, 0.99907, 5.06163e-08, 0.998803, -3.31849e-14, -0.314541,
                6.56072e-06, 2.3231e-05, -0.734035, -0.801636, 0.761566, 1.44226e-12, -0.521044, 0.998576, 2.04797e-08, -0.999782,
                8.46822e-16, 0.236769, -1.06652e-14, -0.000473771, 0.964, 1.61872e-15, -0.918253, -6.50992e-10, 1.19996e-09, -0.976642,
                -1.74187e-06, 2.59573e-19, -2.2928e-13, -8.87904e-17, 0.148158, 4.75517e-15, -0.98799, -6.50965e-10, -0.761594, 0.760215,
                0.000518275, 0.434619, -0.0692393, -0.761594, 0.274058, 6.53858e-08, -0.998366, -3.22841e-06, 1.72996e-06, 0.963795,
                -0.735883, -1.68637e-22, 0.00299617, -2.95948e-14, -0.754872, 0.964028, -0.832263, 2.96769e-05, -1.3113e-06, 0.761594,
                1.00882e-20, -0.700681, 0.251888, -1.44631e-16, -5.78152e-16, 0.964028, -3.13362e-09, 2.38789e-16, -1.29541e-06, -4.12287e-11,
                3.57653e-37, -0.953416, -0.00116461, 4.51481e-11, 0.000654999, -9.88477e-13, -8.25559e-09, 3.24498e-11, 0.631349, -2.22783e-15,
                0.00463415, -0.218827, -3.33063e-20, 2.62387e-15, 0.000655749, -0.761594, -9.3791e-19, 3.55914e-07, 1.21115e-10, 8.51853e-05,
                0.746421, -0.203347, 0.964016, -0.675211, -0.757478, 4.1336e-13, 1.31125e-14, -0.752869, -3.40686e-11, -9.04177e-20,
                0.761593, -0.203337, 0.761593, 4.32757e-13, 0.761594, 1.12104e-12, 1.10405e-22, 2.21747e-11, -0.761594, -0.000232199,
                -7.34674e-20, -0.761592, 1.12101e-05, 0.96248, 1.05897e-15, -0.961035, 0.995049, 0.714666, -0.11871, 0.252246,
                -3.96402e-21, -0.116561, -2.34864e-11, 1.73253e-05, 6.71033e-11, -0.761585, 0.964016, 5.67781e-08, -0.00572334, 0.76158,
                -0.520498, 5.11946e-09, -1.68016e-05, 7.55615e-09, 0.959448, -2.85015e-19, 0.753059, 4.49821e-06, 1.60358e-10, 3.66218e-21,
                -2.05996e-14, 9.10536e-05, 0.443528, 0.000286103, 1.1389e-16, -0.332143, 0.000106997, 1.66753e-09, -0.761594, 7.13778e-13,
                -0.71679, -0.761576, 0.754419, 0.000282388, -0.762039, 9.84022e-09, -0.0872869, 7.7548e-11, 6.80808e-09, 2.60995e-08,
                4.78222e-09, -0.501957, 0.762282, -0.753946, -1.47174e-09, -0.964027, -0.7082, -0.963465, -9.54667e-13, 3.02122e-08,
                7.1591e-12, 6.3354e-14, 2.94622e-22, -0.940787, -6.09716e-15, -2.65188e-14, -0.0115473, -0.758209, 6.95914e-17, 0.760974,
                0.140932, 4.35034e-05, 0.760892, -0.632027, -0.76149, 2.15703e-11, 0.777049, -0.00238567, 5.3589e-28, 0.760506,
                -8.19313e-13, -0.99918, -1.76353e-09, -3.25938e-08, 3.69808e-14, 5.13777e-15, -0.992067, 3.2008e-13, -4.27895e-35, -0.00033198,
                -2.22583e-14, 2.33933e-18, -1.83076e-06, -2.05328e-23, 0.335589, 6.74499e-20, -0.993734, 3.25956e-15, -0.76157, 0.761592,
                -1.39733e-26, 3.51873e-08, -0.000449794, 0.753596, 6.14397e-06, -0.0399566, 6.81031e-16, -1.4436e-16, 6.54251e-37, 6.25711e-16,
                -0.751923, 1.76327e-06, -0.759611, -5.82387e-22, -3.55939e-10, 2.0026e-05, 1.42291e-08, 0.00127849, -0.761586, 0.920489,
                -0.0123201, 1.49775e-11, -0.759593, 0.760956, -1.59088e-19, 1.89399e-06, 1.41353e-17, -3.71678e-07, -7.58637e-09, 4.56301e-13,
                -0.76173, 6.38001e-12, -3.26879e-12, 2.54147e-19, 0.00349909, 1.94641e-15, 0.000533218, -0.680174, -0.798763, -0.749397,
                -1.51134e-16, 0.76157, -0.758462, 0.212695, 2.33021e-17, 1.62019e-15, -5.34576e-05, -2.0166e-15, -0.0958154, -3.4894e-05,
                1.54693e-09, 0.761228, -0.045171, -0.654946, 0.658682, 0.000900979, -0.0121633, -9.55765e-07, -5.83339e-07, -0.697058,
                3.12132e-18, 0.599694, -2.41771e-11, 0.128772, 0.0345568, 0.0222523, -3.01504e-23, -1.42109e-23, 1.32333e-13, 5.34001e-15,
                -6.2253e-22, 8.88893e-22, 0.000599919, -6.13229e-19, 9.45362e-05, 1.03827e-16, -0.00543732, -9.86013e-09, -0.761408, 0.214137,
                -8.00209e-24, 0.679937, 0.917833, -1.34199e-10, 1.89431e-07, -0.761594, -0.0442637, -0.000217974, 0.00476938, 0.0246565,
                -5.376e-08, 0.784315, 0.00101343, 0.0946926, 0.980865, -0.761594, -0.707944, -4.73592e-12, 4.01193e-13, 1.73187e-07,
                0.000424088, 2.85811e-08, 3.59465e-08, -0.472002, 0.867616, -0.757693, -2.0641e-10, -0.994265, -9.17776e-06, 5.91256e-15,
                0.963968, 0.31226, 0.0461947, 1.10725e-08, 2.92814e-07, -5.24128e-08, -0.00704819, -7.04583e-10, -0.000118595, -0.760989,
                0.761239, 0.312348, 0.0461707, -4.89262e-06, 0.762134, 0.0216284, 4.89744e-20, -0.705499, -0.0105222, -2.47122e-06,
                -0.0353277, -0.529026, 0.0371158, 0.0910813, 0.964028, 0.761582, 0.0233926, -2.56495e-19, 2.55893e-06, -4.6058e-09,
                0.0361859, 0.947518, 0.645797, -1.36319e-07, 1.11416e-06, -0.000967128, 1.71085e-06, -3.83432e-05, -4.46048e-12, 1.68634e-09,
                3.07729e-10, 0.0198492, 0.055882, -1.79771e-09, -2.43386e-18, -0.228931, 0.760035, -0.761594, -0.393895, 0.761594,
                0.761594, 0.0695398, -2.10839e-05, -0.761593, -0.761593, -7.67992e-12, -0.000156758, -0.761546, -7.03448e-14, -1.32276e-19,
                -9.58347e-39, 4.26105e-18, -5.41441e-05, 4.03958e-17, 4.40911e-16, -0.747015, -0.879764, 2.22597e-08, -0.0910139, 0.999823,
                0.999852, -0.946979, 4.04764e-10, 1.04581e-21, 1.30145e-14, 9.21577e-17, -0.000708233, 0.761594, 3.99524e-13, 9.18395e-17,
                8.95967e-25, 6.11501e-20, 0.00901172, -2.49613e-41, -1.17426e-18, 3.85668e-09, -0.0678945, 7.0955e-16, -0.761593, 0.792416,
                0.999844, 0.441572, 0.00697713, 5.52616e-05, 2.76337e-24, 0.0128815, -8.97325e-09, 0.0575689, -9.19902e-17, 0.79131,
                0.90745, 0.442877, 1.65895e-09, -9.49956e-12, -0.761592, 0.759226, -0.788664, 0.428198, 4.22296e-08, 0.960446,
                -3.30814e-05, -7.32391e-11, -0.761195, 0.761591, -0.11482, 0.00328929, -0.751446, -0.0329113, -2.38419e-07, -0.742049,
                -0.761814, 2.28806e-08, 9.20868e-10, 7.83398e-05, 0.163255, 0.693679, 0.0233387, -0.0002779, -0.00744079, 0.761594,
                -1.92574e-35, -6.19079e-15, 0.841117, 3.61402e-15, 7.29652e-05, -3.43512e-15, 2.44003e-10, 5.02575e-21, -0.575338, 0.760067,
                -7.58183e-07, -0.411779, 0.937771, -5.07946e-26, -0.0015457, 0.625772, -0.128913, 7.19112e-12, -0.483728, 5.19518e-07,
                2.4981e-17, -2.77282e-10, -6.1325e-23, 4.02951e-22, -0.55648, 0.627102, -6.88607e-19, 9.6838e-17, -0.0142015, 1.0327e-05,
                -0.761481, 3.44292e-08, -0.644774, 5.99614e-06, -2.4661e-12, 0.939834, -0.732183, 0.758408, -0.406918, 0.761563,
                0.624331, 0.959323, -0.757392, 0.00112301, 0.0215218, -7.76243e-10, -6.42156e-20, -3.81184e-11, 2.64018e-18, -0.00122851,
                -9.04986e-17, 0.959198, -0.180509, 8.1107e-09, 4.4799e-16, -0.761594, 0.761594, -3.2702e-11, 6.12426e-16, -7.52802e-11,
                0.727842, 0.736519, 0.00697721, 0.00189766, 0.65557, -8.23888e-06, 1.19768e-19, -0.628732, 4.50315e-07, -0.596042,
                -0.130076, -0.0570775, -0.758032, -0.720559, 0.329393, -0.000139915, -7.64476e-11, -1.93167e-30, 8.91504e-10, -5.52299e-08,
                -0.00538237, -1.59122e-08, -4.0649e-14, -0.747447, 0.871613, 4.90251e-14, 0.760524, 0.623387, 0.761594, -0.963604,
                -0.761594, -0.760084, 0.76159, -0.726583, 0.783823, 2.00219e-05, 1.12777e-06, 1.49013e-16, 0.761591, -0.625537,
                2.84003e-10, 4.32528e-16, 2.39073e-07, -6.75822e-06, 2.44769e-23, -1.39062e-07, 0.768824, -0.000734329, -1.18028e-08, -9.31533e-16,
                1.32778e-09, -5.24861e-07, 0.7686, 1.34084e-06, -0.0222266, -0.775352, 0.000264648, -2.44784e-27, -8.62503e-05, -0.523765,
                8.05469e-07, -8.00327e-17, 1.0076e-08, 0.655086, 1.12223e-07, -2.45992e-13, 0.761594, -0.000735076, -3.16667e-21, -1.0616e-12,
                0.999911, 6.67684e-10, -0.386663, 9.55342e-24, 0.000750192, -3.04876e-14, 1.69092e-13, 2.57222e-30, -0.886215, -8.85092e-18,
                -0.999066, -0.992009, -9.64733e-07, 0.999793, 0.999959, -4.526e-09, -0.993717, 6.25306e-09, 1.5415e-21, 0.761594,
                -5.23755e-08, -0.827688, -0, 5.99811e-22, 0.999958, -0.580146, -6.79704e-17, 1.83394e-07, 1.31284e-13, 3.62265e-10,
                0.35334, -0.00903121, -0.991192, 7.53642e-07, -0.00913154, -1.29673e-06, -1.76794e-11, 1.83017e-07, -0.000150782, 0.761551,
                -0.739773, -0.268164, -0.991253, 0.761593, -8.46913e-07, 4.02483e-07, -0.0547462, -0.0433184, -0.0128592, -0.000826936,
                3.57959e-13, -0.073303, 2.14562e-10, 7.38332e-11, 0.00409962, 0.724394, -1.16275e-22, 1.63268e-25, -1.47741e-05, 3.33773e-08,
                -7.77931e-12, 2.71497e-08, 3.42017e-05, 0.761594, -2.43217e-13, -6.87127e-23, -0.761548, 7.04828e-08, -0.761307, 1.3015e-07,
                -0.00537324, 1.27446e-16, 0.763627, 4.22074e-15, -8.45132e-18, 0.00159667, 0.756707, 0.129373, -0.964029, 0.761554,
                3.34318e-14, 3.74157e-15, 0.763626, 4.1317e-22, -0.734039, 4.81716e-07, 0.753698, 0.758801, -0.995055, -0.76151,
                1.11652e-10, 1.10396e-07, 0.761582, 0.761191, -0.949494, 4.12621e-08, -0.0185743, -5.64389e-13, 1.35275e-28, -0.761587,
                -2.38573e-16, -3.53137e-05, -0.438905, 0.0134316, 4.40739e-11, -3.01366e-05, -1.38874e-32, -3.77945e-21, 0.0780586, 1.6805e-13,
                0.751234, -0.803057, 0.811984, 7.895e-05, 0.761593, -0.00119532, -5.48256e-06, 0.757714, 0.761594, -0.761511,
                -1.92404e-24, -1.41394e-08, 0.983753, 4.33593e-18, 6.11314e-09, -0.000849993, -0.96402, -3.8176e-18, -1.24685e-14, 8.03048e-05,
                0.994964, 0.130676, 0.922726, 1.28861e-07, 0.761594, -6.48442e-30, -0.761584, 0.761594, -3.41314e-14, -1.08584e-06,
                0.963591, 0.130676, 0.542119, -0.725731, 0.761583, -1.63707e-11, 2.25863e-17, 0.766755, -0.761594, -9.47899e-06,
                0.760034, 0.0996578, 0.757886, -0.761808, -0.000816665, -0.000733465, 0.761594, 0.670761, -4.79781e-25, -3.77769e-09,
                -0.071795, -2.09178e-29, -2.96662e-22, -5.45929e-22, 4.72562e-14, 2.23143e-12, 2.66405e-05, 0.948048, -5.47349e-07, 0.761064,
                -0.071795, -1.62295e-13, -2.48377e-11, 0.000322916, 1.0934e-18, 0.745943, 0.000203257, 2.9748e-09, 2.94779e-09, 0.000904395,
                -1.33357e-12, -0.761594, 4.154e-08, 0.761365, -1.23576e-05, -0.118402, 0.758765, 8.3947e-11, 0.757683, 0.00091744,
                6.04679e-13, -8.36692e-05, -7.73738e-13, -0.964028, -2.87151e-05, -0.964027, -0.761415, 2.8511e-12, 0.982895, 0.000913338,
                0.761594, -5.61929e-05, -4.03471e-09, -9.05792e-10, -2.44896e-14, -0.761593, -1.53072e-14, 4.80451e-06, 4.97845e-05, 2.61901e-17},
            std::vector<T>{
                -1.23107e-30, 0.00725055, 0.76182, 1.27929e-05, 1.58846e-07, -0.680477, -5.73051e-15, 6.33457e-20, 0.000288991, 0.00312473,
                0.761596, -0.000445729, 2.03617e-09, 5.11049e-30, 0.964028, -1.18563e-22, 3.80413e-32, -1.97169e-18, -0.759971, -1.27976e-07,
                0.00463415, -0.218827, -3.33063e-20, 2.62387e-15, 0.000655749, -0.761594, -9.3791e-19, 3.55914e-07, 1.21115e-10, 8.51853e-05,
                0.746421, -0.203347, 0.964016, -0.675211, -0.757478, 4.1336e-13, 1.31125e-14, -0.752869, -3.40686e-11, -9.04177e-20,
                -6.2253e-22, 8.88893e-22, 0.000599919, -6.13229e-19, 9.45362e-05, 1.03827e-16, -0.00543732, -9.86013e-09, -0.761408, 0.214137,
                -8.00209e-24, 0.679937, 0.917833, -1.34199e-10, 1.89431e-07, -0.761594, -0.0442637, -0.000217974, 0.00476938, 0.0246565,
                -0.761481, 3.44292e-08, -0.644774, 5.99614e-06, -2.4661e-12, 0.939834, -0.732183, 0.758408, -0.406918, 0.761563,
                0.624331, 0.959323, -0.757392, 0.00112301, 0.0215218, -7.76243e-10, -6.42156e-20, -3.81184e-11, 2.64018e-18, -0.00122851,
                -2.38573e-16, -3.53137e-05, -0.438905, 0.0134316, 4.40739e-11, -3.01366e-05, -1.38874e-32, -3.77945e-21, 0.0780586, 1.6805e-13,
                0.751234, -0.803057, 0.811984, 7.895e-05, 0.761593, -0.00119532, -5.48256e-06, 0.757714, 0.761594, -0.761511},
            std::vector<T>{
                -1.14805e-14, 0.00725081, 1.00065, 1.27929e-05, 1.29939, -0.999801, -5.73051e-15, 3.93105e-11, 1.9999, 0.0243762,
                1, -0.000445729, 2.03617e-09, 5.11049e-30, 2, -1.67708e-22, 1, -0.0034493, -1, -0.591546,
                0.00463418, -2.86822, -0.230709, 0.0172415, 0.000655749, -1, -3.072e-06, 0.746059, 7.45091e-09, 8.51853e-05,
                0.965889, -0.206222, 1.99984, -0.999999, -0.990272, 1.11695e-12, 0.0222863, -0.979547, -2, -1.01401,
                -0.999991, 0.976968, 1, -3.63016e-07, 0.0208492, 3.42118e-07, -0.00543741, -9.86013e-09, -0.999558, 0.219696,
                -1.00056, 0.828996, 1.99999, -2.98023e-07, 1.61781e-05, -1, -0.044346, -1.00563, 0.099386, 0.0246731,
                -0.999731, 3.44351e-08, -0.766302, 5.99614e-06, -0.087751, 1.73662, -0.999998, 1, -0.431925, 0.999926,
                0.732687, 1.93734, -0.99007, 1.41437, 0.0215275, -7.76243e-10, -1.4114e-06, -1.2676, 9.7866e-11, -0.999843,
                -0.0294529, -3.65437e-05, -0.470874, 0.998836, 1.19728e-05, -3.01366e-05, -1.90968e-25, -2.75275e-07, 0.0782207, 0.000366083,
                0.975781, -1.10725, 1.13283, 7.895e-05, 1, -0.00119563, -0.99991, 0.990824, 1, -1}),
    };
    return params;
}

template <element::Type_t ET>
std::vector<LSTMSequenceParams> generateParamsBF16() {
    using T = typename element_type_traits<ET>::value_type;

    std::vector<LSTMSequenceParams> params {
        LSTMSequenceParams(
            5, 10, 10, 10,
            0.7f, op::RecurrentSequenceDirection::FORWARD,
            ET,
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 2.15625,
                5.71875, 1.17188, 1.75, 7.125, 9.25, 2.90625, 9.1875, 3.375, 3.6875, 5.4375,
                6.25, 1.47656, 6.0625, 6.15625, 6.5, 2.3125, 9.625, 6.3125, 3.34375, 7.3125,
                3.07812, 1.92188, 5.8125, 4.71875, 9.5, 7.25, 5.4375, 4.71875, 5.875, 1.45312,
                7.875, 5.8125, 1.40625, 6.96875, 2.25, 5.625, 8.125, 9.5, 1.26562, 6.25,
                8.9375, 9.125, 5.875, 2.23438, 5.03125, 2.25, 9, 8.25, 4.375, 4.5625,
                5.84375, 2.48438, 6.875, 9.375, 4.25, 4.125, 6.125, 7.75, 6.75, 7.53125,
                2.125, 8.9375, 7.1875, 6.625, 6.8125, 7.75, 4.1875, 4.125, 7.875, 3.42188,
                4.1875, 9.0625, 7.75, 4.84375, 8.875, 9.625, 1.10156, 6.96875, 5.46875, 6.59375,
                1.66406, 2.03125, 8.0625, 9.5, 1.57812, 5.0625, 4.1875, 6.1875, 9.5, 4.6875,
                4.40625, 3.125, 7.875, 9.125, 7.9375, 6.15625, 3.71875, 1.02344, 7.9375, 6.5625,
                2.375, 3.9375, 6.1875, 5.75, 1.07812, 9, 7.375, 4.1875, 5.25, 9.125,
                7.875, 6.625, 5.1875, 1.14062, 3.40625, 9.375, 8.5, 7.1875, 5.9375, 10,
                1.625, 2.54688, 5.25, 2.21875, 7.6875, 9.375, 2.71875, 7.25, 5.1875, 1.59375,
                3.0625, 7.8125, 5.5625, 7.78125, 2.875, 9.25, 1.4375, 7.375, 5.65625, 2.125,
                2.54688, 1.17188, 4.5625, 1.23438, 1.96875, 1.25, 5.5625, 3.21875, 1.92188, 8.75,
                3.59375, 5.84375, 3.07812, 5.96875, 9.6875, 8.5625, 3.5, 2.125, 3.09375, 3.5,
                1.82031, 6.25, 6.125, 9.75, 4.75, 6.0625, 4.3125, 1.16406, 8.3125, 8.1875,
                3.59375, 3.09375, 7.4375, 8.25, 6.5, 4.5, 4.8125, 8.75, 7.75, 7.71875,
                4.84375, 6, 4.84375, 2.21875, 4.25, 1.53906, 2.375, 2.09375, 9.375, 1.39844,
                9.25, 1.96875, 8, 3.03125, 6.5625, 7.40625, 1.32031, 6.03125, 6.875, 1.10938,
                2.15625, 1.64062, 3.65625, 9.6875, 4.25, 6.125, 3.46875, 2.82812, 1.66406, 3.26562,
                2.375, 7.6875, 2.45312, 2.75, 9.4375, 6.21875, 4.3125, 9.75, 1.45312, 8.625,
                7.65625, 3.15625, 3.6875, 5.4375, 2.84375, 6.5625, 9.8125, 8.4375, 9, 2.40625,
                7.8125, 1.16406, 6.875, 1.625, 1.35938, 5.375, 8.3125, 6.4375, 7.875, 6.125,
                5.09375, 3.84375, 5.78125, 9.875, 1.98438, 6.1875, 2.3125, 4.40625, 5.5625, 5.9375,
                2.9375, 7.6875, 9.25, 7, 5.15625, 3.375, 2.1875, 1.59375, 7.875, 4.3125,
                2.90625, 6.65625, 1.67188, 2.89062, 1.85938, 7.75, 2.45312, 1.59375, 4.1875, 3.34375,
                1.85938, 8.25, 2.28125, 2.73438, 9.375, 6.75, 6.1875, 5.71875, 8.5, 9.3125,
                6.625, 3.375, 3.90625, 1.59375, 7.5625, 7.625, 5.6875, 7.9375, 7.625, 9.125,
                2.48438, 9.375, 7.1875, 1.125, 4.8125, 3.09375, 7.5625, 6.5625, 7.8125, 9.5,
                4.5625, 9.5, 9.3125, 6, 2.82812, 9.25, 1.07031, 6.75, 9.3125, 4.5,
                3.65625, 5.375, 2.5, 6.4375, 1.21875, 5.9375, 5.0625, 9.3125, 8.25, 9.25,
                4.3125, 4.5625, 6.46875, 9.625, 1.3125, 2.5625, 4.1875, 2.125, 1.70312, 2.21875,
                7.25, 5.5625, 1.10938, 1.1875, 5.125, 9.5, 9.625, 8.4375, 4, 1.13281,
                5.25, 2.57812, 1.94531, 3.98438, 5.5, 2.17188, 9, 8.25, 5.8125, 4.09375,
                3.53125, 9.4375, 4.1875, 6.25, 9.0625, 8.875, 3.17188, 8.625, 1.21875, 9.125,
                9.6875, 5.125, 4.875, 5.90625, 4.125, 8.125, 6.1875, 3.5625, 2.125, 5.40625,
                9.5, 6.375, 3.8125, 1.14062, 9.5625, 6.3125, 2.96875, 4.875, 3.23438, 8.25,
                8.75, 3.84375, 3.125, 9, 8.3125, 6.1875, 5.875, 2.65625, 2.71875, 8.0625,
                6.3125, 6.5, 1.42969, 1.48438, 1.14062, 4.78125, 1.44531, 7.125, 4.59375, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 10},
            std::vector<int64_t>{10, 10, 10, 10, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 2.15625,
                5.71875, 1.17188, 1.75, 7.125, 9.25, 2.90625, 9.1875, 3.375, 3.6875, 5.4375,
                6.25, 1.47656, 6.0625, 6.15625, 6.5, 2.3125, 9.625, 6.3125, 3.34375, 7.3125,
                3.07812, 1.92188, 5.8125, 4.71875, 9.5, 7.25, 5.4375, 4.71875, 5.875, 1.45312,
                7.875, 5.8125, 1.40625, 6.96875, 2.25, 5.625, 8.125, 9.5, 1.26562, 6.25,
                8.9375, 9.125, 5.875, 2.23438, 5.03125, 2.25, 9, 8.25, 4.375, 4.5625,
                5.84375, 2.48438, 6.875, 9.375, 4.25, 4.125, 6.125, 7.75, 6.75, 7.53125,
                2.125, 8.9375, 7.1875, 6.625, 6.8125, 7.75, 4.1875, 4.125, 7.875, 3.42188,
                4.1875, 9.0625, 7.75, 4.84375, 8.875, 9.625, 1.10156, 6.96875, 5.46875, 6.59375,
                1.66406, 2.03125, 8.0625, 9.5, 1.57812, 5.0625, 4.1875, 6.1875, 9.5, 4.6875,
                4.40625, 3.125, 7.875, 9.125, 7.9375, 6.15625, 3.71875, 1.02344, 7.9375, 6.5625,
                2.375, 3.9375, 6.1875, 5.75, 1.07812, 9, 7.375, 4.1875, 5.25, 9.125,
                7.875, 6.625, 5.1875, 1.14062, 3.40625, 9.375, 8.5, 7.1875, 5.9375, 10,
                1.625, 2.54688, 5.25, 2.21875, 7.6875, 9.375, 2.71875, 7.25, 5.1875, 1.59375,
                3.0625, 7.8125, 5.5625, 7.78125, 2.875, 9.25, 1.4375, 7.375, 5.65625, 2.125,
                2.54688, 1.17188, 4.5625, 1.23438, 1.96875, 1.25, 5.5625, 3.21875, 1.92188, 8.75,
                3.59375, 5.84375, 3.07812, 5.96875, 9.6875, 8.5625, 3.5, 2.125, 3.09375, 3.5,
                1.82031, 6.25, 6.125, 9.75, 4.75, 6.0625, 4.3125, 1.16406, 8.3125, 8.1875,
                3.59375, 3.09375, 7.4375, 8.25, 6.5, 4.5, 4.8125, 8.75, 7.75, 7.71875,
                4.84375, 6, 4.84375, 2.21875, 4.25, 1.53906, 2.375, 2.09375, 9.375, 1.39844,
                9.25, 1.96875, 8, 3.03125, 6.5625, 7.40625, 1.32031, 6.03125, 6.875, 1.10938,
                2.15625, 1.64062, 3.65625, 9.6875, 4.25, 6.125, 3.46875, 2.82812, 1.66406, 3.26562,
                2.375, 7.6875, 2.45312, 2.75, 9.4375, 6.21875, 4.3125, 9.75, 1.45312, 8.625,
                7.65625, 3.15625, 3.6875, 5.4375, 2.84375, 6.5625, 9.8125, 8.4375, 9, 2.40625,
                7.8125, 1.16406, 6.875, 1.625, 1.35938, 5.375, 8.3125, 6.4375, 7.875, 6.125,
                5.09375, 3.84375, 5.78125, 9.875, 1.98438, 6.1875, 2.3125, 4.40625, 5.5625, 5.9375,
                2.9375, 7.6875, 9.25, 7, 5.15625, 3.375, 2.1875, 1.59375, 7.875, 4.3125,
                2.90625, 6.65625, 1.67188, 2.89062, 1.85938, 7.75, 2.45312, 1.59375, 4.1875, 3.34375,
                1.85938, 8.25, 2.28125, 2.73438, 9.375, 6.75, 6.1875, 5.71875, 8.5, 9.3125,
                6.625, 3.375, 3.90625, 1.59375, 7.5625, 7.625, 5.6875, 7.9375, 7.625, 9.125,
                2.48438, 9.375, 7.1875, 1.125, 4.8125, 3.09375, 7.5625, 6.5625, 7.8125, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 2.15625,
                5.71875, 1.17188, 1.75, 7.125, 9.25, 2.90625, 9.1875, 3.375, 3.6875, 5.4375,
                6.25, 1.47656, 6.0625, 6.15625, 6.5, 2.3125, 9.625, 6.3125, 3.34375, 7.3125,
                3.07812, 1.92188, 5.8125, 4.71875, 9.5, 7.25, 5.4375, 4.71875, 5.875, 1.45312,
                7.875, 5.8125, 1.40625, 6.96875, 2.25, 5.625, 8.125, 9.5, 1.26562, 6.25,
                8.9375, 9.125, 5.875, 2.23438, 5.03125, 2.25, 9, 8.25, 4.375, 4.5625,
                5.84375, 2.48438, 6.875, 9.375, 4.25, 4.125, 6.125, 7.75, 6.75, 7.53125,
                2.125, 8.9375, 7.1875, 6.625, 6.8125, 7.75, 4.1875, 4.125, 7.875, 3.42188,
                4.1875, 9.0625, 7.75, 4.84375, 8.875, 9.625, 1.10156, 6.96875, 5.46875, 6.59375,
                1.66406, 2.03125, 8.0625, 9.5, 1.57812, 5.0625, 4.1875, 6.1875, 9.5, 4.6875,
                4.40625, 3.125, 7.875, 9.125, 7.9375, 6.15625, 3.71875, 1.02344, 7.9375, 6.5625,
                2.375, 3.9375, 6.1875, 5.75, 1.07812, 9, 7.375, 4.1875, 5.25, 9.125,
                7.875, 6.625, 5.1875, 1.14062, 3.40625, 9.375, 8.5, 7.1875, 5.9375, 10,
                1.625, 2.54688, 5.25, 2.21875, 7.6875, 9.375, 2.71875, 7.25, 5.1875, 1.59375,
                3.0625, 7.8125, 5.5625, 7.78125, 2.875, 9.25, 1.4375, 7.375, 5.65625, 2.125,
                2.54688, 1.17188, 4.5625, 1.23438, 1.96875, 1.25, 5.5625, 3.21875, 1.92188, 8.75,
                3.59375, 5.84375, 3.07812, 5.96875, 9.6875, 8.5625, 3.5, 2.125, 3.09375, 3.5,
                1.82031, 6.25, 6.125, 9.75, 4.75, 6.0625, 4.3125, 1.16406, 8.3125, 8.1875,
                3.59375, 3.09375, 7.4375, 8.25, 6.5, 4.5, 4.8125, 8.75, 7.75, 7.71875,
                4.84375, 6, 4.84375, 2.21875, 4.25, 1.53906, 2.375, 2.09375, 9.375, 1.39844,
                9.25, 1.96875, 8, 3.03125, 6.5625, 7.40625, 1.32031, 6.03125, 6.875, 1.10938,
                2.15625, 1.64062, 3.65625, 9.6875, 4.25, 6.125, 3.46875, 2.82812, 1.66406, 3.26562,
                2.375, 7.6875, 2.45312, 2.75, 9.4375, 6.21875, 4.3125, 9.75, 1.45312, 8.625,
                7.65625, 3.15625, 3.6875, 5.4375, 2.84375, 6.5625, 9.8125, 8.4375, 9, 2.40625,
                7.8125, 1.16406, 6.875, 1.625, 1.35938, 5.375, 8.3125, 6.4375, 7.875, 6.125,
                5.09375, 3.84375, 5.78125, 9.875, 1.98438, 6.1875, 2.3125, 4.40625, 5.5625, 5.9375,
                2.9375, 7.6875, 9.25, 7, 5.15625, 3.375, 2.1875, 1.59375, 7.875, 4.3125,
                2.90625, 6.65625, 1.67188, 2.89062, 1.85938, 7.75, 2.45312, 1.59375, 4.1875, 3.34375,
                1.85938, 8.25, 2.28125, 2.73438, 9.375, 6.75, 6.1875, 5.71875, 8.5, 9.3125,
                6.625, 3.375, 3.90625, 1.59375, 7.5625, 7.625, 5.6875, 7.9375, 7.625, 9.125,
                2.48438, 9.375, 7.1875, 1.125, 4.8125, 3.09375, 7.5625, 6.5625, 7.8125, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 10},
            std::vector<T>{
                0.523438, 0.667969, 0.667969, 0.667969, 0.667969, 0.523438, 0.632812, 0.664062, 0.667969, 0.640625,
                0.539062, 0.664062, 0.667969, 0.667969, 0.667969, 0.539062, 0.617188, 0.65625, 0.667969, 0.625,
                0.546875, 0.648438, 0.667969, 0.664062, 0.667969, 0.546875, 0.601562, 0.640625, 0.667969, 0.609375,
                0.546875, 0.632812, 0.664062, 0.65625, 0.664062, 0.546875, 0.585938, 0.625, 0.664062, 0.59375,
                0.554688, 0.617188, 0.65625, 0.640625, 0.648438, 0.554688, 0.578125, 0.609375, 0.65625, 0.585938,
                0.554688, 0.601562, 0.640625, 0.625, 0.640625, 0.554688, 0.570312, 0.59375, 0.640625, 0.578125,
                0.554688, 0.59375, 0.625, 0.609375, 0.625, 0.554688, 0.570312, 0.585938, 0.625, 0.570312,
                0.554688, 0.585938, 0.609375, 0.59375, 0.609375, 0.554688, 0.5625, 0.578125, 0.609375, 0.570312,
                0.554688, 0.570312, 0.59375, 0.585938, 0.59375, 0.554688, 0.5625, 0.570312, 0.59375, 0.5625,
                0.554688, 0.570312, 0.585938, 0.578125, 0.585938, 0.554688, 0.5625, 0.570312, 0.585938, 0.5625,
                0.65625, 0.617188, 0.664062, 0.648438, 0.664062, 0.664062, 0.667969, 0.664062, 0.667969, 0.667969,
                0.648438, 0.601562, 0.664062, 0.632812, 0.664062, 0.65625, 0.667969, 0.664062, 0.667969, 0.664062,
                0.632812, 0.585938, 0.648438, 0.617188, 0.648438, 0.648438, 0.664062, 0.648438, 0.667969, 0.65625,
                0.617188, 0.578125, 0.632812, 0.601562, 0.632812, 0.632812, 0.65625, 0.632812, 0.664062, 0.648438,
                0.601562, 0.570312, 0.617188, 0.585938, 0.617188, 0.617188, 0.640625, 0.617188, 0.648438, 0.632812,
                0.585938, 0.570312, 0.601562, 0.578125, 0.601562, 0.601562, 0.625, 0.601562, 0.640625, 0.617188,
                0.578125, 0.5625, 0.585938, 0.570312, 0.585938, 0.585938, 0.609375, 0.585938, 0.625, 0.601562,
                0.570312, 0.5625, 0.578125, 0.570312, 0.578125, 0.578125, 0.59375, 0.578125, 0.609375, 0.585938,
                0.570312, 0.5625, 0.570312, 0.5625, 0.570312, 0.570312, 0.585938, 0.570312, 0.59375, 0.578125,
                0.5625, 0.554688, 0.570312, 0.5625, 0.570312, 0.570312, 0.578125, 0.570312, 0.585938, 0.570312,
                0.667969, 0.667969, 0.664062, 0.667969, 0.667969, 0.648438, 0.667969, 0.667969, 0.65625, 0.5625,
                0.667969, 0.664062, 0.65625, 0.667969, 0.664062, 0.640625, 0.664062, 0.667969, 0.640625, 0.5625,
                0.664062, 0.648438, 0.640625, 0.664062, 0.65625, 0.625, 0.65625, 0.664062, 0.625, 0.554688,
                0.65625, 0.632812, 0.625, 0.65625, 0.648438, 0.609375, 0.640625, 0.664062, 0.609375, 0.554688,
                0.648438, 0.617188, 0.609375, 0.640625, 0.632812, 0.59375, 0.625, 0.648438, 0.59375, 0.554688,
                0.632812, 0.601562, 0.59375, 0.625, 0.617188, 0.585938, 0.609375, 0.632812, 0.585938, 0.554688,
                0.617188, 0.59375, 0.585938, 0.609375, 0.601562, 0.578125, 0.59375, 0.617188, 0.578125, 0.554688,
                0.601562, 0.585938, 0.578125, 0.59375, 0.585938, 0.570312, 0.585938, 0.601562, 0.570312, 0.554688,
                0.585938, 0.570312, 0.570312, 0.585938, 0.578125, 0.570312, 0.578125, 0.585938, 0.570312, 0.554688,
                0.578125, 0.570312, 0.570312, 0.578125, 0.570312, 0.5625, 0.570312, 0.578125, 0.5625, 0.554688,
                0.667969, 0.667969, 0.667969, 0.667969, 0.667969, 0.667969, 0.667969, 0.632812, 0.667969, 0.648438,
                0.664062, 0.667969, 0.667969, 0.664062, 0.664062, 0.664062, 0.664062, 0.617188, 0.667969, 0.632812,
                0.65625, 0.664062, 0.667969, 0.648438, 0.65625, 0.65625, 0.648438, 0.601562, 0.667969, 0.617188,
                0.648438, 0.65625, 0.664062, 0.632812, 0.640625, 0.648438, 0.640625, 0.59375, 0.664062, 0.601562,
                0.632812, 0.640625, 0.648438, 0.617188, 0.625, 0.632812, 0.625, 0.585938, 0.648438, 0.59375,
                0.617188, 0.625, 0.632812, 0.601562, 0.609375, 0.617188, 0.609375, 0.570312, 0.640625, 0.585938,
                0.601562, 0.609375, 0.617188, 0.59375, 0.59375, 0.601562, 0.59375, 0.570312, 0.625, 0.570312,
                0.585938, 0.59375, 0.601562, 0.585938, 0.585938, 0.585938, 0.585938, 0.5625, 0.609375, 0.570312,
                0.578125, 0.585938, 0.59375, 0.570312, 0.578125, 0.578125, 0.578125, 0.5625, 0.59375, 0.5625,
                0.570312, 0.578125, 0.585938, 0.570312, 0.570312, 0.570312, 0.570312, 0.5625, 0.585938, 0.5625,
                0.667969, 0.667969, 0.667969, 0.667969, 0.667969, 0.664062, 0.617188, 0.667969, 0.667969, 0.667969,
                0.667969, 0.667969, 0.667969, 0.667969, 0.667969, 0.65625, 0.601562, 0.667969, 0.664062, 0.667969,
                0.664062, 0.664062, 0.664062, 0.667969, 0.664062, 0.640625, 0.585938, 0.664062, 0.65625, 0.667969,
                0.65625, 0.65625, 0.65625, 0.664062, 0.65625, 0.625, 0.578125, 0.65625, 0.648438, 0.664062,
                0.648438, 0.648438, 0.640625, 0.65625, 0.648438, 0.609375, 0.570312, 0.640625, 0.632812, 0.65625,
                0.632812, 0.632812, 0.625, 0.640625, 0.632812, 0.59375, 0.570312, 0.625, 0.617188, 0.640625,
                0.617188, 0.617188, 0.609375, 0.625, 0.617188, 0.585938, 0.5625, 0.609375, 0.601562, 0.625,
                0.601562, 0.601562, 0.59375, 0.609375, 0.601562, 0.578125, 0.5625, 0.59375, 0.585938, 0.609375,
                0.585938, 0.585938, 0.585938, 0.59375, 0.585938, 0.570312, 0.5625, 0.585938, 0.578125, 0.59375,
                0.578125, 0.578125, 0.578125, 0.585938, 0.578125, 0.570312, 0.554688, 0.578125, 0.570312, 0.585938},
            std::vector<T>{
                0.554688, 0.570312, 0.585938, 0.578125, 0.585938, 0.554688, 0.5625, 0.570312, 0.585938, 0.5625,
                0.5625, 0.554688, 0.570312, 0.5625, 0.570312, 0.570312, 0.578125, 0.570312, 0.585938, 0.570312,
                0.578125, 0.570312, 0.570312, 0.578125, 0.570312, 0.5625, 0.570312, 0.578125, 0.5625, 0.554688,
                0.570312, 0.578125, 0.585938, 0.570312, 0.570312, 0.570312, 0.570312, 0.5625, 0.585938, 0.5625,
                0.578125, 0.578125, 0.578125, 0.585938, 0.578125, 0.570312, 0.554688, 0.578125, 0.570312, 0.585938},
            std::vector<T>{
                1.20312, 1.27344, 1.375, 1.32031, 1.35156, 1.20312, 1.22656, 1.25781, 1.375, 1.23438,
                1.25, 1.21875, 1.26562, 1.23438, 1.26562, 1.26562, 1.32031, 1.26562, 1.35156, 1.28906,
                1.34375, 1.27344, 1.25781, 1.32031, 1.28906, 1.24219, 1.28125, 1.34375, 1.24219, 1.21875,
                1.28906, 1.32031, 1.35156, 1.27344, 1.28125, 1.29688, 1.28125, 1.22656, 1.35156, 1.23438,
                1.32812, 1.32812, 1.32031, 1.35938, 1.32812, 1.25781, 1.21875, 1.32031, 1.28906, 1.375}),
        LSTMSequenceParams(
            5, 10, 10, 10,
            0.7f, op::RecurrentSequenceDirection::REVERSE,
            ET,
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 2.15625,
                5.71875, 1.17188, 1.75, 7.125, 9.25, 2.90625, 9.1875, 3.375, 3.6875, 5.4375,
                6.25, 1.47656, 6.0625, 6.15625, 6.5, 2.3125, 9.625, 6.3125, 3.34375, 7.3125,
                3.07812, 1.92188, 5.8125, 4.71875, 9.5, 7.25, 5.4375, 4.71875, 5.875, 1.45312,
                7.875, 5.8125, 1.40625, 6.96875, 2.25, 5.625, 8.125, 9.5, 1.26562, 6.25,
                8.9375, 9.125, 5.875, 2.23438, 5.03125, 2.25, 9, 8.25, 4.375, 4.5625,
                5.84375, 2.48438, 6.875, 9.375, 4.25, 4.125, 6.125, 7.75, 6.75, 7.53125,
                2.125, 8.9375, 7.1875, 6.625, 6.8125, 7.75, 4.1875, 4.125, 7.875, 3.42188,
                4.1875, 9.0625, 7.75, 4.84375, 8.875, 9.625, 1.10156, 6.96875, 5.46875, 6.59375,
                1.66406, 2.03125, 8.0625, 9.5, 1.57812, 5.0625, 4.1875, 6.1875, 9.5, 4.6875,
                4.40625, 3.125, 7.875, 9.125, 7.9375, 6.15625, 3.71875, 1.02344, 7.9375, 6.5625,
                2.375, 3.9375, 6.1875, 5.75, 1.07812, 9, 7.375, 4.1875, 5.25, 9.125,
                7.875, 6.625, 5.1875, 1.14062, 3.40625, 9.375, 8.5, 7.1875, 5.9375, 10,
                1.625, 2.54688, 5.25, 2.21875, 7.6875, 9.375, 2.71875, 7.25, 5.1875, 1.59375,
                3.0625, 7.8125, 5.5625, 7.78125, 2.875, 9.25, 1.4375, 7.375, 5.65625, 2.125,
                2.54688, 1.17188, 4.5625, 1.23438, 1.96875, 1.25, 5.5625, 3.21875, 1.92188, 8.75,
                3.59375, 5.84375, 3.07812, 5.96875, 9.6875, 8.5625, 3.5, 2.125, 3.09375, 3.5,
                1.82031, 6.25, 6.125, 9.75, 4.75, 6.0625, 4.3125, 1.16406, 8.3125, 8.1875,
                3.59375, 3.09375, 7.4375, 8.25, 6.5, 4.5, 4.8125, 8.75, 7.75, 7.71875,
                4.84375, 6, 4.84375, 2.21875, 4.25, 1.53906, 2.375, 2.09375, 9.375, 1.39844,
                9.25, 1.96875, 8, 3.03125, 6.5625, 7.40625, 1.32031, 6.03125, 6.875, 1.10938,
                2.15625, 1.64062, 3.65625, 9.6875, 4.25, 6.125, 3.46875, 2.82812, 1.66406, 3.26562,
                2.375, 7.6875, 2.45312, 2.75, 9.4375, 6.21875, 4.3125, 9.75, 1.45312, 8.625,
                7.65625, 3.15625, 3.6875, 5.4375, 2.84375, 6.5625, 9.8125, 8.4375, 9, 2.40625,
                7.8125, 1.16406, 6.875, 1.625, 1.35938, 5.375, 8.3125, 6.4375, 7.875, 6.125,
                5.09375, 3.84375, 5.78125, 9.875, 1.98438, 6.1875, 2.3125, 4.40625, 5.5625, 5.9375,
                2.9375, 7.6875, 9.25, 7, 5.15625, 3.375, 2.1875, 1.59375, 7.875, 4.3125,
                2.90625, 6.65625, 1.67188, 2.89062, 1.85938, 7.75, 2.45312, 1.59375, 4.1875, 3.34375,
                1.85938, 8.25, 2.28125, 2.73438, 9.375, 6.75, 6.1875, 5.71875, 8.5, 9.3125,
                6.625, 3.375, 3.90625, 1.59375, 7.5625, 7.625, 5.6875, 7.9375, 7.625, 9.125,
                2.48438, 9.375, 7.1875, 1.125, 4.8125, 3.09375, 7.5625, 6.5625, 7.8125, 9.5,
                4.5625, 9.5, 9.3125, 6, 2.82812, 9.25, 1.07031, 6.75, 9.3125, 4.5,
                3.65625, 5.375, 2.5, 6.4375, 1.21875, 5.9375, 5.0625, 9.3125, 8.25, 9.25,
                4.3125, 4.5625, 6.46875, 9.625, 1.3125, 2.5625, 4.1875, 2.125, 1.70312, 2.21875,
                7.25, 5.5625, 1.10938, 1.1875, 5.125, 9.5, 9.625, 8.4375, 4, 1.13281,
                5.25, 2.57812, 1.94531, 3.98438, 5.5, 2.17188, 9, 8.25, 5.8125, 4.09375,
                3.53125, 9.4375, 4.1875, 6.25, 9.0625, 8.875, 3.17188, 8.625, 1.21875, 9.125,
                9.6875, 5.125, 4.875, 5.90625, 4.125, 8.125, 6.1875, 3.5625, 2.125, 5.40625,
                9.5, 6.375, 3.8125, 1.14062, 9.5625, 6.3125, 2.96875, 4.875, 3.23438, 8.25,
                8.75, 3.84375, 3.125, 9, 8.3125, 6.1875, 5.875, 2.65625, 2.71875, 8.0625,
                6.3125, 6.5, 1.42969, 1.48438, 1.14062, 4.78125, 1.44531, 7.125, 4.59375, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 10},
            std::vector<int64_t>{10, 10, 10, 10, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 2.15625,
                5.71875, 1.17188, 1.75, 7.125, 9.25, 2.90625, 9.1875, 3.375, 3.6875, 5.4375,
                6.25, 1.47656, 6.0625, 6.15625, 6.5, 2.3125, 9.625, 6.3125, 3.34375, 7.3125,
                3.07812, 1.92188, 5.8125, 4.71875, 9.5, 7.25, 5.4375, 4.71875, 5.875, 1.45312,
                7.875, 5.8125, 1.40625, 6.96875, 2.25, 5.625, 8.125, 9.5, 1.26562, 6.25,
                8.9375, 9.125, 5.875, 2.23438, 5.03125, 2.25, 9, 8.25, 4.375, 4.5625,
                5.84375, 2.48438, 6.875, 9.375, 4.25, 4.125, 6.125, 7.75, 6.75, 7.53125,
                2.125, 8.9375, 7.1875, 6.625, 6.8125, 7.75, 4.1875, 4.125, 7.875, 3.42188,
                4.1875, 9.0625, 7.75, 4.84375, 8.875, 9.625, 1.10156, 6.96875, 5.46875, 6.59375,
                1.66406, 2.03125, 8.0625, 9.5, 1.57812, 5.0625, 4.1875, 6.1875, 9.5, 4.6875,
                4.40625, 3.125, 7.875, 9.125, 7.9375, 6.15625, 3.71875, 1.02344, 7.9375, 6.5625,
                2.375, 3.9375, 6.1875, 5.75, 1.07812, 9, 7.375, 4.1875, 5.25, 9.125,
                7.875, 6.625, 5.1875, 1.14062, 3.40625, 9.375, 8.5, 7.1875, 5.9375, 10,
                1.625, 2.54688, 5.25, 2.21875, 7.6875, 9.375, 2.71875, 7.25, 5.1875, 1.59375,
                3.0625, 7.8125, 5.5625, 7.78125, 2.875, 9.25, 1.4375, 7.375, 5.65625, 2.125,
                2.54688, 1.17188, 4.5625, 1.23438, 1.96875, 1.25, 5.5625, 3.21875, 1.92188, 8.75,
                3.59375, 5.84375, 3.07812, 5.96875, 9.6875, 8.5625, 3.5, 2.125, 3.09375, 3.5,
                1.82031, 6.25, 6.125, 9.75, 4.75, 6.0625, 4.3125, 1.16406, 8.3125, 8.1875,
                3.59375, 3.09375, 7.4375, 8.25, 6.5, 4.5, 4.8125, 8.75, 7.75, 7.71875,
                4.84375, 6, 4.84375, 2.21875, 4.25, 1.53906, 2.375, 2.09375, 9.375, 1.39844,
                9.25, 1.96875, 8, 3.03125, 6.5625, 7.40625, 1.32031, 6.03125, 6.875, 1.10938,
                2.15625, 1.64062, 3.65625, 9.6875, 4.25, 6.125, 3.46875, 2.82812, 1.66406, 3.26562,
                2.375, 7.6875, 2.45312, 2.75, 9.4375, 6.21875, 4.3125, 9.75, 1.45312, 8.625,
                7.65625, 3.15625, 3.6875, 5.4375, 2.84375, 6.5625, 9.8125, 8.4375, 9, 2.40625,
                7.8125, 1.16406, 6.875, 1.625, 1.35938, 5.375, 8.3125, 6.4375, 7.875, 6.125,
                5.09375, 3.84375, 5.78125, 9.875, 1.98438, 6.1875, 2.3125, 4.40625, 5.5625, 5.9375,
                2.9375, 7.6875, 9.25, 7, 5.15625, 3.375, 2.1875, 1.59375, 7.875, 4.3125,
                2.90625, 6.65625, 1.67188, 2.89062, 1.85938, 7.75, 2.45312, 1.59375, 4.1875, 3.34375,
                1.85938, 8.25, 2.28125, 2.73438, 9.375, 6.75, 6.1875, 5.71875, 8.5, 9.3125,
                6.625, 3.375, 3.90625, 1.59375, 7.5625, 7.625, 5.6875, 7.9375, 7.625, 9.125,
                2.48438, 9.375, 7.1875, 1.125, 4.8125, 3.09375, 7.5625, 6.5625, 7.8125, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 2.78125,
                8, 8.1875, 7.4375, 9.6875, 8.25, 3.8125, 1.82812, 7.21875, 5.65625, 8.875,
                8.75, 9, 8.4375, 1.76562, 8.4375, 1.34375, 3.45312, 2.53125, 1.53125, 8.875,
                7.03125, 1.88281, 6.3125, 4.78125, 7.03125, 9.625, 4.6875, 5.8125, 2.78125, 7.21875,
                3.59375, 3.84375, 2.28125, 7.1875, 8, 8.5, 4.6875, 1.16406, 1.30469, 7.75,
                6.625, 9.875, 6.9375, 7.71875, 3.6875, 3.53125, 5, 8.125, 3, 1.92188,
                1.65625, 5, 5.21875, 9.125, 1.85938, 3.64062, 9.125, 3.59375, 2.0625, 2.15625,
                5.71875, 1.17188, 1.75, 7.125, 9.25, 2.90625, 9.1875, 3.375, 3.6875, 5.4375,
                6.25, 1.47656, 6.0625, 6.15625, 6.5, 2.3125, 9.625, 6.3125, 3.34375, 7.3125,
                3.07812, 1.92188, 5.8125, 4.71875, 9.5, 7.25, 5.4375, 4.71875, 5.875, 1.45312,
                7.875, 5.8125, 1.40625, 6.96875, 2.25, 5.625, 8.125, 9.5, 1.26562, 6.25,
                8.9375, 9.125, 5.875, 2.23438, 5.03125, 2.25, 9, 8.25, 4.375, 4.5625,
                5.84375, 2.48438, 6.875, 9.375, 4.25, 4.125, 6.125, 7.75, 6.75, 7.53125,
                2.125, 8.9375, 7.1875, 6.625, 6.8125, 7.75, 4.1875, 4.125, 7.875, 3.42188,
                4.1875, 9.0625, 7.75, 4.84375, 8.875, 9.625, 1.10156, 6.96875, 5.46875, 6.59375,
                1.66406, 2.03125, 8.0625, 9.5, 1.57812, 5.0625, 4.1875, 6.1875, 9.5, 4.6875,
                4.40625, 3.125, 7.875, 9.125, 7.9375, 6.15625, 3.71875, 1.02344, 7.9375, 6.5625,
                2.375, 3.9375, 6.1875, 5.75, 1.07812, 9, 7.375, 4.1875, 5.25, 9.125,
                7.875, 6.625, 5.1875, 1.14062, 3.40625, 9.375, 8.5, 7.1875, 5.9375, 10,
                1.625, 2.54688, 5.25, 2.21875, 7.6875, 9.375, 2.71875, 7.25, 5.1875, 1.59375,
                3.0625, 7.8125, 5.5625, 7.78125, 2.875, 9.25, 1.4375, 7.375, 5.65625, 2.125,
                2.54688, 1.17188, 4.5625, 1.23438, 1.96875, 1.25, 5.5625, 3.21875, 1.92188, 8.75,
                3.59375, 5.84375, 3.07812, 5.96875, 9.6875, 8.5625, 3.5, 2.125, 3.09375, 3.5,
                1.82031, 6.25, 6.125, 9.75, 4.75, 6.0625, 4.3125, 1.16406, 8.3125, 8.1875,
                3.59375, 3.09375, 7.4375, 8.25, 6.5, 4.5, 4.8125, 8.75, 7.75, 7.71875,
                4.84375, 6, 4.84375, 2.21875, 4.25, 1.53906, 2.375, 2.09375, 9.375, 1.39844,
                9.25, 1.96875, 8, 3.03125, 6.5625, 7.40625, 1.32031, 6.03125, 6.875, 1.10938,
                2.15625, 1.64062, 3.65625, 9.6875, 4.25, 6.125, 3.46875, 2.82812, 1.66406, 3.26562,
                2.375, 7.6875, 2.45312, 2.75, 9.4375, 6.21875, 4.3125, 9.75, 1.45312, 8.625,
                7.65625, 3.15625, 3.6875, 5.4375, 2.84375, 6.5625, 9.8125, 8.4375, 9, 2.40625,
                7.8125, 1.16406, 6.875, 1.625, 1.35938, 5.375, 8.3125, 6.4375, 7.875, 6.125,
                5.09375, 3.84375, 5.78125, 9.875, 1.98438, 6.1875, 2.3125, 4.40625, 5.5625, 5.9375,
                2.9375, 7.6875, 9.25, 7, 5.15625, 3.375, 2.1875, 1.59375, 7.875, 4.3125,
                2.90625, 6.65625, 1.67188, 2.89062, 1.85938, 7.75, 2.45312, 1.59375, 4.1875, 3.34375,
                1.85938, 8.25, 2.28125, 2.73438, 9.375, 6.75, 6.1875, 5.71875, 8.5, 9.3125,
                6.625, 3.375, 3.90625, 1.59375, 7.5625, 7.625, 5.6875, 7.9375, 7.625, 9.125,
                2.48438, 9.375, 7.1875, 1.125, 4.8125, 3.09375, 7.5625, 6.5625, 7.8125, 10},
            std::vector<T>{
                1, 4.75, 10, 7.46875, 9.375, 1, 2.15625, 3.71875, 10, 2.3125,
                3.125, 1.82812, 4.5625, 2.67188, 4.5, 4.125, 7, 4.5625, 9.375, 5.84375,
                8.625, 4.75, 3.8125, 7.15625, 5.71875, 2.84375, 5, 8.875, 3.0625, 1.25,
                5.8125, 7.03125, 9.25, 4.75, 5.125, 6, 4.875, 2.25, 9.4375, 10},
            std::vector<T>{
                0.554688, 0.570312, 0.585938, 0.578125, 0.585938, 0.554688, 0.5625, 0.570312, 0.585938, 0.5625,
                0.554688, 0.570312, 0.59375, 0.585938, 0.59375, 0.554688, 0.5625, 0.570312, 0.59375, 0.5625,
                0.554688, 0.585938, 0.609375, 0.59375, 0.609375, 0.554688, 0.5625, 0.578125, 0.609375, 0.570312,
                0.554688, 0.59375, 0.625, 0.609375, 0.625, 0.554688, 0.570312, 0.585938, 0.625, 0.570312,
                0.554688, 0.601562, 0.640625, 0.625, 0.640625, 0.554688, 0.570312, 0.59375, 0.640625, 0.578125,
                0.554688, 0.617188, 0.65625, 0.640625, 0.648438, 0.554688, 0.578125, 0.609375, 0.65625, 0.585938,
                0.546875, 0.632812, 0.664062, 0.65625, 0.664062, 0.546875, 0.585938, 0.625, 0.664062, 0.59375,
                0.546875, 0.648438, 0.667969, 0.664062, 0.667969, 0.546875, 0.601562, 0.640625, 0.667969, 0.609375,
                0.539062, 0.664062, 0.667969, 0.667969, 0.667969, 0.539062, 0.617188, 0.65625, 0.667969, 0.625,
                0.523438, 0.667969, 0.667969, 0.667969, 0.667969, 0.523438, 0.632812, 0.664062, 0.667969, 0.640625,
                0.5625, 0.554688, 0.570312, 0.5625, 0.570312, 0.570312, 0.578125, 0.570312, 0.585938, 0.570312,
                0.570312, 0.5625, 0.570312, 0.5625, 0.570312, 0.570312, 0.585938, 0.570312, 0.59375, 0.578125,
                0.570312, 0.5625, 0.578125, 0.570312, 0.578125, 0.578125, 0.59375, 0.578125, 0.609375, 0.585938,
                0.578125, 0.5625, 0.585938, 0.570312, 0.585938, 0.585938, 0.609375, 0.585938, 0.625, 0.601562,
                0.585938, 0.570312, 0.601562, 0.578125, 0.601562, 0.601562, 0.625, 0.601562, 0.640625, 0.617188,
                0.601562, 0.570312, 0.617188, 0.585938, 0.617188, 0.617188, 0.640625, 0.617188, 0.648438, 0.632812,
                0.617188, 0.578125, 0.632812, 0.601562, 0.632812, 0.632812, 0.65625, 0.632812, 0.664062, 0.648438,
                0.632812, 0.585938, 0.648438, 0.617188, 0.648438, 0.648438, 0.664062, 0.648438, 0.667969, 0.65625,
                0.648438, 0.601562, 0.664062, 0.632812, 0.664062, 0.65625, 0.667969, 0.664062, 0.667969, 0.664062,
                0.65625, 0.617188, 0.664062, 0.648438, 0.664062, 0.664062, 0.667969, 0.664062, 0.667969, 0.667969,
                0.578125, 0.570312, 0.570312, 0.578125, 0.570312, 0.5625, 0.570312, 0.578125, 0.5625, 0.554688,
                0.585938, 0.570312, 0.570312, 0.585938, 0.578125, 0.570312, 0.578125, 0.585938, 0.570312, 0.554688,
                0.601562, 0.585938, 0.578125, 0.59375, 0.585938, 0.570312, 0.585938, 0.601562, 0.570312, 0.554688,
                0.617188, 0.59375, 0.585938, 0.609375, 0.601562, 0.578125, 0.59375, 0.617188, 0.578125, 0.554688,
                0.632812, 0.601562, 0.59375, 0.625, 0.617188, 0.585938, 0.609375, 0.632812, 0.585938, 0.554688,
                0.648438, 0.617188, 0.609375, 0.640625, 0.632812, 0.59375, 0.625, 0.648438, 0.59375, 0.554688,
                0.65625, 0.632812, 0.625, 0.65625, 0.648438, 0.609375, 0.640625, 0.664062, 0.609375, 0.554688,
                0.664062, 0.648438, 0.640625, 0.664062, 0.65625, 0.625, 0.65625, 0.664062, 0.625, 0.554688,
                0.667969, 0.664062, 0.65625, 0.667969, 0.664062, 0.640625, 0.664062, 0.667969, 0.640625, 0.5625,
                0.667969, 0.667969, 0.664062, 0.667969, 0.667969, 0.648438, 0.667969, 0.667969, 0.65625, 0.5625,
                0.570312, 0.578125, 0.585938, 0.570312, 0.570312, 0.570312, 0.570312, 0.5625, 0.585938, 0.5625,
                0.578125, 0.585938, 0.59375, 0.570312, 0.578125, 0.578125, 0.578125, 0.5625, 0.59375, 0.5625,
                0.585938, 0.59375, 0.601562, 0.585938, 0.585938, 0.585938, 0.585938, 0.5625, 0.609375, 0.570312,
                0.601562, 0.609375, 0.617188, 0.59375, 0.59375, 0.601562, 0.59375, 0.570312, 0.625, 0.570312,
                0.617188, 0.625, 0.632812, 0.601562, 0.609375, 0.617188, 0.609375, 0.570312, 0.640625, 0.585938,
                0.632812, 0.640625, 0.648438, 0.617188, 0.625, 0.632812, 0.625, 0.585938, 0.648438, 0.59375,
                0.648438, 0.65625, 0.664062, 0.632812, 0.640625, 0.648438, 0.640625, 0.59375, 0.664062, 0.601562,
                0.65625, 0.664062, 0.667969, 0.648438, 0.65625, 0.65625, 0.648438, 0.601562, 0.667969, 0.617188,
                0.664062, 0.667969, 0.667969, 0.664062, 0.664062, 0.664062, 0.664062, 0.617188, 0.667969, 0.632812,
                0.667969, 0.667969, 0.667969, 0.667969, 0.667969, 0.667969, 0.667969, 0.632812, 0.667969, 0.648438,
                0.578125, 0.578125, 0.578125, 0.585938, 0.578125, 0.570312, 0.554688, 0.578125, 0.570312, 0.585938,
                0.585938, 0.585938, 0.585938, 0.59375, 0.585938, 0.570312, 0.5625, 0.585938, 0.578125, 0.59375,
                0.601562, 0.601562, 0.59375, 0.609375, 0.601562, 0.578125, 0.5625, 0.59375, 0.585938, 0.609375,
                0.617188, 0.617188, 0.609375, 0.625, 0.617188, 0.585938, 0.5625, 0.609375, 0.601562, 0.625,
                0.632812, 0.632812, 0.625, 0.640625, 0.632812, 0.59375, 0.570312, 0.625, 0.617188, 0.640625,
                0.648438, 0.648438, 0.640625, 0.65625, 0.648438, 0.609375, 0.570312, 0.640625, 0.632812, 0.65625,
                0.65625, 0.65625, 0.65625, 0.664062, 0.65625, 0.625, 0.578125, 0.65625, 0.648438, 0.664062,
                0.664062, 0.664062, 0.664062, 0.667969, 0.664062, 0.640625, 0.585938, 0.664062, 0.65625, 0.667969,
                0.667969, 0.667969, 0.667969, 0.667969, 0.667969, 0.65625, 0.601562, 0.667969, 0.664062, 0.667969,
                0.667969, 0.667969, 0.667969, 0.667969, 0.667969, 0.664062, 0.617188, 0.667969, 0.667969, 0.667969},
            std::vector<T>{
                0.554688, 0.570312, 0.585938, 0.578125, 0.585938, 0.554688, 0.5625, 0.570312, 0.585938, 0.5625,
                0.5625, 0.554688, 0.570312, 0.5625, 0.570312, 0.570312, 0.578125, 0.570312, 0.585938, 0.570312,
                0.578125, 0.570312, 0.570312, 0.578125, 0.570312, 0.5625, 0.570312, 0.578125, 0.5625, 0.554688,
                0.570312, 0.578125, 0.585938, 0.570312, 0.570312, 0.570312, 0.570312, 0.5625, 0.585938, 0.5625,
                0.578125, 0.578125, 0.578125, 0.585938, 0.578125, 0.570312, 0.554688, 0.578125, 0.570312, 0.585938},
            std::vector<T>{
                1.20312, 1.27344, 1.375, 1.32031, 1.35156, 1.20312, 1.22656, 1.25781, 1.375, 1.23438,
                1.25, 1.21875, 1.26562, 1.23438, 1.26562, 1.26562, 1.32031, 1.26562, 1.35156, 1.28906,
                1.34375, 1.27344, 1.25781, 1.32031, 1.28906, 1.24219, 1.28125, 1.34375, 1.24219, 1.21875,
                1.28906, 1.32031, 1.35156, 1.27344, 1.28125, 1.29688, 1.28125, 1.22656, 1.35156, 1.23438,
                1.32812, 1.32812, 1.32031, 1.35938, 1.32812, 1.25781, 1.21875, 1.32031, 1.28906, 1.375}),
        LSTMSequenceParams(
            5, 10, 10, 5,
            0.7f, op::RecurrentSequenceDirection::BIDIRECTIONAL,
            ET,
            std::vector<T>{
                1, 9.375, 9, 3.84375, 2.17188, 2.65625, 1.35938, 2.84375, 8.4375, 6.125,
                5.78125, 6.375, 9.625, 9.625, 5.15625, 6.875, 9.3125, 7.75, 4.375, 6.875,
                2.39062, 7.71875, 9, 9.625, 1.23438, 1.07812, 3.625, 1.95312, 4.5625, 3.6875,
                8.25, 6.90625, 6.625, 8.25, 9.125, 8.875, 6, 9.625, 8.5, 7.5,
                1.45312, 6.78125, 8.25, 7.4375, 9.375, 5.1875, 4.25, 3.9375, 7.1875, 4.9375,
                2.15625, 7.5625, 8.5, 9.9375, 3.85938, 7.0625, 7.625, 8.125, 6.375, 2.53125,
                4.25, 1.23438, 8.125, 8.1875, 7.28125, 9.125, 8.375, 1.21875, 9.125, 5.4375,
                8.5, 5.75, 9.1875, 6.375, 9.75, 1.46875, 6.875, 9, 8.25, 7.5625,
                1.92188, 8.375, 3.125, 5.5, 4.40625, 8.25, 7.28125, 1.85938, 5.375, 2.96875,
                4.75, 3.32812, 6.75, 5.1875, 7.8125, 5.125, 6, 7.375, 7.25, 2.59375,
                9.25, 5.78125, 1.21875, 2.5, 8.5, 7.90625, 4.4375, 9.375, 3.6875, 6.5,
                1.05469, 2.34375, 4.9375, 5.40625, 7.625, 4.375, 4.375, 8.625, 5.4375, 9.1875,
                1.125, 4.4375, 3.25, 3.84375, 4.125, 6.125, 8.125, 2.6875, 9.4375, 2.125,
                1.90625, 7.1875, 7.625, 8.1875, 9.75, 6.15625, 7.34375, 9.75, 9.5625, 6.6875,
                9.375, 9, 4.6875, 5.4375, 4.03125, 4.15625, 7.9375, 7.4375, 4, 5.53125,
                1.74219, 3.03125, 9.0625, 3.20312, 8.0625, 8.125, 7.4375, 5.4375, 5, 9.25,
                7.75, 9.5, 6.90625, 5.8125, 4.25, 3.26562, 4.375, 7.5, 6.84375, 4.3125,
                1.5, 5.5, 1.70312, 3.03125, 1.82812, 4.1875, 8.25, 6.84375, 1.58594, 3.8125,
                3.01562, 7.90625, 2.375, 8, 7.125, 8.625, 2.125, 9.5, 3.3125, 1.96875,
                4.9375, 9.1875, 1.98438, 4, 3.82812, 8.375, 4.15625, 9.0625, 7.84375, 1.38281,
                1.89062, 2.75, 9.375, 3.65625, 3.9375, 6.625, 4.8125, 1.77344, 3.4375, 2.28125,
                8.0625, 5.625, 5.0625, 7.1875, 1.75, 8.6875, 1.63281, 6.8125, 3.96875, 6.25,
                2.71875, 7.375, 8.5, 3.26562, 1.55469, 9.125, 6, 4.96875, 6.125, 1.1875,
                5.15625, 9.625, 9.5, 6.875, 4.09375, 5.625, 8.9375, 7.125, 4.875, 5.375,
                9.9375, 9.3125, 8.1875, 5.625, 9.5, 1.64844, 4.40625, 6.09375, 4.625, 10},
            std::vector<T>{
                -1, 9.1875, 5.5625, 6.625, 2.65625, 8.125, 8.25, 4.875, 9.4375, 5.875,
                1.26562, 4.71875, 8.0625, 1.76562, 2.46875, 8, 4.875, 5.375, 6.15625, 1.45312,
                2.95312, 5.84375, 8.5, 1.375, 3.6875, 8.25, 4.25, 9.9375, 8.8125, 6.75,
                1.10938, 9.5, 7.0625, 7.625, 5.3125, 8.375, 9.75, 2.03125, 2.90625, 7.625,
                2.90625, 8.375, 9.5, 1.66406, 3.9375, 2.875, 9.875, 8.25, 4.9375, 9.25,
                9.6875, 2.34375, 3.3125, 9.625, 7.8125, 4.34375, 2.71875, 7.4375, 3.53125, 1.9375,
                7.9375, 8.875, 1.07031, 6.25, 4.1875, 7.125, 3.09375, 9, 4.4375, 2.60938,
                1.98438, 2.76562, 4.65625, 5.125, 4.625, 8.375, 7.9375, 6.6875, 8.5625, 6.8125,
                4.53125, 5.09375, 5.1875, 6.0625, 1.9375, 7, 2.71875, 6.8125, 7.0625, 1.42188,
                5.25, 3.34375, 8.1875, 1.07812, 3.21875, 8.6875, 7.6875, 1.27344, 7.0625, 10},
            std::vector<T>{
                1, 9.6875, 3.3125, 3.09375, 6, 9.5, 7.375, 2.5625, 5.3125, 7.625,
                9.875, 1.01562, 5.9375, 4, 4.25, 2.4375, 3.25, 6.5625, 6.53125, 5.71875,
                3.26562, 7.875, 6.6875, 1.10938, 9.875, 9.875, 1.11719, 7.75, 7.84375, 6.75,
                2.21875, 6, 10, 9.125, 8, 1.92188, 8.25, 9.1875, 6.625, 5.96875,
                2.28125, 8.0625, 6.5625, 3.84375, 9.125, 7.0625, 1.45312, 1.34375, 2.46875, 4.59375,
                8.9375, 8.5, 4.875, 2.4375, 9.625, 4.15625, 2.9375, 9.875, 2.90625, 1.25,
                9, 5.53125, 7.65625, 4.03125, 3.3125, 3.46875, 3.92188, 8.125, 6.78125, 9.5,
                9.875, 5.6875, 6.4375, 5.0625, 5.75, 6.8125, 4.875, 2.65625, 5.21875, 3.9375,
                3.4375, 4.5, 1.98438, 5.71875, 3.75, 7.5625, 1.11719, 2.29688, 6.75, 6.8125,
                1.0625, 6.78125, 3.65625, 6.4375, 7.3125, 5.625, 1.35156, 8.4375, 4.125, 10},
            std::vector<int64_t>{5, 5, 5, 5, 5},
            std::vector<T>{
                1, 6.9375, 6.75, 6.1875, 2.4375, 3.67188, 6.9375, 3.57812, 4.25, 1.53906,
                4.8125, 9, 6.1875, 7.3125, 9.5, 5.53125, 9.6875, 4.84375, 4.65625, 7.84375,
                3.03125, 6.1875, 8.625, 9.875, 9.5, 9.0625, 8.125, 9.875, 4.875, 2.79688,
                9.5, 4.4375, 1.55469, 2.78125, 8.75, 8.5, 2.34375, 2.90625, 6.25, 2.71875,
                5.25, 6.875, 8, 7.375, 8.9375, 1.875, 3.0625, 5.75, 7.375, 7.75,
                1.125, 6.75, 6.75, 3, 6.65625, 5.125, 3.84375, 6.625, 9.125, 3.92188,
                7, 4.3125, 9.625, 5.53125, 7.6875, 1.85156, 7.03125, 7.375, 3.5, 5.9375,
                9.125, 2.0625, 6.5625, 7.375, 3.4375, 3.17188, 1.25, 9, 2.75, 3.40625,
                5.9375, 5.53125, 3.3125, 6.625, 8.25, 6.84375, 2.82812, 1.27344, 8.625, 5.1875,
                1.79688, 1.97656, 2.125, 9.875, 2.73438, 2.40625, 6.625, 2.76562, 3.76562, 5.59375,
                7.1875, 6.40625, 8.5625, 6.21875, 6.46875, 8.0625, 7.6875, 2.1875, 9.0625, 6.3125,
                9.0625, 2.90625, 2.17188, 7.4375, 3.125, 1.40625, 8.25, 7.0625, 8.375, 4.6875,
                5.125, 1.875, 8.375, 3.4375, 9.6875, 9.3125, 7.5, 8.3125, 4.5, 7.5625,
                1.92188, 1.46094, 8.75, 3.34375, 7.9375, 6, 8.5, 1.23438, 2.75, 7.25,
                7.4375, 4.09375, 8.625, 6.34375, 5.5, 6.3125, 1.23438, 7.875, 8.25, 7.25,
                7.65625, 8.25, 9.25, 9.625, 4.3125, 9.125, 3.64062, 6.4375, 5.9375, 9.125,
                6.1875, 6.625, 3, 1.65625, 5.75, 6.1875, 4.71875, 8.125, 6.25, 2.5,
                3.40625, 3.20312, 9.1875, 9.8125, 8.125, 4.125, 1.08594, 4.78125, 9.5625, 5.0625,
                5.625, 3.5, 8.25, 1.73438, 3.5625, 8.0625, 2.23438, 3.0625, 3.34375, 3.40625,
                3.625, 9.75, 1.14062, 4.84375, 9.125, 5.5, 8, 9.5, 5.125, 7.25,
                9.4375, 2.0625, 5.90625, 9.75, 6.1875, 6, 7.3125, 7.3125, 7.34375, 7.5625,
                6.1875, 9.125, 7.28125, 7.75, 4.59375, 7.5, 7.71875, 9.75, 9.5, 9.9375,
                5.90625, 1.20312, 1.75, 7.75, 8.8125, 3.5625, 8.5, 9.25, 8.375, 7.84375,
                4.9375, 7.4375, 6.75, 1.46875, 3.53125, 7.15625, 6.0625, 3.8125, 5.34375, 1.60938,
                9.6875, 8.4375, 4.65625, 5.46875, 3.65625, 5.125, 9.125, 1.23438, 6.875, 1.79688,
                4.1875, 2.26562, 2.78125, 9.25, 3.28125, 6.5625, 7.3125, 8.1875, 8.9375, 4,
                8.625, 5.3125, 9.875, 4.375, 4.09375, 1.125, 2.84375, 8.375, 7.90625, 4.75,
                4.625, 4.75, 3.17188, 7.59375, 4.5, 2.46875, 4.03125, 3.26562, 2.5, 7.28125,
                6.3125, 7.6875, 1.00781, 1.3125, 7.625, 4.375, 2.0625, 4.1875, 7.625, 4.40625,
                3.6875, 8.75, 4.09375, 2.84375, 5.21875, 3.65625, 9.5625, 1.71875, 5.9375, 1.79688,
                5.8125, 5.625, 1.46875, 1.32812, 1.95312, 4.3125, 1.41406, 3.125, 9.1875, 6.3125,
                5.3125, 4.1875, 1.17969, 1.32031, 9.5, 9.875, 6.78125, 2.32812, 7.1875, 4.3125,
                7.90625, 4.6875, 7.875, 3.59375, 4.6875, 1.3125, 1.71094, 3.98438, 8.75, 8.625,
                2.21875, 9.5, 7.1875, 3.92188, 5.6875, 5.75, 8.75, 6.625, 1.34375, 1.99219,
                8.3125, 1.73438, 5.5625, 7.84375, 1.15625, 2.125, 4.5, 5.75, 7, 3.5,
                7.375, 4.1875, 5.3125, 7.0625, 3.8125, 1.85156, 9, 6.75, 4.3125, 9.9375,
                8.125, 8.75, 1.09375, 9.875, 6.46875, 2.32812, 3.84375, 8.75, 6.6875, 3.17188,
                4.75, 5.25, 5.875, 4.625, 5.59375, 7.25, 8.625, 2.84375, 7, 4.0625,
                6.5, 3.15625, 6.5, 4.65625, 7.8125, 2.6875, 5.6875, 4, 5.9375, 5.53125,
                6.96875, 5.78125, 9.125, 6.53125, 8.125, 2.90625, 5.625, 5.125, 7.4375, 3.78125,
                5.5, 8, 9.4375, 8.75, 6.75, 5.59375, 8.375, 2.4375, 1.27344, 4.90625,
                8.875, 9.5, 5.21875, 6.9375, 9, 3.0625, 3.25, 2.39062, 1.92188, 2.54688,
                3.73438, 7.28125, 4.5, 5.09375, 7.5, 2.14062, 1.10938, 2.32812, 3.875, 4.8125,
                1.95312, 1.98438, 6.34375, 4.8125, 6.84375, 5.3125, 9.1875, 8.3125, 6.5, 7.125,
                2.59375, 4.0625, 1.27344, 5.09375, 5.75, 1.48438, 6.1875, 5.1875, 9.9375, 8.3125,
                2.14062, 5.21875, 5.0625, 1.94531, 6.6875, 7.75, 8.625, 6.3125, 5.8125, 8.75,
                6.375, 8.875, 1.16406, 6.125, 1, 3.71875, 3.1875, 6.96875, 1.14062, 6.09375,
                8.75, 1.5, 5.25, 5.1875, 1.48438, 2.96875, 2.03125, 4.40625, 9.625, 4.125,
                5, 2.45312, 9.25, 5.875, 5.9375, 4.6875, 8.25, 5.3125, 1.73438, 7.4375,
                4.625, 7.6875, 3.4375, 6.53125, 2.64062, 8.75, 2.57812, 7.8125, 4.4375, 6.0625,
                4.5, 9.125, 7.96875, 2.65625, 8.125, 8.8125, 5.25, 5.65625, 4.84375, 7.96875,
                3.09375, 2.3125, 7.34375, 8.5, 6.25, 2.79688, 7.8125, 5.8125, 7.0625, 5.84375,
                6.4375, 9.1875, 5.03125, 2.01562, 1.8125, 6.5625, 9.5625, 4.96875, 5.8125, 4.75,
                6.6875, 9.625, 4.625, 7.71875, 4.6875, 6.4375, 3.59375, 2.25, 3.0625, 6.375,
                2.0625, 6.9375, 1.0625, 9.25, 7.0625, 3.90625, 9.875, 3.09375, 6.125, 9.1875,
                2.9375, 4.0625, 5.6875, 6.4375, 8.25, 8.5, 7.1875, 6.46875, 9.125, 3.71875,
                5.1875, 9.25, 9.375, 6.25, 8.5, 1.67188, 2.09375, 5.78125, 1.78125, 5.8125,
                4.84375, 8.125, 1.64062, 9, 7.8125, 4.5, 3.64062, 1.9375, 1.24219, 9.25,
                4.4375, 1.48438, 1.67188, 8.25, 7.75, 2.875, 3.28125, 3.9375, 8.75, 5.4375,
                2.875, 5.6875, 1.99219, 9.625, 1.39062, 4.46875, 9.5, 7.375, 9.875, 2.98438,
                3.0625, 9.875, 9.5625, 1.14062, 4.6875, 5.53125, 2.98438, 4.5, 9.25, 4.5625,
                7.625, 3.1875, 8.4375, 1.65625, 3, 6, 9.75, 5.3125, 4.375, 6.75,
                3.8125, 2.65625, 8.75, 1.23438, 2.40625, 2.875, 2.20312, 8.8125, 3.42188, 5.25,
                8, 8.375, 2.8125, 4.5625, 6.40625, 4.875, 4.25, 9, 2.25, 5.75,
                7, 8.5, 6.4375, 7.1875, 5.59375, 2.48438, 1.53125, 8.1875, 2.15625, 2.01562,
                9.25, 8.375, 6.1875, 8, 9.125, 3.8125, 6.6875, 8, 8.75, 7.625,
                9.75, 4.8125, 6, 2.78125, 3.5625, 8.5625, 1.07812, 9.9375, 6.8125, 7.125,
                7.3125, 4.5625, 1.50781, 1.15625, 3.79688, 2.8125, 8.6875, 6.28125, 8.1875, 7.75,
                8.875, 7.8125, 5.21875, 5.53125, 9.125, 5.125, 4.125, 1.35938, 6.25, 3.1875,
                8.375, 1, 8, 8.625, 5.25, 8.875, 3.75, 7.5, 1.32812, 1.80469,
                4.5, 2.15625, 6.8125, 2.375, 4.4375, 9.5625, 6.9375, 6.4375, 2.10938, 2.5625,
                9, 1.71875, 6.78125, 6, 2, 7, 7.09375, 4.5, 5.8125, 8.625,
                7.59375, 6.6875, 4.625, 3.35938, 9, 8.625, 1.51562, 9.5625, 6.25, 3.03125,
                3.59375, 1.85156, 5.5625, 8.3125, 7.375, 2.21875, 8.5625, 3.0625, 7.1875, 7,
                2.59375, 3.92188, 8.125, 3.09375, 7.4375, 6.375, 7.375, 8.125, 6.75, 7.78125,
                2.5, 5, 7.5625, 1.42188, 9.0625, 1.875, 4.8125, 3.09375, 5.34375, 3.34375,
                1.92188, 10, 4.3125, 5.03125, 8.75, 6.40625, 9.25, 9.125, 9, 7.4375,
                3.89062, 4.4375, 9, 6.34375, 9.25, 9.875, 1.79688, 5.0625, 9, 3.04688,
                6.6875, 2.98438, 5.40625, 8.125, 9.75, 3.125, 4.125, 3.48438, 8.4375, 8.75,
                4.375, 8.375, 6.4375, 3.5, 3.89062, 5.4375, 9.5, 4.03125, 7.5, 10},
            std::vector<T>{
                1, 5.96875, 3, 8.75, 2.46875, 8.6875, 6.1875, 3.09375, 5, 5.3125,
                3.09375, 3.25, 7.96875, 8.625, 3.34375, 6.8125, 2.8125, 9.5, 3.84375, 1.14062,
                3.8125, 1.09375, 3.375, 7.0625, 6.875, 9.5, 5.125, 2.25, 5.34375, 6.3125,
                1.79688, 5.4375, 8, 6.0625, 7.46875, 7, 9.25, 1.74219, 7.25, 3.5625,
                4.5625, 3.01562, 2.78125, 8.125, 8.375, 2.53125, 4.9375, 6.375, 7.5, 3.78125,
                6, 2.89062, 2.75, 8.4375, 2.85938, 2, 9.4375, 1.32812, 10, 4.3125,
                3.09375, 9.75, 6, 9.125, 7.0625, 5.46875, 9, 4.625, 3.40625, 9.75,
                5.25, 6.75, 5.4375, 3.96875, 3, 6.15625, 9.0625, 5.59375, 2, 2.51562,
                4.75, 7.5625, 4.75, 5.28125, 2.17188, 1.00781, 3.625, 6.1875, 1.96875, 8.625,
                9.25, 9.1875, 2, 2.75, 4.25, 1, 5.46875, 4.5, 6.65625, 3.5,
                7.90625, 7.4375, 6.9375, 5.1875, 8.875, 2.375, 4.46875, 7.1875, 4.125, 6.9375,
                9.5, 1.92969, 6.5, 6.5625, 9.0625, 8.5, 4.75, 4.6875, 6.0625, 9.8125,
                2.34375, 4.65625, 9, 9.5, 6.71875, 5.40625, 3.28125, 6.5625, 4.1875, 1.1875,
                6.1875, 2.8125, 7.875, 8.625, 8.25, 5, 4.9375, 4.625, 5, 7.4375,
                8.5625, 5.3125, 8.4375, 6.5, 1.98438, 2.0625, 5.625, 7.625, 9.5625, 9.5,
                6.125, 8, 10, 2.625, 4.90625, 2.95312, 6.8125, 8.75, 8.9375, 6.5,
                1.65625, 7.84375, 8, 4.125, 9, 8.0625, 5.375, 6.0625, 3.6875, 7.0625,
                2.73438, 7.28125, 7.4375, 5.375, 4.84375, 2.03125, 6.375, 6.5625, 4.0625, 2.125,
                1.84375, 3.0625, 6.03125, 9.375, 4.625, 9.375, 3.57812, 2.84375, 8.0625, 6.65625,
                9.75, 5.125, 1.07812, 8.4375, 5.75, 7.25, 4.5625, 3.54688, 6.9375, 6.625,
                8.875, 6.40625, 2.70312, 6.0625, 2.5, 5.6875, 8.0625, 4.4375, 1.875, 8.1875,
                7.6875, 3.25, 8.125, 7.6875, 3.03125, 9.75, 2.84375, 9.6875, 1.34375, 6.375,
                6.34375, 8.6875, 2.28125, 6.375, 9.9375, 8.375, 1.23438, 6.3125, 8.625, 7.625,
                4.75, 9.6875, 9.125, 4.75, 6.375, 2.6875, 8.75, 2.60938, 5.75, 4.4375,
                8.25, 6.4375, 2.64062, 6.5625, 6.125, 9.625, 3.84375, 2.5, 3.09375, 7.75,
                7.25, 3.96875, 5.9375, 1.09375, 8.125, 5.625, 9.5, 9.25, 5.6875, 3.03125,
                6.1875, 2.85938, 8.0625, 4.75, 2.84375, 7.375, 2, 4.0625, 8.375, 4.1875,
                2.78125, 3.04688, 2.64062, 9.5, 6.625, 9.875, 1.70312, 6.6875, 4.8125, 7,
                1.9375, 1.32031, 7.5, 2.21875, 6.6875, 3.34375, 7.625, 1.53125, 8.875, 1.75781,
                3.35938, 4.8125, 4.40625, 5.46875, 7.5625, 3.29688, 3.21875, 6.25, 4.34375, 5,
                2.34375, 4.875, 2.78125, 7.46875, 3.20312, 5.4375, 6.875, 2, 5.375, 6.71875,
                5.5625, 3.34375, 3.78125, 9.625, 1.8125, 5.125, 2.90625, 1.80469, 3.625, 5.34375,
                1.76562, 5.6875, 9.5625, 9.4375, 10, 1.17188, 7.125, 9.5, 3, 7.1875,
                2.09375, 2.98438, 5.9375, 9.375, 1.9375, 1.89062, 5.75, 3.21875, 7.375, 2.46875,
                6.125, 4.96875, 3.75, 1.0625, 3.53125, 7.875, 4.8125, 9.125, 4.28125, 9.125,
                1.78906, 8.0625, 7.9375, 1.32812, 6.5625, 4.25, 7.375, 6.875, 6.0625, 6.53125,
                3.5625, 4.4375, 4.5, 6.875, 7.9375, 9.625, 4.3125, 1.09375, 2.73438, 7.75,
                6.3125, 9.75, 4.625, 4.71875, 3, 2.40625, 1.17188, 7.625, 5.28125, 6.6875,
                8.125, 9.625, 2.23438, 4.84375, 4.6875, 7.96875, 4.9375, 3.46875, 3, 9.5,
                7.21875, 6.25, 3.0625, 1.34375, 6.25, 1.39844, 5.5625, 5.34375, 2.96875, 9.375,
                2.85938, 7, 7.6875, 4.6875, 8.875, 4.40625, 4.25, 8.625, 5.875, 8,
                4.8125, 1.46875, 5.625, 3, 9.875, 2.03125, 3.82812, 6.375, 8.75, 4.5625,
                9.625, 8.625, 1.625, 6.1875, 1.85938, 9.125, 1.42188, 5.0625, 8.625, 4.5625,
                2.4375, 8.875, 6.84375, 7.3125, 1.63281, 3.28125, 6.25, 3.70312, 1.28125, 6.96875,
                1.67969, 5.875, 9.625, 7.3125, 5.71875, 9.9375, 4.0625, 6.5625, 1.03125, 3.84375,
                4.1875, 9.5, 4.21875, 5.25, 1.64062, 4.78125, 4.28125, 7.8125, 6.1875, 8,
                1.85938, 5.375, 1.9375, 8.125, 6, 4.90625, 1.53125, 2.53125, 6.53125, 1.67188,
                1.5625, 2.48438, 8.9375, 2.75, 4, 3.65625, 4.625, 5.5625, 8.75, 4.9375,
                3.40625, 8.1875, 6.40625, 7.0625, 6.6875, 6.0625, 9.75, 8.375, 1.36719, 9.625,
                8.625, 8.375, 5.28125, 7.90625, 4.25, 9.6875, 5.1875, 6.6875, 8.375, 3.71875,
                5.25, 9.5, 2.25, 7.1875, 3.46875, 7.53125, 5.15625, 7.84375, 2.375, 4,
                5.34375, 8.6875, 4.75, 7, 5.96875, 9, 4.125, 6.34375, 3.28125, 5,
                7.1875, 7.8125, 1.48438, 2.40625, 3.15625, 1.03125, 8.625, 5.78125, 3.5, 8.0625,
                4.71875, 9.75, 3.3125, 5.09375, 3.65625, 4.5, 4.625, 7.6875, 7.375, 9.5,
                4.625, 2.46875, 3.625, 6.125, 1.36719, 8.375, 8.375, 6.53125, 3.125, 5.75,
                9.625, 3.3125, 8.375, 2.65625, 3.375, 3.60938, 6.9375, 4.9375, 1.78906, 9.25,
                3.20312, 6.4375, 2.375, 2.40625, 6.0625, 3.75, 1.3125, 4.3125, 9.125, 9.3125,
                1.09375, 8.3125, 1.98438, 8.1875, 6.1875, 1.21094, 8.25, 6.75, 5.75, 9.25,
                4.75, 7.0625, 4.84375, 4.0625, 7.78125, 2.1875, 2.90625, 3.01562, 9.1875, 5,
                6.5, 1.75, 6.1875, 2.96875, 1.07031, 2.375, 7.625, 6.0625, 3.90625, 3.45312,
                2.57812, 3.3125, 3.96875, 7.875, 4.75, 7, 6.5625, 9.75, 4.5, 1.82812,
                4.375, 4.90625, 9.75, 7.84375, 8.125, 6.3125, 4.875, 2.71875, 2.5, 8.875,
                2.17188, 4.0625, 2.84375, 9, 2.3125, 1.85938, 5.875, 4.5625, 3.32812, 6.71875,
                5.125, 1.67188, 3.8125, 8.5, 5.375, 7.75, 6.5, 8.25, 1.49219, 7.25,
                5.625, 5, 8.6875, 5.5625, 9, 7.375, 6.9375, 6.625, 1.82812, 2.75,
                7.59375, 3.875, 4.78125, 6.5625, 3.03125, 7.15625, 9.5, 9.125, 1.59375, 8.5,
                4.3125, 5.15625, 4.65625, 6.125, 6.5, 4.25, 1.98438, 7.96875, 5.25, 4.40625,
                4.5625, 2.89062, 3, 7.375, 5.75, 8.125, 1.75, 1.5625, 3.40625, 6.125,
                3.29688, 1.39062, 6.625, 1.5, 8.875, 8.5, 5.5, 5, 1.09375, 1.79688,
                9.75, 2.375, 1.07031, 8.25, 9, 9.875, 7.6875, 1.23438, 8.375, 8.0625,
                6.75, 8.25, 8.625, 2.21875, 8.5, 9.25, 4.4375, 4.40625, 8, 3.5,
                3.8125, 7, 2.125, 9, 3.42188, 9.125, 4.1875, 3.04688, 7.65625, 5.9375,
                5.75, 9.5, 4.65625, 6.875, 5.0625, 1.89062, 3.40625, 5.78125, 4.0625, 5.125,
                9, 5.9375, 2.46875, 9.25, 6.875, 7, 8.375, 6.09375, 1.48438, 5.90625,
                8.75, 6.25, 7.3125, 8.5625, 9.9375, 9.125, 1.46875, 9.1875, 1.26562, 9,
                5.5, 6.1875, 1.58594, 7.3125, 7.875, 9.5625, 3.32812, 8, 8.25, 4.59375,
                9.75, 1.125, 8.3125, 4.6875, 3.375, 3.92188, 6.125, 4.46875, 3.625, 2.03125,
                1.625, 9.1875, 2.46875, 3.03125, 6.1875, 9.5625, 5.78125, 2.10938, 3.14062, 4.125,
                5.75, 7.03125, 2.28125, 6.9375, 7.3125, 3.125, 6.0625, 7.75, 7.1875, 8.1875,
                6.90625, 4.875, 8.875, 7.3125, 5.9375, 1.39062, 4.625, 7.75, 5.0625, 10},
            std::vector<T>{
                1, 7.25, 1.08594, 1.90625, 7.3125, 8.1875, 1.84375, 9.8125, 3.65625, 7.6875,
                7.03125, 6.75, 9.875, 9, 2.28125, 4.625, 8.375, 8.125, 4.25, 9.9375,
                2.125, 1.92969, 1.35938, 7.625, 5.6875, 2.65625, 6.9375, 8.0625, 2.53125, 7.375,
                2.8125, 9.1875, 7.3125, 8.75, 7.15625, 7.15625, 3.8125, 6.75, 3.53125, 8.75,
                9.875, 5.75, 7.1875, 5.65625, 7.375, 9.25, 6.03125, 9.75, 8.625, 1.07031,
                6.8125, 2.90625, 6.1875, 4.6875, 7.25, 1.14062, 9.0625, 2.78125, 5.625, 6.875,
                2.0625, 3.3125, 9.875, 8.9375, 5.6875, 5.875, 3.65625, 2.78125, 8.625, 4.1875,
                7.125, 7.09375, 5.125, 2.625, 3.10938, 9.4375, 5.5625, 5.8125, 3.5625, 10},
            std::vector<T>{
                0.523438, 0.667969, 0.65625, 0.65625, 0.667969, 0.667969, 0.667969, 0.648438, 0.667969, 0.667969,
                0.539062, 0.667969, 0.648438, 0.648438, 0.664062, 0.667969, 0.667969, 0.632812, 0.664062, 0.667969,
                0.546875, 0.667969, 0.632812, 0.625, 0.65625, 0.667969, 0.664062, 0.617188, 0.65625, 0.664062,
                0.546875, 0.664062, 0.617188, 0.609375, 0.648438, 0.664062, 0.65625, 0.601562, 0.640625, 0.65625,
                0.554688, 0.65625, 0.601562, 0.59375, 0.632812, 0.648438, 0.640625, 0.585938, 0.625, 0.640625,
                0.65625, 0.554688, 0.632812, 0.609375, 0.617188, 0.585938, 0.601562, 0.632812, 0.632812, 0.632812,
                0.664062, 0.546875, 0.648438, 0.625, 0.632812, 0.59375, 0.617188, 0.648438, 0.648438, 0.648438,
                0.667969, 0.546875, 0.65625, 0.640625, 0.648438, 0.609375, 0.632812, 0.664062, 0.664062, 0.65625,
                0.667969, 0.539062, 0.664062, 0.65625, 0.65625, 0.625, 0.648438, 0.667969, 0.667969, 0.664062,
                0.667969, 0.523438, 0.667969, 0.664062, 0.664062, 0.640625, 0.65625, 0.667969, 0.667969, 0.667969,
                0.65625, 0.667969, 0.667969, 0.539062, 0.667969, 0.667969, 0.546875, 0.667969, 0.667969, 0.667969,
                0.648438, 0.667969, 0.667969, 0.546875, 0.667969, 0.667969, 0.546875, 0.667969, 0.667969, 0.667969,
                0.632812, 0.664062, 0.664062, 0.554688, 0.667969, 0.667969, 0.554688, 0.664062, 0.664062, 0.664062,
                0.617188, 0.65625, 0.648438, 0.554688, 0.664062, 0.664062, 0.554688, 0.65625, 0.65625, 0.648438,
                0.601562, 0.648438, 0.640625, 0.554688, 0.65625, 0.65625, 0.554688, 0.648438, 0.648438, 0.640625,
                0.585938, 0.632812, 0.65625, 0.648438, 0.648438, 0.570312, 0.648438, 0.648438, 0.640625, 0.632812,
                0.59375, 0.648438, 0.664062, 0.664062, 0.65625, 0.585938, 0.65625, 0.664062, 0.648438, 0.648438,
                0.601562, 0.65625, 0.667969, 0.667969, 0.664062, 0.59375, 0.664062, 0.667969, 0.664062, 0.65625,
                0.617188, 0.664062, 0.667969, 0.667969, 0.667969, 0.609375, 0.667969, 0.667969, 0.667969, 0.664062,
                0.632812, 0.667969, 0.667969, 0.667969, 0.667969, 0.625, 0.667969, 0.667969, 0.667969, 0.667969,
                0.640625, 0.667969, 0.667969, 0.664062, 0.667969, 0.667969, 0.585938, 0.570312, 0.648438, 0.667969,
                0.625, 0.667969, 0.667969, 0.65625, 0.667969, 0.667969, 0.578125, 0.570312, 0.625, 0.664062,
                0.609375, 0.664062, 0.664062, 0.640625, 0.667969, 0.664062, 0.570312, 0.5625, 0.609375, 0.648438,
                0.59375, 0.65625, 0.648438, 0.625, 0.664062, 0.65625, 0.570312, 0.5625, 0.59375, 0.632812,
                0.585938, 0.648438, 0.632812, 0.609375, 0.648438, 0.640625, 0.5625, 0.5625, 0.585938, 0.617188,
                0.648438, 0.648438, 0.625, 0.585938, 0.65625, 0.617188, 0.59375, 0.65625, 0.59375, 0.554688,
                0.664062, 0.65625, 0.640625, 0.59375, 0.664062, 0.632812, 0.609375, 0.664062, 0.609375, 0.554688,
                0.667969, 0.664062, 0.648438, 0.609375, 0.667969, 0.648438, 0.625, 0.667969, 0.625, 0.554688,
                0.667969, 0.667969, 0.664062, 0.625, 0.667969, 0.65625, 0.640625, 0.667969, 0.640625, 0.5625,
                0.667969, 0.667969, 0.667969, 0.640625, 0.667969, 0.664062, 0.65625, 0.667969, 0.65625, 0.5625,
                0.667969, 0.667969, 0.667969, 0.664062, 0.65625, 0.664062, 0.664062, 0.667969, 0.667969, 0.667969,
                0.667969, 0.664062, 0.667969, 0.65625, 0.648438, 0.648438, 0.65625, 0.667969, 0.667969, 0.667969,
                0.667969, 0.65625, 0.664062, 0.648438, 0.632812, 0.632812, 0.640625, 0.664062, 0.664062, 0.667969,
                0.664062, 0.648438, 0.65625, 0.625, 0.617188, 0.617188, 0.625, 0.65625, 0.648438, 0.664062,
                0.648438, 0.632812, 0.648438, 0.609375, 0.601562, 0.601562, 0.609375, 0.648438, 0.640625, 0.648438,
                0.65625, 0.632812, 0.632812, 0.625, 0.632812, 0.640625, 0.625, 0.585938, 0.625, 0.609375,
                0.664062, 0.648438, 0.648438, 0.640625, 0.648438, 0.65625, 0.640625, 0.601562, 0.640625, 0.625,
                0.667969, 0.65625, 0.664062, 0.65625, 0.65625, 0.664062, 0.648438, 0.617188, 0.65625, 0.640625,
                0.667969, 0.664062, 0.667969, 0.664062, 0.664062, 0.667969, 0.664062, 0.632812, 0.664062, 0.65625,
                0.667969, 0.667969, 0.667969, 0.667969, 0.667969, 0.667969, 0.667969, 0.648438, 0.667969, 0.664062,
                0.664062, 0.664062, 0.625, 0.667969, 0.664062, 0.667969, 0.546875, 0.640625, 0.667969, 0.667969,
                0.648438, 0.664062, 0.609375, 0.664062, 0.65625, 0.667969, 0.546875, 0.625, 0.667969, 0.667969,
                0.632812, 0.648438, 0.59375, 0.65625, 0.640625, 0.664062, 0.554688, 0.609375, 0.664062, 0.664062,
                0.617188, 0.632812, 0.585938, 0.648438, 0.625, 0.65625, 0.554688, 0.59375, 0.648438, 0.65625,
                0.601562, 0.617188, 0.578125, 0.632812, 0.609375, 0.640625, 0.554688, 0.585938, 0.640625, 0.640625,
                0.554688, 0.640625, 0.609375, 0.632812, 0.640625, 0.632812, 0.5625, 0.648438, 0.617188, 0.65625,
                0.554688, 0.648438, 0.625, 0.648438, 0.65625, 0.648438, 0.5625, 0.65625, 0.632812, 0.664062,
                0.546875, 0.664062, 0.640625, 0.664062, 0.664062, 0.65625, 0.5625, 0.664062, 0.648438, 0.667969,
                0.539062, 0.667969, 0.648438, 0.667969, 0.667969, 0.664062, 0.570312, 0.667969, 0.65625, 0.667969,
                0.539062, 0.667969, 0.664062, 0.667969, 0.667969, 0.667969, 0.570312, 0.667969, 0.664062, 0.667969},
            std::vector<T>{
                0.554688, 0.65625, 0.601562, 0.59375, 0.632812, 0.648438, 0.640625, 0.585938, 0.625, 0.640625,
                0.65625, 0.554688, 0.632812, 0.609375, 0.617188, 0.585938, 0.601562, 0.632812, 0.632812, 0.632812,
                0.601562, 0.648438, 0.640625, 0.554688, 0.65625, 0.65625, 0.554688, 0.648438, 0.648438, 0.640625,
                0.585938, 0.632812, 0.65625, 0.648438, 0.648438, 0.570312, 0.648438, 0.648438, 0.640625, 0.632812,
                0.585938, 0.648438, 0.632812, 0.609375, 0.648438, 0.640625, 0.5625, 0.5625, 0.585938, 0.617188,
                0.648438, 0.648438, 0.625, 0.585938, 0.65625, 0.617188, 0.59375, 0.65625, 0.59375, 0.554688,
                0.648438, 0.632812, 0.648438, 0.609375, 0.601562, 0.601562, 0.609375, 0.648438, 0.640625, 0.648438,
                0.65625, 0.632812, 0.632812, 0.625, 0.632812, 0.640625, 0.625, 0.585938, 0.625, 0.609375,
                0.601562, 0.617188, 0.578125, 0.632812, 0.609375, 0.640625, 0.554688, 0.585938, 0.640625, 0.640625,
                0.554688, 0.640625, 0.609375, 0.632812, 0.640625, 0.632812, 0.5625, 0.648438, 0.617188, 0.65625},
            std::vector<T>{
                1.1875, 2.3125, 1.5, 1.45312, 1.84375, 2.29688, 2.03125, 1.39062, 1.76562, 2.03125,
                2.35938, 1.1875, 1.84375, 1.57812, 1.60938, 1.375, 1.48438, 1.90625, 1.90625, 1.79688,
                1.48438, 2.0625, 1.92188, 1.19531, 2.35938, 2.35938, 1.20312, 2.0625, 2.0625, 1.9375,
                1.35156, 1.84375, 2.35938, 2.23438, 2.10938, 1.30469, 2.125, 2.23438, 1.92188, 1.84375,
                1.35156, 2.10938, 1.90625, 1.5625, 2.23438, 1.98438, 1.24219, 1.22656, 1.375, 1.67188,
                2.23438, 2.17188, 1.70312, 1.375, 2.3125, 1.60938, 1.45312, 2.35938, 1.4375, 1.21875,
                2.23438, 1.78125, 2.0625, 1.57812, 1.5, 1.51562, 1.57812, 2.10938, 1.9375, 2.29688,
                2.35938, 1.79688, 1.89062, 1.73438, 1.8125, 1.95312, 1.70312, 1.40625, 1.73438, 1.57812,
                1.5, 1.64062, 1.32031, 1.79688, 1.54688, 2.03125, 1.20312, 1.35938, 1.9375, 1.95312,
                1.1875, 1.9375, 1.54688, 1.89062, 2, 1.78125, 1.22656, 2.15625, 1.59375, 2.35938}),
    };
    return params;
}

template <element::Type_t ET>
std::vector<LSTMSequenceV1Params> generateV1Params() {
    using T = typename element_type_traits<ET>::value_type;

    std::vector<LSTMSequenceV1Params> params {
        LSTMSequenceV1Params(
            5, 10, 10, 10,
            0.7f, false, op::RecurrentSequenceDirection::FORWARD,
            ET,
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 7.95465,
                2.37637, 6.20771, 1.08107, 7.38138, 5.23577, 7.88133, 5.20653, 3.42101, 8.48523, 5.96192,
                1.63073, 5.25228, 7.68488, 2.7276, 5.1788, 3.07327, 5.57423, 2.87711, 1.44374, 5.66976,
                2.55051, 4.56682, 1.96629, 5.58829, 1.91922, 3.59846, 3.08583, 9.70901, 3.50487, 3.1026,
                1.82401, 6.1306, 4.76134, 4.31059, 8.31695, 3.60784, 7.45652, 6.51653, 4.84219, 7.76686,
                4.85031, 4.85544, 4.25714, 2.38005, 9.43471, 9.24769, 8.03763, 6.54696, 1.32399, 6.88891,
                2.16793, 3.64924, 4.24733, 3.47181, 1.66572, 2.36923, 2.45457, 9.44841, 4.34021, 1.45016,
                7.6686, 3.68812, 2.83922, 9.83581, 9.03719, 7.83414, 6.86009, 1.35715, 8.32489, 7.86316,
                5.09754, 5.78644, 1.98402, 2.31429, 5.5791, 2.94085, 9.24799, 5.15937, 2.19041, 7.87817,
                2.9146, 1.66833, 1.85877, 2.45985, 4.20817, 1.85777, 2.28937, 9.37912, 6.18926, 8.55681,
                6.60963, 3.92066, 7.5521, 5.70463, 7.6313, 2.48866, 7.18352, 4.8413, 7.55702, 7.80702,
                4.5785, 9.3268, 2.83159, 1.07202, 9.33716, 3.6506, 2.50256, 1.21691, 5.06801, 8.27505,
                4.31539, 6.48286, 1.31363, 4.1912, 1.70668, 7.23867, 1.11441, 5.13591, 9.65186, 4.00767,
                5.24875, 1.94852, 5.52768, 8.97121, 5.8094, 3.53329, 4.19126, 9.06652, 3.1734, 1.21496,
                9.69154, 4.86971, 4.1166, 6.19361, 2.13874, 9.55039, 3.8225, 9.57548, 2.96554, 3.2383,
                8.77422, 3.11741, 8.3359, 5.89508, 2.72134, 6.29956, 1.43323, 1.14286, 1.4474, 4.59474,
                6.19214, 8.80766, 8.07546, 3.29232, 1.74029, 2.4198, 2.88544, 4.75644, 4.12921, 7.29896,
                7.27759, 1.67252, 1.32823, 8.1046, 9.10476, 1.04197, 3.37783, 5.2064, 4.23835, 3.16196,
                1.20852, 5.78501, 2.17175, 6.05313, 2.51048, 4.78967, 7.16219, 3.4651, 1.09, 2.9788,
                1.28761, 9.41385, 8.03883, 5.65835, 1.14816, 3.6892, 5.86648, 8.73895, 2.66603, 1.75192,
                1.39845, 4.99427, 1.17387, 1.60329, 8.30594, 6.72662, 7.95565, 7.35114, 3.1439, 1.39976,
                3.53095, 8.78581, 1.65811, 6.94299, 2.68641, 5.70058, 9.13491, 5.27637, 8.6232, 8.54902,
                2.25352, 5.86274, 5.20377, 2.96815, 4.96745, 5.3225, 3.99956, 1.08021, 5.54918, 7.05833,
                1.49913, 2.41822, 6.44593, 3.87301, 9.01465, 8.11336, 2.95749, 2.80188, 7.12396, 2.40595,
                5.59325, 9.89258, 2.30223, 1.4347, 9.09158, 7.43797, 3.79295, 4.53646, 1.72705, 4.16909,
                1.00912, 6.62167, 2.80244, 6.626, 3.89307, 1.42586, 7.51028, 7.83327, 4.65674, 7.33902,
                6.26823, 9.72608, 3.73491, 3.8238, 3.03815, 7.05101, 8.0103, 5.61396, 6.53738, 1.41095,
                5.0149, 9.71211, 4.23604, 5.98629, 4.70219, 9.69442, 2.82752, 9.93544, 6.9328, 8.2817,
                5.12336, 8.98577, 5.80541, 6.19552, 9.25748, 3.82732, 7.53525, 8.24712, 5.32057, 5.38817,
                8.57269, 5.99975, 3.42893, 5.38068, 3.48261, 3.02851, 6.82079, 9.2902, 2.80427, 8.91868,
                5.19227, 7.52482, 3.72584, 5.40107, 2.83307, 1.79755, 2.49121, 5.52697, 8.08823, 9.13242,
                2.97572, 7.64318, 3.32023, 6.07788, 2.19187, 4.34879, 1.7457, 5.55154, 7.24966, 5.1128,
                4.25147, 8.34407, 1.4123, 4.49045, 5.12671, 7.62159, 9.18673, 3.49665, 8.35992, 6.90684,
                1.10152, 7.61818, 6.43145, 7.12017, 6.25564, 6.16169, 4.24916, 9.6283, 9.88249, 4.48422,
                8.52562, 9.83928, 6.26818, 7.03839, 1.77631, 9.92305, 8.0155, 9.94928, 6.88321, 1.33685,
                7.4718, 7.19305, 6.47932, 1.9559, 3.52616, 7.98593, 9.0115, 5.59539, 7.44137, 1.70001,
                6.53774, 8.54023, 7.26405, 5.99553, 8.75071, 7.70789, 3.38094, 9.99792, 6.16359, 6.75153,
                5.4073, 9.00437, 8.87059, 8.63011, 6.82951, 6.27021, 3.53425, 9.92489, 8.19695, 5.51473,
                7.95084, 2.11852, 9.28916, 1.40353, 3.05744, 8.58238, 3.75014, 5.35889, 6.85048, 2.29549,
                3.75218, 8.98228, 8.98158, 5.63695, 3.40379, 8.92309, 5.48185, 4.00095, 9.05227, 2.84035,
                8.37644, 8.54954, 5.70516, 2.45744, 9.54079, 1.53504, 8.9785, 6.1691, 4.40962, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 10},
            std::vector<int64_t>{10, 10, 10, 10, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 7.95465,
                2.37637, 6.20771, 1.08107, 7.38138, 5.23577, 7.88133, 5.20653, 3.42101, 8.48523, 5.96192,
                1.63073, 5.25228, 7.68488, 2.7276, 5.1788, 3.07327, 5.57423, 2.87711, 1.44374, 5.66976,
                2.55051, 4.56682, 1.96629, 5.58829, 1.91922, 3.59846, 3.08583, 9.70901, 3.50487, 3.1026,
                1.82401, 6.1306, 4.76134, 4.31059, 8.31695, 3.60784, 7.45652, 6.51653, 4.84219, 7.76686,
                4.85031, 4.85544, 4.25714, 2.38005, 9.43471, 9.24769, 8.03763, 6.54696, 1.32399, 6.88891,
                2.16793, 3.64924, 4.24733, 3.47181, 1.66572, 2.36923, 2.45457, 9.44841, 4.34021, 1.45016,
                7.6686, 3.68812, 2.83922, 9.83581, 9.03719, 7.83414, 6.86009, 1.35715, 8.32489, 7.86316,
                5.09754, 5.78644, 1.98402, 2.31429, 5.5791, 2.94085, 9.24799, 5.15937, 2.19041, 7.87817,
                2.9146, 1.66833, 1.85877, 2.45985, 4.20817, 1.85777, 2.28937, 9.37912, 6.18926, 8.55681,
                6.60963, 3.92066, 7.5521, 5.70463, 7.6313, 2.48866, 7.18352, 4.8413, 7.55702, 7.80702,
                4.5785, 9.3268, 2.83159, 1.07202, 9.33716, 3.6506, 2.50256, 1.21691, 5.06801, 8.27505,
                4.31539, 6.48286, 1.31363, 4.1912, 1.70668, 7.23867, 1.11441, 5.13591, 9.65186, 4.00767,
                5.24875, 1.94852, 5.52768, 8.97121, 5.8094, 3.53329, 4.19126, 9.06652, 3.1734, 1.21496,
                9.69154, 4.86971, 4.1166, 6.19361, 2.13874, 9.55039, 3.8225, 9.57548, 2.96554, 3.2383,
                8.77422, 3.11741, 8.3359, 5.89508, 2.72134, 6.29956, 1.43323, 1.14286, 1.4474, 4.59474,
                6.19214, 8.80766, 8.07546, 3.29232, 1.74029, 2.4198, 2.88544, 4.75644, 4.12921, 7.29896,
                7.27759, 1.67252, 1.32823, 8.1046, 9.10476, 1.04197, 3.37783, 5.2064, 4.23835, 3.16196,
                1.20852, 5.78501, 2.17175, 6.05313, 2.51048, 4.78967, 7.16219, 3.4651, 1.09, 2.9788,
                1.28761, 9.41385, 8.03883, 5.65835, 1.14816, 3.6892, 5.86648, 8.73895, 2.66603, 1.75192,
                1.39845, 4.99427, 1.17387, 1.60329, 8.30594, 6.72662, 7.95565, 7.35114, 3.1439, 1.39976,
                3.53095, 8.78581, 1.65811, 6.94299, 2.68641, 5.70058, 9.13491, 5.27637, 8.6232, 8.54902,
                2.25352, 5.86274, 5.20377, 2.96815, 4.96745, 5.3225, 3.99956, 1.08021, 5.54918, 7.05833,
                1.49913, 2.41822, 6.44593, 3.87301, 9.01465, 8.11336, 2.95749, 2.80188, 7.12396, 2.40595,
                5.59325, 9.89258, 2.30223, 1.4347, 9.09158, 7.43797, 3.79295, 4.53646, 1.72705, 4.16909,
                1.00912, 6.62167, 2.80244, 6.626, 3.89307, 1.42586, 7.51028, 7.83327, 4.65674, 7.33902,
                6.26823, 9.72608, 3.73491, 3.8238, 3.03815, 7.05101, 8.0103, 5.61396, 6.53738, 1.41095,
                5.0149, 9.71211, 4.23604, 5.98629, 4.70219, 9.69442, 2.82752, 9.93544, 6.9328, 8.2817,
                5.12336, 8.98577, 5.80541, 6.19552, 9.25748, 3.82732, 7.53525, 8.24712, 5.32057, 5.38817,
                8.57269, 5.99975, 3.42893, 5.38068, 3.48261, 3.02851, 6.82079, 9.2902, 2.80427, 8.91868,
                5.19227, 7.52482, 3.72584, 5.40107, 2.83307, 1.79755, 2.49121, 5.52697, 8.08823, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 1.30754,
                6.61627, 6.94572, 3.68646, 5.01521, 2.99912, 1.66028, 5.22315, 1.86555, 9.13033, 2.07541,
                5.72319, 1.75261, 9.25175, 9.19404, 3.69037, 6.2595, 6.09321, 6.52544, 9.60882, 3.34881,
                3.07914, 5.80104, 9.54944, 5.43754, 5.8654, 7.88937, 1.40811, 2.2597, 8.13163, 1.26821,
                8.94813, 5.86709, 5.03182, 9.02922, 4.39826, 5.84582, 6.87069, 4.25135, 6.13908, 6.74053,
                2.13683, 7.21184, 6.82974, 4.18545, 7.8691, 4.20879, 7.77509, 8.93208, 1.10502, 5.48298,
                1.66413, 8.08256, 1.57661, 4.19779, 9.47653, 4.41823, 7.86628, 7.94436, 3.71224, 7.95465,
                2.37637, 6.20771, 1.08107, 7.38138, 5.23577, 7.88133, 5.20653, 3.42101, 8.48523, 5.96192,
                1.63073, 5.25228, 7.68488, 2.7276, 5.1788, 3.07327, 5.57423, 2.87711, 1.44374, 5.66976,
                2.55051, 4.56682, 1.96629, 5.58829, 1.91922, 3.59846, 3.08583, 9.70901, 3.50487, 3.1026,
                1.82401, 6.1306, 4.76134, 4.31059, 8.31695, 3.60784, 7.45652, 6.51653, 4.84219, 7.76686,
                4.85031, 4.85544, 4.25714, 2.38005, 9.43471, 9.24769, 8.03763, 6.54696, 1.32399, 6.88891,
                2.16793, 3.64924, 4.24733, 3.47181, 1.66572, 2.36923, 2.45457, 9.44841, 4.34021, 1.45016,
                7.6686, 3.68812, 2.83922, 9.83581, 9.03719, 7.83414, 6.86009, 1.35715, 8.32489, 7.86316,
                5.09754, 5.78644, 1.98402, 2.31429, 5.5791, 2.94085, 9.24799, 5.15937, 2.19041, 7.87817,
                2.9146, 1.66833, 1.85877, 2.45985, 4.20817, 1.85777, 2.28937, 9.37912, 6.18926, 8.55681,
                6.60963, 3.92066, 7.5521, 5.70463, 7.6313, 2.48866, 7.18352, 4.8413, 7.55702, 7.80702,
                4.5785, 9.3268, 2.83159, 1.07202, 9.33716, 3.6506, 2.50256, 1.21691, 5.06801, 8.27505,
                4.31539, 6.48286, 1.31363, 4.1912, 1.70668, 7.23867, 1.11441, 5.13591, 9.65186, 4.00767,
                5.24875, 1.94852, 5.52768, 8.97121, 5.8094, 3.53329, 4.19126, 9.06652, 3.1734, 1.21496,
                9.69154, 4.86971, 4.1166, 6.19361, 2.13874, 9.55039, 3.8225, 9.57548, 2.96554, 3.2383,
                8.77422, 3.11741, 8.3359, 5.89508, 2.72134, 6.29956, 1.43323, 1.14286, 1.4474, 4.59474,
                6.19214, 8.80766, 8.07546, 3.29232, 1.74029, 2.4198, 2.88544, 4.75644, 4.12921, 7.29896,
                7.27759, 1.67252, 1.32823, 8.1046, 9.10476, 1.04197, 3.37783, 5.2064, 4.23835, 3.16196,
                1.20852, 5.78501, 2.17175, 6.05313, 2.51048, 4.78967, 7.16219, 3.4651, 1.09, 2.9788,
                1.28761, 9.41385, 8.03883, 5.65835, 1.14816, 3.6892, 5.86648, 8.73895, 2.66603, 1.75192,
                1.39845, 4.99427, 1.17387, 1.60329, 8.30594, 6.72662, 7.95565, 7.35114, 3.1439, 1.39976,
                3.53095, 8.78581, 1.65811, 6.94299, 2.68641, 5.70058, 9.13491, 5.27637, 8.6232, 8.54902,
                2.25352, 5.86274, 5.20377, 2.96815, 4.96745, 5.3225, 3.99956, 1.08021, 5.54918, 7.05833,
                1.49913, 2.41822, 6.44593, 3.87301, 9.01465, 8.11336, 2.95749, 2.80188, 7.12396, 2.40595,
                5.59325, 9.89258, 2.30223, 1.4347, 9.09158, 7.43797, 3.79295, 4.53646, 1.72705, 4.16909,
                1.00912, 6.62167, 2.80244, 6.626, 3.89307, 1.42586, 7.51028, 7.83327, 4.65674, 7.33902,
                6.26823, 9.72608, 3.73491, 3.8238, 3.03815, 7.05101, 8.0103, 5.61396, 6.53738, 1.41095,
                5.0149, 9.71211, 4.23604, 5.98629, 4.70219, 9.69442, 2.82752, 9.93544, 6.9328, 8.2817,
                5.12336, 8.98577, 5.80541, 6.19552, 9.25748, 3.82732, 7.53525, 8.24712, 5.32057, 5.38817,
                8.57269, 5.99975, 3.42893, 5.38068, 3.48261, 3.02851, 6.82079, 9.2902, 2.80427, 8.91868,
                5.19227, 7.52482, 3.72584, 5.40107, 2.83307, 1.79755, 2.49121, 5.52697, 8.08823, 10},
            std::vector<T>{
                1, 9.97466, 9.39302, 2.15312, 9.99136, 3.1248, 4.56923, 4.4912, 7.02771, 9.41985,
                8.6168, 3.81946, 5.72093, 4.99108, 3.0662, 5.80973, 9.22566, 5.11484, 4.87629, 9.45215,
                8.0055, 7.44373, 8.22482, 1.83521, 5.66337, 8.78518, 8.46232, 8.46643, 3.45745, 1.53319,
                7.03475, 6.33759, 7.04489, 4.70609, 2.77796, 3.60667, 2.27908, 8.04983, 4.71285, 10},
            std::vector<T>{
                0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1,
                0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1,
                0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1},
            std::vector<T>{
                0.528016, 0.668187, 0.668186, 0.635471, 0.668187, 0.659096, 0.666861, 0.666715, 0.668138, 0.668186,
                0.53964, 0.668141, 0.668109, 0.619255, 0.668141, 0.647193, 0.662341, 0.661921, 0.667534, 0.66811,
                0.54692, 0.667558, 0.667297, 0.604361, 0.667564, 0.631676, 0.652518, 0.651781, 0.664541, 0.667311,
                0.551576, 0.664629, 0.663703, 0.592106, 0.664652, 0.615579, 0.638092, 0.637163, 0.656733, 0.663751,
                0.554596, 0.656917, 0.655047, 0.582718, 0.656967, 0.601233, 0.621878, 0.620939, 0.643723, 0.65514,
                0.556574, 0.643984, 0.641397, 0.575854, 0.644055, 0.589658, 0.606642, 0.605821, 0.627796, 0.641522,
                0.557878, 0.628081, 0.625301, 0.570987, 0.628158, 0.580903, 0.593915, 0.593262, 0.611954, 0.625433,
                0.558742, 0.612216, 0.609684, 0.567605, 0.612287, 0.574556, 0.584071, 0.583581, 0.598219, 0.609803,
                0.559316, 0.598435, 0.596364, 0.565285, 0.598493, 0.57008, 0.576828, 0.576475, 0.587333, 0.596461,
                0.559698, 0.587499, 0.58592, 0.563707, 0.587544, 0.56698, 0.571671, 0.571423, 0.579197, 0.585993,
                0.668182, 0.66458, 0.667903, 0.667432, 0.658361, 0.667935, 0.668185, 0.667547, 0.667307, 0.668186,
                0.66803, 0.656815, 0.666091, 0.664171, 0.646084, 0.666251, 0.668096, 0.66459, 0.663738, 0.668113,
                0.666772, 0.643839, 0.66026, 0.655973, 0.630413, 0.660667, 0.667203, 0.656835, 0.655116, 0.667328,
                0.662084, 0.627922, 0.649014, 0.642661, 0.614386, 0.649671, 0.663395, 0.643868, 0.64149, 0.663807,
                0.652065, 0.61207, 0.633798, 0.626647, 0.600233, 0.634582, 0.654454, 0.627954, 0.625399, 0.65525,
                0.637519, 0.598314, 0.617618, 0.610903, 0.588883, 0.618381, 0.640604, 0.612099, 0.609772, 0.641672,
                0.621298, 0.587406, 0.602959, 0.597357, 0.580333, 0.603611, 0.624467, 0.598338, 0.596436, 0.625592,
                0.606134, 0.57925, 0.591004, 0.586675, 0.57415, 0.591515, 0.608935, 0.587425, 0.585974, 0.609946,
                0.593511, 0.573381, 0.581898, 0.578717, 0.569797, 0.582278, 0.595758, 0.579264, 0.578207, 0.596577,
                0.583768, 0.569262, 0.575267, 0.573003, 0.566785, 0.575539, 0.58546, 0.57339, 0.572642, 0.586082,
                0.668174, 0.668159, 0.668178, 0.618792, 0.66788, 0.668183, 0.66818, 0.66818, 0.662345, 0.595566,
                0.667915, 0.667737, 0.667963, 0.603963, 0.665981, 0.668052, 0.668006, 0.668007, 0.652525, 0.585315,
                0.66615, 0.665341, 0.6664, 0.591792, 0.659985, 0.666907, 0.666636, 0.66664, 0.638101, 0.577728,
                0.660409, 0.658471, 0.661057, 0.582484, 0.648575, 0.662479, 0.661698, 0.661709, 0.621887, 0.572305,
                0.649254, 0.646247, 0.650314, 0.575687, 0.633281, 0.652764, 0.651396, 0.651414, 0.60665, 0.568515,
                0.634083, 0.630598, 0.635357, 0.57087, 0.617117, 0.638404, 0.636684, 0.636707, 0.593922, 0.565907,
                0.617895, 0.614559, 0.619142, 0.567524, 0.602533, 0.622196, 0.62046, 0.620482, 0.584076, 0.564129,
                0.603195, 0.600379, 0.604265, 0.56523, 0.59067, 0.606921, 0.605404, 0.605423, 0.576832, 0.562925,
                0.591189, 0.588995, 0.592029, 0.56367, 0.581651, 0.594139, 0.59293, 0.592946, 0.571674, 0.562114,
                0.582036, 0.580415, 0.582661, 0.562616, 0.57509, 0.584239, 0.583333, 0.583345, 0.568079, 0.561569,
                0.668139, 0.668063, 0.668139, 0.667082, 0.653793, 0.663397, 0.640434, 0.668175, 0.667092, 0.571849,
                0.667538, 0.666978, 0.667544, 0.663011, 0.639734, 0.654459, 0.624289, 0.667925, 0.663042, 0.5682,
                0.664556, 0.66269, 0.664578, 0.653734, 0.623561, 0.640611, 0.608777, 0.666203, 0.653791, 0.565691,
                0.656765, 0.653146, 0.65681, 0.639656, 0.608128, 0.624474, 0.59563, 0.660545, 0.639731, 0.563983,
                0.643768, 0.638894, 0.643833, 0.62348, 0.595107, 0.608942, 0.585363, 0.649473, 0.623558, 0.562827,
                0.627845, 0.622696, 0.627915, 0.608056, 0.584968, 0.595763, 0.577763, 0.634345, 0.608125, 0.562048,
                0.611999, 0.607362, 0.612063, 0.595049, 0.577477, 0.585464, 0.572329, 0.61815, 0.595104, 0.561524,
                0.598256, 0.594491, 0.598309, 0.584924, 0.572127, 0.577836, 0.568532, 0.603413, 0.584966, 0.561173,
                0.587362, 0.584504, 0.587403, 0.577445, 0.568392, 0.572381, 0.565918, 0.591359, 0.577475, 0.560938,
                0.579218, 0.577141, 0.579248, 0.572105, 0.565823, 0.568568, 0.564137, 0.582163, 0.572127, 0.560781,
                0.668102, 0.668132, 0.66388, 0.667456, 0.657447, 0.606385, 0.667634, 0.620685, 0.668185, 0.668187,
                0.667244, 0.667485, 0.655394, 0.664256, 0.644744, 0.59371, 0.664921, 0.6056, 0.668088, 0.668142,
                0.663529, 0.664358, 0.641868, 0.656146, 0.628916, 0.583917, 0.65754, 0.593086, 0.667146, 0.667567,
                0.654712, 0.656356, 0.625799, 0.642901, 0.612988, 0.576717, 0.644878, 0.583449, 0.66321, 0.664664,
                0.640947, 0.643193, 0.610134, 0.626905, 0.599072, 0.571593, 0.629065, 0.57638, 0.654104, 0.656992,
                0.624826, 0.62722, 0.59673, 0.611138, 0.587988, 0.568023, 0.613126, 0.571356, 0.640142, 0.644091,
                0.609258, 0.611426, 0.586197, 0.59755, 0.579676, 0.56557, 0.599186, 0.567859, 0.623984, 0.628198,
                0.596018, 0.597785, 0.578369, 0.586822, 0.573683, 0.563901, 0.588076, 0.565458, 0.608505, 0.612324,
                0.585658, 0.587002, 0.572757, 0.578824, 0.569471, 0.562771, 0.57974, 0.563825, 0.59541, 0.598524,
                0.577977, 0.578955, 0.568828, 0.573079, 0.566562, 0.562011, 0.573728, 0.56272, 0.585197, 0.587567},
            std::vector<T>{
                0.559698, 0.587499, 0.58592, 0.563707, 0.587544, 0.56698, 0.571671, 0.571423, 0.579197, 0.585993,
                0.583768, 0.569262, 0.575267, 0.573003, 0.566785, 0.575539, 0.58546, 0.57339, 0.572642, 0.586082,
                0.582036, 0.580415, 0.582661, 0.562616, 0.57509, 0.584239, 0.583333, 0.583345, 0.568079, 0.561569,
                0.579218, 0.577141, 0.579248, 0.572105, 0.565823, 0.568568, 0.564137, 0.582163, 0.572127, 0.560781,
                0.577977, 0.578955, 0.568828, 0.573079, 0.566562, 0.562011, 0.573728, 0.56272, 0.585197, 0.587567},
            std::vector<T>{
                1.2132, 1.37242, 1.3621, 1.23365, 1.37271, 1.25089, 1.27652, 1.27513, 1.32014, 1.36258,
                1.34833, 1.26322, 1.29695, 1.284, 1.24985, 1.29853, 1.35913, 1.2862, 1.28197, 1.36315,
                1.33748, 1.32752, 1.34137, 1.22801, 1.29593, 1.35132, 1.34559, 1.34566, 1.25679, 1.22266,
                1.32026, 1.30789, 1.32044, 1.27895, 1.24474, 1.25944, 1.23589, 1.33827, 1.27907, 1.21865,
                1.31284, 1.31868, 1.26086, 1.28443, 1.24866, 1.22491, 1.28812, 1.22855, 1.35744, 1.37287}),
        LSTMSequenceV1Params(
            5, 10, 10, 10,
            0.7f, false, op::RecurrentSequenceDirection::REVERSE,
            ET,
            std::vector<T>{
                1, 9.01139, 2.17637, 1.35784, 8.43793, 5.7887, 9.60679, 5.15734, 9.31364, 4.36529,
                2.39476, 9.03109, 1.24111, 3.62352, 4.5887, 8.2656, 6.64385, 9.17132, 6.00758, 8.55927,
                1.45439, 8.25611, 9.37734, 4.27627, 7.21853, 2.16383, 8.49418, 3.86518, 7.63482, 6.37093,
                4.27336, 8.15473, 7.28633, 8.39691, 9.09519, 8.48709, 9.18962, 9.77422, 6.90508, 8.30791,
                1.9247, 3.13776, 4.41411, 7.29312, 5.36451, 4.737, 6.74796, 7.83361, 6.01771, 7.2411,
                9.21933, 1.22152, 8.5364, 4.44885, 3.68894, 1.05667, 4.93802, 7.64144, 4.38184, 5.43925,
                1.12636, 3.24493, 4.12422, 8.17018, 9.44559, 1.9047, 7.61862, 9.78793, 7.35297, 9.56588,
                9.35063, 4.69028, 4.04285, 7.92198, 4.01183, 1.74418, 9.08702, 8.08601, 7.45719, 4.99591,
                7.75803, 6.92108, 4.24628, 4.40491, 6.84669, 1.50073, 1.70433, 1.82713, 8.24594, 1.58777,
                3.01887, 2.3692, 7.12532, 2.13363, 3.31622, 4.94171, 1.98701, 3.83109, 4.15659, 7.85029,
                1.88966, 9.38314, 3.93471, 4.83972, 3.43613, 8.09299, 5.07992, 1.74912, 1.63339, 3.96774,
                2.72895, 8.54079, 1.55517, 5.98811, 6.1355, 5.15899, 9.53946, 4.10297, 8.96004, 4.87936,
                9.95032, 8.20453, 9.4747, 4.41532, 4.64882, 3.58199, 5.05481, 7.886, 6.21909, 4.40311,
                6.60154, 7.65255, 9.22045, 5.78893, 5.66039, 1.92328, 2.09503, 6.79327, 5.01368, 1.68692,
                2.37312, 1.71557, 8.91434, 2.45326, 9.30656, 1.03221, 8.22342, 3.7318, 3.20279, 7.9139,
                7.33134, 8.97559, 9.21827, 3.0278, 4.79335, 2.49276, 2.89751, 8.55908, 9.23905, 7.13432,
                4.68331, 2.99916, 8.5177, 7.99063, 7.63611, 8.72832, 9.7526, 4.68963, 6.37313, 7.64598,
                3.40039, 7.8682, 9.37453, 8.8722, 2.04547, 9.7056, 3.72689, 6.48114, 2.50981, 8.72583,
                7.53753, 1.58606, 5.66289, 6.35937, 5.1213, 6.33018, 3.8751, 8.37849, 1.83465, 2.59183,
                2.98641, 5.26684, 9.25707, 9.92382, 6.43793, 9.28502, 4.29972, 1.07022, 6.89515, 2.51248,
                5.46812, 8.11724, 8.36165, 4.30087, 2.15868, 2.77405, 1.18806, 6.7019, 5.30756, 2.91198,
                3.38931, 2.69973, 9.00829, 7.44529, 1.61716, 3.21075, 9.2832, 2.84414, 5.19293, 7.86764,
                5.7913, 5.9592, 3.11618, 3.48381, 1.15161, 4.80193, 1.42298, 1.5328, 3.87086, 4.98756,
                2.51724, 3.71195, 5.04463, 5.25773, 7.33799, 7.30576, 6.34285, 9.75641, 6.9596, 8.90033,
                4.72902, 8.18267, 2.2418, 3.35014, 3.04117, 5.79207, 7.25244, 3.19579, 2.75589, 8.11128,
                4.64842, 4.91573, 6.63745, 7.10935, 4.88308, 3.6566, 9.97556, 2.77547, 6.02164, 5.25801,
                1.79881, 3.81268, 6.38811, 1.44305, 4.31185, 2.15409, 1.23393, 7.58904, 3.36874, 7.40332,
                7.09691, 8.38579, 8.11905, 6.45184, 5.53603, 4.87629, 1.90757, 5.37873, 4.60355, 6.28454,
                4.99743, 3.10531, 6.7079, 3.06309, 3.26306, 9.80713, 5.01942, 1.72799, 2.07818, 5.165,
                8.00637, 4.58645, 6.62034, 8.186, 5.45095, 1.33642, 7.47286, 7.24008, 3.83915, 8.52919,
                1.36359, 6.66642, 1.19573, 7.67591, 4.57269, 5.67623, 5.17782, 7.06703, 8.65317, 3.84445,
                3.84243, 2.50624, 7.1609, 8.36409, 2.64523, 1.31036, 5.58945, 8.00466, 6.54905, 9.4617,
                3.05165, 8.04617, 4.70239, 7.24866, 3.25719, 5.56595, 3.52864, 2.96697, 9.31387, 8.00382,
                3.33332, 3.04412, 8.14064, 2.59336, 3.64721, 8.35451, 5.01581, 8.64861, 9.45624, 2.77414,
                9.3333, 4.61045, 5.21279, 3.02345, 7.42243, 6.76596, 9.21848, 5.45899, 3.14613, 7.12242,
                6.83178, 7.91725, 6.55954, 7.36386, 7.03253, 5.3523, 9.22608, 1.25059, 9.13047, 5.01358,
                5.90091, 1.1757, 8.76586, 6.71301, 5.49807, 4.45839, 1.03715, 7.6203, 3.26509, 9.74689,
                1.83018, 5.18702, 6.68799, 4.48163, 2.42089, 5.30825, 5.84336, 9.5668, 7.15807, 5.12003,
                5.16425, 2.6135, 9.8642, 9.98883, 6.88379, 8.29995, 3.92791, 1.55486, 2.2537, 2.82402,
                1.36006, 7.34118, 2.78316, 3.29005, 8.66468, 6.91618, 8.25624, 3.83807, 8.26726, 8.96759,
                7.16978, 8.17155, 8.87641, 3.85817, 8.6441, 8.24388, 6.1418, 1.95616, 7.79479, 7.36882,
                2.6179, 8.65872, 8.42148, 9.20951, 8.70195, 2.93495, 4.03916, 1.35492, 3.11572, 2.12385,
                3.49692, 4.94849, 8.73938, 3.90629, 9.88792, 9.68755, 7.25734, 7.03253, 1.32517, 8.06911,
                5.75969, 7.05649, 4.8373, 1.3598, 6.80822, 6.11809, 7.34439, 7.33369, 3.03278, 9.86214,
                2.06297, 2.84922, 9.57812, 9.41124, 4.04918, 9.3727, 1.93448, 2.13315, 4.97116, 6.6088,
                6.47021, 2.53016, 3.99592, 2.34172, 9.62384, 1.58149, 2.05354, 5.14612, 4.35021, 9.56793,
                6.21989, 2.20648, 1.51876, 6.8966, 7.39606, 4.64265, 6.94228, 8.60179, 9.89539, 9.59615,
                7.603, 3.64753, 7.34614, 5.50653, 4.01763, 5.71267, 9.066, 9.4331, 5.01289, 7.5353,
                8.26267, 2.76396, 5.91656, 2.34152, 2.68611, 4.44968, 1.5141, 7.10457, 8.98883, 4.09502,
                3.31662, 2.07621, 4.61676, 4.031, 1.73292, 6.67114, 6.06446, 1.68369, 1.67308, 10},
            std::vector<T>{
                1, 5.58749, 2.65019, 8.27893, 9.44007, 1.26646, 8.06485, 2.4749, 4.90466, 6.16841,
                2.9532, 8.47611, 3.69586, 4.24748, 8.82421, 1.1096, 7.07798, 5.33121, 9.7954, 2.89868,
                2.92113, 9.4959, 3.93547, 9.87516, 4.936, 9.68846, 3.32138, 7.8397, 2.73004, 3.54654,
                7.92584, 1.07364, 4.17543, 3.10202, 4.46595, 1.98213, 4.66613, 4.64841, 7.93022, 8.57683,
                4.53797, 5.19696, 1.93621, 2.71692, 7.05413, 5.27169, 8.21389, 3.22891, 7.67315, 10},
            std::vector<T>{
                1, 2.13438, 5.72392, 9.79342, 4.95939, 5.88606, 6.28873, 1.5905, 7.75203, 6.46966,
                6.53091, 9.13469, 9.27454, 2.62106, 7.6413, 1.27291, 3.02471, 3.92884, 4.92038, 3.27424,
                4.38241, 3.92348, 8.37897, 7.96488, 3.12631, 1.32181, 4.02546, 8.84497, 3.27456, 8.2492,
                5.84422, 9.0774, 6.26614, 7.29545, 7.53335, 1.48872, 5.71874, 6.43082, 9.68644, 3.70349,
                3.64883, 9.97216, 5.6593, 7.06806, 5.38156, 4.63353, 4.82144, 1.5192, 8.87038, 10},
            std::vector<int64_t>{10, 10, 10, 10, 10},
            std::vector<T>{
                1, 1.58235, 1.55715, 9.0725, 8.98434, 6.0259, 2.20956, 5.69598, 5.591, 6.22866,
                3.83043, 8.99042, 6.79015, 5.47362, 3.85033, 1.58001, 6.00399, 9.53966, 8.35195, 2.33561,
                1.43071, 1.10545, 8.23914, 6.88122, 6.16229, 8.65462, 6.19589, 6.19551, 6.01633, 1.76164,
                1.41133, 4.00032, 6.57013, 8.26936, 2.40101, 5.56064, 7.65621, 6.99739, 9.24985, 7.03113,
                5.93531, 6.88895, 7.564, 2.08789, 2.45067, 3.77189, 3.5213, 2.5474, 9.17393, 1.93798,
                4.21552, 4.85012, 3.77817, 2.85546, 3.31723, 4.22566, 8.43396, 6.62641, 9.08403, 9.44927,
                1.33493, 8.96543, 7.55197, 5.28736, 8.38082, 6.82582, 5.29947, 1.47086, 8.33883, 7.2491,
                3.31591, 8.50679, 8.86211, 9.69074, 3.66469, 2.50035, 6.15909, 5.2076, 6.19304, 7.92893,
                9.66382, 7.84886, 9.71917, 2.05874, 6.53135, 9.86027, 5.8924, 5.54567, 4.07782, 1.47401,
                8.71301, 1.85602, 2.73131, 8.98951, 9.9603, 4.3386, 2.81821, 4.36272, 8.3821, 6.64452,
                7.2118, 7.08588, 5.82932, 8.86502, 3.84654, 3.91562, 4.59225, 2.60513, 2.9141, 1.26067,
                9.29858, 9.185, 5.25551, 5.91884, 9.76741, 7.30087, 9.09672, 5.58644, 1.33538, 8.97838,
                8.66129, 8.42606, 6.67734, 5.21469, 3.2893, 7.15482, 6.68333, 6.23113, 1.40497, 3.50662,
                1.04018, 2.64265, 4.43901, 1.74996, 7.92567, 6.25056, 1.80776, 2.43364, 4.19004, 7.16324,
                8.38644, 7.33103, 3.31969, 7.42022, 7.87053, 8.559, 6.21006, 6.56629, 7.032, 5.21333,
                4.12916, 1.09792, 4.91183, 9.98769, 2.63651, 7.47683, 4.09084, 1.11776, 4.98008, 2.05417,
                3.81136, 3.78223, 8.9567, 3.69608, 9.82358, 4.15339, 1.55375, 2.58225, 5.35357, 9.96,
                2.63519, 8.34962, 1.67387, 6.97949, 1.52856, 6.16907, 7.26676, 3.61943, 7.54626, 7.8529,
                9.92461, 5.79463, 4.32859, 7.80883, 2.21124, 3.19625, 8.26345, 3.0258, 4.14905, 4.44074,
                4.40667, 4.3796, 2.65345, 7.52455, 7.24033, 8.20462, 2.70815, 2.24004, 9.89098, 3.3111,
                2.78455, 7.33025, 1.03654, 3.16342, 3.15485, 4.65602, 2.89809, 8.78445, 3.82711, 4.03929,
                5.06706, 7.46336, 2.99334, 6.76448, 5.71191, 5.12153, 5.43331, 1.77758, 6.66328, 3.8654,
                8.21078, 6.80763, 5.15698, 7.09883, 8.90826, 2.80116, 2.05896, 9.43922, 5.59127, 5.10843,
                5.39114, 4.77302, 6.52344, 5.95818, 7.42981, 7.35046, 8.61927, 3.56677, 6.74051, 2.3789,
                8.38043, 4.82216, 4.56786, 3.11453, 4.62395, 1.9779, 6.291, 4.57929, 2.53787, 8.10957,
                1.59414, 1.89001, 7.30965, 2.15147, 4.07816, 8.95942, 9.95274, 9.7535, 6.60151, 6.62482,
                8.13667, 5.02444, 8.33802, 9.46244, 1.53285, 4.86751, 9.00238, 3.6553, 6.14745, 9.38268,
                6.13151, 1.34703, 6.85186, 1.27094, 5.87562, 6.37423, 9.46726, 6.16143, 1.46485, 2.33172,
                4.72484, 2.58218, 8.77431, 2.26195, 2.80098, 6.5379, 9.76863, 2.67786, 1.50383, 9.24777,
                6.70801, 2.48443, 2.93425, 8.84472, 9.48241, 2.54342, 3.29629, 2.93909, 2.07659, 3.51105,
                8.05238, 6.52768, 2.89578, 9.46054, 9.42958, 4.99612, 3.72457, 1.09732, 5.46821, 8.3767,
                4.9284, 4.52297, 9.06671, 3.55751, 7.85551, 9.26148, 8.87619, 3.91563, 8.93108, 7.12851,
                7.38578, 6.00562, 9.88357, 5.96685, 7.98384, 7.18464, 2.08388, 4.19079, 7.13242, 5.10029,
                5.59207, 7.5696, 2.09462, 8.23487, 5.4306, 5.08607, 7.20121, 3.42059, 9.09514, 2.47813,
                7.56546, 3.04737, 4.75688, 6.15676, 5.44893, 6.38723, 5.49974, 9.74185, 2.04256, 6.03927,
                1.62865, 1.22683, 8.57313, 8.65697, 7.51196, 7.66841, 5.93585, 1.01738, 5.79587, 8.98536,
                5.72791, 4.97014, 1.33971, 9.76335, 9.01953, 3.47976, 3.0864, 9.17127, 7.85755, 8.21389,
                2.65647, 8.53283, 6.75763, 5.86692, 4.59959, 4.11153, 4.10978, 6.27941, 8.43579, 8.336,
                9.45283, 1.3166, 3.15401, 9.04877, 7.89529, 1.45832, 2.84421, 4.97627, 4.67164, 8.28176,
                9.46578, 2.64635, 3.16345, 1.4227, 5.32832, 9.22188, 8.92144, 6.89066, 2.86784, 10},
            std::vector<T>{
                1, 4.54659, 6.32448, 5.40494, 9.86159, 9.07852, 3.05904, 9.77817, 3.49926, 7.63737,
                2.39762, 6.18958, 6.24331, 8.25735, 6.53109, 2.65972, 8.42707, 4.09645, 9.46241, 6.56499,
                7.61151, 2.11518, 8.1123, 4.51095, 5.13355, 8.57515, 7.48222, 8.91597, 2.26922, 1.67127,
                6.20972, 5.51447, 5.59529, 7.6583, 2.97689, 8.74954, 3.53749, 9.36144, 2.86963, 9.2585,
                3.89263, 1.92284, 4.57657, 7.99819, 9.3836, 1.19691, 3.23177, 7.07376, 9.90035, 9.54876,
                1.83406, 2.42544, 2.79611, 4.98123, 2.76576, 5.84942, 8.4264, 9.87311, 2.1547, 6.2748,
                4.04931, 4.64838, 7.61106, 4.25832, 2.01333, 6.23769, 7.10437, 7.40875, 9.62977, 9.96437,
                2.11365, 6.23564, 3.32097, 6.85371, 4.39948, 1.80143, 7.94896, 9.80996, 9.71046, 4.03104,
                8.86624, 9.06961, 1.76453, 6.6127, 3.70704, 8.67179, 6.1986, 2.6383, 9.37425, 2.02916,
                1.8428, 7.76668, 2.05601, 6.04629, 3.92551, 9.68333, 5.45613, 5.21474, 3.96363, 1.65678,
                1.63911, 1.51876, 4.37833, 1.04078, 9.77478, 1.27549, 6.44559, 4.64664, 8.92835, 9.88028,
                9.66077, 6.02858, 3.18563, 1.17797, 3.09603, 5.34818, 9.70473, 2.86306, 5.49366, 1.87064,
                7.23007, 7.76858, 2.00428, 2.48244, 7.78903, 7.90607, 8.06873, 2.52497, 2.49137, 2.61559,
                7.75215, 8.75981, 8.30843, 1.23805, 1.52846, 4.92311, 1.50707, 7.50994, 6.00506, 6.58755,
                2.16225, 5.52023, 5.58397, 1.84541, 8.86125, 8.83867, 1.16775, 7.64101, 6.51206, 9.81013,
                6.88963, 9.05285, 6.21656, 4.52452, 4.71779, 1.10079, 3.99572, 8.55103, 6.94, 5.69519,
                9.61978, 7.20197, 7.85556, 1.7112, 3.44624, 4.25074, 7.87477, 9.34275, 4.44811, 3.02249,
                3.0886, 6.17374, 6.47048, 5.63258, 8.19415, 8.7746, 9.22689, 1.18991, 1.6878, 5.10915,
                1.24905, 7.77101, 6.20286, 4.64343, 4.97444, 2.00702, 4.47644, 1.36942, 1.8044, 7.54883,
                7.56364, 2.33436, 4.36203, 7.75994, 3.35254, 1.30727, 8.72577, 6.31045, 1.73164, 3.10143,
                1.75297, 4.90549, 7.79632, 1.42155, 7.67554, 4.11003, 2.45134, 9.93904, 7.65293, 9.55969,
                9.49092, 1.16254, 5.35432, 5.68326, 2.68756, 9.79784, 8.90456, 4.41579, 3.77757, 6.9883,
                3.48931, 4.08603, 3.56546, 5.56486, 5.24488, 1.48558, 7.22185, 5.38926, 3.353, 8.21195,
                7.22835, 8.65753, 1.14195, 5.16396, 2.29447, 6.81389, 6.12026, 7.23296, 9.03696, 6.15992,
                6.61774, 7.49631, 6.15221, 8.23327, 3.52917, 4.44016, 7.59119, 7.20278, 7.87011, 7.4118,
                4.88929, 7.10041, 8.72445, 9.33136, 9.52693, 7.3276, 8.59106, 5.10541, 6.63513, 6.74733,
                8.23243, 4.2018, 8.18058, 4.31184, 2.65255, 3.67934, 8.10169, 4.09561, 9.69242, 5.3705,
                3.02728, 7.75847, 8.23799, 6.83668, 8.52236, 1.39545, 2.02494, 8.31176, 6.58431, 8.52873,
                7.90275, 5.75623, 6.849, 9.04106, 4.84783, 9.78142, 7.13852, 4.52031, 2.7178, 5.59408,
                8.57777, 9.67441, 7.0772, 5.94922, 2.19153, 2.92101, 5.97566, 4.4292, 5.06291, 4.20734,
                5.03142, 3.77804, 3.11829, 4.04353, 6.05138, 2.68516, 3.14739, 9.85841, 1.97082, 6.71148,
                7.21038, 6.83885, 8.15292, 7.97213, 2.2081, 6.63164, 4.6698, 4.86985, 8.82823, 1.55222,
                5.35014, 1.02844, 2.0888, 5.70194, 3.81421, 1.48773, 8.81108, 7.00017, 9.13739, 9.59582,
                1.4752, 2.15818, 5.04522, 7.42531, 5.22722, 9.60355, 7.67216, 8.76329, 4.73203, 5.84448,
                1.55273, 7.13586, 8.91209, 8.22101, 1.03308, 6.54954, 4.1324, 7.53138, 1.53171, 6.15368,
                4.95754, 6.49698, 9.4097, 4.49705, 1.16446, 1.08714, 5.33318, 7.10617, 4.72117, 3.7985,
                2.59871, 6.15397, 3.56235, 9.46913, 8.74236, 1.13157, 5.54921, 4.62576, 7.72262, 8.03736,
                8.69808, 2.61915, 5.70869, 6.4007, 8.62539, 9.05605, 1.76502, 8.7073, 6.7695, 6.44984,
                3.38996, 8.78573, 4.94558, 9.65115, 2.96345, 4.44614, 2.69691, 3.58606, 4.21715, 5.94832,
                7.18326, 7.90754, 6.07498, 3.91409, 3.49496, 7.76969, 6.89076, 8.57066, 5.35908, 10},
            std::vector<T>{
                1, 1.08809, 7.34367, 1.84401, 3.65, 7.04672, 9.90727, 2.28341, 8.40458, 4.26864,
                2.12786, 1.3584, 5.67691, 6.93083, 2.5351, 2.80941, 7.32406, 7.16185, 3.82714, 3.52963,
                9.8522, 7.2122, 7.36617, 6.03843, 8.64737, 6.82174, 6.17929, 7.25098, 9.065, 5.62021,
                2.05619, 9.89887, 5.70674, 3.66863, 8.63875, 7.11328, 5.15165, 3.11287, 5.59069, 10},
            std::vector<T>{
                0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1,
                0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1,
                0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1},
            std::vector<T>{
                0.559698, 0.563643, 0.575276, 0.58701, 0.572904, 0.575772, 0.576994, 0.561768, 0.581308, 0.577538,
                0.559316, 0.56519, 0.581911, 0.597796, 0.578576, 0.582604, 0.584301, 0.562411, 0.590206, 0.585053,
                0.558742, 0.567466, 0.591021, 0.611439, 0.586482, 0.591952, 0.594221, 0.563366, 0.601938, 0.595219,
                0.557878, 0.570786, 0.602981, 0.627234, 0.597104, 0.604167, 0.607024, 0.564781, 0.616415, 0.608267,
                0.556574, 0.575567, 0.617644, 0.643206, 0.610593, 0.619029, 0.622312, 0.566866, 0.632551, 0.623718,
                0.554596, 0.582317, 0.633825, 0.656365, 0.626306, 0.635242, 0.638519, 0.569914, 0.647951, 0.639885,
                0.551576, 0.591568, 0.649036, 0.664363, 0.642343, 0.650219, 0.652853, 0.574318, 0.659587, 0.653909,
                0.54692, 0.603678, 0.660274, 0.667486, 0.655742, 0.661, 0.662528, 0.58057, 0.665818, 0.663105,
                0.53964, 0.61846, 0.666097, 0.668132, 0.664056, 0.666378, 0.666924, 0.589205, 0.667845, 0.667113,
                0.528016, 0.634662, 0.667904, 0.668186, 0.6674, 0.667959, 0.668054, 0.60065, 0.668169, 0.668083,
                0.577721, 0.585209, 0.585595, 0.565295, 0.580988, 0.560659, 0.566647, 0.569616, 0.572781, 0.567475,
                0.585306, 0.595426, 0.595935, 0.56762, 0.589773, 0.560757, 0.569596, 0.573891, 0.578403, 0.570799,
                0.595554, 0.608525, 0.609155, 0.57101, 0.601382, 0.560902, 0.573861, 0.579969, 0.586244, 0.575585,
                0.608683, 0.624007, 0.624712, 0.575886, 0.615756, 0.561119, 0.579927, 0.588388, 0.596791, 0.582342,
                0.624184, 0.640163, 0.640838, 0.582762, 0.631861, 0.561444, 0.58833, 0.599592, 0.610209, 0.591601,
                0.640334, 0.654121, 0.65463, 0.592165, 0.647355, 0.561928, 0.599517, 0.613615, 0.625882, 0.60372,
                0.65425, 0.663218, 0.663487, 0.604436, 0.659202, 0.562648, 0.613525, 0.629591, 0.641946, 0.618509,
                0.663287, 0.667148, 0.667231, 0.619342, 0.665657, 0.563718, 0.629494, 0.645351, 0.655452, 0.634712,
                0.66717, 0.668088, 0.6681, 0.635559, 0.66781, 0.565301, 0.645264, 0.657865, 0.66391, 0.64978,
                0.668091, 0.668185, 0.668185, 0.65048, 0.668166, 0.567629, 0.657805, 0.665069, 0.667358, 0.660733,
                0.571076, 0.569599, 0.583098, 0.58192, 0.566985, 0.560831, 0.569929, 0.584406, 0.567476, 0.58273,
                0.57598, 0.573866, 0.592615, 0.591032, 0.570087, 0.561013, 0.57434, 0.59436, 0.5708, 0.592122,
                0.582893, 0.579934, 0.605006, 0.602996, 0.574566, 0.561285, 0.5806, 0.607198, 0.575587, 0.604382,
                0.592341, 0.58834, 0.620001, 0.617661, 0.580918, 0.561691, 0.589246, 0.622511, 0.582345, 0.619279,
                0.60466, 0.59953, 0.636224, 0.633842, 0.589678, 0.562296, 0.600703, 0.638713, 0.591605, 0.635496,
                0.619601, 0.613541, 0.651024, 0.649051, 0.601259, 0.563195, 0.614947, 0.653005, 0.603726, 0.650428,
                0.635821, 0.62951, 0.66148, 0.660283, 0.61561, 0.564528, 0.631009, 0.662612, 0.618515, 0.661126,
                0.650695, 0.645279, 0.666557, 0.6661, 0.631708, 0.566494, 0.64661, 0.666952, 0.634718, 0.666426,
                0.661285, 0.657816, 0.667992, 0.667905, 0.647221, 0.569372, 0.658712, 0.668059, 0.649785, 0.667968,
                0.666485, 0.665047, 0.66818, 0.668173, 0.659115, 0.57354, 0.665446, 0.668183, 0.660736, 0.668178,
                0.575645, 0.585051, 0.576926, 0.579983, 0.580676, 0.561414, 0.57526, 0.577421, 0.58672, 0.568884,
                0.582425, 0.595217, 0.584206, 0.588407, 0.589349, 0.561883, 0.581889, 0.584892, 0.597416, 0.572837,
                0.591713, 0.608265, 0.594095, 0.599616, 0.600836, 0.562582, 0.590991, 0.595006, 0.610975, 0.578482,
                0.603863, 0.623715, 0.606867, 0.613644, 0.615105, 0.56362, 0.602943, 0.608002, 0.626726, 0.586353,
                0.618675, 0.639882, 0.622134, 0.629621, 0.631176, 0.565157, 0.617599, 0.62342, 0.642735, 0.596934,
                0.634882, 0.653907, 0.638343, 0.645379, 0.646757, 0.567417, 0.633779, 0.639597, 0.656026, 0.610385,
                0.649921, 0.663104, 0.652716, 0.657884, 0.658809, 0.570715, 0.648997, 0.653689, 0.664197, 0.626077,
                0.660819, 0.667112, 0.662452, 0.665078, 0.665489, 0.575465, 0.660249, 0.662987, 0.66744, 0.642128,
                0.666309, 0.668083, 0.666898, 0.667673, 0.667772, 0.582175, 0.666087, 0.667075, 0.668127, 0.655585,
                0.667946, 0.668185, 0.66805, 0.668153, 0.668162, 0.591376, 0.667902, 0.668077, 0.668186, 0.663977,
                0.568706, 0.587492, 0.575078, 0.579316, 0.574221, 0.571875, 0.572469, 0.56152, 0.584476, 0.587567,
                0.57258, 0.598426, 0.581634, 0.587496, 0.580432, 0.577119, 0.577962, 0.562042, 0.594454, 0.598524,
                0.578118, 0.612205, 0.590647, 0.598431, 0.589018, 0.584473, 0.585637, 0.562818, 0.607316, 0.612324,
                0.585852, 0.62807, 0.602503, 0.612212, 0.600409, 0.59445, 0.595992, 0.563969, 0.622644, 0.628198,
                0.596275, 0.643974, 0.617081, 0.628076, 0.614595, 0.60731, 0.609224, 0.565671, 0.638843, 0.644091,
                0.609574, 0.65691, 0.633244, 0.64398, 0.630636, 0.622637, 0.624789, 0.568171, 0.653106, 0.656992,
                0.625179, 0.664625, 0.648544, 0.656914, 0.646281, 0.638837, 0.640912, 0.571807, 0.662668, 0.664664,
                0.641282, 0.667557, 0.659965, 0.664627, 0.658493, 0.653101, 0.654685, 0.577022, 0.666971, 0.667567,
                0.654961, 0.668141, 0.665973, 0.667557, 0.665351, 0.662666, 0.663516, 0.584339, 0.668062, 0.668142,
                0.663659, 0.668187, 0.667878, 0.668141, 0.667739, 0.66697, 0.66724, 0.594272, 0.668184, 0.668187},
            std::vector<T>{
                0.559698, 0.563643, 0.575276, 0.58701, 0.572904, 0.575772, 0.576994, 0.561768, 0.581308, 0.577538,
                0.577721, 0.585209, 0.585595, 0.565295, 0.580988, 0.560659, 0.566647, 0.569616, 0.572781, 0.567475,
                0.571076, 0.569599, 0.583098, 0.58192, 0.566985, 0.560831, 0.569929, 0.584406, 0.567476, 0.58273,
                0.575645, 0.585051, 0.576926, 0.579983, 0.580676, 0.561414, 0.57526, 0.577421, 0.58672, 0.568884,
                0.568706, 0.587492, 0.575078, 0.579316, 0.574221, 0.571875, 0.572469, 0.56152, 0.584476, 0.587567},
            std::vector<T>{
                1.2132, 1.23332, 1.297, 1.3692, 1.28344, 1.29988, 1.30703, 1.22367, 1.33299, 1.31023,
                1.31132, 1.35752, 1.36, 1.24196, 1.33102, 1.21804, 1.24912, 1.26516, 1.28275, 1.25354,
                1.2732, 1.26506, 1.34411, 1.33676, 1.25092, 1.21891, 1.26687, 1.35238, 1.25355, 1.34181,
                1.29914, 1.3565, 1.30662, 1.32489, 1.32911, 1.22187, 1.29691, 1.30955, 1.3673, 1.26116,
                1.26019, 1.37237, 1.29586, 1.32085, 1.29093, 1.27766, 1.28099, 1.22241, 1.35283, 1.37287}),
        LSTMSequenceV1Params(
            5, 10, 10, 5,
            0.7f, true, op::RecurrentSequenceDirection::BIDIRECTIONAL,
            ET,
            std::vector<T>{
                1, 9.01139, 2.17637, 1.35784, 8.43793, 5.7887, 9.60679, 5.15734, 9.31364, 4.36529,
                2.39476, 9.03109, 1.24111, 3.62352, 4.5887, 8.2656, 6.64385, 9.17132, 6.00758, 8.55927,
                1.45439, 8.25611, 9.37734, 4.27627, 7.21853, 2.16383, 8.49418, 3.86518, 7.63482, 6.37093,
                4.27336, 8.15473, 7.28633, 8.39691, 9.09519, 8.48709, 9.18962, 9.77422, 6.90508, 8.30791,
                1.9247, 3.13776, 4.41411, 7.29312, 5.36451, 4.737, 6.74796, 7.83361, 6.01771, 7.2411,
                9.21933, 1.22152, 8.5364, 4.44885, 3.68894, 1.05667, 4.93802, 7.64144, 4.38184, 5.43925,
                1.12636, 3.24493, 4.12422, 8.17018, 9.44559, 1.9047, 7.61862, 9.78793, 7.35297, 9.56588,
                9.35063, 4.69028, 4.04285, 7.92198, 4.01183, 1.74418, 9.08702, 8.08601, 7.45719, 4.99591,
                7.75803, 6.92108, 4.24628, 4.40491, 6.84669, 1.50073, 1.70433, 1.82713, 8.24594, 1.58777,
                3.01887, 2.3692, 7.12532, 2.13363, 3.31622, 4.94171, 1.98701, 3.83109, 4.15659, 7.85029,
                1.88966, 9.38314, 3.93471, 4.83972, 3.43613, 8.09299, 5.07992, 1.74912, 1.63339, 3.96774,
                2.72895, 8.54079, 1.55517, 5.98811, 6.1355, 5.15899, 9.53946, 4.10297, 8.96004, 4.87936,
                9.95032, 8.20453, 9.4747, 4.41532, 4.64882, 3.58199, 5.05481, 7.886, 6.21909, 4.40311,
                6.60154, 7.65255, 9.22045, 5.78893, 5.66039, 1.92328, 2.09503, 6.79327, 5.01368, 1.68692,
                2.37312, 1.71557, 8.91434, 2.45326, 9.30656, 1.03221, 8.22342, 3.7318, 3.20279, 7.9139,
                7.33134, 8.97559, 9.21827, 3.0278, 4.79335, 2.49276, 2.89751, 8.55908, 9.23905, 7.13432,
                4.68331, 2.99916, 8.5177, 7.99063, 7.63611, 8.72832, 9.7526, 4.68963, 6.37313, 7.64598,
                3.40039, 7.8682, 9.37453, 8.8722, 2.04547, 9.7056, 3.72689, 6.48114, 2.50981, 8.72583,
                7.53753, 1.58606, 5.66289, 6.35937, 5.1213, 6.33018, 3.8751, 8.37849, 1.83465, 2.59183,
                2.98641, 5.26684, 9.25707, 9.92382, 6.43793, 9.28502, 4.29972, 1.07022, 6.89515, 2.51248,
                5.46812, 8.11724, 8.36165, 4.30087, 2.15868, 2.77405, 1.18806, 6.7019, 5.30756, 2.91198,
                3.38931, 2.69973, 9.00829, 7.44529, 1.61716, 3.21075, 9.2832, 2.84414, 5.19293, 7.86764,
                5.7913, 5.9592, 3.11618, 3.48381, 1.15161, 4.80193, 1.42298, 1.5328, 3.87086, 4.98756,
                2.51724, 3.71195, 5.04463, 5.25773, 7.33799, 7.30576, 6.34285, 9.75641, 6.9596, 8.90033,
                4.72902, 8.18267, 2.2418, 3.35014, 3.04117, 5.79207, 7.25244, 3.19579, 2.75589, 10},
            std::vector<T>{
                1, 5.58749, 2.65019, 8.27893, 9.44007, 1.26646, 8.06485, 2.4749, 4.90466, 6.16841,
                2.9532, 8.47611, 3.69586, 4.24748, 8.82421, 1.1096, 7.07798, 5.33121, 9.7954, 2.89868,
                2.92113, 9.4959, 3.93547, 9.87516, 4.936, 9.68846, 3.32138, 7.8397, 2.73004, 3.54654,
                7.92584, 1.07364, 4.17543, 3.10202, 4.46595, 1.98213, 4.66613, 4.64841, 7.93022, 8.57683,
                4.53797, 5.19696, 1.93621, 2.71692, 7.05413, 5.27169, 8.21389, 3.22891, 7.67315, 7.06556,
                6.90317, 2.90659, 6.30619, 8.05291, 5.46056, 1.90711, 8.90894, 6.5506, 7.20546, 9.74475,
                4.7904, 1.6435, 2.71316, 3.03569, 1.44884, 2.59864, 6.0138, 9.62099, 9.54249, 5.42718,
                2.24859, 3.33714, 6.17442, 9.87063, 1.9562, 5.23073, 1.41973, 3.51415, 6.39872, 1.10723,
                1.04289, 1.14721, 4.65565, 2.58112, 1.91227, 4.84767, 7.09586, 5.69412, 3.94588, 5.92789,
                7.45168, 4.64152, 6.15237, 9.08577, 2.87424, 8.64111, 2.93614, 2.41379, 4.12032, 10},
            std::vector<T>{
                1, 2.13438, 5.72392, 9.79342, 4.95939, 5.88606, 6.28873, 1.5905, 7.75203, 6.46966,
                6.53091, 9.13469, 9.27454, 2.62106, 7.6413, 1.27291, 3.02471, 3.92884, 4.92038, 3.27424,
                4.38241, 3.92348, 8.37897, 7.96488, 3.12631, 1.32181, 4.02546, 8.84497, 3.27456, 8.2492,
                5.84422, 9.0774, 6.26614, 7.29545, 7.53335, 1.48872, 5.71874, 6.43082, 9.68644, 3.70349,
                3.64883, 9.97216, 5.6593, 7.06806, 5.38156, 4.63353, 4.82144, 1.5192, 8.87038, 9.06057,
                9.99976, 6.25004, 5.44209, 6.97974, 4.3708, 3.26893, 5.25793, 1.41232, 8.75483, 3.70018,
                3.66543, 5.15941, 5.0194, 8.57511, 3.55875, 6.72842, 3.43644, 3.55691, 8.33065, 1.02129,
                7.57898, 8.48318, 2.36557, 4.70784, 4.9721, 9.32466, 5.53333, 9.09969, 2.71381, 8.97326,
                4.91604, 2.48103, 8.44254, 8.4943, 3.80884, 9.94202, 3.91681, 5.81107, 8.72025, 5.34059,
                8.33047, 8.90756, 9.69971, 7.08959, 4.25775, 3.27698, 5.6478, 2.44651, 2.74736, 10},
            std::vector<int64_t>{5, 5, 5, 5, 5},
            std::vector<T>{
                1, 1.58235, 1.55715, 9.0725, 8.98434, 6.0259, 2.20956, 5.69598, 5.591, 6.22866,
                3.83043, 8.99042, 6.79015, 5.47362, 3.85033, 1.58001, 6.00399, 9.53966, 8.35195, 2.33561,
                1.43071, 1.10545, 8.23914, 6.88122, 6.16229, 8.65462, 6.19589, 6.19551, 6.01633, 1.76164,
                1.41133, 4.00032, 6.57013, 8.26936, 2.40101, 5.56064, 7.65621, 6.99739, 9.24985, 7.03113,
                5.93531, 6.88895, 7.564, 2.08789, 2.45067, 3.77189, 3.5213, 2.5474, 9.17393, 1.93798,
                4.21552, 4.85012, 3.77817, 2.85546, 3.31723, 4.22566, 8.43396, 6.62641, 9.08403, 9.44927,
                1.33493, 8.96543, 7.55197, 5.28736, 8.38082, 6.82582, 5.29947, 1.47086, 8.33883, 7.2491,
                3.31591, 8.50679, 8.86211, 9.69074, 3.66469, 2.50035, 6.15909, 5.2076, 6.19304, 7.92893,
                9.66382, 7.84886, 9.71917, 2.05874, 6.53135, 9.86027, 5.8924, 5.54567, 4.07782, 1.47401,
                8.71301, 1.85602, 2.73131, 8.98951, 9.9603, 4.3386, 2.81821, 4.36272, 8.3821, 6.64452,
                7.2118, 7.08588, 5.82932, 8.86502, 3.84654, 3.91562, 4.59225, 2.60513, 2.9141, 1.26067,
                9.29858, 9.185, 5.25551, 5.91884, 9.76741, 7.30087, 9.09672, 5.58644, 1.33538, 8.97838,
                8.66129, 8.42606, 6.67734, 5.21469, 3.2893, 7.15482, 6.68333, 6.23113, 1.40497, 3.50662,
                1.04018, 2.64265, 4.43901, 1.74996, 7.92567, 6.25056, 1.80776, 2.43364, 4.19004, 7.16324,
                8.38644, 7.33103, 3.31969, 7.42022, 7.87053, 8.559, 6.21006, 6.56629, 7.032, 5.21333,
                4.12916, 1.09792, 4.91183, 9.98769, 2.63651, 7.47683, 4.09084, 1.11776, 4.98008, 2.05417,
                3.81136, 3.78223, 8.9567, 3.69608, 9.82358, 4.15339, 1.55375, 2.58225, 5.35357, 9.96,
                2.63519, 8.34962, 1.67387, 6.97949, 1.52856, 6.16907, 7.26676, 3.61943, 7.54626, 7.8529,
                9.92461, 5.79463, 4.32859, 7.80883, 2.21124, 3.19625, 8.26345, 3.0258, 4.14905, 4.44074,
                4.40667, 4.3796, 2.65345, 7.52455, 7.24033, 8.20462, 2.70815, 2.24004, 9.89098, 3.3111,
                2.78455, 7.33025, 1.03654, 3.16342, 3.15485, 4.65602, 2.89809, 8.78445, 3.82711, 4.03929,
                5.06706, 7.46336, 2.99334, 6.76448, 5.71191, 5.12153, 5.43331, 1.77758, 6.66328, 3.8654,
                8.21078, 6.80763, 5.15698, 7.09883, 8.90826, 2.80116, 2.05896, 9.43922, 5.59127, 5.10843,
                5.39114, 4.77302, 6.52344, 5.95818, 7.42981, 7.35046, 8.61927, 3.56677, 6.74051, 2.3789,
                8.38043, 4.82216, 4.56786, 3.11453, 4.62395, 1.9779, 6.291, 4.57929, 2.53787, 8.10957,
                1.59414, 1.89001, 7.30965, 2.15147, 4.07816, 8.95942, 9.95274, 9.7535, 6.60151, 6.62482,
                8.13667, 5.02444, 8.33802, 9.46244, 1.53285, 4.86751, 9.00238, 3.6553, 6.14745, 9.38268,
                6.13151, 1.34703, 6.85186, 1.27094, 5.87562, 6.37423, 9.46726, 6.16143, 1.46485, 2.33172,
                4.72484, 2.58218, 8.77431, 2.26195, 2.80098, 6.5379, 9.76863, 2.67786, 1.50383, 9.24777,
                6.70801, 2.48443, 2.93425, 8.84472, 9.48241, 2.54342, 3.29629, 2.93909, 2.07659, 3.51105,
                8.05238, 6.52768, 2.89578, 9.46054, 9.42958, 4.99612, 3.72457, 1.09732, 5.46821, 8.3767,
                4.9284, 4.52297, 9.06671, 3.55751, 7.85551, 9.26148, 8.87619, 3.91563, 8.93108, 7.12851,
                7.38578, 6.00562, 9.88357, 5.96685, 7.98384, 7.18464, 2.08388, 4.19079, 7.13242, 5.10029,
                5.59207, 7.5696, 2.09462, 8.23487, 5.4306, 5.08607, 7.20121, 3.42059, 9.09514, 2.47813,
                7.56546, 3.04737, 4.75688, 6.15676, 5.44893, 6.38723, 5.49974, 9.74185, 2.04256, 6.03927,
                1.62865, 1.22683, 8.57313, 8.65697, 7.51196, 7.66841, 5.93585, 1.01738, 5.79587, 8.98536,
                5.72791, 4.97014, 1.33971, 9.76335, 9.01953, 3.47976, 3.0864, 9.17127, 7.85755, 8.21389,
                2.65647, 8.53283, 6.75763, 5.86692, 4.59959, 4.11153, 4.10978, 6.27941, 8.43579, 8.336,
                9.45283, 1.3166, 3.15401, 9.04877, 7.89529, 1.45832, 2.84421, 4.97627, 4.67164, 8.28176,
                9.46578, 2.64635, 3.16345, 1.4227, 5.32832, 9.22188, 8.92144, 6.89066, 2.86784, 8.8094,
                3.84588, 1.23466, 9.78789, 1.42303, 3.01417, 9.11293, 4.56955, 3.41493, 8.52977, 1.46027,
                9.6304, 3.7595, 8.82851, 3.39647, 5.96831, 9.46836, 7.06139, 1.00386, 4.40682, 3.34827,
                4.47685, 6.86575, 7.61416, 6.26863, 7.95864, 9.60114, 3.60174, 2.12447, 5.49191, 5.58725,
                9.57542, 1.23677, 2.3244, 1.14875, 6.63691, 6.05344, 1.17406, 5.24456, 6.94052, 3.63667,
                6.6326, 7.56277, 8.43608, 6.25921, 8.02039, 1.0978, 5.06817, 8.34008, 3.511, 7.46162,
                7.85162, 8.85299, 7.65666, 3.49309, 3.28998, 9.05593, 6.38533, 5.95336, 3.44796, 6.04198,
                2.35335, 7.15819, 2.51471, 4.67303, 6.04976, 7.07969, 8.89531, 8.28522, 3.21776, 6.08729,
                3.63002, 6.91288, 8.43328, 9.77926, 2.00687, 6.17069, 7.84192, 6.11165, 7.56512, 6.71043,
                9.9949, 2.20909, 6.52124, 3.8416, 3.56753, 9.80769, 9.40458, 5.57479, 8.94376, 7.007,
                9.15643, 7.34537, 2.77838, 4.32169, 5.46074, 2.82916, 5.01243, 7.20892, 7.11125, 9.59418,
                4.04632, 9.70847, 9.16347, 9.22927, 5.53325, 1.07013, 9.42891, 2.02416, 8.10225, 3.57743,
                2.48204, 3.85673, 7.87652, 9.5724, 9.96488, 1.22926, 9.93237, 7.98604, 3.93368, 5.40463,
                8.27822, 9.25339, 8.12387, 9.82377, 5.85865, 9.73178, 8.74985, 6.61779, 5.20904, 2.06291,
                7.4461, 9.19566, 3.64798, 4.03986, 6.28108, 5.8935, 3.28368, 7.23357, 3.69042, 1.18574,
                2.97844, 4.92947, 6.3979, 5.05038, 9.67588, 8.6172, 9.25392, 7.47116, 9.72858, 2.39344,
                1.65939, 6.65531, 6.01121, 8.64874, 4.35918, 3.78387, 7.42273, 5.83712, 3.12879, 3.05836,
                1.83374, 4.63502, 4.31904, 7.06468, 5.90065, 9.69154, 1.78742, 5.74906, 4.37218, 3.95907,
                4.392, 2.80048, 3.14789, 7.02904, 3.02179, 6.69492, 2.01936, 6.49056, 5.4559, 8.08234,
                6.56281, 6.79647, 2.82158, 9.3293, 6.74985, 7.46573, 5.85924, 6.46302, 1.75108, 2.51813,
                4.0386, 4.23293, 8.95259, 9.31798, 6.29865, 2.13308, 6.6108, 7.12425, 1.61579, 2.58459,
                7.30912, 9.12349, 5.95058, 7.13723, 8.84, 9.7293, 9.66431, 5.26864, 1.00005, 6.20084,
                4.38823, 1.99631, 5.24372, 5.5035, 6.1096, 9.0086, 5.62153, 4.48418, 4.89917, 8.4891,
                6.06718, 6.09795, 5.61604, 4.05636, 7.16928, 2.85497, 1.87784, 4.09056, 1.19954, 3.65072,
                1.02866, 4.28399, 3.71394, 9.3255, 4.37615, 2.17663, 3.74709, 8.02241, 4.53525, 1.40447,
                3.20265, 8.01579, 5.0947, 8.39444, 6.70833, 1.97415, 2.69876, 7.17428, 5.09109, 4.61213,
                1.70647, 9.68497, 4.87501, 1.7283, 2.43997, 5.65806, 9.24942, 6.33399, 8.78482, 7.74617,
                1.39981, 2.79742, 7.02529, 3.34156, 8.11078, 3.08428, 3.9854, 4.30715, 4.98405, 1.10623,
                4.9921, 1.07542, 8.80374, 5.7398, 1.246, 4.76494, 4.82886, 7.94031, 9.49241, 4.36517,
                6.15431, 6.39414, 4.94161, 6.58667, 2.51963, 1.48029, 5.60639, 7.02553, 1.24623, 2.7569,
                3.87485, 1.45887, 5.81507, 1.29477, 8.10444, 2.09316, 7.38267, 8.64263, 3.0875, 4.03357,
                7.37073, 7.01626, 2.00842, 3.51386, 9.36265, 3.36725, 9.14829, 8.19237, 3.57092, 7.91152,
                8.00399, 6.64666, 4.91302, 5.85436, 2.76866, 8.73306, 3.07039, 4.64437, 1.0429, 1.19034,
                8.57202, 2.47873, 5.76703, 7.85474, 4.77875, 9.10824, 3.30109, 7.75884, 4.46161, 1.67611,
                6.95084, 4.45923, 9.98302, 4.35653, 4.57055, 1.62008, 8.52016, 8.87614, 2.9365, 9.98647,
                7.19558, 3.4473, 6.94256, 2.52838, 1.8037, 4.13277, 8.07921, 2.76578, 5.34165, 2.3978,
                5.00802, 2.44953, 2.21856, 2.5814, 8.93366, 2.55804, 9.74033, 5.7804, 6.50438, 2.19987,
                5.58994, 9.04511, 4.09384, 6.76115, 8.06394, 9.79993, 1.02129, 9.63807, 4.05821, 2.03345,
                9.2003, 8.09304, 2.76742, 6.32525, 3.01142, 2.35557, 8.09354, 9.66952, 7.11964, 5.88259,
                6.04748, 5.05317, 9.25398, 1.62705, 8.36628, 2.22643, 4.24874, 9.11405, 2.267, 8.49208,
                3.461, 5.52411, 8.69717, 1.3226, 1.44982, 4.06619, 4.39646, 6.37047, 6.59565, 10},
            std::vector<T>{
                1, 4.54659, 6.32448, 5.40494, 9.86159, 9.07852, 3.05904, 9.77817, 3.49926, 7.63737,
                2.39762, 6.18958, 6.24331, 8.25735, 6.53109, 2.65972, 8.42707, 4.09645, 9.46241, 6.56499,
                7.61151, 2.11518, 8.1123, 4.51095, 5.13355, 8.57515, 7.48222, 8.91597, 2.26922, 1.67127,
                6.20972, 5.51447, 5.59529, 7.6583, 2.97689, 8.74954, 3.53749, 9.36144, 2.86963, 9.2585,
                3.89263, 1.92284, 4.57657, 7.99819, 9.3836, 1.19691, 3.23177, 7.07376, 9.90035, 9.54876,
                1.83406, 2.42544, 2.79611, 4.98123, 2.76576, 5.84942, 8.4264, 9.87311, 2.1547, 6.2748,
                4.04931, 4.64838, 7.61106, 4.25832, 2.01333, 6.23769, 7.10437, 7.40875, 9.62977, 9.96437,
                2.11365, 6.23564, 3.32097, 6.85371, 4.39948, 1.80143, 7.94896, 9.80996, 9.71046, 4.03104,
                8.86624, 9.06961, 1.76453, 6.6127, 3.70704, 8.67179, 6.1986, 2.6383, 9.37425, 2.02916,
                1.8428, 7.76668, 2.05601, 6.04629, 3.92551, 9.68333, 5.45613, 5.21474, 3.96363, 1.65678,
                1.63911, 1.51876, 4.37833, 1.04078, 9.77478, 1.27549, 6.44559, 4.64664, 8.92835, 9.88028,
                9.66077, 6.02858, 3.18563, 1.17797, 3.09603, 5.34818, 9.70473, 2.86306, 5.49366, 1.87064,
                7.23007, 7.76858, 2.00428, 2.48244, 7.78903, 7.90607, 8.06873, 2.52497, 2.49137, 2.61559,
                7.75215, 8.75981, 8.30843, 1.23805, 1.52846, 4.92311, 1.50707, 7.50994, 6.00506, 6.58755,
                2.16225, 5.52023, 5.58397, 1.84541, 8.86125, 8.83867, 1.16775, 7.64101, 6.51206, 9.81013,
                6.88963, 9.05285, 6.21656, 4.52452, 4.71779, 1.10079, 3.99572, 8.55103, 6.94, 5.69519,
                9.61978, 7.20197, 7.85556, 1.7112, 3.44624, 4.25074, 7.87477, 9.34275, 4.44811, 3.02249,
                3.0886, 6.17374, 6.47048, 5.63258, 8.19415, 8.7746, 9.22689, 1.18991, 1.6878, 5.10915,
                1.24905, 7.77101, 6.20286, 4.64343, 4.97444, 2.00702, 4.47644, 1.36942, 1.8044, 7.54883,
                7.56364, 2.33436, 4.36203, 7.75994, 3.35254, 1.30727, 8.72577, 6.31045, 1.73164, 3.10143,
                1.75297, 4.90549, 7.79632, 1.42155, 7.67554, 4.11003, 2.45134, 9.93904, 7.65293, 9.55969,
                9.49092, 1.16254, 5.35432, 5.68326, 2.68756, 9.79784, 8.90456, 4.41579, 3.77757, 6.9883,
                3.48931, 4.08603, 3.56546, 5.56486, 5.24488, 1.48558, 7.22185, 5.38926, 3.353, 8.21195,
                7.22835, 8.65753, 1.14195, 5.16396, 2.29447, 6.81389, 6.12026, 7.23296, 9.03696, 6.15992,
                6.61774, 7.49631, 6.15221, 8.23327, 3.52917, 4.44016, 7.59119, 7.20278, 7.87011, 7.4118,
                4.88929, 7.10041, 8.72445, 9.33136, 9.52693, 7.3276, 8.59106, 5.10541, 6.63513, 6.74733,
                8.23243, 4.2018, 8.18058, 4.31184, 2.65255, 3.67934, 8.10169, 4.09561, 9.69242, 5.3705,
                3.02728, 7.75847, 8.23799, 6.83668, 8.52236, 1.39545, 2.02494, 8.31176, 6.58431, 8.52873,
                7.90275, 5.75623, 6.849, 9.04106, 4.84783, 9.78142, 7.13852, 4.52031, 2.7178, 5.59408,
                8.57777, 9.67441, 7.0772, 5.94922, 2.19153, 2.92101, 5.97566, 4.4292, 5.06291, 4.20734,
                5.03142, 3.77804, 3.11829, 4.04353, 6.05138, 2.68516, 3.14739, 9.85841, 1.97082, 6.71148,
                7.21038, 6.83885, 8.15292, 7.97213, 2.2081, 6.63164, 4.6698, 4.86985, 8.82823, 1.55222,
                5.35014, 1.02844, 2.0888, 5.70194, 3.81421, 1.48773, 8.81108, 7.00017, 9.13739, 9.59582,
                1.4752, 2.15818, 5.04522, 7.42531, 5.22722, 9.60355, 7.67216, 8.76329, 4.73203, 5.84448,
                1.55273, 7.13586, 8.91209, 8.22101, 1.03308, 6.54954, 4.1324, 7.53138, 1.53171, 6.15368,
                4.95754, 6.49698, 9.4097, 4.49705, 1.16446, 1.08714, 5.33318, 7.10617, 4.72117, 3.7985,
                2.59871, 6.15397, 3.56235, 9.46913, 8.74236, 1.13157, 5.54921, 4.62576, 7.72262, 8.03736,
                8.69808, 2.61915, 5.70869, 6.4007, 8.62539, 9.05605, 1.76502, 8.7073, 6.7695, 6.44984,
                3.38996, 8.78573, 4.94558, 9.65115, 2.96345, 4.44614, 2.69691, 3.58606, 4.21715, 5.94832,
                7.18326, 7.90754, 6.07498, 3.91409, 3.49496, 7.76969, 6.89076, 8.57066, 5.35908, 9.39983,
                4.72367, 5.7885, 5.50385, 5.60082, 6.13971, 6.75445, 4.19049, 1.55002, 2.17919, 4.86608,
                2.83434, 2.54872, 6.78346, 5.17315, 5.68842, 9.60461, 7.62238, 8.55562, 9.20677, 1.99981,
                5.2998, 1.78484, 1.25381, 3.41057, 4.2545, 8.49597, 6.98817, 4.90633, 7.60492, 9.93473,
                9.60203, 8.31804, 3.88664, 2.34239, 2.57365, 6.30953, 7.92014, 4.19319, 6.73355, 2.84212,
                7.38741, 3.73816, 3.77684, 4.04575, 4.62478, 9.56991, 6.35541, 8.23529, 7.66465, 5.89206,
                6.40386, 2.00967, 9.06787, 8.96033, 6.39315, 6.09508, 8.71021, 7.24184, 5.04831, 6.63348,
                7.41792, 8.8327, 7.9134, 5.5876, 7.79328, 7.87538, 5.41605, 2.74772, 3.88932, 7.89934,
                9.8233, 5.03149, 2.1468, 1.45979, 4.32571, 5.31263, 4.43767, 9.98486, 3.49411, 8.23258,
                5.19501, 5.87229, 2.6117, 1.36902, 1.22311, 2.73083, 2.61211, 9.50159, 7.23431, 6.90859,
                6.11484, 7.05027, 7.73243, 1.53322, 4.02925, 3.35174, 5.23253, 9.18545, 4.2252, 4.3851,
                5.4527, 8.23178, 7.43629, 5.49068, 3.10742, 6.21584, 3.57903, 7.43574, 4.15479, 4.90666,
                7.84751, 3.20987, 5.93707, 5.80065, 7.3108, 8.2003, 2.45924, 2.65312, 9.98448, 7.86027,
                4.41315, 1.29331, 6.86571, 9.36546, 6.37493, 2.67899, 1.42097, 6.94499, 4.97888, 8.32001,
                1.59289, 6.30954, 6.55959, 3.61335, 9.37199, 8.13811, 2.92084, 9.5581, 3.05901, 4.57234,
                6.70284, 5.98927, 5.8783, 5.00032, 5.46558, 5.609, 6.44573, 9.38336, 7.30968, 2.05593,
                2.70677, 3.13242, 2.30241, 6.14404, 5.85338, 7.28636, 4.81769, 9.25817, 2.41154, 6.88612,
                2.93221, 1.95278, 2.00515, 5.93273, 7.80319, 4.1318, 4.74696, 2.18834, 1.75543, 1.35063,
                6.294, 4.75432, 6.07164, 5.49934, 1.39722, 4.40799, 7.92895, 4.76634, 4.56078, 1.90916,
                7.00981, 2.50819, 6.31957, 3.46231, 2.75408, 5.33776, 5.67472, 4.72021, 8.5595, 3.42135,
                9.40977, 5.0546, 7.58948, 7.83337, 4.94762, 7.46493, 1.74705, 3.73444, 8.03541, 5.32388,
                1.99355, 7.29553, 7.27851, 9.03012, 1.43236, 2.39153, 2.41475, 3.99028, 3.46309, 4.02156,
                8.97672, 9.23768, 4.02787, 8.05564, 8.04031, 3.48179, 6.04413, 4.15226, 1.24342, 1.01373,
                1.82379, 7.45745, 9.19248, 3.87985, 4.5868, 1.71719, 2.50581, 2.41056, 9.03223, 5.30532,
                1.4033, 9.71298, 6.57269, 2.46664, 8.00804, 5.28823, 9.80983, 3.90576, 2.8259, 9.10474,
                1.03754, 9.21314, 5.31505, 2.49488, 7.36028, 2.5922, 5.87357, 7.79499, 3.27024, 3.66111,
                8.68053, 9.44535, 7.72449, 8.18736, 2.98887, 5.96881, 1.96588, 5.81309, 5.08468, 5.78022,
                3.48668, 1.16396, 5.56096, 7.47946, 6.94378, 5.22245, 6.63033, 9.72782, 7.27977, 2.82604,
                4.6663, 2.51966, 3.46937, 6.84506, 7.7865, 1.43957, 8.07416, 3.61659, 4.01118, 3.4731,
                9.81749, 6.91879, 8.72981, 5.55872, 6.12225, 1.3078, 3.71204, 7.14632, 1.45034, 6.46249,
                2.86946, 2.77546, 7.8614, 8.86595, 9.12923, 9.287, 3.09589, 7.65446, 7.24201, 7.03379,
                6.62077, 1.72055, 2.96869, 5.11422, 9.71284, 6.79057, 1.55773, 1.54237, 8.82527, 7.19741,
                9.51219, 6.03872, 3.71177, 3.04635, 5.78192, 7.2869, 9.19943, 8.73105, 2.6986, 8.51169,
                2.42288, 6.83215, 8.51947, 2.67462, 7.18716, 3.92524, 3.64838, 6.45243, 8.06931, 7.97245,
                7.70631, 8.17218, 2.62555, 3.87636, 6.24838, 2.62702, 3.4494, 9.96515, 7.14449, 8.13208,
                2.82945, 8.73997, 6.34161, 7.51236, 8.24821, 8.52709, 8.23385, 4.06273, 7.85741, 7.51928,
                3.33662, 9.07837, 6.40373, 6.78942, 7.70122, 6.72775, 9.35052, 1.78962, 5.69662, 9.55886,
                1.28139, 6.11816, 7.42325, 2.96934, 5.04452, 3.33527, 3.21557, 7.72912, 2.65684, 9.88658,
                2.13663, 6.46773, 3.83655, 6.00923, 8.59554, 5.53451, 3.37365, 7.61857, 6.51566, 6.2357,
                9.20489, 1.67264, 7.15646, 3.73925, 4.29553, 7.37171, 2.62476, 7.72404, 1.78335, 6.55747,
                4.85313, 2.12074, 8.00724, 2.16874, 8.1091, 1.97776, 3.19266, 8.89558, 6.5867, 10},
            std::vector<T>{
                1, 1.08809, 7.34367, 1.84401, 3.65, 7.04672, 9.90727, 2.28341, 8.40458, 4.26864,
                2.12786, 1.3584, 5.67691, 6.93083, 2.5351, 2.80941, 7.32406, 7.16185, 3.82714, 3.52963,
                9.8522, 7.2122, 7.36617, 6.03843, 8.64737, 6.82174, 6.17929, 7.25098, 9.065, 5.62021,
                2.05619, 9.89887, 5.70674, 3.66863, 8.63875, 7.11328, 5.15165, 3.11287, 5.59069, 3.56016,
                7.04003, 9.88518, 5.09578, 8.85682, 1.44654, 5.53915, 1.61339, 1.18572, 3.6579, 1.24913,
                6.82981, 6.45494, 8.94083, 8.33236, 9.24103, 7.20789, 9.35252, 8.34363, 7.27433, 8.89566,
                8.11744, 8.27532, 9.8568, 3.84764, 7.58856, 1.72907, 6.9287, 5.69146, 2.46782, 4.1626,
                7.37082, 4.00921, 6.75911, 6.68399, 7.11442, 9.12597, 4.31382, 5.93337, 4.63211, 10},
            std::vector<T>{
                0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1,
                0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1,
                0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1,
                0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1,
                0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1,
                0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1, 0.1},
            std::vector<T>{
                0.468216, 0.618468, 0.667762, 0.668186, 0.667005, 0.667845, 0.667987, 0.569274, 0.668159, 0.66803,
                0.436517, 0.573615, 0.664074, 0.668079, 0.66007, 0.664627, 0.6657, 0.520998, 0.667513, 0.666071,
                0.413202, 0.525179, 0.649639, 0.666531, 0.639151, 0.651328, 0.654893, 0.478152, 0.662603, 0.656241,
                0.396648, 0.481646, 0.618295, 0.658034, 0.601404, 0.621309, 0.628053, 0.444053, 0.645491, 0.630768,
                0.385148, 0.446733, 0.573404, 0.634552, 0.553825, 0.577138, 0.585828, 0.418653, 0.611283, 0.589475,
                0.590675, 0.628293, 0.629709, 0.46966, 0.609654, 0.401027, 0.487158, 0.521658, 0.552734, 0.497312,
                0.631646, 0.655014, 0.655722, 0.510668, 0.644485, 0.419431, 0.531691, 0.569965, 0.60041, 0.54341,
                0.656666, 0.665734, 0.66593, 0.558256, 0.662223, 0.445122, 0.580236, 0.615459, 0.638483, 0.591706,
                0.666183, 0.667991, 0.668015, 0.605382, 0.667443, 0.479548, 0.623757, 0.647998, 0.659784, 0.632394,
                0.668043, 0.668183, 0.668184, 0.641772, 0.668155, 0.522673, 0.652657, 0.66351, 0.666941, 0.657024,
                0.665632, 0.663477, 0.668175, 0.668166, 0.654609, 0.530933, 0.664075, 0.668181, 0.657029, 0.668173,
                0.654654, 0.647903, 0.667802, 0.66763, 0.627494, 0.486513, 0.649644, 0.667934, 0.632404, 0.667755,
                0.627581, 0.615297, 0.664341, 0.663266, 0.585088, 0.450492, 0.618303, 0.665272, 0.59172, 0.664032,
                0.585204, 0.56977, 0.650444, 0.647309, 0.536576, 0.423354, 0.573413, 0.653419, 0.543424, 0.649515,
                0.536694, 0.521472, 0.619718, 0.614293, 0.491353, 0.403801, 0.524983, 0.625194, 0.497325, 0.618077,
                0.576187, 0.627698, 0.585362, 0.604275, 0.608023, 0.413104, 0.573282, 0.588705, 0.633605, 0.513641,
                0.620548, 0.654713, 0.627701, 0.641052, 0.643462, 0.436382, 0.618196, 0.6302, 0.657595, 0.561469,
                0.650907, 0.665649, 0.654715, 0.660863, 0.661828, 0.468037, 0.649583, 0.655964, 0.666422, 0.608212,
                0.664492, 0.667981, 0.66565, 0.667174, 0.667368, 0.508666, 0.664055, 0.665996, 0.668068, 0.643581,
                0.667825, 0.668183, 0.667981, 0.668136, 0.66815, 0.556073, 0.667759, 0.668022, 0.668186, 0.661874,
                0.661398, 0.668186, 0.667723, 0.668117, 0.667515, 0.66636, 0.666765, 0.560186, 0.668181, 0.668183,
                0.642374, 0.668095, 0.66383, 0.666946, 0.662609, 0.65735, 0.659013, 0.51245, 0.667939, 0.667978,
                0.606315, 0.666698, 0.648921, 0.659806, 0.645507, 0.633083, 0.636719, 0.471112, 0.665316, 0.665624,
                0.559309, 0.658727, 0.617043, 0.638534, 0.611309, 0.592663, 0.597827, 0.438701, 0.653566, 0.654623,
                0.511639, 0.636078, 0.571877, 0.600486, 0.565051, 0.544415, 0.549923, 0.414776, 0.625475, 0.627521,
                0.636308, 0.585028, 0.566575, 0.598958, 0.536327, 0.497101, 0.561873, 0.408877, 0.624184, 0.51352,
                0.65883, 0.627448, 0.612609, 0.637495, 0.584843, 0.54317, 0.608564, 0.430487, 0.652885, 0.561339,
                0.666722, 0.654586, 0.646298, 0.659355, 0.627308, 0.591478, 0.643802, 0.460154, 0.665112, 0.608099,
                0.668097, 0.665613, 0.662901, 0.666844, 0.654514, 0.632229, 0.661961, 0.49883, 0.667912, 0.643509,
                0.668186, 0.667977, 0.667567, 0.668108, 0.665593, 0.656946, 0.667394, 0.54513, 0.66818, 0.661847,
                0.661547, 0.667282, 0.667096, 0.668178, 0.660535, 0.668076, 0.659185, 0.660516, 0.668175, 0.473006,
                0.642746, 0.661391, 0.660491, 0.667864, 0.640257, 0.666507, 0.637108, 0.640212, 0.667785, 0.440135,
                0.606897, 0.642355, 0.640151, 0.664764, 0.603065, 0.657937, 0.598393, 0.602997, 0.664228, 0.415812,
                0.559969, 0.606286, 0.602906, 0.651762, 0.555663, 0.634343, 0.550536, 0.555588, 0.650103, 0.39848,
                0.51225, 0.559276, 0.555486, 0.6221, 0.508292, 0.594429, 0.503649, 0.508223, 0.619113, 0.386411,
                0.608718, 0.620998, 0.457879, 0.54662, 0.554179, 0.630205, 0.568834, 0.62793, 0.473801, 0.626594,
                0.643899, 0.651157, 0.495954, 0.594746, 0.601725, 0.655966, 0.614515, 0.654831, 0.515734, 0.654148,
                0.661998, 0.664572, 0.541864, 0.634567, 0.639366, 0.665997, 0.64744, 0.665683, 0.563712, 0.665487,
                0.667401, 0.667837, 0.590227, 0.658041, 0.660161, 0.668022, 0.663313, 0.667985, 0.610158, 0.667961,
                0.668152, 0.668177, 0.631318, 0.666532, 0.667025, 0.668184, 0.667638, 0.668183, 0.644798, 0.668182,
                0.666934, 0.636465, 0.668176, 0.668177, 0.6627, 0.668186, 0.663435, 0.667808, 0.66818, 0.667477,
                0.659751, 0.59746, 0.667824, 0.66784, 0.645753, 0.668092, 0.647783, 0.664381, 0.667904, 0.662402,
                0.638407, 0.549527, 0.664483, 0.664596, 0.611711, 0.666671, 0.615095, 0.650567, 0.665048, 0.644956,
                0.600299, 0.502744, 0.650882, 0.651231, 0.565521, 0.658614, 0.569527, 0.619937, 0.652675, 0.610413,
                0.552612, 0.463272, 0.620503, 0.621133, 0.517435, 0.635826, 0.52124, 0.575428, 0.62379, 0.564008,
                0.619111, 0.625883, 0.633724, 0.600854, 0.532704, 0.497421, 0.571602, 0.461671, 0.475281, 0.63631,
                0.650102, 0.653779, 0.65765, 0.638782, 0.581251, 0.543534, 0.616817, 0.500738, 0.517531, 0.658831,
                0.664228, 0.665379, 0.666436, 0.659912, 0.624548, 0.591824, 0.64879, 0.547281, 0.565623, 0.666723,
                0.667785, 0.667947, 0.668069, 0.66697, 0.653078, 0.632479, 0.663785, 0.595367, 0.611798, 0.668097,
                0.668175, 0.668182, 0.668186, 0.668119, 0.66517, 0.657065, 0.667716, 0.635005, 0.645806, 0.668186},
            std::vector<T>{
                0.385148, 0.446733, 0.573404, 0.634552, 0.553825, 0.577138, 0.585828, 0.418653, 0.611283, 0.589475,
                0.590675, 0.628293, 0.629709, 0.46966, 0.609654, 0.401027, 0.487158, 0.521658, 0.552734, 0.497312,
                0.536694, 0.521472, 0.619718, 0.614293, 0.491353, 0.403801, 0.524983, 0.625194, 0.497325, 0.618077,
                0.576187, 0.627698, 0.585362, 0.604275, 0.608023, 0.413104, 0.573282, 0.588705, 0.633605, 0.513641,
                0.511639, 0.636078, 0.571877, 0.600486, 0.565051, 0.544415, 0.549923, 0.414776, 0.625475, 0.627521,
                0.636308, 0.585028, 0.566575, 0.598958, 0.536327, 0.497101, 0.561873, 0.408877, 0.624184, 0.51352,
                0.51225, 0.559276, 0.555486, 0.6221, 0.508292, 0.594429, 0.503649, 0.508223, 0.619113, 0.386411,
                0.608718, 0.620998, 0.457879, 0.54662, 0.554179, 0.630205, 0.568834, 0.62793, 0.473801, 0.626594,
                0.552612, 0.463272, 0.620503, 0.621133, 0.517435, 0.635826, 0.52124, 0.575428, 0.62379, 0.564008,
                0.619111, 0.625883, 0.633724, 0.600854, 0.532704, 0.497421, 0.571602, 0.461671, 0.475281, 0.63631},
            std::vector<T>{
                0.657064, 0.808159, 1.28627, 1.82832, 1.18444, 1.30787, 1.3615, 0.735716, 1.55641, 1.3856,
                1.39376, 1.74058, 1.7592, 0.872984, 1.54166, 0.693415, 0.926748, 1.04718, 1.17924, 0.959985,
                1.10759, 1.04646, 1.63992, 1.58476, 0.940282, 0.699929, 1.06005, 1.70199, 0.960028, 1.62263,
                1.3023, 1.73294, 1.3585, 1.4956, 1.52728, 0.722161, 1.28558, 1.38043, 1.81407, 1.01716,
                1.00988, 1.85212, 1.27767, 1.46531, 1.24067, 1.14104, 1.16607, 0.72622, 1.70537, 1.7307,
                1.8558, 1.35635, 1.24873, 1.45354, 1.10604, 0.959278, 1.22421, 0.711984, 1.68998, 1.01672,
                1.01209, 1.21108, 1.19243, 1.66604, 0.997881, 1.42007, 0.98159, 0.997636, 1.63348, 0.6599,
                1.53336, 1.6538, 0.838953, 1.15094, 1.18613, 1.76588, 1.26089, 1.73591, 0.885339, 1.71907,
                1.17867, 0.854332, 1.64838, 1.65528, 1.03119, 1.84811, 1.04557, 1.29788, 1.68537, 1.23522,
                1.63346, 1.71032, 1.81584, 1.46818, 1.09099, 0.960351, 1.27613, 0.849735, 0.889806, 1.85583}),
    };
    return params;
}

template <element::Type_t ET>
std::vector<LSTMSequenceV1Params> generateV1ParamsBF16() {
    using T = typename element_type_traits<ET>::value_type;

    std::vector<LSTMSequenceV1Params> params {
        LSTMSequenceV1Params(
            5, 10, 10, 10,
            0.7f, false, op::RecurrentSequenceDirection::FORWARD,
            ET,
            std::vector<T>{
                1, 9.375, 9, 3.84375, 2.17188, 2.65625, 1.35938, 2.84375, 8.4375, 6.125,
                5.78125, 6.375, 9.625, 9.625, 5.15625, 6.875, 9.3125, 7.75, 4.375, 6.875,
                2.39062, 7.71875, 9, 9.625, 1.23438, 1.07812, 3.625, 1.95312, 4.5625, 3.6875,
                8.25, 6.90625, 6.625, 8.25, 9.125, 8.875, 6, 9.625, 8.5, 7.5,
                1.45312, 6.78125, 8.25, 7.4375, 9.375, 5.1875, 4.25, 3.9375, 7.1875, 4.9375,
                2.15625, 7.5625, 8.5, 9.9375, 3.85938, 7.0625, 7.625, 8.125, 6.375, 2.53125,
                4.25, 1.23438, 8.125, 8.1875, 7.28125, 9.125, 8.375, 1.21875, 9.125, 5.4375,
                8.5, 5.75, 9.1875, 6.375, 9.75, 1.46875, 6.875, 9, 8.25, 7.5625,
                1.92188, 8.375, 3.125, 5.5, 4.40625, 8.25, 7.28125, 1.85938, 5.375, 2.96875,
                4.75, 3.32812, 6.75, 5.1875, 7.8125, 5.125, 6, 7.375, 7.25, 2.59375,
                9.25, 5.78125, 1.21875, 2.5, 8.5, 7.90625, 4.4375, 9.375, 3.6875, 6.5,
                1.05469, 2.34375, 4.9375, 5.40625, 7.625, 4.375, 4.375, 8.625, 5.4375, 9.1875,
                1.125, 4.4375, 3.25, 3.84375, 4.125, 6.125, 8.125, 2.6875, 9.4375, 2.125,
                1.90625, 7.1875, 7.625, 8.1875, 9.75, 6.15625, 7.34375, 9.75, 9.5625, 6.6875,
                9.375, 9, 4.6875, 5.4375, 4.03125, 4.15625, 7.9375, 7.4375, 4, 5.53125,
                1.74219, 3.03125, 9.0625, 3.20312, 8.0625, 8.125, 7.4375, 5.4375, 5, 9.25,
                7.75, 9.5, 6.90625, 5.8125, 4.25, 3.26562, 4.375, 7.5, 6.84375, 4.3125,
                1.5, 5.5, 1.70312, 3.03125, 1.82812, 4.1875, 8.25, 6.84375, 1.58594, 3.8125,
                3.01562, 7.90625, 2.375, 8, 7.125, 8.625, 2.125, 9.5, 3.3125, 1.96875,
                4.9375, 9.1875, 1.98438, 4, 3.82812, 8.375, 4.15625, 9.0625, 7.84375, 1.38281,
                1.89062, 2.75, 9.375, 3.65625, 3.9375, 6.625, 4.8125, 1.77344, 3.4375, 2.28125,
                8.0625, 5.625, 5.0625, 7.1875, 1.75, 8.6875, 1.63281, 6.8125, 3.96875, 6.25,
                2.71875, 7.375, 8.5, 3.26562, 1.55469, 9.125, 6, 4.96875, 6.125, 1.1875,
                5.15625, 9.625, 9.5, 6.875, 4.09375, 5.625, 8.9375, 7.125, 4.875, 5.375,
                9.9375, 9.3125, 8.1875, 5.625, 9.5, 1.64844, 4.40625, 6.09375, 4.625, 6.53125,
                3.57812, 9.5, 5.0625, 4.75, 7.875, 3.375, 6.21875, 1.875, 4.375, 5.375,
                6.59375, 5.1875, 7.625, 1.26562, 9.25, 7.25, 5.78125, 7.4375, 5.65625, 7.5625,
                1.92188, 4.71875, 2.09375, 1.13281, 6.78125, 9.125, 5, 8.125, 1.6875, 2.48438,
                2.375, 3.8125, 1.71875, 6.5, 8.875, 4.25, 2.45312, 2.40625, 9.25, 2.59375,
                1.03125, 8.75, 8.25, 3.60938, 3.71875, 6.25, 3.1875, 5.0625, 7.90625, 4.6875,
                7.3125, 8.9375, 9, 7.21875, 9.1875, 3.5, 3.03125, 1.57812, 4.78125, 2.78125,
                2.5, 9.375, 2.89062, 8.6875, 8.5, 9.5625, 9.25, 1.46875, 7.125, 6.1875,
                4.6875, 5.3125, 3, 1.19531, 8.5, 4.375, 8, 4.71875, 7.625, 6.4375,
                8.75, 7.03125, 9.75, 8.5, 4.6875, 8, 6.375, 4.59375, 7.625, 8.125,
                3.40625, 9, 7.875, 3.35938, 9.375, 9.875, 8.875, 8.625, 2.03125, 7.5625,
                9.6875, 4.1875, 3.71875, 8.9375, 6.46875, 8.75, 2.5, 9.625, 8.75, 1,
                7.53125, 1.14844, 1.58594, 3.8125, 5.65625, 9.9375, 6.34375, 2.34375, 5.125, 2.5,
                6.3125, 7.8125, 3.875, 1.625, 8.375, 7.34375, 1.82812, 5.21875, 2.59375, 1.09375,
                2.98438, 7.96875, 5.25, 8.125, 9.25, 2.34375, 9.875, 1.21094, 6.4375, 7.84375,
                9.25, 3, 4.3125, 3.35938, 1.0625, 5.125, 6.875, 3.25, 2.5, 6.125,
                5.4375, 8.625, 8.125, 4.375, 8.375, 4.875, 4.3125, 8.5, 2.15625, 4.3125,
                2.78125, 1.35938, 1.1875, 6, 6.6875, 5.0625, 5.3125, 7.5, 2.90625, 4.375,
                3.375, 8.5625, 2.6875, 5.21875, 9, 6.0625, 7.4375, 6.9375, 1.60938, 5.15625,
                3.20312, 6.625, 9.25, 3, 2.84375, 7.59375, 5.1875, 4.4375, 7.875, 2.75,
                5.78125, 3.4375, 5.9375, 3.25, 3.10938, 2.375, 3.46875, 7.9375, 1.14844, 3.29688,
                4.8125, 2.14062, 1.42188, 7, 1.53125, 4.6875, 3.875, 7, 5, 6.9375,
                2.51562, 3.75, 3.71875, 2.8125, 5.03125, 3, 5.25, 2.07812, 7.3125, 1.32812,
                7.3125, 1.30469, 6.3125, 3.0625, 9.75, 3.0625, 6.9375, 6.625, 8.875, 9,
                4.71875, 8, 8.125, 7.5, 2.23438, 3.78125, 3.34375, 4.25, 3.03125, 2.75,
                5.78125, 9.375, 7.25, 6.0625, 3.1875, 8.375, 2.75, 4.125, 8.125, 10},
            std::vector<T>{
                1, 9.1875, 5.5625, 6.625, 2.65625, 8.125, 8.25, 4.875, 9.4375, 5.875,
                1.26562, 4.71875, 8.0625, 1.76562, 2.46875, 8, 4.875, 5.375, 6.15625, 1.45312,
                2.95312, 5.84375, 8.5, 1.375, 3.6875, 8.25, 4.25, 9.9375, 8.8125, 6.75,
                1.10938, 9.5, 7.0625, 7.625, 5.3125, 8.375, 9.75, 2.03125, 2.90625, 7.625,
                2.90625, 8.375, 9.5, 1.66406, 3.9375, 2.875, 9.875, 8.25, 4.9375, 10},
            std::vector<T>{
                1, 6.59375, 2.125, 2.89062, 5.71875, 5.0625, 9.75, 6, 4.9375, 3.21875,
                5.875, 1.85156, 6.28125, 9.875, 1.59375, 3.40625, 7.75, 9.25, 6.46875, 8.75,
                6.5, 4.0625, 9.125, 4.4375, 9.25, 5.5625, 2.625, 1.8125, 7.625, 1.875,
                1.26562, 6.3125, 3.03125, 7.3125, 3.92188, 2.21875, 4.90625, 2.10938, 3.28125, 6.875,
                4.375, 7.25, 3.92188, 3.09375, 8.375, 7.96875, 7.9375, 1.875, 3.125, 10},
            std::vector<int64_t>{10, 10, 10, 10, 10},
            std::vector<T>{
                1, 7.5, 1.57812, 10, 1.55469, 3.5, 9.0625, 7.5625, 9, 1.30469,
                6, 7.1875, 2.20312, 9, 5.6875, 2, 5.5625, 5.5625, 6.21875, 3.09375,
                3.82812, 7.28125, 9, 1.625, 6.78125, 9.875, 5.46875, 4.1875, 3.84375, 2.21875,
                1.57812, 9.125, 6, 6, 9.5, 2.28125, 8.375, 9.875, 2.32812, 5.96875,
                1.42969, 4.5625, 1.10156, 5.4375, 8.25, 1.625, 6.875, 4.6875, 6.15625, 6.15625,
                8.625, 4.125, 6.1875, 1.375, 6.1875, 8.875, 6, 3.40625, 1.75781, 4.8125,
                1.40625, 4.21875, 4, 6.4375, 6.5625, 4.1875, 8.25, 7.40625, 2.40625, 7,
                5.5625, 7.9375, 7.625, 4.75, 7, 1.625, 9.25, 7, 7, 2.5,
                5.9375, 7.1875, 6.875, 5.84375, 7.5625, 6.5, 2.09375, 6.0625, 2.4375, 3.34375,
                3.76562, 2.1875, 3.51562, 5.875, 2.54688, 8.3125, 9.125, 2.96875, 1.9375, 3.76562,
                4.1875, 6.28125, 4.84375, 7.5, 3.78125, 5.375, 2.84375, 8.5, 3.3125, 8.1875,
                4.21875, 6.375, 8.375, 1.46875, 6.625, 6.5625, 9.0625, 2.90625, 9.4375, 5.375,
                1.32812, 5.9375, 8.9375, 4.4375, 7.5625, 8.9375, 5.28125, 2.59375, 8.375, 5.15625,
                6.8125, 9.25, 5.3125, 4.375, 1.46875, 7.5625, 8.3125, 2.82812, 7.25, 7.03125,
                3.3125, 3.03125, 8.5, 3.21875, 8.875, 7, 9.6875, 5.28125, 3.65625, 1.8125,
                2.5, 4.0625, 6.15625, 8, 5.1875, 2.45312, 6.1875, 1.375, 7.9375, 7.1875,
                9.625, 7.5, 7.84375, 7.40625, 9.75, 7.09375, 2.0625, 1.5, 6.53125, 5,
                9.875, 6.6875, 5.875, 6.8125, 5.53125, 9.25, 4.0625, 4.1875, 1.46875, 5,
                8.6875, 3.70312, 1.85938, 1.21094, 2.71875, 1.82812, 9, 2.71875, 9.9375, 7.8125,
                4.3125, 8.8125, 2.8125, 3.23438, 4.375, 4.1875, 8.375, 8.75, 6.625, 5.34375,
                7.1875, 3.45312, 7.0625, 8.5, 5.8125, 4.875, 8.875, 3.5625, 3.84375, 6.1875,
                3.90625, 4.9375, 4.5625, 6.625, 2.59375, 9.875, 2.90625, 1.82031, 1.25781, 6,
                9.25, 4.09375, 9.125, 2.5625, 5.25, 2.34375, 5.90625, 7.90625, 9.75, 6.875,
                7.3125, 1.14062, 9.125, 7.75, 5.5625, 2.8125, 1.32812, 7.5625, 9, 7.125,
                8.625, 4.4375, 8.375, 6.375, 6.6875, 3.82812, 5.1875, 7.8125, 3.28125, 1.17188,
                7.125, 8.9375, 6.6875, 5.4375, 6.21875, 4.125, 1.40625, 3.51562, 3.5, 3.0625,
                1.03906, 7.28125, 2.64062, 8.125, 4.4375, 5.25, 1.75, 1.96875, 7.9375, 4.46875,
                6.25, 8.75, 1.80469, 3.375, 2.4375, 8.9375, 4.1875, 4.1875, 7.15625, 6.65625,
                8.375, 1.13281, 7.3125, 7.375, 3.3125, 1.36719, 7.40625, 6.375, 7.875, 4,
                8.5, 8.5, 6.1875, 2.3125, 6.5625, 6.25, 7.03125, 7.625, 5.1875, 6.71875,
                4.125, 7.4375, 1.09375, 5, 4.90625, 5.5625, 10, 8.0625, 2.625, 1.21875,
                7.46875, 3.125, 4.0625, 8.3125, 1.11719, 5.40625, 4.96875, 1.35156, 2.04688, 2.5,
                3.8125, 6.125, 3.78125, 2.82812, 8.9375, 6.90625, 3.6875, 1.95312, 9.8125, 7.25,
                4.125, 4.875, 1.54688, 6.40625, 2.57812, 6.46875, 5.34375, 7.8125, 9.9375, 5.5625,
                2.625, 7.1875, 8.375, 9.75, 1.67188, 3, 6.96875, 6, 1.53125, 2.09375,
                6.15625, 5.3125, 7.25, 7.3125, 3.625, 1.10938, 7.53125, 1.375, 7.84375, 7.96875,
                9.875, 9.8125, 5.78125, 7.9375, 4.3125, 8.0625, 7.8125, 6.5625, 2.21875, 4.0625,
                3.1875, 4.75, 8.25, 6.125, 3.03125, 6.1875, 4.125, 7.71875, 4.4375, 1.46875,
                4.40625, 6.375, 4.375, 1.48438, 2.65625, 1.80469, 7.5, 9.875, 7.25, 7.875,
                8.1875, 5.4375, 2.70312, 3.39062, 2.23438, 2.5625, 9.875, 1.76562, 3.3125, 10},
            std::vector<T>{
                1, 6.125, 4.53125, 4.4375, 6.3125, 6.6875, 5.375, 8.25, 9.875, 9.375,
                9.0625, 5.9375, 3.0625, 6.75, 9.75, 8.8125, 3.5, 2.40625, 7.625, 2.21875,
                2.39062, 9.25, 6.1875, 4.0625, 6.25, 5.375, 8.25, 7.3125, 6.5, 7.8125,
                2.65625, 2.10938, 8.375, 7.65625, 4.09375, 2.71875, 9.4375, 7.0625, 6.5625, 8.625,
                7.625, 8.25, 2.10938, 8.625, 8.125, 7.40625, 4.5, 2.76562, 5.125, 9.625,
                8.5625, 8.75, 7.46875, 9.625, 8.875, 4.40625, 2.26562, 9.3125, 1.67188, 9.1875,
                6.1875, 4.78125, 5.5, 6.40625, 5.59375, 8.125, 7.65625, 1.75, 2.96875, 3.875,
                8.75, 1.99219, 3.53125, 2.53125, 9.375, 2.1875, 2.875, 1.86719, 9.25, 7.125,
                3.89062, 9.5, 1.92188, 5.4375, 4.5625, 1, 8, 3.96875, 9.375, 9.1875,
                1.19531, 6.875, 3.21875, 8.5625, 7.0625, 8.6875, 9.875, 4.65625, 9.5, 6.25,
                1.82812, 1.51562, 2.42188, 4.21875, 2.78125, 7.75, 4.96875, 3.89062, 2.76562, 9.25,
                5.84375, 5.84375, 8.375, 3.34375, 9.875, 2.625, 2.15625, 1.73438, 6.25, 8.875,
                4.0625, 7.8125, 4.625, 7.59375, 7.625, 2.71875, 4.25, 3.96875, 2, 2.9375,
                6.25, 7.3125, 7.09375, 7.1875, 7.40625, 1.13281, 9.625, 5.90625, 9.9375, 1.0625,
                2.10938, 9.5, 6.25, 5.1875, 3.3125, 9.4375, 6.84375, 8.25, 4.375, 4.59375,
                1.79688, 9, 7.9375, 2.64062, 9.75, 8.75, 9.6875, 3.59375, 4, 10,
                8.875, 2.82812, 9.0625, 8.8125, 1.76562, 2.23438, 6.625, 2.375, 3.70312, 8.375,
                8.625, 4, 6.1875, 9.8125, 2.625, 2.60938, 9.375, 1.625, 2.03125, 9.5,
                1.84375, 9.125, 7.75, 4.75, 2.0625, 2.375, 6.03125, 5.65625, 3.92188, 4.125,
                9.625, 4.25, 5.4375, 1.34375, 5.1875, 7.53125, 3.96875, 4.875, 1.65625, 3.54688,
                1.64062, 4.625, 1.51562, 4.4375, 4.375, 5.1875, 1.03906, 1.58594, 9.75, 2.78125,
                1.27344, 6.25, 6.4375, 9.5625, 4.625, 3.89062, 8.875, 5.875, 9.875, 4,
                9.625, 8.1875, 6, 6.5625, 3.1875, 5.9375, 1.17188, 6.375, 3.09375, 3.5,
                5.34375, 6.625, 9.6875, 1.85938, 2.85938, 9.75, 5.5, 7.6875, 1.86719, 1.03125,
                7.21875, 7.125, 7.75, 6.1875, 2, 2.59375, 2.46875, 8.25, 7.78125, 3.75,
                7.875, 4.6875, 8.0625, 7.9375, 2.53125, 6.5, 2.48438, 6.75, 2.60938, 1.60938,
                7.75, 2.28125, 8.75, 7.6875, 8.25, 8.0625, 1.23438, 6.0625, 1.53125, 6.96875,
                4.9375, 5.21875, 1.5, 5.125, 7.5, 7.1875, 6, 2.71875, 6.5625, 9.875,
                2.15625, 4.40625, 5.5, 3.98438, 5.5625, 8.875, 1.84375, 8.75, 8.875, 9.6875,
                8.8125, 2.78125, 1.16406, 6.03125, 7.625, 2.73438, 6.5, 5.6875, 9.75, 8.125,
                6.875, 7.34375, 9, 5.0625, 6.1875, 3.95312, 4.5, 2.35938, 4.6875, 9.875,
                1.09375, 6.09375, 4, 9.75, 8.5, 1.70312, 6.9375, 1.29688, 5.6875, 7.8125,
                9.625, 3.125, 7.1875, 6.6875, 7.84375, 1.21094, 1.71094, 5.875, 3.4375, 5.8125,
                4.25, 4.125, 7.875, 5.1875, 9.3125, 9, 4.4375, 1.0625, 3.01562, 5.25,
                3.09375, 7.375, 6.1875, 3.4375, 6.46875, 9.375, 5.625, 4.8125, 8.1875, 2.03125,
                8.75, 4.21875, 9.25, 8.3125, 1.1875, 3.0625, 1.6875, 3.375, 5.09375, 1.95312,
                1.25, 3.76562, 7.75, 5.09375, 6.1875, 4.625, 4.625, 1.49219, 4.96875, 9.875,
                2, 9, 4.46875, 7.25, 1.36719, 7.9375, 1.79688, 3, 7.5625, 2.625,
                7.5625, 1.92969, 2.32812, 3.25, 4.375, 1.125, 7.75, 2.4375, 3.34375, 5.6875,
                1.30469, 7.9375, 8.75, 3.625, 6.3125, 6.75, 1.73438, 2.5, 3.09375, 10},
            std::vector<T>{
                1, 7.25, 1.08594, 1.90625, 7.3125, 8.1875, 1.84375, 9.8125, 3.65625, 7.6875,
                7.03125, 6.75, 9.875, 9, 2.28125, 4.625, 8.375, 8.125, 4.25, 9.9375,
                2.125, 1.92969, 1.35938, 7.625, 5.6875, 2.65625, 6.9375, 8.0625, 2.53125, 7.375,
                2.8125, 9.1875, 7.3125, 8.75, 7.15625, 7.15625, 3.8125, 6.75, 3.53125, 10},
            std::vector<T>{
                0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094,
                0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094,
                0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094},
            std::vector<T>{
                0.523438, 0.667969, 0.632812, 0.65625, 0.667969, 0.667969, 0.667969, 0.667969, 0.667969, 0.65625,
                0.539062, 0.667969, 0.617188, 0.640625, 0.664062, 0.664062, 0.667969, 0.664062, 0.664062, 0.648438,
                0.546875, 0.664062, 0.601562, 0.625, 0.65625, 0.65625, 0.667969, 0.65625, 0.65625, 0.632812,
                0.546875, 0.648438, 0.585938, 0.609375, 0.648438, 0.640625, 0.664062, 0.648438, 0.640625, 0.617188,
                0.554688, 0.632812, 0.578125, 0.59375, 0.632812, 0.625, 0.65625, 0.632812, 0.625, 0.601562,
                0.554688, 0.617188, 0.570312, 0.585938, 0.617188, 0.609375, 0.640625, 0.617188, 0.609375, 0.585938,
                0.554688, 0.601562, 0.570312, 0.578125, 0.601562, 0.59375, 0.625, 0.601562, 0.59375, 0.578125,
                0.554688, 0.59375, 0.5625, 0.570312, 0.585938, 0.585938, 0.609375, 0.585938, 0.585938, 0.570312,
                0.554688, 0.585938, 0.5625, 0.570312, 0.578125, 0.578125, 0.59375, 0.578125, 0.578125, 0.570312,
                0.554688, 0.570312, 0.5625, 0.5625, 0.570312, 0.570312, 0.585938, 0.570312, 0.570312, 0.5625,
                0.667969, 0.617188, 0.667969, 0.667969, 0.601562, 0.664062, 0.667969, 0.667969, 0.667969, 0.667969,
                0.664062, 0.601562, 0.664062, 0.667969, 0.585938, 0.648438, 0.667969, 0.667969, 0.667969, 0.667969,
                0.65625, 0.585938, 0.664062, 0.667969, 0.578125, 0.632812, 0.664062, 0.667969, 0.664062, 0.664062,
                0.648438, 0.578125, 0.648438, 0.664062, 0.570312, 0.617188, 0.65625, 0.664062, 0.648438, 0.664062,
                0.632812, 0.570312, 0.632812, 0.65625, 0.570312, 0.601562, 0.648438, 0.648438, 0.632812, 0.648438,
                0.617188, 0.570312, 0.617188, 0.640625, 0.5625, 0.585938, 0.632812, 0.632812, 0.617188, 0.632812,
                0.601562, 0.5625, 0.601562, 0.625, 0.5625, 0.578125, 0.617188, 0.617188, 0.601562, 0.617188,
                0.585938, 0.5625, 0.585938, 0.609375, 0.5625, 0.570312, 0.601562, 0.601562, 0.59375, 0.601562,
                0.578125, 0.5625, 0.578125, 0.59375, 0.554688, 0.570312, 0.585938, 0.59375, 0.585938, 0.585938,
                0.570312, 0.554688, 0.570312, 0.585938, 0.554688, 0.5625, 0.578125, 0.585938, 0.570312, 0.578125,
                0.667969, 0.664062, 0.667969, 0.664062, 0.667969, 0.667969, 0.648438, 0.617188, 0.667969, 0.617188,
                0.667969, 0.65625, 0.667969, 0.65625, 0.667969, 0.664062, 0.632812, 0.601562, 0.667969, 0.601562,
                0.664062, 0.648438, 0.667969, 0.648438, 0.667969, 0.65625, 0.617188, 0.585938, 0.664062, 0.59375,
                0.648438, 0.632812, 0.664062, 0.632812, 0.664062, 0.648438, 0.601562, 0.578125, 0.65625, 0.585938,
                0.632812, 0.617188, 0.648438, 0.617188, 0.648438, 0.632812, 0.585938, 0.570312, 0.640625, 0.570312,
                0.617188, 0.601562, 0.632812, 0.601562, 0.632812, 0.617188, 0.578125, 0.570312, 0.625, 0.570312,
                0.601562, 0.585938, 0.617188, 0.585938, 0.617188, 0.601562, 0.570312, 0.5625, 0.609375, 0.5625,
                0.59375, 0.578125, 0.601562, 0.578125, 0.601562, 0.585938, 0.570312, 0.5625, 0.59375, 0.5625,
                0.585938, 0.570312, 0.59375, 0.570312, 0.59375, 0.578125, 0.5625, 0.5625, 0.585938, 0.5625,
                0.570312, 0.570312, 0.585938, 0.570312, 0.585938, 0.570312, 0.5625, 0.554688, 0.578125, 0.554688,
                0.5625, 0.667969, 0.65625, 0.667969, 0.664062, 0.632812, 0.667969, 0.632812, 0.65625, 0.667969,
                0.5625, 0.664062, 0.640625, 0.667969, 0.65625, 0.617188, 0.664062, 0.617188, 0.648438, 0.667969,
                0.5625, 0.664062, 0.625, 0.664062, 0.640625, 0.601562, 0.65625, 0.601562, 0.632812, 0.664062,
                0.554688, 0.648438, 0.609375, 0.65625, 0.625, 0.59375, 0.640625, 0.585938, 0.617188, 0.65625,
                0.554688, 0.632812, 0.59375, 0.640625, 0.609375, 0.585938, 0.625, 0.578125, 0.601562, 0.640625,
                0.554688, 0.617188, 0.585938, 0.625, 0.59375, 0.570312, 0.609375, 0.570312, 0.585938, 0.625,
                0.554688, 0.601562, 0.578125, 0.609375, 0.585938, 0.570312, 0.59375, 0.570312, 0.578125, 0.609375,
                0.554688, 0.585938, 0.570312, 0.59375, 0.578125, 0.5625, 0.585938, 0.5625, 0.570312, 0.59375,
                0.554688, 0.578125, 0.570312, 0.585938, 0.570312, 0.5625, 0.578125, 0.5625, 0.570312, 0.585938,
                0.554688, 0.570312, 0.5625, 0.578125, 0.570312, 0.5625, 0.570312, 0.5625, 0.5625, 0.578125,
                0.664062, 0.667969, 0.664062, 0.65625, 0.667969, 0.667969, 0.667969, 0.617188, 0.65625, 0.667969,
                0.65625, 0.667969, 0.65625, 0.648438, 0.667969, 0.667969, 0.667969, 0.601562, 0.648438, 0.667969,
                0.648438, 0.664062, 0.640625, 0.625, 0.664062, 0.664062, 0.664062, 0.59375, 0.632812, 0.667969,
                0.632812, 0.65625, 0.625, 0.609375, 0.65625, 0.65625, 0.65625, 0.585938, 0.617188, 0.664062,
                0.617188, 0.640625, 0.609375, 0.59375, 0.648438, 0.648438, 0.648438, 0.570312, 0.601562, 0.65625,
                0.601562, 0.625, 0.59375, 0.585938, 0.632812, 0.632812, 0.632812, 0.570312, 0.585938, 0.640625,
                0.585938, 0.609375, 0.585938, 0.578125, 0.617188, 0.617188, 0.617188, 0.5625, 0.578125, 0.625,
                0.578125, 0.59375, 0.578125, 0.570312, 0.601562, 0.601562, 0.601562, 0.5625, 0.570312, 0.609375,
                0.570312, 0.585938, 0.570312, 0.570312, 0.585938, 0.585938, 0.585938, 0.5625, 0.570312, 0.59375,
                0.570312, 0.578125, 0.570312, 0.5625, 0.578125, 0.578125, 0.578125, 0.554688, 0.5625, 0.585938},
            std::vector<T>{
                0.554688, 0.570312, 0.5625, 0.5625, 0.570312, 0.570312, 0.585938, 0.570312, 0.570312, 0.5625,
                0.570312, 0.554688, 0.570312, 0.585938, 0.554688, 0.5625, 0.578125, 0.585938, 0.570312, 0.578125,
                0.570312, 0.570312, 0.585938, 0.570312, 0.585938, 0.570312, 0.5625, 0.554688, 0.578125, 0.554688,
                0.554688, 0.570312, 0.5625, 0.578125, 0.570312, 0.5625, 0.570312, 0.5625, 0.5625, 0.578125,
                0.570312, 0.578125, 0.570312, 0.5625, 0.578125, 0.578125, 0.578125, 0.554688, 0.5625, 0.585938},
            std::vector<T>{
                1.20312, 1.30469, 1.22656, 1.24219, 1.28906, 1.28125, 1.35938, 1.29688, 1.28125, 1.25,
                1.28906, 1.21875, 1.29688, 1.375, 1.21875, 1.25, 1.32812, 1.35156, 1.30469, 1.34375,
                1.30469, 1.26562, 1.35156, 1.26562, 1.35156, 1.28906, 1.23438, 1.21875, 1.32031, 1.21875,
                1.21875, 1.29688, 1.24219, 1.32031, 1.25781, 1.22656, 1.28125, 1.22656, 1.25, 1.3125,
                1.26562, 1.32031, 1.25781, 1.24219, 1.34375, 1.32812, 1.32812, 1.21875, 1.25, 1.375}),
        LSTMSequenceV1Params(
            5, 10, 10, 10,
            0.7f, false, op::RecurrentSequenceDirection::REVERSE,
            ET,
            std::vector<T>{
                1, 9.375, 9, 3.84375, 2.17188, 2.65625, 1.35938, 2.84375, 8.4375, 6.125,
                5.78125, 6.375, 9.625, 9.625, 5.15625, 6.875, 9.3125, 7.75, 4.375, 6.875,
                2.39062, 7.71875, 9, 9.625, 1.23438, 1.07812, 3.625, 1.95312, 4.5625, 3.6875,
                8.25, 6.90625, 6.625, 8.25, 9.125, 8.875, 6, 9.625, 8.5, 7.5,
                1.45312, 6.78125, 8.25, 7.4375, 9.375, 5.1875, 4.25, 3.9375, 7.1875, 4.9375,
                2.15625, 7.5625, 8.5, 9.9375, 3.85938, 7.0625, 7.625, 8.125, 6.375, 2.53125,
                4.25, 1.23438, 8.125, 8.1875, 7.28125, 9.125, 8.375, 1.21875, 9.125, 5.4375,
                8.5, 5.75, 9.1875, 6.375, 9.75, 1.46875, 6.875, 9, 8.25, 7.5625,
                1.92188, 8.375, 3.125, 5.5, 4.40625, 8.25, 7.28125, 1.85938, 5.375, 2.96875,
                4.75, 3.32812, 6.75, 5.1875, 7.8125, 5.125, 6, 7.375, 7.25, 2.59375,
                9.25, 5.78125, 1.21875, 2.5, 8.5, 7.90625, 4.4375, 9.375, 3.6875, 6.5,
                1.05469, 2.34375, 4.9375, 5.40625, 7.625, 4.375, 4.375, 8.625, 5.4375, 9.1875,
                1.125, 4.4375, 3.25, 3.84375, 4.125, 6.125, 8.125, 2.6875, 9.4375, 2.125,
                1.90625, 7.1875, 7.625, 8.1875, 9.75, 6.15625, 7.34375, 9.75, 9.5625, 6.6875,
                9.375, 9, 4.6875, 5.4375, 4.03125, 4.15625, 7.9375, 7.4375, 4, 5.53125,
                1.74219, 3.03125, 9.0625, 3.20312, 8.0625, 8.125, 7.4375, 5.4375, 5, 9.25,
                7.75, 9.5, 6.90625, 5.8125, 4.25, 3.26562, 4.375, 7.5, 6.84375, 4.3125,
                1.5, 5.5, 1.70312, 3.03125, 1.82812, 4.1875, 8.25, 6.84375, 1.58594, 3.8125,
                3.01562, 7.90625, 2.375, 8, 7.125, 8.625, 2.125, 9.5, 3.3125, 1.96875,
                4.9375, 9.1875, 1.98438, 4, 3.82812, 8.375, 4.15625, 9.0625, 7.84375, 1.38281,
                1.89062, 2.75, 9.375, 3.65625, 3.9375, 6.625, 4.8125, 1.77344, 3.4375, 2.28125,
                8.0625, 5.625, 5.0625, 7.1875, 1.75, 8.6875, 1.63281, 6.8125, 3.96875, 6.25,
                2.71875, 7.375, 8.5, 3.26562, 1.55469, 9.125, 6, 4.96875, 6.125, 1.1875,
                5.15625, 9.625, 9.5, 6.875, 4.09375, 5.625, 8.9375, 7.125, 4.875, 5.375,
                9.9375, 9.3125, 8.1875, 5.625, 9.5, 1.64844, 4.40625, 6.09375, 4.625, 6.53125,
                3.57812, 9.5, 5.0625, 4.75, 7.875, 3.375, 6.21875, 1.875, 4.375, 5.375,
                6.59375, 5.1875, 7.625, 1.26562, 9.25, 7.25, 5.78125, 7.4375, 5.65625, 7.5625,
                1.92188, 4.71875, 2.09375, 1.13281, 6.78125, 9.125, 5, 8.125, 1.6875, 2.48438,
                2.375, 3.8125, 1.71875, 6.5, 8.875, 4.25, 2.45312, 2.40625, 9.25, 2.59375,
                1.03125, 8.75, 8.25, 3.60938, 3.71875, 6.25, 3.1875, 5.0625, 7.90625, 4.6875,
                7.3125, 8.9375, 9, 7.21875, 9.1875, 3.5, 3.03125, 1.57812, 4.78125, 2.78125,
                2.5, 9.375, 2.89062, 8.6875, 8.5, 9.5625, 9.25, 1.46875, 7.125, 6.1875,
                4.6875, 5.3125, 3, 1.19531, 8.5, 4.375, 8, 4.71875, 7.625, 6.4375,
                8.75, 7.03125, 9.75, 8.5, 4.6875, 8, 6.375, 4.59375, 7.625, 8.125,
                3.40625, 9, 7.875, 3.35938, 9.375, 9.875, 8.875, 8.625, 2.03125, 7.5625,
                9.6875, 4.1875, 3.71875, 8.9375, 6.46875, 8.75, 2.5, 9.625, 8.75, 1,
                7.53125, 1.14844, 1.58594, 3.8125, 5.65625, 9.9375, 6.34375, 2.34375, 5.125, 2.5,
                6.3125, 7.8125, 3.875, 1.625, 8.375, 7.34375, 1.82812, 5.21875, 2.59375, 1.09375,
                2.98438, 7.96875, 5.25, 8.125, 9.25, 2.34375, 9.875, 1.21094, 6.4375, 7.84375,
                9.25, 3, 4.3125, 3.35938, 1.0625, 5.125, 6.875, 3.25, 2.5, 6.125,
                5.4375, 8.625, 8.125, 4.375, 8.375, 4.875, 4.3125, 8.5, 2.15625, 4.3125,
                2.78125, 1.35938, 1.1875, 6, 6.6875, 5.0625, 5.3125, 7.5, 2.90625, 4.375,
                3.375, 8.5625, 2.6875, 5.21875, 9, 6.0625, 7.4375, 6.9375, 1.60938, 5.15625,
                3.20312, 6.625, 9.25, 3, 2.84375, 7.59375, 5.1875, 4.4375, 7.875, 2.75,
                5.78125, 3.4375, 5.9375, 3.25, 3.10938, 2.375, 3.46875, 7.9375, 1.14844, 3.29688,
                4.8125, 2.14062, 1.42188, 7, 1.53125, 4.6875, 3.875, 7, 5, 6.9375,
                2.51562, 3.75, 3.71875, 2.8125, 5.03125, 3, 5.25, 2.07812, 7.3125, 1.32812,
                7.3125, 1.30469, 6.3125, 3.0625, 9.75, 3.0625, 6.9375, 6.625, 8.875, 9,
                4.71875, 8, 8.125, 7.5, 2.23438, 3.78125, 3.34375, 4.25, 3.03125, 2.75,
                5.78125, 9.375, 7.25, 6.0625, 3.1875, 8.375, 2.75, 4.125, 8.125, 10},
            std::vector<T>{
                1, 9.1875, 5.5625, 6.625, 2.65625, 8.125, 8.25, 4.875, 9.4375, 5.875,
                1.26562, 4.71875, 8.0625, 1.76562, 2.46875, 8, 4.875, 5.375, 6.15625, 1.45312,
                2.95312, 5.84375, 8.5, 1.375, 3.6875, 8.25, 4.25, 9.9375, 8.8125, 6.75,
                1.10938, 9.5, 7.0625, 7.625, 5.3125, 8.375, 9.75, 2.03125, 2.90625, 7.625,
                2.90625, 8.375, 9.5, 1.66406, 3.9375, 2.875, 9.875, 8.25, 4.9375, 10},
            std::vector<T>{
                1, 6.59375, 2.125, 2.89062, 5.71875, 5.0625, 9.75, 6, 4.9375, 3.21875,
                5.875, 1.85156, 6.28125, 9.875, 1.59375, 3.40625, 7.75, 9.25, 6.46875, 8.75,
                6.5, 4.0625, 9.125, 4.4375, 9.25, 5.5625, 2.625, 1.8125, 7.625, 1.875,
                1.26562, 6.3125, 3.03125, 7.3125, 3.92188, 2.21875, 4.90625, 2.10938, 3.28125, 6.875,
                4.375, 7.25, 3.92188, 3.09375, 8.375, 7.96875, 7.9375, 1.875, 3.125, 10},
            std::vector<int64_t>{10, 10, 10, 10, 10},
            std::vector<T>{
                1, 7.5, 1.57812, 10, 1.55469, 3.5, 9.0625, 7.5625, 9, 1.30469,
                6, 7.1875, 2.20312, 9, 5.6875, 2, 5.5625, 5.5625, 6.21875, 3.09375,
                3.82812, 7.28125, 9, 1.625, 6.78125, 9.875, 5.46875, 4.1875, 3.84375, 2.21875,
                1.57812, 9.125, 6, 6, 9.5, 2.28125, 8.375, 9.875, 2.32812, 5.96875,
                1.42969, 4.5625, 1.10156, 5.4375, 8.25, 1.625, 6.875, 4.6875, 6.15625, 6.15625,
                8.625, 4.125, 6.1875, 1.375, 6.1875, 8.875, 6, 3.40625, 1.75781, 4.8125,
                1.40625, 4.21875, 4, 6.4375, 6.5625, 4.1875, 8.25, 7.40625, 2.40625, 7,
                5.5625, 7.9375, 7.625, 4.75, 7, 1.625, 9.25, 7, 7, 2.5,
                5.9375, 7.1875, 6.875, 5.84375, 7.5625, 6.5, 2.09375, 6.0625, 2.4375, 3.34375,
                3.76562, 2.1875, 3.51562, 5.875, 2.54688, 8.3125, 9.125, 2.96875, 1.9375, 3.76562,
                4.1875, 6.28125, 4.84375, 7.5, 3.78125, 5.375, 2.84375, 8.5, 3.3125, 8.1875,
                4.21875, 6.375, 8.375, 1.46875, 6.625, 6.5625, 9.0625, 2.90625, 9.4375, 5.375,
                1.32812, 5.9375, 8.9375, 4.4375, 7.5625, 8.9375, 5.28125, 2.59375, 8.375, 5.15625,
                6.8125, 9.25, 5.3125, 4.375, 1.46875, 7.5625, 8.3125, 2.82812, 7.25, 7.03125,
                3.3125, 3.03125, 8.5, 3.21875, 8.875, 7, 9.6875, 5.28125, 3.65625, 1.8125,
                2.5, 4.0625, 6.15625, 8, 5.1875, 2.45312, 6.1875, 1.375, 7.9375, 7.1875,
                9.625, 7.5, 7.84375, 7.40625, 9.75, 7.09375, 2.0625, 1.5, 6.53125, 5,
                9.875, 6.6875, 5.875, 6.8125, 5.53125, 9.25, 4.0625, 4.1875, 1.46875, 5,
                8.6875, 3.70312, 1.85938, 1.21094, 2.71875, 1.82812, 9, 2.71875, 9.9375, 7.8125,
                4.3125, 8.8125, 2.8125, 3.23438, 4.375, 4.1875, 8.375, 8.75, 6.625, 5.34375,
                7.1875, 3.45312, 7.0625, 8.5, 5.8125, 4.875, 8.875, 3.5625, 3.84375, 6.1875,
                3.90625, 4.9375, 4.5625, 6.625, 2.59375, 9.875, 2.90625, 1.82031, 1.25781, 6,
                9.25, 4.09375, 9.125, 2.5625, 5.25, 2.34375, 5.90625, 7.90625, 9.75, 6.875,
                7.3125, 1.14062, 9.125, 7.75, 5.5625, 2.8125, 1.32812, 7.5625, 9, 7.125,
                8.625, 4.4375, 8.375, 6.375, 6.6875, 3.82812, 5.1875, 7.8125, 3.28125, 1.17188,
                7.125, 8.9375, 6.6875, 5.4375, 6.21875, 4.125, 1.40625, 3.51562, 3.5, 3.0625,
                1.03906, 7.28125, 2.64062, 8.125, 4.4375, 5.25, 1.75, 1.96875, 7.9375, 4.46875,
                6.25, 8.75, 1.80469, 3.375, 2.4375, 8.9375, 4.1875, 4.1875, 7.15625, 6.65625,
                8.375, 1.13281, 7.3125, 7.375, 3.3125, 1.36719, 7.40625, 6.375, 7.875, 4,
                8.5, 8.5, 6.1875, 2.3125, 6.5625, 6.25, 7.03125, 7.625, 5.1875, 6.71875,
                4.125, 7.4375, 1.09375, 5, 4.90625, 5.5625, 10, 8.0625, 2.625, 1.21875,
                7.46875, 3.125, 4.0625, 8.3125, 1.11719, 5.40625, 4.96875, 1.35156, 2.04688, 2.5,
                3.8125, 6.125, 3.78125, 2.82812, 8.9375, 6.90625, 3.6875, 1.95312, 9.8125, 7.25,
                4.125, 4.875, 1.54688, 6.40625, 2.57812, 6.46875, 5.34375, 7.8125, 9.9375, 5.5625,
                2.625, 7.1875, 8.375, 9.75, 1.67188, 3, 6.96875, 6, 1.53125, 2.09375,
                6.15625, 5.3125, 7.25, 7.3125, 3.625, 1.10938, 7.53125, 1.375, 7.84375, 7.96875,
                9.875, 9.8125, 5.78125, 7.9375, 4.3125, 8.0625, 7.8125, 6.5625, 2.21875, 4.0625,
                3.1875, 4.75, 8.25, 6.125, 3.03125, 6.1875, 4.125, 7.71875, 4.4375, 1.46875,
                4.40625, 6.375, 4.375, 1.48438, 2.65625, 1.80469, 7.5, 9.875, 7.25, 7.875,
                8.1875, 5.4375, 2.70312, 3.39062, 2.23438, 2.5625, 9.875, 1.76562, 3.3125, 10},
            std::vector<T>{
                1, 6.125, 4.53125, 4.4375, 6.3125, 6.6875, 5.375, 8.25, 9.875, 9.375,
                9.0625, 5.9375, 3.0625, 6.75, 9.75, 8.8125, 3.5, 2.40625, 7.625, 2.21875,
                2.39062, 9.25, 6.1875, 4.0625, 6.25, 5.375, 8.25, 7.3125, 6.5, 7.8125,
                2.65625, 2.10938, 8.375, 7.65625, 4.09375, 2.71875, 9.4375, 7.0625, 6.5625, 8.625,
                7.625, 8.25, 2.10938, 8.625, 8.125, 7.40625, 4.5, 2.76562, 5.125, 9.625,
                8.5625, 8.75, 7.46875, 9.625, 8.875, 4.40625, 2.26562, 9.3125, 1.67188, 9.1875,
                6.1875, 4.78125, 5.5, 6.40625, 5.59375, 8.125, 7.65625, 1.75, 2.96875, 3.875,
                8.75, 1.99219, 3.53125, 2.53125, 9.375, 2.1875, 2.875, 1.86719, 9.25, 7.125,
                3.89062, 9.5, 1.92188, 5.4375, 4.5625, 1, 8, 3.96875, 9.375, 9.1875,
                1.19531, 6.875, 3.21875, 8.5625, 7.0625, 8.6875, 9.875, 4.65625, 9.5, 6.25,
                1.82812, 1.51562, 2.42188, 4.21875, 2.78125, 7.75, 4.96875, 3.89062, 2.76562, 9.25,
                5.84375, 5.84375, 8.375, 3.34375, 9.875, 2.625, 2.15625, 1.73438, 6.25, 8.875,
                4.0625, 7.8125, 4.625, 7.59375, 7.625, 2.71875, 4.25, 3.96875, 2, 2.9375,
                6.25, 7.3125, 7.09375, 7.1875, 7.40625, 1.13281, 9.625, 5.90625, 9.9375, 1.0625,
                2.10938, 9.5, 6.25, 5.1875, 3.3125, 9.4375, 6.84375, 8.25, 4.375, 4.59375,
                1.79688, 9, 7.9375, 2.64062, 9.75, 8.75, 9.6875, 3.59375, 4, 10,
                8.875, 2.82812, 9.0625, 8.8125, 1.76562, 2.23438, 6.625, 2.375, 3.70312, 8.375,
                8.625, 4, 6.1875, 9.8125, 2.625, 2.60938, 9.375, 1.625, 2.03125, 9.5,
                1.84375, 9.125, 7.75, 4.75, 2.0625, 2.375, 6.03125, 5.65625, 3.92188, 4.125,
                9.625, 4.25, 5.4375, 1.34375, 5.1875, 7.53125, 3.96875, 4.875, 1.65625, 3.54688,
                1.64062, 4.625, 1.51562, 4.4375, 4.375, 5.1875, 1.03906, 1.58594, 9.75, 2.78125,
                1.27344, 6.25, 6.4375, 9.5625, 4.625, 3.89062, 8.875, 5.875, 9.875, 4,
                9.625, 8.1875, 6, 6.5625, 3.1875, 5.9375, 1.17188, 6.375, 3.09375, 3.5,
                5.34375, 6.625, 9.6875, 1.85938, 2.85938, 9.75, 5.5, 7.6875, 1.86719, 1.03125,
                7.21875, 7.125, 7.75, 6.1875, 2, 2.59375, 2.46875, 8.25, 7.78125, 3.75,
                7.875, 4.6875, 8.0625, 7.9375, 2.53125, 6.5, 2.48438, 6.75, 2.60938, 1.60938,
                7.75, 2.28125, 8.75, 7.6875, 8.25, 8.0625, 1.23438, 6.0625, 1.53125, 6.96875,
                4.9375, 5.21875, 1.5, 5.125, 7.5, 7.1875, 6, 2.71875, 6.5625, 9.875,
                2.15625, 4.40625, 5.5, 3.98438, 5.5625, 8.875, 1.84375, 8.75, 8.875, 9.6875,
                8.8125, 2.78125, 1.16406, 6.03125, 7.625, 2.73438, 6.5, 5.6875, 9.75, 8.125,
                6.875, 7.34375, 9, 5.0625, 6.1875, 3.95312, 4.5, 2.35938, 4.6875, 9.875,
                1.09375, 6.09375, 4, 9.75, 8.5, 1.70312, 6.9375, 1.29688, 5.6875, 7.8125,
                9.625, 3.125, 7.1875, 6.6875, 7.84375, 1.21094, 1.71094, 5.875, 3.4375, 5.8125,
                4.25, 4.125, 7.875, 5.1875, 9.3125, 9, 4.4375, 1.0625, 3.01562, 5.25,
                3.09375, 7.375, 6.1875, 3.4375, 6.46875, 9.375, 5.625, 4.8125, 8.1875, 2.03125,
                8.75, 4.21875, 9.25, 8.3125, 1.1875, 3.0625, 1.6875, 3.375, 5.09375, 1.95312,
                1.25, 3.76562, 7.75, 5.09375, 6.1875, 4.625, 4.625, 1.49219, 4.96875, 9.875,
                2, 9, 4.46875, 7.25, 1.36719, 7.9375, 1.79688, 3, 7.5625, 2.625,
                7.5625, 1.92969, 2.32812, 3.25, 4.375, 1.125, 7.75, 2.4375, 3.34375, 5.6875,
                1.30469, 7.9375, 8.75, 3.625, 6.3125, 6.75, 1.73438, 2.5, 3.09375, 10},
            std::vector<T>{
                1, 7.25, 1.08594, 1.90625, 7.3125, 8.1875, 1.84375, 9.8125, 3.65625, 7.6875,
                7.03125, 6.75, 9.875, 9, 2.28125, 4.625, 8.375, 8.125, 4.25, 9.9375,
                2.125, 1.92969, 1.35938, 7.625, 5.6875, 2.65625, 6.9375, 8.0625, 2.53125, 7.375,
                2.8125, 9.1875, 7.3125, 8.75, 7.15625, 7.15625, 3.8125, 6.75, 3.53125, 10},
            std::vector<T>{
                0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094,
                0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094,
                0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094},
            std::vector<T>{
                0.554688, 0.570312, 0.5625, 0.5625, 0.570312, 0.570312, 0.585938, 0.570312, 0.570312, 0.5625,
                0.554688, 0.585938, 0.5625, 0.570312, 0.578125, 0.578125, 0.59375, 0.578125, 0.578125, 0.570312,
                0.554688, 0.59375, 0.5625, 0.570312, 0.585938, 0.585938, 0.609375, 0.585938, 0.585938, 0.570312,
                0.554688, 0.601562, 0.570312, 0.578125, 0.601562, 0.59375, 0.625, 0.601562, 0.59375, 0.578125,
                0.554688, 0.617188, 0.570312, 0.585938, 0.617188, 0.609375, 0.640625, 0.617188, 0.609375, 0.585938,
                0.554688, 0.632812, 0.578125, 0.59375, 0.632812, 0.625, 0.65625, 0.632812, 0.625, 0.601562,
                0.546875, 0.648438, 0.585938, 0.609375, 0.648438, 0.640625, 0.664062, 0.648438, 0.640625, 0.617188,
                0.546875, 0.664062, 0.601562, 0.625, 0.65625, 0.65625, 0.667969, 0.65625, 0.65625, 0.632812,
                0.539062, 0.667969, 0.617188, 0.640625, 0.664062, 0.664062, 0.667969, 0.664062, 0.664062, 0.648438,
                0.523438, 0.667969, 0.632812, 0.65625, 0.667969, 0.667969, 0.667969, 0.667969, 0.667969, 0.65625,
                0.570312, 0.554688, 0.570312, 0.585938, 0.554688, 0.5625, 0.578125, 0.585938, 0.570312, 0.578125,
                0.578125, 0.5625, 0.578125, 0.59375, 0.554688, 0.570312, 0.585938, 0.59375, 0.585938, 0.585938,
                0.585938, 0.5625, 0.585938, 0.609375, 0.5625, 0.570312, 0.601562, 0.601562, 0.59375, 0.601562,
                0.601562, 0.5625, 0.601562, 0.625, 0.5625, 0.578125, 0.617188, 0.617188, 0.601562, 0.617188,
                0.617188, 0.570312, 0.617188, 0.640625, 0.5625, 0.585938, 0.632812, 0.632812, 0.617188, 0.632812,
                0.632812, 0.570312, 0.632812, 0.65625, 0.570312, 0.601562, 0.648438, 0.648438, 0.632812, 0.648438,
                0.648438, 0.578125, 0.648438, 0.664062, 0.570312, 0.617188, 0.65625, 0.664062, 0.648438, 0.664062,
                0.65625, 0.585938, 0.664062, 0.667969, 0.578125, 0.632812, 0.664062, 0.667969, 0.664062, 0.664062,
                0.664062, 0.601562, 0.664062, 0.667969, 0.585938, 0.648438, 0.667969, 0.667969, 0.667969, 0.667969,
                0.667969, 0.617188, 0.667969, 0.667969, 0.601562, 0.664062, 0.667969, 0.667969, 0.667969, 0.667969,
                0.570312, 0.570312, 0.585938, 0.570312, 0.585938, 0.570312, 0.5625, 0.554688, 0.578125, 0.554688,
                0.585938, 0.570312, 0.59375, 0.570312, 0.59375, 0.578125, 0.5625, 0.5625, 0.585938, 0.5625,
                0.59375, 0.578125, 0.601562, 0.578125, 0.601562, 0.585938, 0.570312, 0.5625, 0.59375, 0.5625,
                0.601562, 0.585938, 0.617188, 0.585938, 0.617188, 0.601562, 0.570312, 0.5625, 0.609375, 0.5625,
                0.617188, 0.601562, 0.632812, 0.601562, 0.632812, 0.617188, 0.578125, 0.570312, 0.625, 0.570312,
                0.632812, 0.617188, 0.648438, 0.617188, 0.648438, 0.632812, 0.585938, 0.570312, 0.640625, 0.570312,
                0.648438, 0.632812, 0.664062, 0.632812, 0.664062, 0.648438, 0.601562, 0.578125, 0.65625, 0.585938,
                0.664062, 0.648438, 0.667969, 0.648438, 0.667969, 0.65625, 0.617188, 0.585938, 0.664062, 0.59375,
                0.667969, 0.65625, 0.667969, 0.65625, 0.667969, 0.664062, 0.632812, 0.601562, 0.667969, 0.601562,
                0.667969, 0.664062, 0.667969, 0.664062, 0.667969, 0.667969, 0.648438, 0.617188, 0.667969, 0.617188,
                0.554688, 0.570312, 0.5625, 0.578125, 0.570312, 0.5625, 0.570312, 0.5625, 0.5625, 0.578125,
                0.554688, 0.578125, 0.570312, 0.585938, 0.570312, 0.5625, 0.578125, 0.5625, 0.570312, 0.585938,
                0.554688, 0.585938, 0.570312, 0.59375, 0.578125, 0.5625, 0.585938, 0.5625, 0.570312, 0.59375,
                0.554688, 0.601562, 0.578125, 0.609375, 0.585938, 0.570312, 0.59375, 0.570312, 0.578125, 0.609375,
                0.554688, 0.617188, 0.585938, 0.625, 0.59375, 0.570312, 0.609375, 0.570312, 0.585938, 0.625,
                0.554688, 0.632812, 0.59375, 0.640625, 0.609375, 0.585938, 0.625, 0.578125, 0.601562, 0.640625,
                0.554688, 0.648438, 0.609375, 0.65625, 0.625, 0.59375, 0.640625, 0.585938, 0.617188, 0.65625,
                0.5625, 0.664062, 0.625, 0.664062, 0.640625, 0.601562, 0.65625, 0.601562, 0.632812, 0.664062,
                0.5625, 0.664062, 0.640625, 0.667969, 0.65625, 0.617188, 0.664062, 0.617188, 0.648438, 0.667969,
                0.5625, 0.667969, 0.65625, 0.667969, 0.664062, 0.632812, 0.667969, 0.632812, 0.65625, 0.667969,
                0.570312, 0.578125, 0.570312, 0.5625, 0.578125, 0.578125, 0.578125, 0.554688, 0.5625, 0.585938,
                0.570312, 0.585938, 0.570312, 0.570312, 0.585938, 0.585938, 0.585938, 0.5625, 0.570312, 0.59375,
                0.578125, 0.59375, 0.578125, 0.570312, 0.601562, 0.601562, 0.601562, 0.5625, 0.570312, 0.609375,
                0.585938, 0.609375, 0.585938, 0.578125, 0.617188, 0.617188, 0.617188, 0.5625, 0.578125, 0.625,
                0.601562, 0.625, 0.59375, 0.585938, 0.632812, 0.632812, 0.632812, 0.570312, 0.585938, 0.640625,
                0.617188, 0.640625, 0.609375, 0.59375, 0.648438, 0.648438, 0.648438, 0.570312, 0.601562, 0.65625,
                0.632812, 0.65625, 0.625, 0.609375, 0.65625, 0.65625, 0.65625, 0.585938, 0.617188, 0.664062,
                0.648438, 0.664062, 0.640625, 0.625, 0.664062, 0.664062, 0.664062, 0.59375, 0.632812, 0.667969,
                0.65625, 0.667969, 0.65625, 0.648438, 0.667969, 0.667969, 0.667969, 0.601562, 0.648438, 0.667969,
                0.664062, 0.667969, 0.664062, 0.65625, 0.667969, 0.667969, 0.667969, 0.617188, 0.65625, 0.667969},
            std::vector<T>{
                0.554688, 0.570312, 0.5625, 0.5625, 0.570312, 0.570312, 0.585938, 0.570312, 0.570312, 0.5625,
                0.570312, 0.554688, 0.570312, 0.585938, 0.554688, 0.5625, 0.578125, 0.585938, 0.570312, 0.578125,
                0.570312, 0.570312, 0.585938, 0.570312, 0.585938, 0.570312, 0.5625, 0.554688, 0.578125, 0.554688,
                0.554688, 0.570312, 0.5625, 0.578125, 0.570312, 0.5625, 0.570312, 0.5625, 0.5625, 0.578125,
                0.570312, 0.578125, 0.570312, 0.5625, 0.578125, 0.578125, 0.578125, 0.554688, 0.5625, 0.585938},
            std::vector<T>{
                1.20312, 1.30469, 1.22656, 1.24219, 1.28906, 1.28125, 1.35938, 1.29688, 1.28125, 1.25,
                1.28906, 1.21875, 1.29688, 1.375, 1.21875, 1.25, 1.32812, 1.35156, 1.30469, 1.34375,
                1.30469, 1.26562, 1.35156, 1.26562, 1.35156, 1.28906, 1.23438, 1.21875, 1.32031, 1.21875,
                1.21875, 1.29688, 1.24219, 1.32031, 1.25781, 1.22656, 1.28125, 1.22656, 1.25, 1.3125,
                1.26562, 1.32031, 1.25781, 1.24219, 1.34375, 1.32812, 1.32812, 1.21875, 1.25, 1.375}),
        LSTMSequenceV1Params(
            5, 10, 10, 5,
            0.7f, true, op::RecurrentSequenceDirection::BIDIRECTIONAL,
            ET,
            std::vector<T>{
                1, 9.375, 9, 3.84375, 2.17188, 2.65625, 1.35938, 2.84375, 8.4375, 6.125,
                5.78125, 6.375, 9.625, 9.625, 5.15625, 6.875, 9.3125, 7.75, 4.375, 6.875,
                2.39062, 7.71875, 9, 9.625, 1.23438, 1.07812, 3.625, 1.95312, 4.5625, 3.6875,
                8.25, 6.90625, 6.625, 8.25, 9.125, 8.875, 6, 9.625, 8.5, 7.5,
                1.45312, 6.78125, 8.25, 7.4375, 9.375, 5.1875, 4.25, 3.9375, 7.1875, 4.9375,
                2.15625, 7.5625, 8.5, 9.9375, 3.85938, 7.0625, 7.625, 8.125, 6.375, 2.53125,
                4.25, 1.23438, 8.125, 8.1875, 7.28125, 9.125, 8.375, 1.21875, 9.125, 5.4375,
                8.5, 5.75, 9.1875, 6.375, 9.75, 1.46875, 6.875, 9, 8.25, 7.5625,
                1.92188, 8.375, 3.125, 5.5, 4.40625, 8.25, 7.28125, 1.85938, 5.375, 2.96875,
                4.75, 3.32812, 6.75, 5.1875, 7.8125, 5.125, 6, 7.375, 7.25, 2.59375,
                9.25, 5.78125, 1.21875, 2.5, 8.5, 7.90625, 4.4375, 9.375, 3.6875, 6.5,
                1.05469, 2.34375, 4.9375, 5.40625, 7.625, 4.375, 4.375, 8.625, 5.4375, 9.1875,
                1.125, 4.4375, 3.25, 3.84375, 4.125, 6.125, 8.125, 2.6875, 9.4375, 2.125,
                1.90625, 7.1875, 7.625, 8.1875, 9.75, 6.15625, 7.34375, 9.75, 9.5625, 6.6875,
                9.375, 9, 4.6875, 5.4375, 4.03125, 4.15625, 7.9375, 7.4375, 4, 5.53125,
                1.74219, 3.03125, 9.0625, 3.20312, 8.0625, 8.125, 7.4375, 5.4375, 5, 9.25,
                7.75, 9.5, 6.90625, 5.8125, 4.25, 3.26562, 4.375, 7.5, 6.84375, 4.3125,
                1.5, 5.5, 1.70312, 3.03125, 1.82812, 4.1875, 8.25, 6.84375, 1.58594, 3.8125,
                3.01562, 7.90625, 2.375, 8, 7.125, 8.625, 2.125, 9.5, 3.3125, 1.96875,
                4.9375, 9.1875, 1.98438, 4, 3.82812, 8.375, 4.15625, 9.0625, 7.84375, 1.38281,
                1.89062, 2.75, 9.375, 3.65625, 3.9375, 6.625, 4.8125, 1.77344, 3.4375, 2.28125,
                8.0625, 5.625, 5.0625, 7.1875, 1.75, 8.6875, 1.63281, 6.8125, 3.96875, 6.25,
                2.71875, 7.375, 8.5, 3.26562, 1.55469, 9.125, 6, 4.96875, 6.125, 1.1875,
                5.15625, 9.625, 9.5, 6.875, 4.09375, 5.625, 8.9375, 7.125, 4.875, 5.375,
                9.9375, 9.3125, 8.1875, 5.625, 9.5, 1.64844, 4.40625, 6.09375, 4.625, 10},
            std::vector<T>{
                1, 9.1875, 5.5625, 6.625, 2.65625, 8.125, 8.25, 4.875, 9.4375, 5.875,
                1.26562, 4.71875, 8.0625, 1.76562, 2.46875, 8, 4.875, 5.375, 6.15625, 1.45312,
                2.95312, 5.84375, 8.5, 1.375, 3.6875, 8.25, 4.25, 9.9375, 8.8125, 6.75,
                1.10938, 9.5, 7.0625, 7.625, 5.3125, 8.375, 9.75, 2.03125, 2.90625, 7.625,
                2.90625, 8.375, 9.5, 1.66406, 3.9375, 2.875, 9.875, 8.25, 4.9375, 9.25,
                9.6875, 2.34375, 3.3125, 9.625, 7.8125, 4.34375, 2.71875, 7.4375, 3.53125, 1.9375,
                7.9375, 8.875, 1.07031, 6.25, 4.1875, 7.125, 3.09375, 9, 4.4375, 2.60938,
                1.98438, 2.76562, 4.65625, 5.125, 4.625, 8.375, 7.9375, 6.6875, 8.5625, 6.8125,
                4.53125, 5.09375, 5.1875, 6.0625, 1.9375, 7, 2.71875, 6.8125, 7.0625, 1.42188,
                5.25, 3.34375, 8.1875, 1.07812, 3.21875, 8.6875, 7.6875, 1.27344, 7.0625, 10},
            std::vector<T>{
                1, 6.59375, 2.125, 2.89062, 5.71875, 5.0625, 9.75, 6, 4.9375, 3.21875,
                5.875, 1.85156, 6.28125, 9.875, 1.59375, 3.40625, 7.75, 9.25, 6.46875, 8.75,
                6.5, 4.0625, 9.125, 4.4375, 9.25, 5.5625, 2.625, 1.8125, 7.625, 1.875,
                1.26562, 6.3125, 3.03125, 7.3125, 3.92188, 2.21875, 4.90625, 2.10938, 3.28125, 6.875,
                4.375, 7.25, 3.92188, 3.09375, 8.375, 7.96875, 7.9375, 1.875, 3.125, 4.9375,
                1.32031, 3.67188, 4, 7.4375, 8.875, 6.75, 3.28125, 1.71875, 8.25, 2.39062,
                5.84375, 6.84375, 9.0625, 1.71875, 6.25, 7, 7.28125, 3.8125, 7.53125, 6.0625,
                1.48438, 2.64062, 5.6875, 2.34375, 6.4375, 9.375, 9.625, 7.25, 3.70312, 7.5,
                3.65625, 8.375, 10, 6.5, 5.65625, 5.9375, 7.0625, 5.5625, 5.375, 9,
                4.625, 7.3125, 4.8125, 1.71875, 1.51562, 2.3125, 8.875, 8.625, 9, 10},
            std::vector<int64_t>{5, 5, 5, 5, 5},
            std::vector<T>{
                1, 7.5, 1.57812, 10, 1.55469, 3.5, 9.0625, 7.5625, 9, 1.30469,
                6, 7.1875, 2.20312, 9, 5.6875, 2, 5.5625, 5.5625, 6.21875, 3.09375,
                3.82812, 7.28125, 9, 1.625, 6.78125, 9.875, 5.46875, 4.1875, 3.84375, 2.21875,
                1.57812, 9.125, 6, 6, 9.5, 2.28125, 8.375, 9.875, 2.32812, 5.96875,
                1.42969, 4.5625, 1.10156, 5.4375, 8.25, 1.625, 6.875, 4.6875, 6.15625, 6.15625,
                8.625, 4.125, 6.1875, 1.375, 6.1875, 8.875, 6, 3.40625, 1.75781, 4.8125,
                1.40625, 4.21875, 4, 6.4375, 6.5625, 4.1875, 8.25, 7.40625, 2.40625, 7,
                5.5625, 7.9375, 7.625, 4.75, 7, 1.625, 9.25, 7, 7, 2.5,
                5.9375, 7.1875, 6.875, 5.84375, 7.5625, 6.5, 2.09375, 6.0625, 2.4375, 3.34375,
                3.76562, 2.1875, 3.51562, 5.875, 2.54688, 8.3125, 9.125, 2.96875, 1.9375, 3.76562,
                4.1875, 6.28125, 4.84375, 7.5, 3.78125, 5.375, 2.84375, 8.5, 3.3125, 8.1875,
                4.21875, 6.375, 8.375, 1.46875, 6.625, 6.5625, 9.0625, 2.90625, 9.4375, 5.375,
                1.32812, 5.9375, 8.9375, 4.4375, 7.5625, 8.9375, 5.28125, 2.59375, 8.375, 5.15625,
                6.8125, 9.25, 5.3125, 4.375, 1.46875, 7.5625, 8.3125, 2.82812, 7.25, 7.03125,
                3.3125, 3.03125, 8.5, 3.21875, 8.875, 7, 9.6875, 5.28125, 3.65625, 1.8125,
                2.5, 4.0625, 6.15625, 8, 5.1875, 2.45312, 6.1875, 1.375, 7.9375, 7.1875,
                9.625, 7.5, 7.84375, 7.40625, 9.75, 7.09375, 2.0625, 1.5, 6.53125, 5,
                9.875, 6.6875, 5.875, 6.8125, 5.53125, 9.25, 4.0625, 4.1875, 1.46875, 5,
                8.6875, 3.70312, 1.85938, 1.21094, 2.71875, 1.82812, 9, 2.71875, 9.9375, 7.8125,
                4.3125, 8.8125, 2.8125, 3.23438, 4.375, 4.1875, 8.375, 8.75, 6.625, 5.34375,
                7.1875, 3.45312, 7.0625, 8.5, 5.8125, 4.875, 8.875, 3.5625, 3.84375, 6.1875,
                3.90625, 4.9375, 4.5625, 6.625, 2.59375, 9.875, 2.90625, 1.82031, 1.25781, 6,
                9.25, 4.09375, 9.125, 2.5625, 5.25, 2.34375, 5.90625, 7.90625, 9.75, 6.875,
                7.3125, 1.14062, 9.125, 7.75, 5.5625, 2.8125, 1.32812, 7.5625, 9, 7.125,
                8.625, 4.4375, 8.375, 6.375, 6.6875, 3.82812, 5.1875, 7.8125, 3.28125, 1.17188,
                7.125, 8.9375, 6.6875, 5.4375, 6.21875, 4.125, 1.40625, 3.51562, 3.5, 3.0625,
                1.03906, 7.28125, 2.64062, 8.125, 4.4375, 5.25, 1.75, 1.96875, 7.9375, 4.46875,
                6.25, 8.75, 1.80469, 3.375, 2.4375, 8.9375, 4.1875, 4.1875, 7.15625, 6.65625,
                8.375, 1.13281, 7.3125, 7.375, 3.3125, 1.36719, 7.40625, 6.375, 7.875, 4,
                8.5, 8.5, 6.1875, 2.3125, 6.5625, 6.25, 7.03125, 7.625, 5.1875, 6.71875,
                4.125, 7.4375, 1.09375, 5, 4.90625, 5.5625, 10, 8.0625, 2.625, 1.21875,
                7.46875, 3.125, 4.0625, 8.3125, 1.11719, 5.40625, 4.96875, 1.35156, 2.04688, 2.5,
                3.8125, 6.125, 3.78125, 2.82812, 8.9375, 6.90625, 3.6875, 1.95312, 9.8125, 7.25,
                4.125, 4.875, 1.54688, 6.40625, 2.57812, 6.46875, 5.34375, 7.8125, 9.9375, 5.5625,
                2.625, 7.1875, 8.375, 9.75, 1.67188, 3, 6.96875, 6, 1.53125, 2.09375,
                6.15625, 5.3125, 7.25, 7.3125, 3.625, 1.10938, 7.53125, 1.375, 7.84375, 7.96875,
                9.875, 9.8125, 5.78125, 7.9375, 4.3125, 8.0625, 7.8125, 6.5625, 2.21875, 4.0625,
                3.1875, 4.75, 8.25, 6.125, 3.03125, 6.1875, 4.125, 7.71875, 4.4375, 1.46875,
                4.40625, 6.375, 4.375, 1.48438, 2.65625, 1.80469, 7.5, 9.875, 7.25, 7.875,
                8.1875, 5.4375, 2.70312, 3.39062, 2.23438, 2.5625, 9.875, 1.76562, 3.3125, 3.10938,
                2.78125, 6.9375, 7.3125, 5.59375, 1.03125, 5.9375, 3.15625, 6.1875, 3.15625, 7.5,
                4.625, 5.65625, 2.89062, 2.71875, 8.75, 1.85938, 3.8125, 2.71875, 4.03125, 6.25,
                5.0625, 8, 7.4375, 5.625, 3, 1.1875, 6.75, 4.4375, 5.6875, 7.625,
                5.125, 9.625, 5.4375, 3.28125, 1.78125, 2.03125, 6.65625, 6.875, 3.85938, 7.0625,
                8.1875, 2.15625, 6.8125, 3.84375, 5.15625, 8.875, 7.09375, 8.375, 8.875, 8.75,
                2.79688, 3.1875, 2.0625, 9.75, 9.4375, 4.3125, 5.5625, 8.8125, 5.09375, 4,
                5.375, 7.375, 4.75, 8.875, 6.5, 1.25, 5.9375, 8.125, 7.4375, 5.625,
                7.34375, 9.75, 8.625, 7.125, 3.5625, 1.67969, 6.75, 3.35938, 2.375, 5.9375,
                8.375, 1.95312, 4.8125, 4.0625, 4.5625, 1.21875, 3.10938, 1.96094, 4.625, 3.4375,
                1.97656, 2.32812, 6.28125, 5.0625, 4.5625, 1.375, 2.53125, 1.23438, 8.125, 9.75,
                1.59375, 4.6875, 1.89062, 2.625, 7.3125, 5.5, 2.15625, 6.84375, 4.0625, 9.875,
                8.9375, 5.4375, 9.9375, 3.96875, 9.75, 7.5, 6.59375, 2.96875, 6.625, 1.05469,
                8.125, 7.6875, 5, 3.92188, 8.3125, 6.65625, 9.4375, 8.125, 1.53125, 3.75,
                4.875, 9.375, 9, 5.53125, 3.65625, 2.29688, 6.125, 4.21875, 9.375, 1.39062,
                6.125, 8, 1.34375, 6.90625, 6.84375, 6.4375, 1.26562, 4.375, 5.875, 5.0625,
                6.375, 2.73438, 9.4375, 7.90625, 6.15625, 6.3125, 1.46875, 6.125, 2.32812, 2.53125,
                4.71875, 7.40625, 2.57812, 1.9375, 8.75, 5.0625, 2.25, 1.95312, 2.79688, 3.40625,
                6.53125, 3.98438, 9.75, 4.875, 2.67188, 9.25, 1.5, 6.9375, 9.25, 2.46875,
                6.6875, 2.1875, 2.48438, 6.40625, 2.9375, 1.25, 8.875, 3.8125, 9.5, 2.375,
                2.53125, 8.5, 3.28125, 7.4375, 2.9375, 6.53125, 2.0625, 3.45312, 3.5, 1.60938,
                8, 5.25, 6.5, 8.875, 2.89062, 4.5625, 9.4375, 9.625, 9.375, 3.4375,
                5, 7.46875, 3.71875, 2.875, 1.09375, 6.0625, 5.4375, 4.4375, 8.375, 9.25,
                4.9375, 9.625, 4.5, 4.28125, 9.0625, 1.92188, 3.5625, 6.15625, 7.84375, 7.09375,
                9.25, 3.04688, 8.875, 7.5, 3.90625, 6.40625, 8.875, 8.9375, 7.125, 9.5,
                7.375, 8.875, 6, 1.03125, 9.875, 1.27344, 5.9375, 3.8125, 7.96875, 5.6875,
                7.1875, 9.625, 2.07812, 2.5, 4.1875, 6.375, 7.125, 6.4375, 5.09375, 9.375,
                5.5625, 6.40625, 7.5625, 3.3125, 2.09375, 1.07812, 8.25, 9.0625, 5.4375, 8.375,
                5.0625, 2.5, 7.1875, 1.04688, 3.40625, 7, 9.125, 1.21875, 2.46875, 2.53125,
                7.5625, 6.125, 3.04688, 9.375, 4.75, 10, 6.15625, 2.0625, 5.4375, 8.5,
                6.375, 7.875, 5.5, 3.375, 9.75, 7.9375, 2.03125, 2.0625, 6.03125, 5.5625,
                1.625, 1.13281, 1.22656, 7.09375, 8.5625, 1.77344, 8.625, 7.6875, 7.5, 7.34375,
                7.65625, 3.20312, 5.9375, 6.1875, 1.01562, 5.65625, 5.78125, 2.21875, 9, 2.04688,
                5.71875, 5.125, 4.96875, 2.53125, 1.33594, 4.1875, 9.75, 3.0625, 9, 7.1875,
                3.46875, 3.75, 3.09375, 2.85938, 9.125, 1.35938, 7.84375, 3.82812, 8.1875, 4.75,
                2.65625, 1.40625, 8.5, 5.375, 6.75, 1.69531, 5.875, 6.75, 4.59375, 2.625,
                4.125, 5.375, 4.125, 9.375, 6.25, 7.3125, 8.375, 1.55469, 8.3125, 5.125,
                9.4375, 5.03125, 1.3125, 8.625, 3.15625, 1.30469, 9, 7.90625, 7.875, 7.8125,
                1.45312, 5.4375, 2.84375, 1.23438, 4.96875, 7.59375, 4.65625, 2.04688, 8.25, 4.21875,
                9.4375, 5.9375, 2.64062, 5.75, 3.15625, 9.5, 1.42188, 8.4375, 5.3125, 5,
                9.25, 7.875, 8.875, 4, 6.875, 8.625, 2.875, 2.54688, 8.75, 10},
            std::vector<T>{
                1, 6.125, 4.53125, 4.4375, 6.3125, 6.6875, 5.375, 8.25, 9.875, 9.375,
                9.0625, 5.9375, 3.0625, 6.75, 9.75, 8.8125, 3.5, 2.40625, 7.625, 2.21875,
                2.39062, 9.25, 6.1875, 4.0625, 6.25, 5.375, 8.25, 7.3125, 6.5, 7.8125,
                2.65625, 2.10938, 8.375, 7.65625, 4.09375, 2.71875, 9.4375, 7.0625, 6.5625, 8.625,
                7.625, 8.25, 2.10938, 8.625, 8.125, 7.40625, 4.5, 2.76562, 5.125, 9.625,
                8.5625, 8.75, 7.46875, 9.625, 8.875, 4.40625, 2.26562, 9.3125, 1.67188, 9.1875,
                6.1875, 4.78125, 5.5, 6.40625, 5.59375, 8.125, 7.65625, 1.75, 2.96875, 3.875,
                8.75, 1.99219, 3.53125, 2.53125, 9.375, 2.1875, 2.875, 1.86719, 9.25, 7.125,
                3.89062, 9.5, 1.92188, 5.4375, 4.5625, 1, 8, 3.96875, 9.375, 9.1875,
                1.19531, 6.875, 3.21875, 8.5625, 7.0625, 8.6875, 9.875, 4.65625, 9.5, 6.25,
                1.82812, 1.51562, 2.42188, 4.21875, 2.78125, 7.75, 4.96875, 3.89062, 2.76562, 9.25,
                5.84375, 5.84375, 8.375, 3.34375, 9.875, 2.625, 2.15625, 1.73438, 6.25, 8.875,
                4.0625, 7.8125, 4.625, 7.59375, 7.625, 2.71875, 4.25, 3.96875, 2, 2.9375,
                6.25, 7.3125, 7.09375, 7.1875, 7.40625, 1.13281, 9.625, 5.90625, 9.9375, 1.0625,
                2.10938, 9.5, 6.25, 5.1875, 3.3125, 9.4375, 6.84375, 8.25, 4.375, 4.59375,
                1.79688, 9, 7.9375, 2.64062, 9.75, 8.75, 9.6875, 3.59375, 4, 10,
                8.875, 2.82812, 9.0625, 8.8125, 1.76562, 2.23438, 6.625, 2.375, 3.70312, 8.375,
                8.625, 4, 6.1875, 9.8125, 2.625, 2.60938, 9.375, 1.625, 2.03125, 9.5,
                1.84375, 9.125, 7.75, 4.75, 2.0625, 2.375, 6.03125, 5.65625, 3.92188, 4.125,
                9.625, 4.25, 5.4375, 1.34375, 5.1875, 7.53125, 3.96875, 4.875, 1.65625, 3.54688,
                1.64062, 4.625, 1.51562, 4.4375, 4.375, 5.1875, 1.03906, 1.58594, 9.75, 2.78125,
                1.27344, 6.25, 6.4375, 9.5625, 4.625, 3.89062, 8.875, 5.875, 9.875, 4,
                9.625, 8.1875, 6, 6.5625, 3.1875, 5.9375, 1.17188, 6.375, 3.09375, 3.5,
                5.34375, 6.625, 9.6875, 1.85938, 2.85938, 9.75, 5.5, 7.6875, 1.86719, 1.03125,
                7.21875, 7.125, 7.75, 6.1875, 2, 2.59375, 2.46875, 8.25, 7.78125, 3.75,
                7.875, 4.6875, 8.0625, 7.9375, 2.53125, 6.5, 2.48438, 6.75, 2.60938, 1.60938,
                7.75, 2.28125, 8.75, 7.6875, 8.25, 8.0625, 1.23438, 6.0625, 1.53125, 6.96875,
                4.9375, 5.21875, 1.5, 5.125, 7.5, 7.1875, 6, 2.71875, 6.5625, 9.875,
                2.15625, 4.40625, 5.5, 3.98438, 5.5625, 8.875, 1.84375, 8.75, 8.875, 9.6875,
                8.8125, 2.78125, 1.16406, 6.03125, 7.625, 2.73438, 6.5, 5.6875, 9.75, 8.125,
                6.875, 7.34375, 9, 5.0625, 6.1875, 3.95312, 4.5, 2.35938, 4.6875, 9.875,
                1.09375, 6.09375, 4, 9.75, 8.5, 1.70312, 6.9375, 1.29688, 5.6875, 7.8125,
                9.625, 3.125, 7.1875, 6.6875, 7.84375, 1.21094, 1.71094, 5.875, 3.4375, 5.8125,
                4.25, 4.125, 7.875, 5.1875, 9.3125, 9, 4.4375, 1.0625, 3.01562, 5.25,
                3.09375, 7.375, 6.1875, 3.4375, 6.46875, 9.375, 5.625, 4.8125, 8.1875, 2.03125,
                8.75, 4.21875, 9.25, 8.3125, 1.1875, 3.0625, 1.6875, 3.375, 5.09375, 1.95312,
                1.25, 3.76562, 7.75, 5.09375, 6.1875, 4.625, 4.625, 1.49219, 4.96875, 9.875,
                2, 9, 4.46875, 7.25, 1.36719, 7.9375, 1.79688, 3, 7.5625, 2.625,
                7.5625, 1.92969, 2.32812, 3.25, 4.375, 1.125, 7.75, 2.4375, 3.34375, 5.6875,
                1.30469, 7.9375, 8.75, 3.625, 6.3125, 6.75, 1.73438, 2.5, 3.09375, 1.28906,
                1.75, 8.4375, 4.875, 3.5, 7.78125, 6.8125, 1.42188, 5.125, 7.6875, 3.90625,
                4.125, 9.8125, 2.4375, 4.625, 9.9375, 9.75, 7.625, 5.875, 9.5, 9.125,
                9.5, 1.03125, 1.15625, 8.5625, 5.34375, 2.75, 5.6875, 6.03125, 2.6875, 3.15625,
                9.75, 2.04688, 8.875, 4.6875, 4.40625, 5, 3.78125, 1.17188, 7, 3.92188,
                3.48438, 9.0625, 4.0625, 3.34375, 3.5625, 6.6875, 5.5625, 8.875, 5.25, 2.34375,
                1.48438, 1.14844, 7.21875, 7.25, 5.375, 7.1875, 3.34375, 1.5625, 8.1875, 1.95312,
                7.21875, 1.48438, 8.625, 9.25, 1.14062, 4.65625, 5.15625, 9.75, 2.28125, 1.71875,
                6.8125, 9.25, 6.125, 2.84375, 7.21875, 4.09375, 9, 4.8125, 6.15625, 7.625,
                6.625, 6, 7.5, 2.46875, 6.125, 1.33594, 8.25, 5.125, 3.53125, 6.875,
                4.4375, 3.10938, 7.5625, 9.25, 7.1875, 1.15625, 7.875, 3.21875, 7.40625, 6.46875,
                4.875, 5.3125, 7.09375, 9.375, 8.75, 3.75, 9.3125, 2.25, 9.5, 3.375,
                7.3125, 6.875, 8.5625, 6.5, 5.09375, 6.4375, 6.625, 4.28125, 6.75, 2.65625,
                8.25, 7, 4.1875, 5.4375, 8.125, 2.71875, 4.3125, 8, 2.65625, 6.375,
                3.67188, 3.6875, 8.125, 8.125, 4.09375, 2.25, 9.6875, 1.04688, 5.375, 1.96875,
                3.03125, 2.42188, 7.75, 5.125, 8.25, 6.9375, 6.8125, 8.375, 8.5, 1.04688,
                1.39062, 4.21875, 2.03125, 5.40625, 8.25, 9, 6.5625, 4.25, 8.5, 4.9375,
                7.875, 5.875, 5.75, 9.4375, 6.84375, 4.0625, 9, 3.375, 4.84375, 4.375,
                9.75, 6.5, 7.125, 9.375, 4.5, 1.40625, 2.71875, 8.75, 5.59375, 8.875,
                8.5625, 4.375, 9.625, 5.5625, 7.0625, 9, 5.9375, 4.1875, 2.1875, 8.75,
                2.90625, 6.5, 5.96875, 5.15625, 4.4375, 7.25, 5.0625, 1.65625, 4.1875, 5.46875,
                5.03125, 2.59375, 3.78125, 7.84375, 3.125, 2.46875, 4.03125, 4.5625, 6.0625, 9.25,
                2.6875, 6.75, 3.14062, 3.71875, 9.875, 5.9375, 1.96875, 4.375, 6.6875, 7.6875,
                7.1875, 6.375, 6.8125, 1.76562, 8.125, 8.625, 7.96875, 2.45312, 2.20312, 1.65625,
                6.625, 7.8125, 4.65625, 5.25, 4.875, 3.71875, 8.8125, 7.3125, 1.54688, 4.3125,
                5.34375, 1.8125, 1.03125, 7.84375, 2.09375, 1.8125, 5.6875, 4.3125, 3.8125, 8.25,
                1.48438, 2.75, 8.75, 6.4375, 7, 1.67188, 9.125, 7.875, 9.625, 7.875,
                1.46875, 10, 2.15625, 7.28125, 5.03125, 4.34375, 7.4375, 8.1875, 5.21875, 2.625,
                9.625, 10, 7.6875, 7.75, 8.75, 2.25, 4.71875, 6.375, 5.84375, 8.8125,
                1.54688, 6.4375, 7.125, 7.25, 8.875, 5.25, 8.25, 10, 1.03125, 1.45312,
                6.5625, 4.5625, 4.125, 4.125, 7.53125, 8.75, 1.53125, 7.4375, 6.125, 2.71875,
                4.9375, 6.25, 6.5, 2.95312, 9.375, 9.625, 4.5, 5.40625, 1.16406, 2.71875,
                1.08594, 8.875, 5.3125, 2.34375, 7.09375, 8.875, 4.71875, 3.95312, 3.79688, 5.375,
                2.59375, 9.0625, 6.125, 8.25, 3.5625, 5.375, 9.5, 7.5625, 8.75, 3.75,
                1.125, 3.21875, 5.5625, 3.59375, 4.625, 8.5, 7.71875, 3.15625, 8, 8.25,
                8.6875, 4.78125, 2.625, 4.28125, 5.6875, 1.625, 6.375, 2.4375, 8.625, 5.75,
                9, 2.39062, 1.76562, 1.5, 8.6875, 3.96875, 6.75, 2.40625, 6.4375, 8.8125,
                3.375, 2.15625, 8.75, 3.625, 4.9375, 7.9375, 9.625, 8.125, 2.96875, 9.625,
                4.4375, 9.4375, 2.6875, 3.625, 3.59375, 9.25, 4.1875, 2.6875, 5.9375, 7.875,
                7.1875, 8, 7.90625, 4.28125, 6.0625, 8.875, 3.90625, 1.69531, 3.5, 3.34375,
                7.75, 2.46875, 6.875, 4.1875, 8.5625, 5.1875, 5.34375, 6.15625, 9.375, 10},
            std::vector<T>{
                1, 7.25, 1.08594, 1.90625, 7.3125, 8.1875, 1.84375, 9.8125, 3.65625, 7.6875,
                7.03125, 6.75, 9.875, 9, 2.28125, 4.625, 8.375, 8.125, 4.25, 9.9375,
                2.125, 1.92969, 1.35938, 7.625, 5.6875, 2.65625, 6.9375, 8.0625, 2.53125, 7.375,
                2.8125, 9.1875, 7.3125, 8.75, 7.15625, 7.15625, 3.8125, 6.75, 3.53125, 8.75,
                9.875, 5.75, 7.1875, 5.65625, 7.375, 9.25, 6.03125, 9.75, 8.625, 1.07031,
                6.8125, 2.90625, 6.1875, 4.6875, 7.25, 1.14062, 9.0625, 2.78125, 5.625, 6.875,
                2.0625, 3.3125, 9.875, 8.9375, 5.6875, 5.875, 3.65625, 2.78125, 8.625, 4.1875,
                7.125, 7.09375, 5.125, 2.625, 3.10938, 9.4375, 5.5625, 5.8125, 3.5625, 10},
            std::vector<T>{
                0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094,
                0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094,
                0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094,
                0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094,
                0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094,
                0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094, 0.0996094},
            std::vector<T>{
                0.466797, 0.667969, 0.617188, 0.648438, 0.667969, 0.667969, 0.667969, 0.667969, 0.667969, 0.65625,
                0.433594, 0.664062, 0.570312, 0.617188, 0.664062, 0.65625, 0.667969, 0.664062, 0.65625, 0.625,
                0.410156, 0.65625, 0.523438, 0.570312, 0.648438, 0.632812, 0.664062, 0.648438, 0.632812, 0.585938,
                0.390625, 0.625, 0.480469, 0.523438, 0.617188, 0.601562, 0.65625, 0.617188, 0.601562, 0.539062,
                0.380859, 0.585938, 0.443359, 0.480469, 0.570312, 0.554688, 0.632812, 0.578125, 0.554688, 0.490234,
                0.570312, 0.427734, 0.578125, 0.632812, 0.417969, 0.5, 0.609375, 0.625, 0.585938, 0.617188,
                0.617188, 0.458984, 0.625, 0.65625, 0.443359, 0.546875, 0.640625, 0.648438, 0.625, 0.648438,
                0.648438, 0.5, 0.648438, 0.664062, 0.476562, 0.59375, 0.664062, 0.664062, 0.65625, 0.664062,
                0.664062, 0.546875, 0.664062, 0.667969, 0.523438, 0.632812, 0.667969, 0.667969, 0.664062, 0.667969,
                0.667969, 0.59375, 0.667969, 0.667969, 0.570312, 0.65625, 0.667969, 0.667969, 0.667969, 0.667969,
                0.667969, 0.664062, 0.667969, 0.664062, 0.667969, 0.667969, 0.640625, 0.585938, 0.667969, 0.59375,
                0.664062, 0.648438, 0.667969, 0.648438, 0.667969, 0.664062, 0.601562, 0.539062, 0.667969, 0.546875,
                0.65625, 0.617188, 0.664062, 0.625, 0.664062, 0.648438, 0.554688, 0.496094, 0.65625, 0.5,
                0.625, 0.570312, 0.648438, 0.578125, 0.648438, 0.609375, 0.507812, 0.458984, 0.640625, 0.464844,
                0.585938, 0.523438, 0.625, 0.53125, 0.625, 0.5625, 0.466797, 0.427734, 0.601562, 0.433594,
                0.396484, 0.578125, 0.484375, 0.601562, 0.515625, 0.449219, 0.546875, 0.443359, 0.492188, 0.59375,
                0.417969, 0.625, 0.53125, 0.632812, 0.5625, 0.484375, 0.59375, 0.480469, 0.539062, 0.632812,
                0.443359, 0.648438, 0.578125, 0.65625, 0.609375, 0.53125, 0.632812, 0.523438, 0.585938, 0.65625,
                0.476562, 0.664062, 0.625, 0.667969, 0.648438, 0.578125, 0.65625, 0.570312, 0.632812, 0.664062,
                0.523438, 0.667969, 0.648438, 0.667969, 0.664062, 0.625, 0.667969, 0.617188, 0.65625, 0.667969,
                0.664062, 0.667969, 0.664062, 0.648438, 0.667969, 0.667969, 0.667969, 0.59375, 0.648438, 0.667969,
                0.648438, 0.667969, 0.648438, 0.625, 0.667969, 0.667969, 0.667969, 0.546875, 0.625, 0.65625,
                0.625, 0.65625, 0.609375, 0.578125, 0.664062, 0.664062, 0.664062, 0.5, 0.578125, 0.632812,
                0.578125, 0.632812, 0.5625, 0.53125, 0.648438, 0.648438, 0.648438, 0.464844, 0.53125, 0.601562,
                0.53125, 0.601562, 0.515625, 0.484375, 0.617188, 0.609375, 0.609375, 0.433594, 0.484375, 0.554688,
                0.398438, 0.507812, 0.523438, 0.601562, 0.625, 0.585938, 0.492188, 0.421875, 0.617188, 0.458984,
                0.417969, 0.554688, 0.570312, 0.640625, 0.648438, 0.632812, 0.539062, 0.449219, 0.648438, 0.496094,
                0.445312, 0.601562, 0.617188, 0.65625, 0.664062, 0.65625, 0.585938, 0.484375, 0.664062, 0.539062,
                0.484375, 0.640625, 0.648438, 0.667969, 0.667969, 0.664062, 0.632812, 0.53125, 0.667969, 0.585938,
                0.523438, 0.65625, 0.664062, 0.667969, 0.667969, 0.667969, 0.65625, 0.578125, 0.667969, 0.632812,
                0.667969, 0.667969, 0.667969, 0.578125, 0.667969, 0.667969, 0.667969, 0.664062, 0.667969, 0.667969,
                0.664062, 0.664062, 0.667969, 0.53125, 0.664062, 0.664062, 0.667969, 0.640625, 0.667969, 0.664062,
                0.648438, 0.65625, 0.664062, 0.484375, 0.648438, 0.65625, 0.65625, 0.609375, 0.65625, 0.648438,
                0.617188, 0.632812, 0.648438, 0.449219, 0.625, 0.632812, 0.632812, 0.554688, 0.640625, 0.625,
                0.570312, 0.59375, 0.625, 0.421875, 0.578125, 0.59375, 0.601562, 0.507812, 0.601562, 0.578125,
                0.410156, 0.46875, 0.570312, 0.453125, 0.585938, 0.625, 0.632812, 0.601562, 0.507812, 0.601562,
                0.433594, 0.507812, 0.617188, 0.490234, 0.625, 0.65625, 0.65625, 0.632812, 0.554688, 0.640625,
                0.464844, 0.554688, 0.648438, 0.539062, 0.65625, 0.664062, 0.664062, 0.65625, 0.601562, 0.65625,
                0.507812, 0.601562, 0.664062, 0.585938, 0.664062, 0.667969, 0.667969, 0.667969, 0.640625, 0.667969,
                0.554688, 0.640625, 0.667969, 0.625, 0.667969, 0.667969, 0.667969, 0.667969, 0.65625, 0.667969,
                0.65625, 0.667969, 0.667969, 0.667969, 0.667969, 0.667969, 0.667969, 0.667969, 0.667969, 0.667969,
                0.640625, 0.667969, 0.667969, 0.664062, 0.664062, 0.664062, 0.664062, 0.664062, 0.664062, 0.667969,
                0.601562, 0.664062, 0.664062, 0.65625, 0.648438, 0.648438, 0.65625, 0.648438, 0.640625, 0.664062,
                0.554688, 0.648438, 0.65625, 0.625, 0.617188, 0.617188, 0.632812, 0.609375, 0.609375, 0.648438,
                0.507812, 0.617188, 0.632812, 0.585938, 0.570312, 0.578125, 0.59375, 0.5625, 0.554688, 0.625,
                0.539062, 0.601562, 0.546875, 0.421875, 0.412109, 0.453125, 0.625, 0.617188, 0.625, 0.632812,
                0.585938, 0.632812, 0.59375, 0.449219, 0.4375, 0.490234, 0.648438, 0.648438, 0.648438, 0.65625,
                0.632812, 0.65625, 0.632812, 0.484375, 0.46875, 0.539062, 0.664062, 0.664062, 0.664062, 0.664062,
                0.65625, 0.667969, 0.65625, 0.53125, 0.507812, 0.585938, 0.667969, 0.667969, 0.667969, 0.667969,
                0.664062, 0.667969, 0.664062, 0.578125, 0.554688, 0.625, 0.667969, 0.667969, 0.667969, 0.667969},
            std::vector<T>{
                0.380859, 0.585938, 0.443359, 0.480469, 0.570312, 0.554688, 0.632812, 0.578125, 0.554688, 0.490234,
                0.570312, 0.427734, 0.578125, 0.632812, 0.417969, 0.5, 0.609375, 0.625, 0.585938, 0.617188,
                0.585938, 0.523438, 0.625, 0.53125, 0.625, 0.5625, 0.466797, 0.427734, 0.601562, 0.433594,
                0.396484, 0.578125, 0.484375, 0.601562, 0.515625, 0.449219, 0.546875, 0.443359, 0.492188, 0.59375,
                0.53125, 0.601562, 0.515625, 0.484375, 0.617188, 0.609375, 0.609375, 0.433594, 0.484375, 0.554688,
                0.398438, 0.507812, 0.523438, 0.601562, 0.625, 0.585938, 0.492188, 0.421875, 0.617188, 0.458984,
                0.570312, 0.59375, 0.625, 0.421875, 0.578125, 0.59375, 0.601562, 0.507812, 0.601562, 0.578125,
                0.410156, 0.46875, 0.570312, 0.453125, 0.585938, 0.625, 0.632812, 0.601562, 0.507812, 0.601562,
                0.507812, 0.617188, 0.632812, 0.585938, 0.570312, 0.578125, 0.59375, 0.5625, 0.554688, 0.625,
                0.539062, 0.601562, 0.546875, 0.421875, 0.412109, 0.453125, 0.625, 0.617188, 0.625, 0.632812},
            std::vector<T>{
                0.648438, 1.375, 0.800781, 0.902344, 1.28125, 1.17969, 1.78125, 1.3125, 1.17969, 0.945312,
                1.29688, 0.761719, 1.34375, 1.8125, 0.730469, 0.972656, 1.53125, 1.71875, 1.375, 1.65625,
                1.375, 1.05469, 1.71875, 1.09375, 1.71875, 1.25, 0.867188, 0.761719, 1.51562, 0.769531,
                0.6875, 1.34375, 0.925781, 1.46875, 1.03125, 0.816406, 1.17188, 0.800781, 0.949219, 1.42188,
                1.09375, 1.46875, 1.03125, 0.929688, 1.625, 1.57812, 1.57812, 0.769531, 0.929688, 1.17969,
                0.691406, 1, 1.05469, 1.5, 1.6875, 1.40625, 0.949219, 0.746094, 1.625, 0.839844,
                1.29688, 1.42188, 1.71875, 0.746094, 1.34375, 1.4375, 1.46875, 1.01562, 1.5, 1.32812,
                0.714844, 0.878906, 1.28125, 0.832031, 1.375, 1.75, 1.78125, 1.46875, 1.01562, 1.5,
                1, 1.625, 1.82812, 1.375, 1.26562, 1.3125, 1.4375, 1.25, 1.21875, 1.6875,
                1.125, 1.46875, 1.15625, 0.746094, 0.722656, 0.832031, 1.6875, 1.65625, 1.6875, 1.82812}),
    };
    return params;
}

std::vector<LSTMSequenceParams> generateCombinedParams() {
    const std::vector<std::vector<LSTMSequenceParams>> generatedParams {
        generateParams<element::Type_t::f64>(),
        generateParams<element::Type_t::f32>(),
        generateParamsBF16<element::Type_t::f16>(),
        generateParamsBF16<element::Type_t::bf16>(),
    };
    std::vector<LSTMSequenceParams> combinedParams;

    for (const auto& params : generatedParams) {
        combinedParams.insert(combinedParams.end(), params.begin(), params.end());
    }
    return combinedParams;
}

std::vector<LSTMSequenceV1Params> generateV1CombinedParams() {
    const std::vector<std::vector<LSTMSequenceV1Params>> generatedParams {
        generateV1Params<element::Type_t::f64>(),
        generateV1Params<element::Type_t::f32>(),
        generateV1Params<element::Type_t::f16>(),
        generateV1ParamsBF16<element::Type_t::bf16>(),
    };
    std::vector<LSTMSequenceV1Params> combinedParams;

    for (const auto& params : generatedParams) {
        combinedParams.insert(combinedParams.end(), params.begin(), params.end());
    }
    return combinedParams;
}

INSTANTIATE_TEST_SUITE_P(smoke_LSTMSequence_With_Hardcoded_Refs, ReferenceLSTMSequenceTest,
    testing::ValuesIn(generateCombinedParams()), ReferenceLSTMSequenceTest::getTestCaseName);

INSTANTIATE_TEST_SUITE_P(smoke_LSTMSequence_With_Hardcoded_Refs, ReferenceLSTMSequenceV1Test,
    testing::ValuesIn(generateV1CombinedParams()), ReferenceLSTMSequenceV1Test::getTestCaseName);

} // namespace