"""
Copyright (c) 2018-2022 Intel Corporation

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
"""

from ..representation import ClassificationAnnotation
from ..utils import get_path, read_csv, check_file_existence
from ..config import StringField, PathField
from .format_converter import FileBasedAnnotationConverter
from .format_converter import ConverterReturn

class MalwareClassificationDatasetConverter(FileBasedAnnotationConverter):
    __provider__ = 'malware_classification_converter'

    @classmethod
    def parameters(cls):
        parameters = super().parameters()
        parameters.update({
            'known_folder': StringField(optional=True, default='KNOWN_1000',
                                        description="Known files folder"),
            'malicious_folder': StringField(optional=True, default='MALICIOUS_1000',
                                            description="Malicious files folder"),
            'data_dir': PathField(optional=False, default=None, is_directory=True, description="Dataset root folder"),
        })

        return parameters

    def configure(self):
        self.annotation_file = self.get_value_from_config('annotation_file')
        self.data_folder = self.get_value_from_config('data_dir')
        self.known_folder = self.get_value_from_config('known_folder')
        self.malicious_folder = self.get_value_from_config('malicious_folder')

        self.data_dir = get_path(self.data_folder, is_directory=True)
        self.known_dir = get_path(self.data_folder / self.known_folder, is_directory=True)
        self.malicious_dir = get_path(self.data_folder / self.malicious_folder, is_directory=True)

    def convert(self, check_content=False, progress_callback=None, progress_interval=100, **kwargs):
        annotations = []
        content_errors = []
        meta = None

        annotation = read_csv(self.annotation_file, is_dict=False)

        for entry_id, (identifier, kind) in enumerate(annotation):
            identifier = identifier.replace(':', '_')
            kind = int(kind)
            sample_path = self.malicious_dir / identifier if kind == 1 else self.known_dir / identifier
            if check_content:
                if not check_file_existence(sample_path):
                    content_errors.append('{}: does not exist'.format(sample_path))
            annotations.append(ClassificationAnnotation(str(sample_path.relative_to(self.data_dir)), [kind, ]))
            if progress_callback is not None and entry_id % progress_interval == 0:
                progress_callback(entry_id * 100 / len(annotation))

        return ConverterReturn(annotations, meta, content_errors)
