// *****************************************************************************
//
// Copyright (c) 2015, Southwest Research Institute® (SwRI®)
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright
//       notice, this list of conditions and the following disclaimer in the
//       documentation and/or other materials provided with the distribution.
//     * Neither the name of Southwest Research Institute® (SwRI®) nor the
//       names of its contributors may be used to endorse or promote products
//       derived from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
// DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
// ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// *****************************************************************************

#ifndef TILE_MAP_TILE_MAP_PLUGIN_H_
#define TILE_MAP_TILE_MAP_PLUGIN_H_

// C++ standard libraries
#include <string>
#include <map>

// Boost libraries
#include <boost/filesystem.hpp>
#include <boost/algorithm/string/trim.hpp>

#include <mapviz/mapviz_plugin.h>

// QT libraries
#include <QGLWidget>
#include <QObject>
#include <QWidget>

#include <mapviz/map_canvas.h>
#include <tile_map/tile_map_view.h>

// QT autogenerated files
#include "ui_tile_map_config.h"

namespace tile_map
{
  class TileSource;

  class TileMapPlugin : public mapviz::MapvizPlugin
  {
    Q_OBJECT

  public:
    TileMapPlugin();
    ~TileMapPlugin() override = default;

    bool Initialize(QGLWidget* canvas) override;
    void Shutdown() override {}

    void Draw(double x, double y, double scale) override;

    void Transform() override;

    void LoadConfig(const YAML::Node& node, const std::string& path) override;
    void SaveConfig(YAML::Emitter& emitter, const std::string& path) override;

    QWidget* GetConfigWidget(QWidget* parent) override;

    void SetNode(rclcpp::Node& node) override;

  protected Q_SLOTS:
    void PrintError(const std::string& message) override;
    void PrintInfo(const std::string& message) override;
    void PrintWarning(const std::string& message) override;

    void DeleteTileSource();
    void SelectSource(const QString& source_name);
    void SaveCustomSource();
    void ResetTileCache();

  private:
    void selectTileSource(const std::shared_ptr<TileSource>& tile_source);
    void startCustomEditing();
    void stopCustomEditing();

    Ui::tile_map_config ui_;
    QWidget* config_widget_;

    swri_transform_util::Transform transform_;
    swri_transform_util::Transform inverse_transform_;

    bool transformed_;
    
    TileMapView tile_map_;
    std::map<QString, std::shared_ptr<TileSource> > tile_sources_;

    double last_center_x_;
    double last_center_y_;
    double last_scale_;
    int32_t last_height_;
    int32_t last_width_;

    static std::string BASE_URL_KEY;
    static std::string BING_API_KEY;
    static std::string CUSTOM_SOURCES_KEY;
    static std::string MAX_ZOOM_KEY;
    static std::string NAME_KEY;
    static std::string SOURCE_KEY;
    static std::string TYPE_KEY;
    static QString BING_NAME;
    static QString STAMEN_TERRAIN_NAME;
    static QString STAMEN_TONER_NAME;
    static QString STAMEN_WATERCOLOR_NAME;
  };
}

#endif  // TILE_MAP_TILE_MAP_PLUGIN_H_
