/* ========================= eCAL LICENSE =================================
 *
 * Copyright (C) 2016 - 2020 Continental Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * ========================= eCAL LICENSE =================================
*/

#include "upload.h"

#include <rec_server_core/proto_helpers.h>

#include <ecal_utils/string.h>

namespace eCAL
{
  namespace rec_cli
  {
    namespace command
    {

      std::string Upload::Usage() const
      {
        return "[MeasID]";
      }

      std::string Upload::Help() const
      {
        return "Uploads a measurement. If no ID is given, all measurements that can be uploaded will be uploaded.";
      }

      std::string Upload::Example() const
      {
        return "1234";
      }

      eCAL::rec::Error Upload::Execute(const std::shared_ptr<eCAL::rec_server::RecServer>& rec_server_instance, const std::vector<std::string>& argv) const
      {
        // Arg check
        if (argv.size() > 1)
          return eCAL::rec::Error(eCAL::rec::Error::ErrorCode::TOO_MANY_PARAMETERS, EcalUtils::String::Join(" ", std::vector<std::string>(std::next(argv.begin()), argv.end())));

        // (Try to) parse measurement ID
        int64_t meas_id = 0;
        if (argv.size() == 1)
        {
          try
          {
            meas_id = std::stoll(argv[0]);
          }
          catch (const std::exception& e)
          {
            return eCAL::rec::Error(eCAL::rec::Error::ErrorCode::PARAMETER_ERROR, "Unable to parse ID " + argv[0] + ": " + std::string(e.what()));
          }
        }

        if (meas_id != 0)
        {
          return rec_server_instance->UploadMeasurement(meas_id);
        }
        else
        {
          rec_server_instance->UploadNonUploadedMeasurements();
          return eCAL::rec::Error::ErrorCode::OK;
        }
      }

      eCAL::rec::Error Upload::Execute(const std::string& hostname, const std::shared_ptr<eCAL::protobuf::CServiceClient<eCAL::pb::rec_server::EcalRecServerService>>& remote_rec_server_service, const std::vector<std::string>& argv) const
      {
        // Arg check
        if (argv.size() > 1)
          return eCAL::rec::Error(eCAL::rec::Error::ErrorCode::TOO_MANY_PARAMETERS, EcalUtils::String::Join(" ", std::vector<std::string>(std::next(argv.begin()), argv.end())));

        // (Try to) parse measurement ID
        int64_t meas_id = 0;
        if (argv.size() == 1)
        {
          try
          {
            meas_id = std::stoll(argv[0]);
          }
          catch (const std::exception& e)
          {
            return eCAL::rec::Error(eCAL::rec::Error::ErrorCode::PARAMETER_ERROR, "Unable to parse ID " + argv[0] + ": " + std::string(e.what()));
          }
        }
        
        // Check if measurement ID exists
        if (meas_id != 0)
        {
          eCAL::rec_server::RecServerStatus status;
          auto error = GetRemoteStatus(hostname, remote_rec_server_service, status);
          if (error)
            return error;

          bool meas_id_found(false);
          for (const auto& job_history_entry : status.job_history_)
          {
            if (job_history_entry.local_evaluated_job_config_.GetJobId() == meas_id)
            {
              meas_id_found = true;
              break;
            }
          }

          if (!meas_id_found)
            return eCAL::rec::Error(eCAL::rec::Error::ErrorCode::MEAS_ID_NOT_FOUND, std::to_string(meas_id));
        }

        // Service call
        eCAL::pb::rec_server::GenericMeasurementRequest request_pb;
        eCAL::pb::rec_server::ServiceResult             response_pb;

        request_pb.set_meas_id(meas_id);
        eCAL::rec::Error service_call_error = CallRemoteEcalrecService(remote_rec_server_service, hostname, "UploadMeasurement", request_pb, response_pb);

        // Service call failed
        if (service_call_error)
          return service_call_error;

        // Service call reported error
        {
          eCAL::rec::Error error = eCAL::rec_server::proto_helpers::FromProtobuf(response_pb);
          if (error)
            return error;
        }

        // Success!
        return eCAL::rec::Error::ErrorCode::OK;
      }
    }
  }
}
