// Copyright (c) 2008, Willow Garage, Inc.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//    * Redistributions of source code must retain the above copyright
//      notice, this list of conditions and the following disclaimer.
//
//    * Redistributions in binary form must reproduce the above copyright
//      notice, this list of conditions and the following disclaimer in the
//      documentation and/or other materials provided with the distribution.
//
//    * Neither the name of the Willow Garage nor the names of its
//      contributors may be used to endorse or promote products derived from
//      this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.

/* Author: Wim Meeussen */

#include <iostream>

#include "gtest/gtest.h"
#include "kdl_parser/kdl_parser.hpp"

const char test_robot[] =
  "<?xml version=\"1.0\"?>"
  "<robot name=\"r2d2\">"
  "  <material name=\"blue\">"
  "    <color rgba=\"0 0 .8 1\"/>"
  "  </material>"
  "  <material name=\"black\">"
  "    <color rgba=\"0 0 0 1\"/>"
  "  </material>"
  "  <material name=\"white\">"
  "    <color rgba=\"1 1 1 1\"/>"
  "  </material>"
  "  <link name=\"dummy_link\"/>"
  "  <link name=\"base_link\">"
  "    <visual>"
  "      <geometry>"
  "        <cylinder length=\"0.6\" radius=\"0.2\"/>"
  "      </geometry>"
  "      <material name=\"blue\"/>"
  "    </visual>"
  "    <collision>"
  "      <geometry>"
  "        <cylinder length=\"0.6\" radius=\"0.2\"/>"
  "      </geometry>"
  "    </collision>"
  "    <inertial>"
  "      <mass value=\"10\"/>"
  "      <inertia ixx=\"1.0\" ixy=\"0.0\" ixz=\"0.0\" iyy=\"1.0\" iyz=\"0.0\" izz=\"1.0\"/>"
  "    </inertial>"
  "  </link>"
  "  <joint name=\"dummy_to_base\" type=\"fixed\">"
  "    <parent link=\"dummy_link\"/>"
  "    <child link=\"base_link\"/>"
  "  </joint>"
  "  <link name=\"right_leg\">"
  "    <visual>"
  "      <geometry>"
  "        <box size=\"0.6 .1 .2\"/>"
  "      </geometry>"
  "      <origin rpy=\"0 1.57075 0\" xyz=\"0 0 -0.3\"/>"
  "      <material name=\"white\"/>"
  "    </visual>"
  "    <collision>"
  "      <geometry>"
  "        <box size=\"0.6 .1 .2\"/>"
  "      </geometry>"
  "      <origin rpy=\"0 1.57075 0\" xyz=\"0 0 -0.3\"/>"
  "    </collision>"
  "    <inertial>"
  "      <mass value=\"10\"/>"
  "      <inertia ixx=\"1.0\" ixy=\"0.0\" ixz=\"0.0\" iyy=\"1.0\" iyz=\"0.0\" izz=\"1.0\"/>"
  "    </inertial>"
  "  </link>"
  "  <joint name=\"base_to_right_leg\" type=\"fixed\">"
  "    <parent link=\"base_link\"/>"
  "    <child link=\"right_leg\"/>"
  "    <origin xyz=\"0 -0.22 .25\"/>"
  "  </joint>"
  "  <link name=\"right_base\">"
  "    <visual>"
  "      <geometry>"
  "        <box size=\"0.4 .1 .1\"/>"
  "      </geometry>"
  "      <material name=\"white\"/>"
  "    </visual>"
  "    <collision>"
  "      <geometry>"
  "        <box size=\"0.4 .1 .1\"/>"
  "      </geometry>"
  "    </collision>"
  "    <inertial>"
  "      <mass value=\"10\"/>"
  "      <inertia ixx=\"1.0\" ixy=\"0.0\" ixz=\"0.0\" iyy=\"1.0\" iyz=\"0.0\" izz=\"1.0\"/>"
  "    </inertial>"
  "  </link>"
  "  <joint name=\"right_base_joint\" type=\"fixed\">"
  "    <parent link=\"right_leg\"/>"
  "    <child link=\"right_base\"/>"
  "    <origin xyz=\"0 0 -0.6\"/>"
  "  </joint>"
  "  <link name=\"right_front_wheel\">"
  "    <visual>"
  "      <origin rpy=\"1.57075 0 0\" xyz=\"0 0 0\"/>"
  "      <geometry>"
  "        <cylinder length=\"0.1\" radius=\"0.035\"/>"
  "      </geometry>"
  "      <material name=\"black\"/>"
  "    </visual>"
  "    <collision>"
  "      <origin rpy=\"1.57075 0 0\" xyz=\"0 0 0\"/>"
  "      <geometry>"
  "        <cylinder length=\"0.1\" radius=\"0.035\"/>"
  "      </geometry>"
  "    </collision>"
  "    <inertial>"
  "      <mass value=\"1\"/>"
  "      <inertia ixx=\"1.0\" ixy=\"0.0\" ixz=\"0.0\" iyy=\"1.0\" iyz=\"0.0\" izz=\"1.0\"/>"
  "    </inertial>"
  "  </link>"
  "  <joint name=\"right_front_wheel_joint\" type=\"continuous\">"
  "    <axis rpy=\"0 0 0\" xyz=\"0 1 0\"/>"
  "    <parent link=\"right_base\"/>"
  "    <child link=\"right_front_wheel\"/>"
  "    <origin rpy=\"0 0 0\" xyz=\"0.133333333333 0 -0.085\"/>"
  "  </joint>"
  "  <link name=\"right_back_wheel\">"
  "    <visual>"
  "      <origin rpy=\"1.57075 0 0\" xyz=\"0 0 0\"/>"
  "      <geometry>"
  "        <cylinder length=\"0.1\" radius=\"0.035\"/>"
  "      </geometry>"
  "      <material name=\"black\"/>"
  "    </visual>"
  "    <collision>"
  "      <origin rpy=\"1.57075 0 0\" xyz=\"0 0 0\"/>"
  "      <geometry>"
  "        <cylinder length=\"0.1\" radius=\"0.035\"/>"
  "      </geometry>"
  "    </collision>"
  "    <inertial>"
  "      <mass value=\"1\"/>"
  "      <inertia ixx=\"1.0\" ixy=\"0.0\" ixz=\"0.0\" iyy=\"1.0\" iyz=\"0.0\" izz=\"1.0\"/>"
  "    </inertial>"
  "  </link>"
  "  <joint name=\"right_back_wheel_joint\" type=\"continuous\">"
  "    <axis rpy=\"0 0 0\" xyz=\"0 1 0\"/>"
  "    <parent link=\"right_base\"/>"
  "    <child link=\"right_back_wheel\"/>"
  "    <origin rpy=\"0 0 0\" xyz=\"-0.133333333333 0 -0.085\"/>"
  "  </joint>"
  "  <link name=\"left_leg\">"
  "    <visual>"
  "      <geometry>"
  "        <box size=\"0.6 .1 .2\"/>"
  "      </geometry>"
  "      <origin rpy=\"0 1.57075 0\" xyz=\"0 0 -0.3\"/>"
  "      <material name=\"white\"/>"
  "    </visual>"
  "    <collision>"
  "      <geometry>"
  "        <box size=\"0.6 .1 .2\"/>"
  "      </geometry>"
  "      <origin rpy=\"0 1.57075 0\" xyz=\"0 0 -0.3\"/>"
  "    </collision>"
  "    <inertial>"
  "      <mass value=\"10\"/>"
  "      <inertia ixx=\"1.0\" ixy=\"0.0\" ixz=\"0.0\" iyy=\"1.0\" iyz=\"0.0\" izz=\"1.0\"/>"
  "    </inertial>"
  "  </link>"
  "  <joint name=\"base_to_left_leg\" type=\"fixed\">"
  "    <parent link=\"base_link\"/>"
  "    <child link=\"left_leg\"/>"
  "    <origin xyz=\"0 0.22 .25\"/>"
  "  </joint>"
  "  <link name=\"left_base\">"
  "    <visual>"
  "      <geometry>"
  "        <box size=\"0.4 .1 .1\"/>"
  "      </geometry>"
  "      <material name=\"white\"/>"
  "    </visual>"
  "    <collision>"
  "      <geometry>"
  "        <box size=\"0.4 .1 .1\"/>"
  "      </geometry>"
  "    </collision>"
  "    <inertial>"
  "      <mass value=\"10\"/>"
  "      <inertia ixx=\"1.0\" ixy=\"0.0\" ixz=\"0.0\" iyy=\"1.0\" iyz=\"0.0\" izz=\"1.0\"/>"
  "    </inertial>"
  "  </link>"
  "  <joint name=\"left_base_joint\" type=\"fixed\">"
  "    <parent link=\"left_leg\"/>"
  "    <child link=\"left_base\"/>"
  "    <origin xyz=\"0 0 -0.6\"/>"
  "  </joint>"
  "  <link name=\"left_front_wheel\">"
  "    <visual>"
  "      <origin rpy=\"1.57075 0 0\" xyz=\"0 0 0\"/>"
  "      <geometry>"
  "        <cylinder length=\"0.1\" radius=\"0.035\"/>"
  "      </geometry>"
  "      <material name=\"black\"/>"
  "    </visual>"
  "    <collision>"
  "      <origin rpy=\"1.57075 0 0\" xyz=\"0 0 0\"/>"
  "      <geometry>"
  "        <cylinder length=\"0.1\" radius=\"0.035\"/>"
  "      </geometry>"
  "    </collision>"
  "    <inertial>"
  "      <mass value=\"1\"/>"
  "      <inertia ixx=\"1.0\" ixy=\"0.0\" ixz=\"0.0\" iyy=\"1.0\" iyz=\"0.0\" izz=\"1.0\"/>"
  "    </inertial>"
  "  </link>"
  "  <joint name=\"left_front_wheel_joint\" type=\"continuous\">"
  "    <axis rpy=\"0 0 0\" xyz=\"0 1 0\"/>"
  "    <parent link=\"left_base\"/>"
  "    <child link=\"left_front_wheel\"/>"
  "    <origin rpy=\"0 0 0\" xyz=\"0.133333333333 0 -0.085\"/>"
  "  </joint>"
  "  <link name=\"left_back_wheel\">"
  "    <visual>"
  "      <origin rpy=\"1.57075 0 0\" xyz=\"0 0 0\"/>"
  "      <geometry>"
  "        <cylinder length=\"0.1\" radius=\"0.035\"/>"
  "      </geometry>"
  "      <material name=\"black\"/>"
  "    </visual>"
  "    <collision>"
  "      <origin rpy=\"1.57075 0 0\" xyz=\"0 0 0\"/>"
  "      <geometry>"
  "        <cylinder length=\"0.1\" radius=\"0.035\"/>"
  "      </geometry>"
  "    </collision>"
  "    <inertial>"
  "      <mass value=\"1\"/>"
  "      <inertia ixx=\"1.0\" ixy=\"0.0\" ixz=\"0.0\" iyy=\"1.0\" iyz=\"0.0\" izz=\"1.0\"/>"
  "    </inertial>"
  "  </link>"
  "  <joint name=\"left_back_wheel_joint\" type=\"continuous\">"
  "    <axis rpy=\"0 0 0\" xyz=\"0 1 0\"/>"
  "    <parent link=\"left_base\"/>"
  "    <child link=\"left_back_wheel\"/>"
  "    <origin rpy=\"0 0 0\" xyz=\"-0.133333333333 0 -0.085\"/>"
  "  </joint>"
  "  <joint name=\"gripper_extension\" type=\"prismatic\">"
  "    <parent link=\"base_link\"/>"
  "    <child link=\"gripper_pole\"/>"
  "    <limit effort=\"1000.0\" lower=\"-0.38\" upper=\"0\" velocity=\"0.5\"/>"
  "    <origin rpy=\"0 0 0\" xyz=\"0.19 0 .2\"/>"
  "  </joint>"
  "  <link name=\"gripper_pole\">"
  "    <visual>"
  "      <geometry>"
  "        <cylinder length=\"0.2\" radius=\".01\"/>"
  "      </geometry>"
  "      <origin rpy=\"0 1.57075 0 \" xyz=\"0.1 0 0\"/>"
  "    </visual>"
  "    <collision>"
  "      <geometry>"
  "        <cylinder length=\"0.2\" radius=\".01\"/>"
  "      </geometry>"
  "      <origin rpy=\"0 1.57075 0 \" xyz=\"0.1 0 0\"/>"
  "    </collision>"
  "    <inertial>"
  "      <mass value=\"0.05\"/>"
  "      <inertia ixx=\"1.0\" ixy=\"0.0\" ixz=\"0.0\" iyy=\"1.0\" iyz=\"0.0\" izz=\"1.0\"/>"
  "    </inertial>"
  "  </link>"
  "  <joint name=\"left_gripper_joint\" type=\"revolute\">"
  "    <axis xyz=\"0 0 1\"/>"
  "    <limit effort=\"1000.0\" lower=\"0.0\" upper=\"0.548\" velocity=\"0.5\"/>"
  "    <origin rpy=\"0 0 0\" xyz=\"0.2 0.01 0\"/>"
  "    <parent link=\"gripper_pole\"/>"
  "    <child link=\"left_gripper\"/>"
  "  </joint>"
  "  <link name=\"left_gripper\">"
  "    <visual>"
  "      <origin rpy=\"0.0 0 0\" xyz=\"0 0 0\"/>"
  "      <geometry>"
  "        <mesh filename=\"package://pr2_description/meshes/gripper_v0/l_finger.dae\"/>"
  "      </geometry>"
  "    </visual>"
  "    <collision>"
  "      <geometry>"
  "        <mesh filename=\"package://pr2_description/meshes/gripper_v0/l_finger.dae\"/>"
  "      </geometry>"
  "      <origin rpy=\"0.0 0 0\" xyz=\"0 0 0\"/>"
  "    </collision>"
  "    <inertial>"
  "      <mass value=\"0.05\"/>"
  "      <inertia ixx=\"1.0\" ixy=\"0.0\" ixz=\"0.0\" iyy=\"1.0\" iyz=\"0.0\" izz=\"1.0\"/>"
  "    </inertial>"
  "  </link>"
  "  <joint name=\"left_tip_joint\" type=\"fixed\">"
  "    <parent link=\"left_gripper\"/>"
  "    <child link=\"left_tip\"/>"
  "  </joint>"
  "  <link name=\"left_tip\">"
  "    <visual>"
  "      <origin rpy=\"0.0 0 0\" xyz=\"0.09137 0.00495 0\"/>"
  "      <geometry>"
  "        <mesh filename=\"package://pr2_description/meshes/gripper_v0/l_finger_tip.dae\"/>"
  "      </geometry>"
  "    </visual>"
  "    <collision>"
  "      <geometry>"
  "        <mesh filename=\"package://pr2_description/meshes/gripper_v0/l_finger_tip.dae\"/>"
  "      </geometry>"
  "      <origin rpy=\"0.0 0 0\" xyz=\"0.09137 0.00495 0\"/>"
  "    </collision>"
  "    <inertial>"
  "      <mass value=\"0.05\"/>"
  "      <inertia ixx=\"1.0\" ixy=\"0.0\" ixz=\"0.0\" iyy=\"1.0\" iyz=\"0.0\" izz=\"1.0\"/>"
  "    </inertial>"
  "  </link>"
  "  <joint name=\"right_gripper_joint\" type=\"revolute\">"
  "    <axis xyz=\"0 0 -1\"/>"
  "    <limit effort=\"1000.0\" lower=\"0.0\" upper=\"0.548\" velocity=\"0.5\"/>"
  "    <origin rpy=\"0 0 0\" xyz=\"0.2 -0.01 0\"/>"
  "    <parent link=\"gripper_pole\"/>"
  "    <child link=\"right_gripper\"/>"
  "  </joint>"
  "  <link name=\"right_gripper\">"
  "    <visual>"
  "      <origin rpy=\"-3.1415 0 0\" xyz=\"0 0 0\"/>"
  "      <geometry>"
  "        <mesh filename=\"package://pr2_description/meshes/gripper_v0/l_finger.dae\"/>"
  "      </geometry>"
  "    </visual>"
  "    <collision>"
  "      <geometry>"
  "        <mesh filename=\"package://pr2_description/meshes/gripper_v0/l_finger.dae\"/>"
  "      </geometry>"
  "      <origin rpy=\"-3.1415 0 0\" xyz=\"0 0 0\"/>"
  "    </collision>"
  "    <inertial>"
  "      <mass value=\"0.05\"/>"
  "      <inertia ixx=\"1.0\" ixy=\"0.0\" ixz=\"0.0\" iyy=\"1.0\" iyz=\"0.0\" izz=\"1.0\"/>"
  "    </inertial>"
  "  </link>"
  "  <joint name=\"right_tip_joint\" type=\"fixed\">"
  "    <parent link=\"right_gripper\"/>"
  "    <child link=\"right_tip\"/>"
  "  </joint>"
  "  <link name=\"right_tip\">"
  "    <visual>"
  "      <origin rpy=\"-3.1415 0 0\" xyz=\"0.09137 0.00495 0\"/>"
  "      <geometry>"
  "        <mesh filename=\"package://pr2_description/meshes/gripper_v0/l_finger_tip.dae\"/>"
  "      </geometry>"
  "    </visual>"
  "    <collision>"
  "      <geometry>"
  "        <mesh filename=\"package://pr2_description/meshes/gripper_v0/l_finger_tip.dae\"/>"
  "      </geometry>"
  "      <origin rpy=\"-3.1415 0 0\" xyz=\"0.09137 0.00495 0\"/>"
  "    </collision>"
  "    <inertial>"
  "      <mass value=\"0.05\"/>"
  "      <inertia ixx=\"1.0\" ixy=\"0.0\" ixz=\"0.0\" iyy=\"1.0\" iyz=\"0.0\" izz=\"1.0\"/>"
  "    </inertial>"
  "  </link>"
  "  <link name=\"head\">"
  "    <visual>"
  "      <geometry>"
  "        <sphere radius=\"0.2\"/>"
  "      </geometry>"
  "      <material name=\"white\"/>"
  "    </visual>"
  "    <collision>"
  "      <geometry>"
  "        <sphere radius=\"0.2\"/>"
  "      </geometry>"
  "    </collision>"
  "    <inertial>"
  "      <mass value=\"2\"/>"
  "      <inertia ixx=\"1.0\" ixy=\"0.0\" ixz=\"0.0\" iyy=\"1.0\" iyz=\"0.0\" izz=\"1.0\"/>"
  "    </inertial>"
  "  </link>"
  "  <joint name=\"head_swivel\" type=\"continuous\">"
  "    <parent link=\"base_link\"/>"
  "    <child link=\"head\"/>"
  "    <axis xyz=\"0 0 1\"/>"
  "    <origin xyz=\"0 0 0.3\"/>"
  "  </joint>"
  "  <link name=\"box\">"
  "    <visual>"
  "      <geometry>"
  "        <box size=\".08 .08 .08\"/>"
  "      </geometry>"
  "      <material name=\"blue\"/>"
  "      <origin xyz=\"-0.04 0 0\"/>"
  "    </visual>"
  "    <collision>"
  "      <geometry>"
  "        <box size=\".08 .08 .08\"/>"
  "      </geometry>"
  "    </collision>"
  "    <inertial>"
  "      <mass value=\"1\"/>"
  "      <inertia ixx=\"1.0\" ixy=\"0.0\" ixz=\"0.0\" iyy=\"1.0\" iyz=\"0.0\" izz=\"1.0\"/>"
  "    </inertial>"
  "  </link>"
  "  <joint name=\"tobox\" type=\"fixed\">"
  "    <parent link=\"head\"/>"
  "    <child link=\"box\"/>"
  "    <origin xyz=\"0.1814 0 0.1414\"/>"
  "  </joint>"
  "</robot>";


TEST(TestParser, test)
{
  KDL::Tree my_tree;

  ASSERT_TRUE(kdl_parser::treeFromString(test_robot, my_tree));
  ASSERT_EQ(8, my_tree.getNrOfJoints());
  ASSERT_EQ(16, my_tree.getNrOfSegments());
  ASSERT_TRUE(my_tree.getSegment("dummy_link") == my_tree.getRootSegment());
  ASSERT_EQ((unsigned int)1, my_tree.getRootSegment()->second.children.size());
  ASSERT_TRUE(my_tree.getSegment("base_link")->second.parent == my_tree.getRootSegment());
  ASSERT_EQ(10.0, my_tree.getSegment("base_link")->second.segment.getInertia().getMass());
  ASSERT_NEAR(
    1.000, my_tree.getSegment(
      "base_link")->second.segment.getInertia().getRotationalInertia().data[0], 0.001);
}

int main(int argc, char ** argv)
{
  testing::InitGoogleTest(&argc, argv);
  return RUN_ALL_TESTS();
}
