// Copyright 2023, Foxglove Technologies.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#ifndef ROSBAG2_STORAGE__MESSAGE_DEFINITION_HPP_
#define ROSBAG2_STORAGE__MESSAGE_DEFINITION_HPP_

#include <string>
#include <cassert>

#include "rosbag2_storage/visibility_control.hpp"

namespace rosbag2_storage
{

struct MessageDefinition
{
  /// @brief  The name of the type.
  ///
  /// Should match the `type` in TopicMetadata for all topics using this message definition.
  std::string topic_type;
  /// @brief The encoding technique used in `encoded_message_definition`.
  ///
  /// See docs/message_definition_encoding.md for details of each encoding.
  std::string encoding;
  /// @brief The full encoded message definition for this type.
  std::string encoded_message_definition;
  /// @brief  REP-2011 type description hash if available for topic, "" otherwise.
  std::string type_hash;

  /// @brief used when no message definition is available for a given topic type.
  static MessageDefinition empty_message_definition_for(const std::string & topic_type)
  {
    MessageDefinition self;
    self.topic_type = topic_type;
    self.encoding = "";
    self.encoded_message_definition = "";
    self.type_hash = "";
    return self;
  }

  bool operator==(const MessageDefinition & rhs) const
  {
    return topic_type == rhs.topic_type && encoding == rhs.encoding &&
           encoded_message_definition == rhs.encoded_message_definition &&
           type_hash == rhs.type_hash;
  }
};

}  // namespace rosbag2_storage

#endif  // ROSBAG2_STORAGE__MESSAGE_DEFINITION_HPP_
