// Copyright 2015 Open Source Robotics Foundation, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#ifndef RCL__TIME_H_
#define RCL__TIME_H_

#if __cplusplus
extern "C"
{
#endif

#include "rcl/macros.h"
#include "rcl/types.h"
#include "rcl/visibility_control.h"
#include "rcutils/time.h"

/// Convenience macro to convert seconds to nanoseconds.
#define RCL_S_TO_NS RCUTILS_S_TO_NS
/// Convenience macro to convert milliseconds to nanoseconds.
#define RCL_MS_TO_NS RCUTILS_MS_TO_NS
/// Convenience macro to convert microseconds to nanoseconds.
#define RCL_US_TO_NS RCUTILS_US_TO_NS

/// Convenience macro to convert nanoseconds to seconds.
#define RCL_NS_TO_S RCUTILS_NS_TO_S
/// Convenience macro to convert nanoseconds to milliseconds.
#define RCL_NS_TO_MS RCUTILS_NS_TO_MS
/// Convenience macro to convert nanoseconds to microseconds.
#define RCL_NS_TO_US RCUTILS_NS_TO_US

/// A single point in time, measured in nanoseconds since the Unix epoch.
typedef rcutils_time_point_value_t rcl_time_point_value_t;
/// A duration of time, measured in nanoseconds.
typedef rcutils_duration_value_t rcl_duration_value_t;

/// Time source type, used to indicate the source of a time measurement.
enum rcl_time_source_type_t
{
  RCL_TIME_SOURCE_UNINITIALIZED = 0,
  RCL_ROS_TIME,
  RCL_SYSTEM_TIME,
  RCL_STEADY_TIME
};

/// Encapsulation of a time source.
typedef struct rcl_time_source_t
{
  enum rcl_time_source_type_t type;
  void (* pre_update)(void);
  void (* post_update)(void);
  rcl_ret_t (* get_now)(void * data, rcl_time_point_value_t * now);
  // void (*set_now) (rcl_time_point_value_t);
  void * data;
} rcl_time_source_t;

struct rcl_ros_time_source_storage_t;

/// A single point in time, measured in nanoseconds, the reference point is based on the source.
typedef struct rcl_time_point_t
{
  rcl_time_point_value_t nanoseconds;
  rcl_time_source_t * time_source;
} rcl_time_point_t;

/// A duration of time, measured in nanoseconds and its source.
typedef struct rcl_duration_t
{
  rcl_duration_value_t nanoseconds;
  rcl_time_source_t * time_source;
} rcl_duration_t;

// typedef struct rcl_rate_t
// {
//   rcl_time_point_value_t trigger_time;
//   int64_t period;
//   rcl_time_source_t * time_source;
// } rcl_rate_t;
// TODO(tfoote) integrate rate and timer implementations

/// Check if the time_source has valid values.
/**
 * This function returns true if the time source appears to be valid.
 * It will check that the type is not uninitialized, and that pointers
 * are not invalid.
 * Note that if data is uninitialized it may give a false positive.
 *
 * \param[in] time_source the handle to the time_source which is being queried
 * \return true if the source is believed to be valid, otherwise return false.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
bool
rcl_time_source_valid(rcl_time_source_t * time_source);

/// Initialize a time_source based on the passed type.
/**
 * This will allocate all necessary internal structures, and initialize variables.
 *
 * \param[in] time_source_type the type identifying the time source to provide
 * \param[in] time_source the handle to the time_source which is being initialized
 * \return `RCL_RET_OK` if the time source was successfully initialized, or
 * \return `RCL_RET_INVALID_ARGUMENT` if any arguments are invalid, or
 * \return `RCL_RET_ERROR` an unspecified error occur.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_ret_t
rcl_time_source_init(
  enum rcl_time_source_type_t time_source_type, rcl_time_source_t * time_source
);

/// Finalize a time_source.
/**
 * This will deallocate all necessary internal structures, and clean up any variables.
 * It can be combined with any of the init functions.
 *
 * Passing a time_source with type RCL_TIME_SOURCE_UNINITIALIZED will result in
 * RCL_RET_INVALID_ARGUMENT being returned.
 *
 * \param[in] time_source the handle to the time_source which is being finalized
 * \return `RCL_RET_OK` if the time source was successfully finalized, or
 * \return `RCL_RET_INVALID_ARGUMENT` if any arguments are invalid, or
 * \return `RCL_RET_ERROR` an unspecified error occur.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_ret_t
rcl_time_source_fini(rcl_time_source_t * time_source);

/// Initialize a time_source as a RCL_ROS_TIME time source.
/**
 * This will allocate all necessary internal structures, and initialize variables.
 * It is specifically setting up a RCL_ROS_TIME time source.
 *
 * \param[in] time_source the handle to the time_source which is being initialized
 * \return `RCL_RET_OK` if the time source was successfully initialized, or
 * \return `RCL_RET_INVALID_ARGUMENT` if any arguments are invalid, or
 * \return `RCL_RET_ERROR` an unspecified error occur.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_ret_t
rcl_ros_time_source_init(rcl_time_source_t * time_source);

/// Finalize a time_source as a `RCL_ROS_TIME` time source.
/**
 * This will deallocate all necessary internal structures, and clean up any variables.
 * It is specifically setting up a `RCL_ROS_TIME` time source. It is expected
 * to be paired with the init fuction.
 *
 * \param[in] time_source the handle to the time_source which is being initialized
 * \return `RCL_RET_OK` if the time source was successfully finalized, or
 * \return `RCL_RET_INVALID_ARGUMENT` if any arguments are invalid, or
 * \return `RCL_RET_ERROR` an unspecified error occur.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_ret_t
rcl_ros_time_source_fini(rcl_time_source_t * time_source);

/// Initialize a time_source as a `RCL_STEADY_TIME` time source.
/**
 * This will allocate all necessary internal structures, and initialize variables.
 * It is specifically setting up a `RCL_STEADY_TIME` time source.
 *
 * \param[in] time_source the handle to the time_source which is being initialized
 * \return `RCL_RET_OK` if the time source was successfully initialized, or
 * \return `RCL_RET_INVALID_ARGUMENT` if any arguments are invalid, or
 * \return `RCL_RET_ERROR` an unspecified error occur.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_ret_t
rcl_steady_time_source_init(rcl_time_source_t * time_source);

/// Finalize a time_source as a `RCL_STEADY_TIME` time source.
/**
 * Finalize the time_source as a `RCL_STEADY_TIME` time source.
 *
 * This will deallocate all necessary internal structures, and clean up any variables.
 * It is specifically setting up a steady time source. It is expected to be
 * paired with the init fuction.
 *
 * \param[in] time_source the handle to the time_source which is being initialized
 * \return `RCL_RET_OK` if the time source was successfully finalized, or
 * \return `RCL_RET_INVALID_ARGUMENT` if any arguments are invalid, or
 * \return `RCL_RET_ERROR` an unspecified error occur.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_ret_t
rcl_steady_time_source_fini(rcl_time_source_t * time_source);

/// Initialize a time_source as a `RCL_SYSTEM_TIME` time source.
/**
 * Initialize the time_source as a `RCL_SYSTEM_TIME` time source.
 *
 * This will allocate all necessary internal structures, and initialize variables.
 * It is specifically setting up a system time source.
 *
 * \param[in] time_source the handle to the time_source which is being initialized
 * \return `RCL_RET_OK` if the time source was successfully initialized, or
 * \return `RCL_RET_INVALID_ARGUMENT` if any arguments are invalid, or
 * \return `RCL_RET_ERROR` an unspecified error occur.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_ret_t
rcl_system_time_source_init(rcl_time_source_t * time_source);

/// Finalize a time_source as a `RCL_SYSTEM_TIME` time source.
/**
 * Finalize the time_source as a `RCL_SYSTEM_TIME` time source.
 *
 * This will deallocate all necessary internal structures, and clean up any variables.
 * It is specifically setting up a system time source. It is expected to be paired with
 * the init fuction.
 *
 * \param[in] time_source the handle to the time_source which is being initialized.
 * \return `RCL_RET_OK` if the time source was successfully finalized, or
 * \return `RCL_RET_INVALID_ARGUMENT` if any arguments are invalid, or
 * \return `RCL_RET_ERROR` an unspecified error occur.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_ret_t
rcl_system_time_source_fini(rcl_time_source_t * time_source);

/// Initialize a time point using the time_source.
/**
 * This function will initialize the time_point using the time_source
 * as a reference.
 * If the time_source is null it will use the system default time_source.
 *
 * This will allocate all necessary internal structures, and initialize variables.
 * The time_source may be of types `RCL_ROS_TIME`, `RCL_STEADY_TIME`, or
 * `RCL_SYSTEM_TIME`.
 *
 * \param[in] time_point the handle to the time_source which is being initialized.
 * \param[in] time_source the handle to the time_source will be used for reference.
 * \return `RCL_RET_OK` if the last call time was retrieved successfully, or
 * \return `RCL_RET_INVALID_ARGUMENT` if any arguments are invalid, or
 * \return `RCL_RET_ERROR` an unspecified error occur.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_ret_t
rcl_time_point_init(rcl_time_point_t * time_point, rcl_time_source_t * time_source);

/// Finalize a time_point
/**
 * Finalize the time_point such that it is ready for deallocation.
 *
 * This will deallocate all necessary internal structures, and clean up any variables.
 *
 * \param[in] time_point the handle to the time_source which is being finalized.
 * \return `RCL_RET_OK` if the last call time was retrieved successfully, or
 * \return `RCL_RET_INVALID_ARGUMENT` if any arguments are invalid, or
 * \return `RCL_RET_ERROR` an unspecified error occur.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_ret_t
rcl_time_point_fini(rcl_time_point_t * time_point);

/// Initialize a duration using the time_source.
/**
 * This function will initialize the duration using the time_source as a reference.
 * If the time_source is null it will use the system default time_source.
 *
 * This will allocate all necessary internal structures, and initialize variables.
 * The time_source may be of types ros, steady, or system.
 *
 * \param[in] duration the handle to the duration which is being initialized.
 * \param[in] time_source the handle to the time_source will be used for reference.
 * \return `RCL_RET_OK` if the last call time was retrieved successfully, or
 * \return `RCL_RET_INVALID_ARGUMENT` if any arguments are invalid, or
 * \return `RCL_RET_ERROR` an unspecified error occur.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_ret_t
rcl_duration_init(rcl_duration_t * duration, rcl_time_source_t * time_source);

/// Finalize a duration
/**
 * Finalize the duration such that it is ready for deallocation.
 *
 * This will deallocate all necessary internal structures, and clean up any variables.
 *
 * \param[in] duration the handle to the duration which is being finalized.
 * \return `RCL_RET_OK` if the last call time was retrieved successfully, or
 * \return `RCL_RET_INVALID_ARGUMENT` if any arguments are invalid, or
 * \return `RCL_RET_ERROR` an unspecified error occur.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_ret_t
rcl_duration_fini(rcl_duration_t * duration);

/// Get the default `RCL_ROS_TIME` time source
/**
 * This function will get the process default time source.
 * This time source is specifically of the ROS time abstraction,
 * and may be overridden by updates.
 *
 * If the default has not yet been used it will allocate
 * and initialize the time source.
 *
 * \return rcl_time_source_t if it successfully found or allocated a
 *         time source. If an error occurred it will return NULL.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_time_source_t *
rcl_get_default_ros_time_source(void);

/// Get the default `RCL_STEADY_TIME` time source
/**
 * This function will get the process default time source.
 * This time source is specifically of the steady time abstraction,
 * it should not be able to be overridden..
 *
 * If the default has not yet been used it will allocate
 * and initialize the time source.
 *
 * \return rcl_time_source_t if it successfully found or allocated a
 *         time source. If an error occurred it will return NULL.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_time_source_t *
rcl_get_default_steady_time_source(void);

/// Get the default `RCL_SYSTEM_TIME` time source
/**
 * This function will get the process default time source.
 * This time source is specifically of the system time abstraction,
 * and may be overridden by updates to the system clock.
 *
 * If the default has not yet been used it will allocate
 * and initialize the time source.
 *
 * \return rcl_time_source_t if it successfully found or allocated a
 *         time source. If an error occurred it will return NULL.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_time_source_t *
rcl_get_default_system_time_source(void);

/// Set the current time on the `RCL_ROS_TIME` time source
/**
 * This function is used to set the time on a ROS time source.
 * It will error if passed a different time source.
 *
 * This should not block, except on Windows. One caveat is that
 * if the ROS time abstraction is active, it will invoke the user
 * defined callbacks, for pre and post update notifications. The
 * callbacks are supposed to be short running and non-blocking.
 *
 * \param[in] process_time_source The time source on which to set the value.
 * \return `RCL_RET_OK` if the value was set successfully, or
 * \return `RCL_RET_INVALID_ARGUMENT` if any arguments are invalid, or
 * \return `RCL_RET_ERROR` an unspecified error occur.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_ret_t
rcl_set_default_ros_time_source(rcl_time_source_t * process_time_source);

/// Compute the difference between two time points
/**
 * This function takes two time points and computes the duration between them.
 * The two time points must be using the same time abstraction, and the
 * resultant duration will also be of the same abstraction.
 *
 * The value will be computed as duration = finish - start. If start is after
 * finish the duration will be negative.
 *
 * \param[in] start The time point for the start of the duration.
 * \param[in] finish The time point for the end of the duration.
 * \param[out] delta The duration between the start and finish.
 * \return `RCL_RET_OK` if the difference was computed successfully, or
 * \return `RCL_RET_INVALID_ARGUMENT` if any arguments are invalid, or
 * \return `RCL_RET_ERROR` an unspecified error occur.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_ret_t
rcl_difference_times(rcl_time_point_t * start, rcl_time_point_t * finish,
  rcl_duration_t * delta);

/// Fill the time point with the current value of the associated clock.
/**
 * This function will populate the data of the time_point object with the
 * current value from it's associated time abstraction.
 *
 * \param[out] time_point The time_point to populate.
 * \return `RCL_RET_OK` if the last call time was retrieved successfully, or
 * \return `RCL_RET_INVALID_ARGUMENT` if any arguments are invalid, or
 * \return `RCL_RET_ERROR` an unspecified error occur.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_ret_t
rcl_time_point_get_now(rcl_time_point_t * time_point);


/// Enable the ROS time abstraction override.
/**
 * This method will enable the ROS time abstraction override values,
 * such that the time source will report the set value instead of falling
 * back to system time.
 *
 * \param[in] time_source The time_source to enable.
 * \return `RCL_RET_OK` if the time source was enabled successfully, or
 * \return `RCL_RET_INVALID_ARGUMENT` if any arguments are invalid, or
 * \return `RCL_RET_ERROR` an unspecified error occur.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_ret_t
rcl_enable_ros_time_override(rcl_time_source_t * time_source);

/// Disable the ROS time abstraction override.
/**
 * This method will disable the `RCL_ROS_TIME` time abstraction override values,
 * such that the time source will report the system time even if a custom
 * value has been set.
 *
 * \param[in] time_source The time_source to disable.
 * \return `RCL_RET_OK` if the time source was disabled successfully, or
 * \return `RCL_RET_INVALID_ARGUMENT` if any arguments are invalid, or
 * \return `RCL_RET_ERROR` an unspecified error occur.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_ret_t
rcl_disable_ros_time_override(rcl_time_source_t * time_source);


/// Check if the `RCL_ROS_TIME` time source has the override enabled.
/**
 * This will populate the is_enabled object to indicate if the
 * time overide is enabled. If it is enabled, the set value will be returned.
 * Otherwise this time source will return the equivalent to system time abstraction.
 *
 * \param[in] time_source The time_source to query.
 * \param[out] is_enabled Whether the override is enabled..
 * \return `RCL_RET_OK` if the time source was queried successfully, or
 * \return `RCL_RET_INVALID_ARGUMENT` if any arguments are invalid, or
 * \return `RCL_RET_ERROR` an unspecified error occur.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_ret_t
rcl_is_enabled_ros_time_override(rcl_time_source_t * time_source,
  bool * is_enabled);

/// Set the current time for this `RCL_ROS_TIME` time source.
/**
 * This function will update the internal storage for the `RCL_ROS_TIME`
 * time source.
 * If queried and override enabled the time source will return this value,
 * otherwise it will return the system time.
 *
 * \param[in] time_source The time_source to update.
 * \param[in] time_value The new current time.
 * \return `RCL_RET_OK` if the time source was set successfully, or
 * \return `RCL_RET_INVALID_ARGUMENT` if any arguments are invalid, or
 * \return `RCL_RET_ERROR` an unspecified error occur.
 */
RCL_PUBLIC
RCL_WARN_UNUSED
rcl_ret_t
rcl_set_ros_time_override(rcl_time_source_t * time_source,
  rcl_time_point_value_t time_value);

#if __cplusplus
}
#endif

#endif  // RCL__TIME_H_
