# Copyright 2017 Open Source Robotics Foundation, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import rclpy
from std_msgs.msg import String


class Talker(rclpy.Node):
    """
    A node with a single publisher.

    This class creates a node which regularly publishes messages on a topic. Creating a node by
    inheriting from rclpy.Node is recommended because it allows it to be imported and used by
    other scripts.
    """

    def __init__(self):
        # Calls rclpy.Node.__init__('talker')
        super().__init__('talker')
        self.i = 0
        self.pub = self.create_publisher(String, 'chatter')
        # Create a timer that calls a callback every second. A timer is recommended for executing
        # periodic tasks because it does not block the main thread while it's waiting. This allows
        # an executor to do other work when mutliple nodes are run in the same process.
        self.timer = self.create_timer(1.0, self.timer_callback)

    def timer_callback(self):
        msg = String()
        msg.data = 'Hello World: {0}'.format(self.i)
        self.i += 1
        print('Publishing: "{0}"'.format(msg.data))
        self.pub.publish(msg)


def main(args=None):
    """
    Run a Talker node standalone.

    This function is called directly when using an entrypoint. Entrypoints are configured in
    setup.py. This along with the script installation in setup.cfg allows a talker node to be run
    with the command `ros2 run examples_rclpy_executors talker`.

    :param args: Arguments passed in from the command line.
    """
    # Run standalone
    rclpy.init(args=args)
    try:
        rclpy.spin(Talker())
    finally:
        rclpy.shutdown()


if __name__ == '__main__':
    # Runs a talker node when this script is run directly (not through an entrypoint)
    main()
